// -*- C++ -*-

// Deal with keeping files up to date and transferring them.
// Also copes with ordinary FS reads of files.  The complexity
// is in allowing a file to be read as it's being updated without
// getting problems.

// (C) Copyright 1994 Jeremy Fitzhardinge
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#pragma implementation

#include "ftpfile.h"
#include "ftpconn.h"
#include "ftpdir.h"
#include "ftpfs.h"

#include <LWP.h>
#include <io.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <utime.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>

#include <iostream.h>

ftpfile::ftpfile(ftpfs &fs, const String &name, ftpdir *dir, ftpconn *conn)
	: dir(dir), ino(fs), conn(conn), name(name)
{
	fd = -1;
	updating = 0;
	wlist = NULL;
	mode = S_IFREG | 0444;
	conn->incref();
	upd_thread = NULL;

	open();
	if (fd != -1)
	{
		struct stat st;

		if (fstat(fd, &st) == 0)
		{
			size = st.st_size;
			mtime = st.st_mtime;
		}
		close();
	}
}

ftpfile::~ftpfile()
{
	if (upd_thread != NULL)
		delete upd_thread;
	close();
	conn->decref();
}

struct waiting
{
	off_t off;
	Semaphore sem;
	int err;
	waiting *next, *prev;

	waiting(off_t, waiting *next);
};

waiting *
ftpfile::addwlist(off_t off)
{
	waiting *w = new waiting(off, wlist);

	wlist = w;

	return w;
}

void
ftpfile::delwlist(waiting *w)
{
	if (w->next)
		w->next->prev = w->prev;
	if (w->prev)
		w->prev->next = w->next;
	else
		wlist = w->next;
}

waiting::waiting(off_t off, waiting *n)
	:off(off), sem(0), next(n)
{
	err = 0;
	next = prev = NULL;
}

// This class is passed as a callback to the ftpconn method for
// getting a file
class docopy: public ftpxfer
{
	ftpfile &ftp;
	off_t last_off;
	int err;
	
public:
	docopy(String path, int cfd, ftpfile &, ftpconn *);
	~docopy();

	void notify(int);
	void error(int e)	{ err = e; }
	// Things for passing through the thread args
	int cfd;
	String path;
	ftpconn *conn;
};

static void
copy_cache(int, char **, void *v)
{
	cout << "update thread starting\n";
	docopy *dc = (docopy *)v;
	dc->conn->getfile(dc->path, dc->cfd, dc);
	delete dc;
	cout << "update thread finishing\n";
	suicide();
	abort();
}

docopy::docopy(String path, int cfd, ftpfile &f, ftpconn *conn)
       : ftp(f), cfd(cfd), path(path), conn(conn)
{
	assert(ftp.updating == 0);
	ftp.updating = 1;
	last_off = 0;
	err = 0;
}

docopy::~docopy()
{
	assert(ftp.updating == 1);
	// Wake up everything outstanding
	for(waiting *w = ftp.wlist; w != NULL; w = w->next)
	{
		ftp.delwlist(w);
		w->err = err;
		w->sem.signal();
		delete w;
	}

	String nm;
	cat(ftp.dir->path(), "/", ftp.name, nm);
	struct utimbuf ut;
	ut.actime = ftp.atime;
	ut.modtime = ftp.mtime;
	utime(nm, &ut);

	ftp.updating = 0;
	ftp.close();
}

void
docopy::notify(int infd)
{
	off_t size = lseek(cfd, 0, SEEK_CUR);

	// Force thread change if it wouldn't otherwise.
	// If we're doing this we re-queue it with a low select
	// dispatch priority.
	// If it's yielding a lot because of blocking reads anyhow,
	// it means other threads get good response
	
	if (size >= (last_off + 8*1024))
	{
		class copyyield: public DispatchFD
		{
			Semaphore sem;
		public:
			copyyield(): sem(0)	{}

			int dispatch(int,int)	{ sem.signal(); return 0; }
			void wait()		{ sem.wait(); }
		};

		copyyield *cy = new copyyield();
		ftp.filesys.comm()->addDispatch(infd, cy, DISP_R, 0);
		cy->wait();
		last_off = size;
	}
	for(waiting *w = ftp.wlist; w != NULL; w = w->next)
		if (w->off < size)
		{
			cout << "passing offset " << w->off <<
				" at " << size << "\n";
			ftp.delwlist(w);
			w->sem.signal();
			delete w;
		}
	ftp.size = size;
}

void
ftpfile::decref()
{
	if (nlink-- == 0)
	{
		String cn;
		cat(dir->path(), "/", name, cn);
		::unlink(cn);
		delete this;
	}
}

int
ftpfile::is_uptodate() const
{
	struct stat st;
	String cn;
	cat(dir->path(), "/", name, cn);

	if (stat((const char *)cn, &st) == 0 &&
	    size == st.st_size &&
	    size != 0)
			return 1;
	return 0;
}

// Start updating file if needed
// This starts an update thread.  Things wanting to read the file should
// use ftpfile::read, which checks to see if the cache file is being updated.
// If so, it puts an entry on the waiting list.  If the update thread passes
// a point being waited for by a reading thread, it is woken.
int
ftpfile::updatefile()
{
	if (updating)
		return 0;

	if (is_uptodate())
		return 0;
	
	String cn;
	cat(dir->path(), "/", name, cn);

	String fn;
	cat(dir->ftppath(), "/", name, fn);

	::unlink(cn);
	close();
	if (open(1, 0444))
		return -1;

	// Start cache update thread
	docopy *dc = new docopy(fn, fd, *this, conn);
	new LWP(UPD_TPRI, copy_cache, 16*1024, 0, NULL, dc);
	
	return 0;
}

/* Open the cache file */
int
ftpfile::open(int create, int mode)
{
	String cp;

	close();
	cat(dir->path(), "/", name, cp);
	fd = ::open(cp, create ? O_RDWR|O_TRUNC|O_CREAT|O_APPEND : O_RDONLY, mode);
//	cout << "ftpfile::open(" << create << ", " << mode << ") opened " << cp << " fd="<< fd << '\n';

	return fd == -1 ? -1 : 0;
}

void
ftpfile::close()
{
	::close(fd);
	fd = -1;
}

// Read from the file
// If the file is being updated from the ftp site, we wait until
// the update thread says some data we want has arrived.  Otherwise,
// just go direct to the file
int
ftpfile::read(unsigned char *buf, size_t sz, off_t off)
{
	if (fd == -1 && open() == -1)
		return -1;
	
	int cfd = dup(fd);
	if (cfd == -1)
		perror("dup failed");
	
	if (updating)
	{
		waiting *w = addwlist(off);
		w->sem.wait();
		if (w->err != 0)
		{
			cout << "read failed in wait: " << strerror(w->err)
				<< '\n';
			ftruncate(cfd, 0);
			close();
			errno = w->err;
			return -1;
		}
	}
	
	lseek(cfd, off, SEEK_SET);
	int ret = ::read(cfd, buf, sz);
	if (ret == -1)
		perror("read failed");
	::close(cfd);

	return ret;
}

int
ftpfile::do_iput(const up_preamble &, upp_repl &,
		 const upp_iput_s &arg)
{
	if (!updating)
		close();
	return 0;
}

int
ftpfile::do_read(const up_preamble &, upp_repl &,
		 const upp_read_s &arg, upp_read_r &ret)
{
	updatefile();

	unsigned char buf[arg.size];
	int rd = read(buf, arg.size, arg.off);

	if (rd == -1)
		return errno;

	ret.data.alloc(rd);
	ret.data.nelem = rd;
	memcpy(ret.data.elems, buf, rd);

	return 0;
}

void
ftpfile::setstats(size_t sz, time_t time)
{
	size = sz;
	mtime = time;
}
