// -*- C++ -*-

// Top directory
// This is initialized from the cache directory, but users can
// create directories with the names of hostnames.  Directories
// are always created with the official hostname of a machine,
// with all the alises appearing as symlinks.

// (C) Copyright 1994 Jeremy Fitzhardinge <jeremy@sw.oz.au>
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#pragma implementation

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <String.h>

#include <unistd.h>
#include <netdb.h>
#include <stdio.h>

#include "ftpfs.h"
#include "topdir.h"
#include "sitetopdir.h"
#include "ftpconn.h"
#include "ftplink.h"

int 
topdir::do_create(const up_preamble &, upp_repl &,
		  const upp_create_s &arg, upp_create_r &ret)
{
	if (!S_ISDIR(arg.mode))
		return EPERM;
	
	String name((char *)arg.name.elems, (int)arg.name.nelem);
	
	struct hostent *he = gethostbyname((const char *)name);

	if (he == NULL)
	{
		cerr << "Hostname lookup for " << name << " failed:",
		herror("");
		
		return ENOENT;
	}
	cout << "Official name of " << name << " is " << he->h_name << '\n';
	String hostname = he->h_name;

	if (lookup(hostname) != NULL)
		return EEXIST;
	else
	{
		Inode *ino = new sitetopdir(filesys, hostname, this);
		link(hostname, ino);
		if (name != hostname)
		{
			Inode *ino = new ftplink(filesys, hostname, this, name);
			if (link(name, ino))
				delete ino;
		}
		ret.file.handle = ino->gethandle();
	}
	
	for (char **aliaspp = he->h_aliases; aliaspp && *aliaspp; aliaspp++)
	{
		String alias(*aliaspp);

		cout << "aliasing to " << *aliaspp << '\n';
		
		Inode *ino = new ftplink(filesys, hostname, this, alias);
		if (link(alias, ino))
			delete ino;
	}
	
	return 0;
}

topdir::topdir(ftpfs &fs, const String &cache)
       :dir(fs, cache, NULL)
{
	mode = S_IFDIR | 0777;
	init();
}

Inode *
topdir::newinode(dir *d, const struct stat *st, const String &name)
{
	Inode *ino = NULL;
	
	if (S_ISDIR(st->st_mode))
	{
		if (name == "." || name == "..")
			return NULL;
		ino = new sitetopdir(filesys, name, this);
	}
	else if (S_ISLNK(st->st_mode))
	{
		char buf[1024];
		int len = readlink(name, buf, sizeof(buf));
		if (len == -1)
			return NULL;
			
		ino = new ftplink(filesys, String(buf, len), this, name);
	}
	
	return ino;
}
