// -*- C++ -*-

// Intensive filesystem - the opposite of an extensive filesystem.

// The basic idea is that rather than having file which literally
// describe their contents, we have files which can describe how their
// contents are built.

// This is done by having symbolic links which contain strings of
// shell commands rather than literal file names.  Such links are
// identified by having the 1st 2 characters "#!".

// The filesystem is a layer over another filesystem to provide this
// functionality.  For each normal file in the base filesystem there
// is a symbolic link in the intfs filesystem.  Symbolic links in
// the base filesystem appear literally in the intfs.  Normal
// symlinks are treated normally, but ones with command syntax are
// interpreted to generate the contents of the file.  In such cases
// the readlink operation returns the command, but followlink returns
// the name of the temp file or named pipe with the actual contents
// of the file.

#pragma implementation

#include <DeferComm.h>
#include <GetOpt.h>
#include <stdio.h>

#include "intfs.h"
#include "intfsDirIno.h"
#include "intfsFifoDir.h"

#include "dbg.h"

int
intfs::Enquire(up_ops op)
{
	switch(op)
	{
	case up_mount:
	case up_iread:
	case up_readdir:
	case up_multireaddir:
	case up_lookup:
	case up_readlink:
	case up_followlink:
	case up_unlink:
	case up_rename:
	case up_create:
	case up_symlink:
	case up_iput:
//	case up_iwrite:
	case up_read:
	case up_write:
	case up_open:
	case up_close:
		return 0;

	default:
		return DeferFilesys::Enquire(op);
	}
}

intfs::intfs(const char *mpoint, String base)
	: DeferFilesys(mpoint), base(base)
{
	inocnt = 2;
	root = new ifsDirIno(*this, genhand(), NULL, base);
	fifod = new ifsFifoDir(*this, genhand(), root);
	root->link(".fifo", fifod);
}

intfs::~intfs()
{
}

int
intfs::do_mount(const up_preamble &, upp_repl &, upp_mount_r &ret)
{
	ret.root.handle = root->gethandle();
	return 0;
}

int
main(int argc, char **argv)
{
	GetOpt getopt(argc, argv, "i:o:m:");
	int optch, err = 0;
	int infd, outfd;
	const char *mpoint = "";
	const char *base;

	infd = outfd = -1;
	while((optch = getopt()) != EOF)
		switch(optch)
		{
		case 'i':
			infd = atoi(getopt.optarg);
			break;
		case 'o':
			outfd = atoi(getopt.optarg);
			break;
		case 'm':
			mpoint = getopt.optarg;
			break;
		default:
			err++;
		}

	if (err || infd == -1 || outfd == -1 || getopt.optind == argc)
	{
		fprintf(stderr, "Usage: %s -i infd -o outfd -m mpoint basedir\n",
			argv[0]);
		exit(1);
	}

	base = argv[getopt.optind];

#if 0 && DEBUG
	printf("pid=%d, return to cont\n", getpid());
	getchar();
#endif
	intfs ifs(mpoint, base);
	DeferComm comm(ifs, outfd, infd);

	if (comm.Run() == -1)
		fprintf(stderr, "%s: Comm.Run() failed!\n", argv[0]);
	
	return 0;
}
