/* (Read-only) archive filesystem.  */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <utime.h>
#include <dirent.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <getopt.h>
#include <assert.h>

#include <time.h>

#include <sys/types.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/fcntl.h>

#include <linux/fs.h>
#include <linux/userfs_fs.h>

#include <assert.h>

#include "arcfs.h"

static char *mpoint;
static ARC_DIRENT *root;

/* No point in this being in a seperate file.  */
#include <linux/userfs_types.c>

static int
myread (int fd, void *buf, int sz)
{
  int rd = 0;
  int ret;
	
  while (sz > 0)
    {
      ret = read (fd, buf, sz);

      if (ret == -1)
	{
	  if (errno == EINTR)
	    continue;

	  perror ("myread");
	  exit (1);
	}

      if (ret == 0)
	break;

      sz -= ret;
      buf += ret;
      rd += ret;
    }

  return rd;
}

static int
mywrite (int fd, const void *buf, int sz)
{
  int wr = 0;
  int ret;

  while (sz > 0)
    {
      ret = write (fd, buf, sz);

      if (ret == -1)
	{
	  if (errno == EINTR)
	    continue;
	  FAIL ("write");
	  exit (1);
	}

      if (ret == 0)
	{
	  fprintf (stderr, "write wrote 0 bytes\n");
	  break;
	}

      sz -= ret;
      buf += ret;
      wr += ret;
    }

  return wr;
}

static int
readpkt (int fd, up_preamble *pre, unsigned char *buf)
{
  int presz = sizeof_up_preamble (pre);
  unsigned int rd;
  unsigned char pbuf[presz];

  memset (pbuf, 0, sizeof (pbuf));

  rd = myread (fd, pbuf, presz);
  if (rd != presz)
    {
      if (rd == 0)
	return 0;
      if (rd > 0)
	fprintf (stderr, "readpkt, incomplete read (%d, not %d)\n",
		 rd, presz);
      perror ("readpkt, header read");
      return -1;
    }

  decode_up_preamble (pre, pbuf);

  if (pre->isreq != UP_ENQ && pre->isreq != UP_REQ)
    fprintf (stderr, "pre->isreq = %d\n", pre->isreq);

  if (pre->version != UP_VERSION)
    {
      fprintf (stderr, "Version mismatch: us=%d them=%d\n",
	       UP_VERSION, pre->version);
      return -1;
    }

  if (pre->size != 0)
    {
      if (myread (fd, buf, pre->size) != pre->size)
	{
	  perror ("readpkt, body read");
	  return -1;
	}
    }

  return 1;
}

static int
writepkt (int fd, upp_repl *repl, const unsigned char *buf)
{
  int replsz = sizeof_upp_repl (repl);
  unsigned char pbuf[replsz];
  unsigned int wr;
  unsigned char *p;

  assert (repl->version == UP_VERSION);
  assert (repl->isreq == UP_REPL);

  p = encode_upp_repl (repl, pbuf);

  assert (replsz == p - pbuf);

  wr = mywrite (fd, pbuf, replsz);
  if (wr != replsz)
    {
      perror ("writepkt, header write");
      return -1;
    }

  if (repl->size > 0 && repl->errno == 0)
    {
      if ( mywrite(fd, buf, repl->size) != repl->size)
	{
	  perror ("writepkt, body write");
	  return -1;
	}
    }

  return 1;
}

static void
genpkt (upp_repl *repl, int seq, up_ops op)
{
  repl->version = UP_VERSION;
  repl->seq = seq;
  repl->op = op;
  repl->isreq = 0;
  repl->size = 0;
  repl->errno = 0;
}

#define DIRINO 1

static int
do_iread (unsigned char *buf, upp_repl *repl)
{
  upp_iread_s rcv;
  upp_iread_r snd;
  time_t t;

  decode_upp_iread_s(&rcv, buf);

  memset(&snd, 0, sizeof(snd));

  snd.ino.mode = 0444;
  snd.handle = rcv.handle;
  snd.ino.nlink = 1;

  if (rcv.handle.handle == DIRINO)
    {
      snd.ino.mode |= S_IFDIR | 0111;
      t = time (0);
    }
  else
    {
      ARC_DIRENT *ent = (ARC_DIRENT *) rcv.handle.handle;

      if (ent->size != -1)
	{
	  snd.ino.mode |= S_IFREG;
	  snd.ino.size = ent->size;
	  /* These numbers seem to be magic for Linux.  */
	  snd.ino.blksize = 512;
	  snd.ino.blocks = (ent->data.file.csize + 511) / 512;
	}
      else
	{
	  snd.ino.mode |= S_IFDIR | 0111;
	  snd.ino.size = 0;
	}
      t = 0;
    }
	
  snd.ino.mtime = snd.ino.ctime = t;
  snd.ino.atime = time (0);

  repl->size = encode_upp_iread_r(&snd, buf) - buf;

  return 0;
}

static int
do_lookup (unsigned char *buf, upp_repl *repl)
{
  upp_lookup_s rcv;
  upp_lookup_r snd;
  ARC_DIRENT *ent;
	
  decode_upp_lookup_s(&rcv, buf);

  ent = (ARC_DIRENT *) rcv.dir.handle;
  if (ent->size != -1)
    return ENOTDIR;

  if (rcv.name.nelem == 2
	   && rcv.name.elems[0] == '.' && rcv.name.elems[1] == '.')
    {
      ent = ent->pdir;
    }
  else if (rcv.name.nelem != 1 || rcv.name.elems[0] != '.')
    {
      ent = ent->data.dir.first;
      while (ent)
	{
	  /* Probably bombs out on prefixes...  */
	  if (!strncmp(ent->name, rcv.name.elems, rcv.name.nelem))
	    break;
	  ent = ent->next;
	}
    }
  FREE(rcv.name.elems);
  if (!ent)
    return ENOENT;
  snd.handle.handle = (Ulong) ent;

  repl->size = encode_upp_lookup_r (&snd, buf) - buf;

  return 0;
}

static int
do_readdir (unsigned char *buf, upp_repl *repl)
{
  upp_readdir_s rcv;
  upp_readdir_r snd;
  ARC_DIRENT *ent;

  decode_upp_readdir_s(&rcv, buf);

  ent = (ARC_DIRENT *) rcv.dir.handle;
  if (ent->size != -1)
    return ENOTDIR;

  snd.off = 0;
  snd.name.nelem = 0;

  if (!rcv.off--)
    {
      snd.name.elems = ".";
    }
  else if (!rcv.off--)
    {
      snd.name.elems = "..";
      ent = ent->pdir;
    }
  else
    {
      ent = ent->data.dir.first;
      while (ent && rcv.off--)
	ent = ent->next;
      if (ent)
	snd.name.elems = ent->name;
    }

  if (ent)
    {
      snd.off = 1;
      snd.name.nelem = strlen (snd.name.elems);
      snd.file.handle = (Ulong) ent;
    }

  repl->size = encode_upp_readdir_r(&snd, buf) - buf;

  return 0;
}

static int
do_open (unsigned char *buf, upp_repl *repl)
{
  upp_open_s rcv;
  ARC_DIRENT *ent;
	
  decode_upp_open_s (&rcv, buf);

  ent = (ARC_DIRENT *) rcv.file.handle;
  /* Directories are opened here, but read using _readdir instead of _read,
     so all we do is return success!  */
  if (ent->size == -1)
    return 0;

  return a_open (ent);
}

static int
do_close (unsigned char *buf, upp_repl *repl)
{
  upp_close_s rcv;
  ARC_DIRENT *ent;
	
  decode_upp_close_s (&rcv, buf);

  ent = (ARC_DIRENT *) rcv.file.handle;
  if (ent->size == -1)
    return 0;

  return a_close (ent);
}

static int
do_read (unsigned char *mbuf, upp_repl *repl)
{
  upp_read_s rcv;
  upp_read_r snd;
  ARC_DIRENT *ent;
  int rc;

  snd.data.nelem = 0;
	
  decode_upp_read_s (&rcv, mbuf);

  ent = (ARC_DIRENT *) rcv.file.handle;
  if (ent->size == -1)
    return EISDIR;

  rc = a_read (ent, rcv.off, rcv.size, &snd.data.nelem, &snd.data.elems);
  if (rc)
    return rc;
  repl->size = encode_upp_read_r (&snd, mbuf) - mbuf;
  return 0;
}

static int
do_mount (unsigned char *buf, upp_repl *repl)
{
  upp_mount_r snd;

  snd.root.handle = (Ulong) root;
  repl->size = encode_upp_mount_r(&snd, buf) - buf;
  /* INIT */
  repl->size = encode_upp_mount_r(&snd, buf) - buf;
  return 0;
}

static void
run (int wr, int rd)
{
  up_preamble pre;
  upp_repl repl;
  /* Ah!  The bogey man's a-coming to get me!  */
  unsigned char	buf[4096];
  int error;

  while(readpkt(rd, &pre, buf) == 1)
    {
      genpkt(&repl, pre.seq, pre.op);

      switch(pre.op)
	{
#define DO(op, b, r)  case up_##op: \
			  error = (pre.isreq == UP_ENQ ? 0 : do_##op(b, r)); \
			  break;
	DO(mount, buf, &repl);
	DO(iread, buf, &repl);
	DO(lookup, buf, &repl);
	DO(readdir, buf, &repl);
	DO(open, buf, &repl);
	DO(close, buf, &repl);
	DO(read, buf, &repl);
#undef DO
	default:
	  error = ENOSYS;
	  break;
	}
		
      repl.errno = error;
      if (error != 0)
	repl.size = 0;

      if (writepkt(wr, &repl, buf) != 1)
	break;
    }
}
		
int
main (int argc, char **argv)
{
  int infd = 0;
  int outfd = 1;
  int c;

  while ((c = getopt (argc, argv, "i:o:m:")) != EOF)
    switch(c)
      {
      case 'i':
	infd = atoi(optarg);
	break;
      case 'o':
	outfd = atoi(optarg);
	break;
      case 'm':
	mpoint = optarg;
	break;
      default:
	fprintf (stderr, "%s: bad option\n", argv[0]);
	return 1;
      }

  if (optind != argc - 1)
    {
      fprintf (stderr, "%s: no zipfile named\n", argv[0]);
      return 1;
    }

  root = arc_read (argv[optind++]);
  if (!root)
    return 1;

  signal (SIGCHLD, SIG_IGN);
	
  run(outfd, infd);

  return 0;
}
