// -*- C++ -*-

// Cope with deferred replies in the form of sub-processes

#pragma implementation

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>

#include "DeferComm.h"
#include "DeferFilesys.h"
#include "io.h"

class Defer_doreq: public Comm_doreq
{
	int deferred;
	
	int dispatch(int fd, int flag);
public:
	Defer_doreq(const CommBase *, Filesystem &);
	void defer(int fd);
};

class copy : public DispToKern
{
public:
	copy(const CommBase *);
	dispatch(int fd, int);
};

copy::copy(const CommBase *c)
     :DispToKern(c)
{
}

int
copy::dispatch(int from, int)
{
	int rd;
	char buf[1024];

 again:
	while((rd = read(from, buf, sizeof(buf))) > 0)
	{
		char *wrp = buf;

		while(rd > 0)
		{
			int wr = write(tokern, wrp, rd);

			if (wr == -1 && errno == EINTR)
				continue;
			
			if (wr == -1 || wr == 0)
				return -1;
			wrp += wr;
			rd -= wr;
		}
	}
	
	if (rd == -1)
	{
		if (errno == EINTR)
			goto again;
		return -1;
	}
	
	return 0;
}	

DeferComm::DeferComm(DeferFilesys &fs, unsigned int to, unsigned int from)
	  :CommBase(fs, to, from),filesys(fs)
{
	req = ourreq = new Defer_doreq(this, filesys);
	addDispatch(from, req);
	fs.setcomm(this);
}

// This can be (indirectly) called by filesystem methods so they
// can defer replying until later, allowing the filesystem to be
// used by other processes in the meantime.  This would happen,
// for example, if a read takes a long time to generate and the
// filesystem should be available for directory scans and name
// lookups.
// The function uses fork(); the parent returns so the main
// filesystem process can keep accepting requests.  The child
// works away and sends the reply when its ready.  This goes
// through a pipe from the child to the parent for copying
// to the kernel in one piece.  Everything is set up so the
// Comm class in the child knows what to do, and exits when
// done.
int
DeferComm::DeferRepl()
{
	int p[2];
	int pid;
	
	if (pipe(p) == -1)
	{
		perror("DeferComm::DeferRepl pipe() failed");
		return -1;
	}
	
	switch(pid = fork())
	{
	case -1:
		perror("DeferComm::DeferRepl fork() failed");
		return -1;
	case 0:
		CloseFDs(0);
		ourreq->defer(p[1]);
		::close(p[0]);
		break;
	default:
		::close(p[1]);
		addDispatch(p[0], new copy(this));
		break;
	}

	return pid;
}

DispatchFD *
DeferComm::kern_disp()
{
	return req;
}

Defer_doreq::Defer_doreq(const CommBase *comm, Filesystem &fs)
	    : Comm_doreq(comm, fs)
{
	deferred = 0;
}

// Process a request from the kernel
// Returns 0 on EOF, -1 on error and 1 for OK
int
Defer_doreq::dispatch(int fd, int)
{
	up_preamble pre;
	size_t presize = sizeof_up_preamble(&pre);
	Uchar hbuf[256];
	Uchar buf[4096];
	int ret;

	ret = fullread(fd, hbuf, presize);
	if (ret == 0)
		return 0;
	
	if (ret != presize)
	{
		fprintf(stderr, "Defer_doreq::dispatch failed to get whole header (%d wanted, %d got)\n",
			presize, ret);
		return -1;
	}

	upp_repl repl;
	size_t replsize = sizeof_upp_repl(&repl);

	assert(sizeof(hbuf) > replsize);
		
	decode_up_preamble(&pre, hbuf);
		
	if (pre.size != 0 && (ret = fullread(fd, buf, pre.size)) != pre.size)
	{
		fprintf(stderr, "Defer_doreq::dispatch: failed to get whole body (%d wanted, %d got)\n",
			pre.size, ret);
		return -1;
	}
	
	ret = filesys.DoOp(pre, repl, buf);

	if (ret == -1 && !deferred)
		return 1;

	if (!ret)
		return -1;

#ifdef DEBUG
	if (repl.errno != 0)
	{
		fprintf(stderr, "Request %d failing with %d %s\n",
			pre.op, repl.errno, strerror(repl.errno));
	}
#endif
		
	encode_upp_repl(&repl, hbuf);

	if (fullwrite(tokern, hbuf, replsize) != replsize)
		return -1;
		
	if (repl.size != 0 && fullwrite(tokern, buf, repl.size) != repl.size)
		return -1;

	return deferred ? 0 : 1;
}

void
Defer_doreq::defer(int fd)
{
	::close(tokern);
	tokern = fd;
	deferred = 1;
}
