// -*- C++ -*-

// Generic directory inode by Jeremy Fitzhardinge
// This program is distributed under the terms of the
// Free Software Foundation General Public Licence.
// See COPYING for details

// A directory

#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <assert.h>
#include <sys/time.h>

#pragma implementation

#include "Filesystem.h"
#include "SimpleInode.h"
#include "DirInode.h"
#include "String.h"
#include "dbg.h"

DirInode::DirInode(Filesystem &fs, Handle h, Inode *p, int mm)
	:SimpleInode(fs, h), max_multird(mm)
{
	LOG("DirInode::DirInode");

	mode = S_IFDIR;
	
	parent = p == NULL ? this : p;

	size = 0;
	
	entries = end = NULL;
	link(".", this);
	link("..", parent);
}

DirInode::~DirInode()
{
	DirEntry *next;

	for(; entries; entries = next)
	{
		next = entries->next;
		delete entries;
	}

	entries = end = NULL;
}
		
int
DirInode::link(const String &name, Inode *ino)
{
	if (lookup(name) != NULL)
		return EEXIST;

	DirEntry *de = new DirEntry(name, ino);

	de->prev = end;
	de->next = NULL;
	if (de->prev)
		de->prev->next = de;
	else
		entries = de;
	end = de;

	ino->incref();
	
	size++;

	return 0;
}

int
DirInode::unlink(const String &name)
{
	DirEntry *de = lookup(name);

	if (de == NULL)
		return ENOENT;

	de->geti()->decref();

	if (de->next)
		de->next->prev = de->prev;
	else
		end = de->prev;
	
	if (de->prev)
		de->prev->next = de->next;
	else
		entries = de->next;

	delete de;
	
	size--;
	
	return 0;
}

DirEntry *
DirInode::lookup(const String &name) const
{
	for(DirEntry *de = entries; de != NULL; de = de->next)
		if (de->name == name)
			return de;

	return NULL;
}

DirEntry *
DirInode::scan(DirEntry * &de) const
{
	if (de == NULL)
		de = entries;
	
	if (de == NULL)
		return NULL;

	DirEntry *ret = de;
	de = de->next;

	return ret;
}

DirEntry *
DirInode::scan(int &ind) const
{
	if (ind >= size)
		return NULL;

	int i;
	DirEntry *de;

	for(de = entries, i = 0; de != NULL && i < ind; de = de->next, i++)
		;
	
	if (de == NULL)
		return de;

	ind++;
	return de;
}

// Implementations of userfs operations
int
DirInode::do_unlink(const up_preamble &, upp_repl &,
		    const upp_unlink_s &arg)
{
	update();
	
	return unlink(String((char *)arg.name.elems, (int)arg.name.nelem));
}

int
DirInode::do_link(const up_preamble &, upp_repl &,
		  const upp_link_s &args)
{
	update();
	
	Inode *ino = findino(args.ofile.handle);

	if (!ino)
		return EINVAL;

	return link(String((char *)args.name.elems, (int)args.name.nelem), ino);
}

int
DirInode::do_rename(const up_preamble &, upp_repl &,
		    const upp_rename_s &arg)
{
	Inode *ndir = findino(arg.ndir.handle);

	if (ndir == NULL || !S_ISDIR(ndir->mode))
		return ENOTDIR;

	update();

	String oname((char *)arg.oname.elems, (int)arg.oname.nelem);
	
	DirEntry *fde = lookup(oname);

	if (!fde)
		return ENOENT;
	
	Inode *file = fde->geti();
	
	if (file == NULL)
		return EINVAL;

	int ret = ndir->link((char *)arg.nname.elems, arg.nname.nelem, file);
	if (ret)
		return ret;
	ret = unlink(oname);

	ndir->mtime = mtime = time(0);
	
	return ret;
}

int
DirInode::do_lookup(const up_preamble &pre, upp_repl &rep,
		    const upp_lookup_s &arg, upp_lookup_r &res)
{
	update();

	DirEntry *de = lookup(String((char *)arg.name.elems, (int)arg.name.nelem));

	atime = time(0);
	
	if (de != NULL)
	{
		res.handle.handle = de->geti()->gethandle();
		return 0;
	}
	return ENOENT;
}

int
DirInode::do_readdir(const up_preamble &pre, upp_repl &repl, 
		     const upp_readdir_s &arg, upp_readdir_r &res)
{
	LOG("readdir");

	update();
	
	res.off=1;
	
	int ind = arg.off;
	
	DirEntry *de = scan(ind);

	if(de == NULL)
	{
		res.off=0;
		res.name.nelem=0;
		return 0;
	}

	atime = time(0);
	
	res.name.nelem = strlen(de->name);
	res.name.elems = (char *)de->name.chars();
	res.file.handle = de->ino->gethandle();
	
	DB(printf("Readdir:'%s',%d,%d\n",res.name.elems,res.name.nelem,res.file.handle));
	return 0;
}

int
DirInode::do_read(const up_preamble &, upp_repl &,
		  const upp_read_s &, upp_read_r &)
{
	return EISDIR;
}

int
DirInode::do_write(const up_preamble &, upp_repl &,
		   const upp_write_s &, upp_write_r &)
{
	return EISDIR;
}

int
DirInode::do_multireaddir(const up_preamble &pre, upp_repl &repl,
			  const upp_multireaddir_s &arg,
			  upp_multireaddir_r &res)
{
	update();
	
	LOG("multireaddir");

	int ind = arg.off;
	DirEntry *de = scan(ind);

	if (de == NULL)
	{
		res.nelem = 0;
		return 0;
	}
	
	atime = time(0);

	res.nelem = 0;
	res.alloc(max_multird);
	upp_readdir_r *ents = res.elems;

	int i;
	for(i = 0;
	    de != NULL && i < max_multird &&
	    sizeof_upp_multireaddir_r(&res) < USERFS_MAX_XFER;
	    i++, scan(de))
	{
		int len = de->name.length();
		ents[i].off = 1;
		ents[i].name.nelem = len;
		ents[i].name.alloc(len);
		memcpy(ents[i].name.elems, (const char *)de->name, len);
		ents[i].file.handle = de->ino->gethandle();
		DB(printf("Multireaddir:%d '%s',nlen=%d,ino=%d,off=%d\n",
			  i, (const char *)de->name,
			  len, ents[i].file.handle, off));
		res.nelem = i+1;
	}
	res.nelem = i;
	
	while(sizeof_upp_multireaddir_r(&res) >= USERFS_MAX_XFER)
		res.nelem--;
	
	return 0;
}

DirEntry::DirEntry(const String name, Inode *i)
	: name(name), ino(i)
{
}

DirEntry::~DirEntry()
{
}
