/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: lpq.c
 ***************************************************************************/

#include "lp.h"
#include "common/displayq.h"
#include "library/errormsg.h"
#include "library/readconf.h"
#include "LPQ/termclear.h"
#include "patchlevel.h"

/***************************************************************************
 *   lpq - spool queue examination program
 *   lpq [-Pprinter ][-a][-s][+[n]][-D[n]][ job # ... ][ user ... ]
 * DESCRIPTION
 *   lpq examines the spooling area used by lpd(8) for printing
 *   files on the line printer, and reports the status of the
 *   specified jobs or all jobs associated with a user. lpq
 *   invoked without any arguments reports on the printer given
 *   by the default printer (see -P option).  For each job sub-
 *   mitted (i.e. invocation of lpr(1)) lpq reports the user's
 *   name, current rank in the queue, the names of files compris-
 *   ing the job, the job identifier (a number which may be sup-
 *   plied to lprm(1) for removing a specific job), and the total
 *   size in bytes.  Job ordering is dependent on the algorithm
 *   used to scan the spooling directory and is FIFO (First in
 *   First Out), in order of priority level.  File names compris-
 *   ing a job may be unavailable (when lpr(1) is used as a sink
 *   in a pipeline) in which case the file is indicated as
 *   ``(stdin)''.  The following options are available.
 *   -P printer  spool queue name
 *   -a   All printers listed in /etc/printcap(5) file are reported.
 *   -s   alternate display format
 *   [+[n]] redisplay the spool queue
 *   -D[n]  debug level n
 *   [ job # ... ][ user ... ]
 ***************************************************************************/

static int Delay;		/* time to wait */
static int Do_all_pr;		/* do all printers */
static int Name_all_pr;		/* list all printers by name */
static char **Printer_name;
static int Printer_count;
static int Max_printer_count;

static void Usemsg(void)
{
    (void) fprintf (stderr,"Usage: lpq [-Pprinter] [-al] [+[n]] [-Dn] (job# | user)*\n");
}

plp_signal_t cleanup(int signal)
{
    Term_finish ();
    exit (1);
}

static void Check_already(int c, int *v)
{
    if (*v) {
	Usemsg ();
	Diemsg ("-%c has already been specified", c);
    }
    *v = 1;
}

/***************************************************************************
 * 1. set up the name routine
 * 2. extract parameters
 *  -Pprinter sets Printer to printer
 *  -Dn set Debug to n
 *  -a   sets Do_all_pr
 *  -n   sets Name_all_pr
 *  -s   set Short_format
 *  +nn
 ***************************************************************************/
static void Get_parms(int argc, char *argv[])
{
    int c;			/* ACME Integers, Inc. */
    char *arg;			/* argument from list */
    int Long_format;

    Long_format = Short_format = 0;
    Opt_flag = "+";
    while ((c = Getopt (argc, argv, "P:D:talsn+?")) != EOF) {
	switch (c) {
	case 'P':
	    if (!Optarg) {
		break;
	    }
	    if (Printer_count == Max_printer_count) {
		if (Max_printer_count == 0) {
		    Max_printer_count = 3;
		    Printer_name = (char **) malloc
			    (Max_printer_count * sizeof (char *));
		} else {
		    Max_printer_count += 3;
		    Printer_name = (char **) realloc (Printer_name,
			    Max_printer_count * sizeof (char *));
		}
	    }
	    Printer_name[Printer_count] = Optarg;
	    ++Printer_count;
	    break;
	case 'D':
	    if (!Optarg)
		break;
	    Check_already (c, &Debug);
	    Parse_debug (Optarg);
	    break;
	case 'a':
	    Check_already (c, &Do_all_pr);
	    break;
	case 'n':
	    Check_already (c, &Name_all_pr);
	    break;
	case 'l':
	    Check_already (c, &Long_format);
	    break;
	case 's':
	    Check_already (c, &Short_format);
	    break;
	case '+':
	    /**************************************
	     [+[n]] set Delay to n
	     **************************************/
	    Check_already (c, &Delay);
	    if (Optarg) {
		Delay = atoi (Optarg);
	    }
	    if (Delay == 0) {
		Delay = 30;
	    }
	    break;
	default:
	    Usemsg ();
	    exit (1);
	}
    }
    if (Long_format && Short_format) {
	Usemsg ();
	Diemsg ("-l and -s both specified");
    }

    allocParmsIfNeeded ();
    for (; Optind < argc; ++Optind) {
	arg = argv[Optind];
	Parms[Parmcount].str = arg;
	if (isdigit (*arg)) {
	    Parms[Parmcount].num = atoi (arg);
	} else {
	    Parms[Parmcount].num = -1;
	}
	++Parmcount;
	growParmsIfNeeded ();
    }
}

/***************************************************************************
 * 1. get the printer list
 * 2. for each printer in the list call Displayq
 ***************************************************************************/

static void Display_all(void)
{
    char **list;

    for (list = All_printers (); *list; ++list) {
	Printer = *list;
	Displayq (1);
    }
}

/*
 * Name_all: added by jmason for shell-script writers; name all the printers.
 */

static void Name_all(void)
{
    char **list;

    for (list = All_printers (); *list; ++list) {
	printf ("%s\n", *list);
    }
}

int main(int argc, char *argv[], char *envp[])
{
    struct passwd *pw_ent;	/* user entry in /etc/passwd */
    int i;

    /*
     * set umask to avoid problems with user umask
     */
    (void) umask (0);
    set_user_uid (0);
    root_to_user ();

    /*
     * Set fd 0, 1, 2 to /dev/null if not open
     */
    Std_environ (argc, argv, envp);
    Get_parms (argc, argv);

    Readconf ();

    /*
     * pick up the default printer name
     */
    if (Printer_count == 0) {
	Get_Printer (0);
    }

    /*
     * set up the from information
     */
    FQDN = Host;
    From = ShortHost;
    /*
     * get the user name
     */
    if ((pw_ent = getpwuid (UserUID)) == 0) {
	logerr_die (XLOG_INFO, "getpwuid failed on uid %d", UserUID);
    }
    (void) strcpy (LOGNAME, pw_ent->pw_name);
    Person = LOGNAME;

    (void) plp_signal (SIGPIPE, (plp_sigfunc_t)SIG_IGN);
    (void) plp_signal (SIGHUP, cleanup);
    (void) plp_signal (SIGINT, cleanup);
    (void) plp_signal (SIGQUIT, cleanup);
    (void) plp_signal (SIGTERM, cleanup);

    do {
	/*
	 * if we are to clear the terminal, do so
	 */
	if (Delay) {
	    Term_clear ();
	}
	/*
	 * if we have all printers to display, do so
	 */
	if (Do_all_pr) {
	    Display_all ();
	} else if (Name_all_pr) {
	    Name_all ();
	} else if (Printer_count == 0) {
	    Displayq (1);
	} else {
	    for (i = 0; i < Printer_count; ++i) {
		Printer = Printer_name[i];
		Displayq (1);
	    }
	}
	if (Delay) {
	    sleep ((unsigned) Delay);
	}
    } while (Delay);
    return (0);
}
