/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: lpr_filters.c
 * filter programs for LPR
 * Supports the reading and enqueuing of input
 ***************************************************************************/

#include <assert.h>
#include "lpr.h"
#include "lpr_filters.h"
#include "lpr_global.h"
#include "lpr_temp.h"
#include "common/setup_filter.h"
#include "library/errormsg.h"
#include "library/filestack.h"

/***************************************************************************
 * Run_pr()
 *   Copies input the PR program; temp file name left in Read_stdin
 *   Returns: 0 if nothing, 1 otherwise
 * Copy_stdin()
 *   Copies input to a temporary file; temp file name left in Read_stdin
 *   Returns: 0 if nothing, 1 otherwise
 * char *Copy_file( int infile, char *name )
 *    copies file to temp file, returns name of temp file
 * int Open_SD(char *s) Open the file in the spool directory
 ***************************************************************************/

static char *estr3(char *s, char *s1, char *s2, char *s3, char *e)
{
    if (s1)
	s = estrcp (s, s1, e);
    if (s2)
	s = estrcp (s, s2, e);
    if (s3)
	s = estrcp (s, s3, e);
    return (s);
}

/***************************************************************************
 * 1. Gets the name of a temporary file for the output
 * 2. Sets up the PR command
 * 3. Forks a process
 * 4. Arranges the process output goes to the output file
 * 5. Waits for PR to complete
 ***************************************************************************/

int Run_pr(void)
{
    int fd;			/* Data file */
    char cmdbuf[BUFSIZ];	/* build up the command here */
    char buf[BUFSIZ];		/* local buf up the command here */
    char *bp;			/* bp points to cmdbuf */
    char *ep;			/* ep is limit */
    pid_t pid;			/* pid of pr process */
    plp_status_t status;	/* the status.w_status is the integer value */
    int i;			/* ACME Integer, Inc. */
    struct stat statb;		/* for stating the output file */

    /*
     * get name of the data file
     */
    Pr_stdin = Get_tmp_data ();
    fd = Open_SD (Pr_stdin);
    /*
     * set up the printer name
     */
    if (PR == 0 || *PR == 0) {
	Diemsg ("there is no pr program specified for -p format");
    }
    /* set up the command */
    ep = cmdbuf + sizeof (cmdbuf);
    bp = estr3 (cmdbuf, PR, (char *) 0, (char *) 0, ep);
    /*
     * set the width, length, and title flags
     */

    if (PWIDTH[0]) {
	bp = estr3 (bp, " -w", PWIDTH, " ", ep);
    } else if (PW) {
	(void) sprintf (buf, " -w%d ", PW);
	bp = estrcp (bp, buf, ep);
    }
    if (PL) {
	(void) sprintf (buf, "-l%d ", PL);
	bp = estrcp (bp, buf, ep);
    }
    if (PRTITLE[0]) {
	bp = estr3 (bp, "-h '", PRTITLE, "' ", ep);
    } else if (JOBNAME[0]) {
	bp = estr3 (bp, "-h '", JOBNAME, "' ", ep);
    } else if (Parmcount == 0) {
	bp = estr3 (bp, "-h '", "(stdin)", "' ", ep);
    }
    /*
     * Put the file names in the list
     */
    for (i = 0; i < Parmcount; i++) {
	bp = estr3 (bp, Parms[i].str, " ", (char *) 0, ep);
    }
    if (bp == 0) {
	Diemsg ("pr command is too long: %s", cmdbuf);
    }
    if (Debug > 2)
	log (XLOG_DEBUG, "pr command is '%s'", cmdbuf);

    if ((pid = fork ()) < 0) {
	logerr_die (XLOG_INFO, "Run_pr: fork failed");
    } else if (pid == 0) {
	if (dup2 (fd, 1) < 0) {
	    logerr_die (XLOG_INFO, "Run_pr: dup2 failed");
	}

	setup_close_on_exec (); /* this'll close the unused pipe fds */

	/* pr runs as the user. */
	full_user_perms (); secure_exec (cmdbuf);
    }
    /* wait for printer */
    if ((plp_waitpid (pid, &status, 0)) < 0 || PLP_WEXITSTATUS(status)) {
	fatal (XLOG_INFO, "pr failed (%s)", Decode_status (&status));
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "pr done successfully");
    if (fstat (fd, &statb) < 0) {
	logerr_die (XLOG_INFO, "Run_pr: cannot stat output file %s", Pr_stdin);
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "Run_pr: %s is %d", Pr_stdin, statb.st_size);
    (void) close (fd);
    return (statb.st_size != 0);
}

/***************************************************************************
 * Open the name of the file in the spool directory
 * 1. Prepend the SD name
 * 2. Create the file with the desired perms
 * Returns: fd if successful; dies otherwise
 ***************************************************************************/

int Open_SD (char *s)
{
    static char buf[MAXPATHLEN];

    assert(s!=(char*)0);
    fix_SD ();			/* just in case the perms need fixing */

    *buf = '\0';		/* workaround for bug in SunOS sprintf() */
    (void) sprintf (buf, "%s/%s", SD, s);
    if (Debug > 3) {
	log (XLOG_DEBUG, "Open_SD: exclusive-opening spool file '%s'", buf);
    }
    return (Exlockcf (buf));
}

/***************************************************************************
 * 1. Gets the name of a temporary file for the output
 * 2. Copies stdin to the file
 * 3. Returns 0 if empty file, 1 otherwise
 ***************************************************************************/

int Copy_stdin(void)
{
    int fd;			/* Data file */
    static char buf[MAXPATHLEN]; /* local buf up the command here */
    int n, i;			/* bytes read */
    int count;			/* number of bytes */
    long blocks;		/* number of blocks */

    /*
     * get name of the data file
     */
    Read_stdin = Get_tmp_data ();
    fd = Open_SD (Read_stdin);

    count = 0;
    blocks = 0;

    do {
	if (((n = fread (buf, 1, sizeof (buf), stdin)) > 0)) {
	    count += n;		/* update count */
	    while (count >= 1024) {
		count -= 1024;
		++blocks;
	    }
	    if (MX > 0 && blocks > MX) {
		Diemsg ("input from stdin exceeds maximum file size limits");
	    }
	    i = write (fd, buf, n);
	    if (i != n) {
		logerr_die (XLOG_INFO, "Copy_stdin: cannot write %s", Read_stdin);
	    }
	}
    } while (!feof (stdin));

    if (ferror (stdin)) {
	Diemsg ("error reading from stdin");
    }
    (void) close (fd);
    if (count) {
	++blocks;
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "Copy_stdin: %s is %d blocks", Read_stdin, blocks);
    return (blocks != 0);
}

/***************************************************************************
 * 1. Gets the name of a temporary file for the output
 * 2. Copies file to the temporary file
 * Returns: temporary file name
 ***************************************************************************/

char *Copy_file (int in_fd, char *in_file, struct stat *statb)
{
    int out_fd;			/* Data file */
    char buf[BUFSIZ];		/* local buf for IO */
    int n, i;			/* bytes read */
    int count;			/* number of bytes */
    long blocks;		/* number of blocks */
    char *fname;		/* file name */

    assert(in_file!=(char*)0);
    assert(statb!=(struct stat*)0);
    /*
     * get name of the data file
     */
    fname = Get_tmp_data ();
    out_fd = Open_SD (fname);

    count = 0;
    blocks = 0;
    while ((n = read (in_fd, buf, sizeof (buf))) > 0) {
	count += n;		/* update count */
	while (count >= 1024) {
	    count -= 1024;
	    ++blocks;
	}
	if (MX > 0 && blocks > MX) {
	    if (Debug)
		log (XLOG_DEBUG, "file %d > max %d", blocks, MX);
	    Diemsg ("file %s too large", in_file);
	}
	i = write (out_fd, buf, n);
	if (i != n) {
	    logerr_die (XLOG_INFO, "Copy_file: cannot write to file %s", in_file);
	}
    }
    if (n < 0) {
	logerr_die (XLOG_INFO, "Copy_file: error reading from %s", in_file);
    }
    if (fstat (out_fd, statb) < 0) {
	logerr_die (XLOG_INFO, "Copy_file: cannot stat %s", fname);
    }
    (void) close (out_fd);
    return (fname);
}

/***************************************************************************
 *	Prefilter handler
 *  1. Sets up a prefilter command.
 *  filtername arguments \   <- from filtername
 *      -PPrinter -wwidth -llength -xwidth -ylength [-c] [-iindent] \
 *      [-Zoptions] [-Cclass] [-Jjob] [-Raccntname] -nlogin -hHost
 *      -Fformat [file]
 *     Note: the filter parameter is of the form "filter,X", where
 *     output format is specified.  If there is no specified format,
 *     'f' will be used.
 *  2. Gets a temporary file for output
 *  3. If file is explicitly named,  uses that for input instead of
 *     parameter files.
 *  4. Forks and runs the command.
 *  5. Filter must return 0 for successful status, otherwise we abort
 ***************************************************************************/

int Do_prefilter(char *cmd, struct parm *pent)
{
    char cmdbuf[BUFSIZ];	/* build up the command here */
    char fnamebuf[BUFSIZ];	/* build up the filename here */
    plp_status_t status;	/* the status.w_status is the integer value */
    char *cp, *ep;		/* ACME Pointers, Inc. */
    int l;			/* ACME Integers, Inc. */
    pid_t pid;
    int new_format = 'f';	/* new format */
    char *Filter_out;		/* the output filename */
    int fd;			/* the output file */
    struct stat statb;		/* for stating the output file */

    assert(cmd!=(char*)0);
    /*
     * get the output format
     */
    assert(strlen(cmd)<sizeof(cmdbuf));
    (void) strcpy (cmdbuf, cmd);
    l = strlen (cmdbuf);
    if (l > 2) {
	if (cmdbuf[l - 2] == ',') {
	    new_format = cmdbuf[l - 1];
	    cmdbuf[l - 2] = 0;
	}
    }
    if (!isascii (new_format) || !islower (new_format)) {
	fatal (XLOG_INFO, "bad prefilter output format, command: %s", cmd);
    }
    /*
     * set up the basic filter command
     */
    AF = 0;
    cp = Setup_filter (Format, cmdbuf);
    Format = new_format;
    ep = cmdbuf + sizeof (cmdbuf);
    /*
     * copy command to buffer
     */
    cp = estrcp (cmdbuf, cp, ep);

    /*
     * add the file names
     */
    if (Pr_stdin) {
  	/*
  	 * we have an already filtered file (through 'pr') -- open it
  	 */
  	if (Debug > 3)
  	    log (XLOG_DEBUG, "prefilter: reading from file '%s'", Pr_stdin);
  
        assert((strlen(SD)+2+strlen(Pr_stdin))<=sizeof(fnamebuf));
        (void) sprintf(fnamebuf, "%s/%s", SD, Pr_stdin);
  
  	if ((fd = open_daemon (fnamebuf, O_RDONLY, 0644)) < 0) {
  	    logerr_die (XLOG_INFO, "open failed");
  	}
  	if (dup2 (fd, 0) < 0) {
  	    logerr_die (XLOG_INFO, "Do_prefilter: dup2 (read) failed");
  	}
  	(void) close (fd);
  
      } else if (pent) {
	/*
	 * we have an explicitly named file -- open it
	 */
	if (Debug > 3)
	    log (XLOG_DEBUG, "prefilter: reading from file '%s'", pent->str);

	if ((fd = open (pent->str, O_RDONLY)) < 0) {
	    logerr_die (XLOG_INFO, "open failed");
	}
	if (dup2 (fd, 0) < 0) {
	    logerr_die (XLOG_INFO, "Do_prefilter: dup2 (read) failed");
	}
	(void) close (fd);

    } else {
	if (Debug > 3)
	    log (XLOG_DEBUG, "prefilter: reading from stdin");
    }

    if (cp == 0) {
	fatal (XLOG_INFO, "Do_prefilter: command too long: %s", cmd);
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "prefilter command is '%s'", cmdbuf);

    /*
     * get the output file name
     */
    Filter_out = Get_tmp_data ();
    fd = Open_SD (Filter_out);

    if (pent) {
	strcpy (pent->filename, Filter_out);
	Filter_stdin = NULL;
    } else {
	Filter_stdin = Filter_out;
    }

    /*
     * start the prefilter up and wait for output
     */
    if ((pid = fork ()) < 0) {
	logerr_die (XLOG_INFO, "Do_prefilter: fork failed");

    } else if (pid == 0) {
	if (dup2 (fd, 1) < 0) {
	    logerr_die (XLOG_INFO, "Do_prefilter: dup2 failed");
	}
	chdir_SD ();
	setup_close_on_exec (); /* this'll close the unused pipe fds */

	/* prefilters should be run as "daemon". */
	user_to_root (); full_daemon_perms (); secure_exec (cmdbuf);
    }
    /*
     * wait for prefilter
     */
    if ((plp_waitpid (pid, &status, 0)) < 0 || PLP_WEXITSTATUS(status)) {
	logerr_die (XLOG_INFO, "Do_prefilter: prefilter failed (%s)",
		    Decode_status (&status));
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "Do_prefilter: prefilter successful");
    if (fstat (fd, &statb) < 0) {
	logerr_die (XLOG_INFO, "Do_prefilter: cannot stat output file %s",
		    Filter_out);
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "Do_prefilter: output file is %s: %ld bytes",
	     Filter_out, statb.st_size);

    (void) close (fd);

    /* if already filtered by 'pr', remove its input file */
    if( Pr_stdin ) {
	if (Debug > 3)
           log (XLOG_DEBUG, "Do_prefilter: removing temporary file %s", Pr_stdin);
        (void)sprintf(cmdbuf, "%s/%s", SD, Pr_stdin);
        if (unlink_daemon( cmdbuf ) < 0){
            logerr( XLOG_INFO, "could not remove %s- %s",Read_stdin,
                   Errormsg(errno) );
        }
        Remove_tempfile(Pr_stdin); /* delete it from temp file list */
    }
    return (statb.st_size != 0);
}
