/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 * Changlog:
 * 16.08.1994: Added functions: downq, changepriority (Stefano Ianigro)
 *             Changed the ouptut format; it distinguishes between queues
 *                                        and printers
 * 04.09.1994: Changed the output format so that the contents of the queue
 *             and the printer status will be displayed 
 *
 ***************************************************************************
 * MODULE: Control_ops.c
 */

#include "lp.h"
#include "library/errormsg.h"
#include "common/control_ops.h"
#include "common/displayq.h"
#include "common/setstatus.h"
#include "common/requeue.h"
#include "common/hold.h"
#include "common/move.h"
#ifdef HAVE_UTIME_H
#include <utime.h>
#endif

/***************************************************************************
 *  1. get the printcap entries
 *  2. check for permissions
 *  3. determine the function to be carried out
 *  4. carry out the function
 ***************************************************************************/
static int op_init;		/* flag for operation initialization */
static void Show_ops (), Print_sum (), touch ();
static int find_command (), service (), demote (), changepriority (), 
    promote (), remote_cando ();

static int j = 1;              /* flag for display operations */
static int m = 1;              /* flag for display operations */

/*
 * dispatch is a data structure that contains the names and flags of control commands.
 */
struct dispatch {
    char *name;
    int distinct;
    int (*routine) ();
    int flags;
    char *summary;
};

static int
#ifdef C_BAD
    C_bad (),			/* not used normally */
#endif
#ifdef EUCS_ZOPTIONS
    C_forms (),
#endif
    C_help (), C_abort (), C_clean (), C_disable (),
    C_enable (), C_exit (), C_kill (), C_restart (), C_lpq (), C_lprm (),
    C_start (), C_status (), C_stop (), C_topq (), C_remote (), C_lpd (),
    C_down (), C_up (), C_attach (), C_unattach (),
    C_hold (), C_release (), C_move (), C_downq (), C_chprio (), C_requeue ();

#define NEED_OPT		1	/* we check for other options */
#define ALL_ALLOWED		2	/* check for all */
#define NEED_PRIV		4	/* privileged operation */
#define ALL_DEF			8	/* if no parameter, all is default */
#define NO_PR_LIST		0x10	/* printer list */
#define REMOTE_OP		0x20	/* remote allowed */
#define SERVER_OP		0x40	/* server allowed */
#define ON_REMOTE		0x80	/* not if you have RM and NW */
#define IS_OPT(x) ((x)&NEED_OPT)
#define IS_ALL(x) ((x)&ALL_ALLOWED)
#define IS_PRIV(x) ((x)&NEED_PRIV)
#define IS_ALL_DEF(x) ((x)&ALL_DEF)
#define IS_NO_PR_LIST(x) ((x)&NO_PR_LIST)
#define IS_REMOTE(x) ((x)&REMOTE_OP)
#define IS_SERVER(x) ((x)&SERVER_OP)
#define IS_ON_REMOTE(x) ((x)&ON_REMOTE)

static struct dispatch dispatch[] = 
{
  {"", 0, 0, 0, "not a command"},
  {"?", 1, C_help, 0, "? is same as 'help'"},
  {"abort", 1, C_abort,
   NEED_OPT | NEED_PRIV | ALL_ALLOWED | REMOTE_OP | SERVER_OP | ON_REMOTE,
   "abort ( all | Printer ):                  kill off server and disable printing"},
  {"attach", 2, C_attach, NEED_OPT | NO_PR_LIST | REMOTE_OP | ON_REMOTE | NEED_PRIV,
   "attach Printer destPrinter:               attach queue to another printer"},
  {"chprio", 1, C_chprio, NEED_PRIV | NEED_OPT | REMOTE_OP | ON_REMOTE | NO_PR_LIST,
   "chprio Printer (user|host|job)* priority: change priority of a job"},
  {"clean", 1, C_clean, NEED_OPT | NEED_PRIV | ALL_ALLOWED | REMOTE_OP | ON_REMOTE,
   "clean ( all | Printer ):                  remove all crud from queue \n                                          (dangerous!)"},
  {"disable", 2, C_disable, NEED_OPT | NEED_PRIV | ALL_ALLOWED | REMOTE_OP | ON_REMOTE,
   "disable ( all | Printer ):                disable queueing"},
  {"down", 2, C_down, NEED_OPT | NEED_PRIV | NO_PR_LIST | ALL_ALLOWED | REMOTE_OP | SERVER_OP | ON_REMOTE,
   "down ( all | Printer ) [message]:         disable queueing and printing"},
  {"downq", 1, C_downq, NEED_OPT | NO_PR_LIST | REMOTE_OP | ON_REMOTE | NEED_PRIV,
   "downq Printer (user|host|job)*:           move job to the end of queue"},
  {"enable", 2, C_enable, NEED_OPT | NEED_PRIV | ALL_ALLOWED | ON_REMOTE | REMOTE_OP,
   "enable ( all | Printer ):                 enable queuing"},
  {"exit", 2, C_exit, 0, "exit:                                     terminate"},
#ifdef EUCS_ZOPTIONS
  {"forms", 1, C_forms, NEED_OPT | NO_PR_LIST | REMOTE_OP | ON_REMOTE | NEED_PRIV,
   "forms Printer forms:                      change loaded forms"},
#endif /* EUCS_ZOPTIONS */
  {"help", 1, C_help, 0, "help [all] [command]:                     print command summary"},
  {"hold", 2, C_hold, NEED_OPT | NO_PR_LIST | REMOTE_OP | ON_REMOTE | NEED_PRIV,
   "hold Printer jobnumber:                   hold specified job"},
  {"kill", 1, C_kill,
   NEED_OPT | NEED_PRIV | ALL_ALLOWED | REMOTE_OP | SERVER_OP | ON_REMOTE,
   "kill ( all | Printer ):                   kill off server and then \n                                          restart printing"},
  {"lpd", 3, C_lpd, 0,
   "lpd :                                     check out lpd process"},
  {"lpq", 3, C_lpq, 0,
   "lpq ( parms ) :                           call lpq"},
  {"lprm", 3, C_lprm, 0,
   "lprm ( parms ) :                          call lprm"},
  {"move", 1, C_move, NEED_OPT | NO_PR_LIST | REMOTE_OP | ON_REMOTE,
   "move Printer jobnumber destPrinter:       move specified job to another \n                                          printer"},
  {"mv", 1, C_move, NEED_OPT | NO_PR_LIST | REMOTE_OP | ON_REMOTE,
   "mv Printer jobnumber destPrinter:         same as move"},
  {"quit", 1, C_exit, 0, "quit (same as exit):                      terminate"},
  {"remote", 3, C_remote, 0,
   "remote command:                           do the action on remote printer"},
  {"release", 1, C_release, NEED_OPT | NO_PR_LIST | REMOTE_OP | ON_REMOTE | NEED_PRIV,
   "release Printer jobnumber:                release specified job"},
  {"requeue", 1, C_requeue, NEED_OPT | NO_PR_LIST | ON_REMOTE | NEED_PRIV,
   "requeue Printer jobnumber:                requeue specified job to the same \n                                          printer"},
  {"rq", 1, C_requeue, NEED_OPT | NO_PR_LIST | ON_REMOTE | NEED_PRIV,
   "rq Printer jobnumber:                     same as requeue"},
  {"restart", 3, C_restart, NEED_OPT | ALL_ALLOWED | REMOTE_OP | SERVER_OP,
   "restart ( all | Printer ):                start a server"},
  {"start", 4, C_start,
   NEED_OPT | NEED_PRIV | ALL_ALLOWED | REMOTE_OP | SERVER_OP | ON_REMOTE,
   "start ( all | Printer ):                  enable printing and start server"},
  {"status", 4, C_status, ALL_DEF | ALL_ALLOWED | NEED_OPT,
   "status [all] [Printer]:                   print status"},
  {"stop", 3, C_stop,
   NEED_OPT | NEED_PRIV | ALL_ALLOWED | REMOTE_OP | SERVER_OP | ON_REMOTE,
   "stop ( all | Printer ):                   disable further printing"},
  {"topq", 1, C_topq, NEED_OPT | NO_PR_LIST | REMOTE_OP | ON_REMOTE | NEED_PRIV,
   "topq Printer (user|host|job)*:            move job to the top of queue"},
  {"unattach", 2, C_unattach, NEED_OPT | NO_PR_LIST | REMOTE_OP | ON_REMOTE | NEED_PRIV,
   "unattach Printer:                         revert queue original printer"},
  {"up", 2, C_up, NEED_OPT | NEED_PRIV | ALL_ALLOWED | REMOTE_OP | SERVER_OP | ON_REMOTE,
   "up ( all | Printer ):                     enable queuing and printing"},
};

static int ncmds = sizeof (dispatch) / sizeof (struct dispatch);

/***************************************************************************
 * This routine is called with the parameters in the Parms[] array.
 * It will check to see if the command is valid,  and if so, will
 * do any other actions.
 ***************************************************************************/
int
Control_ops (void) {
    int command;		/* command we are going to do */
    char **list;		/* list of Printers */
    int all;			/* doing for all Printers */

    /*
     * set the flags needed
     */
    Is_local = (hostcmp (From, Host) == 0);
    Is_root = strsame (Person, "root");
    op_init = 0;
    command = 0;
    all = 0;
    /*
     * check to see that we have a valid command
     */
    if (Debug > 4)
	Show_ops ();
    if (Parmcount < 1) {
	(void) fprintf (stdout, "No command.\n");
	return (0);
    }
    command = find_command (Parms[0].str);
    if (command == 0) {
	(void) fprintf (stdout, "Unknown command %s.\n", Parms[0].str);
	Parmcount = 0;
	return (0);
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "Control_ops: command %s, flags 0x%x, %s",
	     dispatch[command].name, dispatch[command].flags,
	     dispatch[command].summary);

    if (Log_LPCs) {
	if (Printer && *Printer) {
	    log_request ("%s@%s: control operation '%s %s'", Person, From,
			 dispatch[command].name, Printer);
	} else {
	    log_request ("%s@%s: control operation '%s'", Person, From,
			 dispatch[command].name);
	}
    }
    if (!Is_local && !IS_REMOTE (dispatch[command].flags)) {
	log (XLOG_INFO, "Command %s cannot be done remotely", Parms[0].str);
	Parmcount = 0;
	return (0);
    }
    /*
     * call the appropriate routine
     */
    Shift_parms (1);
    /*
     * If no option processing just dispatch
     */
    if (!IS_OPT (dispatch[command].flags)) {
	return ((*dispatch[command].routine) (&dispatch[command]));
    }
    /*
     * if no options and IS_ALL is the default, set up the all flag
     */
    if (Parmcount == 0) {
	if (IS_ALL_DEF (dispatch[command].flags)) {
	    all = 1;
	} else {
	    (void) fprintf (stdout, "No parameters, use: %s.\n",
			    dispatch[command].summary);
	    return (0);
	}
    } else if (strsame (Parms[0].str, "all")) {
	if (IS_ALL (dispatch[command].flags)) {
	    all = 1;
	    Shift_parms (1);	/* AJCD */
	} else {
	    (void) fprintf (stdout, "'all' not allowed, use: %s.\n",
			    dispatch[command].summary);
	    return (0);
	}
    }
    /*
     * set up Printer from parameters
     */
    if (all) {
	if (Debug > 2)
	    log (XLOG_DEBUG, "'all' parameter");
	for (list = All_printers (); *list; ++list) {
	    Printer = *list;
	    if (service (&dispatch[command]) == 0) {
		return (0);
	    }
	}
    } else {
	while (Parmcount > 0) {
	    Printer = Parms[0].str;
	    Shift_parms (1);
	    if (service (&dispatch[command]) == 0) {
		return (0);
	    }
	    if (IS_NO_PR_LIST (dispatch[command].flags)) {
		break;
	    }
	}
    }
    return (1);
}

/***************************************************************************
 * 1. check on the printcap entry
 * 2. chdir to the spool directory
 * 3. dispatch the particular routine
 ***************************************************************************/

static int
service (struct dispatch *cmd) {
    char *st, *ps;
    int perms = 'C';		/* Permission Checking */

    if (Debug > 4)
	log (XLOG_DEBUG, "service: printer '%s', cmd '%s'",
	     Printer, cmd->name);
    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	printf ("Printer '%s' does not have printcap entry.\n", Printer);
	return (0);
    }
    if (RM && NW && IS_ON_REMOTE (cmd->flags)) {
	(void) fprintf (stdout,
		"%s: Remote printer and network file system, use \'remote %s %s\'.\n",
		Printer, cmd->name, Printer);
	return (0);
    }
    /*
     * we may have a server specified
     */
    if (SS && *SS) {
	ps = PS;
	st = ST;
	if (Set_pc_entry (SS, Status_pc_vars, Status_pc_len) == 0) {
	    printf ("Server '%s' queue '%s' does not have printcap entry.\n",
		    Printer, SS);
	    return (0);
	}
	PS = ps;
	ST = st;
	LO = Printer;
	SV = 0;
    }
    if (SD == 0 || *SD == 0) {
	log (XLOG_INFO,
	     "service: Printer %s does not have spool directory", Printer);
	return (0);
    }
    chdir_SD ();
    /*
     * check on the privileges needed You have to be local and root OR have C privs
     */
    if (IS_PRIV (cmd->flags) && !(Is_local && Is_root)) {
	if (Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) <= 0) {
	    (void) fprintf (stdout,
		    "Sorry %s, but you do not have printer control perms on '%s'!\n", 
		    Person, Printer);
	    return (0);
	}
    }
    return ((*cmd->routine) (cmd));
}

/***************************************************************************
 * look in the command table for a match.  The "distinct" entry is used
 * to determine the numbers of characters for a match.
 ***************************************************************************/

static int
find_command (char *str) {
    int i;

    for (i = 1; i < ncmds; ++i) {
	if (strncmp (str, dispatch[i].name, dispatch[i].distinct) == 0
	    && strncmp (str, dispatch[i].name, strlen (str)) == 0) {
	    return (i);
	}
    }
    return (0);
}

#ifdef C_BAD
/***************************************************************************
 * Cannot decide what to do with the command
 ***************************************************************************/
static int
C_bad (void) {
    (void) fprintf (stdout, "Bad command %s.\n", Parms[0].str);
    return (1);
}
#endif /* C_BAD */

/***************************************************************************
 * Print a help message for each command listed,
 * or a simple list of commands
 ***************************************************************************/
static int
C_help (void)
{
    int i, cmd;

    if (Parmcount < 2 || strsame (Parms[1].str, "all")) {
	for (i = 1; i < ncmds; ++i) {
	    /*
	     * Do a kind of VERY crude paging...
	     */
	    if ( i == 15 ) {
		(void)fprintf(stdout, "Press Return to continue...");
		(void)getchar();
	    }
	    Print_sum (i);
	}
    } else {
	for (i = 1; i < Parmcount; ++i) {
	    cmd = find_command (Parms[i].str);
	    if (cmd > 0) {
		Print_sum (cmd);
	    } else {
		(void) fprintf (stdout, "Not a command: %s.", Parms[i].str);
	    }
	}
    }
    return (1);
}

/***************************************************************************
 * prints the command summary line
 ***************************************************************************/
static void
Print_sum (int cmd) {
    (void) fprintf (stdout, "%s\n", dispatch[cmd].summary);
}


/***************************************************************************
 * terminate gracefully
 ***************************************************************************/
static int
C_exit (void)
{
    fprintf(stdout, "LPC terminated.\n");
    exit (0);
    /* NOTREACHED */
}

/***************************************************************************
 * Sets the DISABLE_PRINT perm bit in the lockfile perms
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
static int
C_stop (void)
{
    int s;

    if (Debug > 4)
	log (XLOG_DEBUG, "C_stop: printer %s, lock '%s'", Printer, LO);
    (void) Checklockfile (LO, (pid_t *) 0, (char *) 0, 0, &LO_statb);
    if ((s = chmod_daemon (LO, (int) (LO_statb.st_mode & 0777) | DISABLE_PRINT)) < 0) {
	logerr (XLOG_INFO, "cannot chmod lockfile %s", LO);
    } else {
	if (j) {
	    Displayq (0);
	}
    }
    return (s >= 0);
}

/***************************************************************************
 * 1. Clears the DISABLE_PRINT perm bit in the lockfile perms
 * 2. Starts the Printer
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
static int
C_start (void)
{
    int s;

    if (Debug > 4)
	log (XLOG_DEBUG, "C_start: printer %s, lock '%s'", Printer, LO);
    (void) Checklockfile (LO, (pid_t *) 0, (char *) 0, 0, &LO_statb);
    if ((s = chmod_daemon (LO, (int) (LO_statb.st_mode & ENABLE_PRINT))) < 0) {
	logerr (XLOG_INFO, "cannot chmod lockfile %s", LO);
    }
    
    /*
     * start the server
     */
    if (s >= 0) {
	s = C_restart ();
       (void)sleep (1);
	if (j) {
           Displayq (0);
       }
    }
    return (s >= 0);
}

/***************************************************************************
 * Sets the DISABLE_QUEUE perm bit in the lockfile perms
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
static int
C_disable (void)
{
    int s;

    if (Debug > 4)
	log (XLOG_DEBUG, "C_disable: printer %s, lock '%s'", Printer, LO);
    (void) Checklockfile (LO, (pid_t *) 0, (char *) 0, 0, &LO_statb);
    if ((s = chmod_daemon (LO, (int) (LO_statb.st_mode & 0777) | DISABLE_QUEUE)) < 0) {
	logerr (XLOG_INFO, "cannot chmod lockfile %s", LO);
    } else {
	if (j) {
	    Displayq (0);
	}
    }
    return (s >= 0);
}

/***************************************************************************
 * 1. Clears the DISABLE_QUEUE perm bit in the lockfile perms
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
static int
C_enable (void)
{
    int s;

    if (Debug > 4)
	log (XLOG_DEBUG, "C_enable: printer %s, lock '%s'", Printer, LO);
    (void) Checklockfile (LO, (pid_t *) 0, (char *) 0, 0, &LO_statb);
    if ((s = chmod_daemon (LO, (int) (LO_statb.st_mode & ENABLE_QUEUE))) < 0) {
	logerr (XLOG_INFO, "cannot chmod lockfile %s", LO);
    } else {
       if (j) {
           Displayq (0);
       }
    }
    return (s >= 0);
}

/***************************************************************************
 * 1. Attempts to fire up the server
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
static int
C_restart (void)
{
    /*
     * start the server
     */

    if (Debug > 4)
	log (XLOG_DEBUG, "C_restart: printer %s, lock '%s'", Printer, LO);
    (void) Startserver ();
    (void) utime (SD, NULL);
    return (1);
}

/*
 * killserver(): kills the current server to stop printing
 */
static int
killserver (void)
{
    int s;			/* Success of operation */
    pid_t pid;			/* server PID */
    /*
     * Kill the current server to stop printing now.
     */
    s = 1;
    if (Checklockfile (LO, &pid, (char *) 0, 0, &LO_statb)) {
	if (kill (pid, SIGINT) < 0) {
	    (void) logerr (XLOG_INFO, "server (pid %d) not killed", pid);
	    s = 0;
	} else {
	    (void) fprintf (stdout, "Printer '%s': server (pid %d) killed.\n",
			    Printer, (int) pid);
	}
    } 

    (void) utime (SD, NULL);
    return (s);
}

/***************************************************************************
 * 1. Does C_stop(void)
 * 2. kills off server if there is one
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
static int C_abort (void)
{
    int s;			/* Success of operation */

    if (Debug > 4)
	log (XLOG_DEBUG, "C_abort: printer %s, lock '%s'", Printer, LO);
    j = 0;
    s = C_stop ();
    j = 1;
    if (s) {
	s = killserver ();
	if (m == 1) {
	    Displayq (0);
	}
    }
    return (s);
}

/***************************************************************************
 * C_kill(void)
 * 1. Does C_abort(void)
 * 2. Does C_start(void)
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
static int
C_kill (void)
{
    int s;			/* Success of operation */

    if (Debug > 4)
	log (XLOG_DEBUG, "C_kill: printer %s, lock '%s'", Printer, LO);
    j = 0;
    s = C_abort ();
    j = 1;
    if (s) {
	j = 0;
	s = C_start ();
	j = 1;
    }
    return (s);
}

/***************************************************************************
 * 1. Removes all entries in the specified spool directory.
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
static int
C_clean (void)
{
    int c;			/* ACME Integers, Inc. */
    DIR *dirp;
    plp_dir_t *d;
    pid_t pid;			/* server PID */

    if (Debug > 4)
	log (XLOG_DEBUG, "C_clean: printer %s, lock '%s'", Printer, LO);

    if ((dirp = opendir_SD ()) == NULL) {
	logerr (XLOG_INFO, "cannot open spool directory %s");
	return (0);
    }

    /* First the active server will be killed if there is one */
    if (Checklockfile (LO, &pid, (char *) 0, 0, &LO_statb)) {
	if (kill (pid, SIGINT) < 0) {
	    (void) logerr (XLOG_INFO, "server (pid %d) not killed", pid);
	} else {
	    (void) fprintf (stdout, "Printer '%s': server (pid %d) killed.\n",
			Printer, (int) pid);
	}
    }

    while ((d = readdir (dirp)) != NULL) {
	c = d->d_name[0];
	if ((c == 'c' || c == 't' || c == 'd') && d->d_name[1] == 'f') {
            if (unlink_daemon (d->d_name) < 0) {
		(void) fprintf (stdout, "Cannot remove %s from queue %s.\n", 
                              d->d_name, Printer);
            } else {
		(void) fprintf (stdout, "Removed %s from queue %s.\n", 
                              d->d_name, Printer);
	    }
	    (void) fflush (stdout);
	}
    }
    closedir (dirp);
    return (1);
}

/***************************************************************************
 * C_status (void)
 * Print the status of each queue listed or all the queues.
 ***************************************************************************/
static char *hdr_format  = "%-12.12s         %-4s  %-10s %s\n";
static char *data_format = "%-20.20s %4d  %-10s %s%s%s\n";
static char *attach_data_format = "%-8.8s%-4s%-8.8s %4d  %-10s %s%s%s\n";
static int
C_status (void)
{
    FILE *afp;                  /* attach file pointer */  
    static int att_mark;        /* marker for attached printer */
    int active;
    pid_t pid;			/* active server and its pid */
    char afname[MAXPARMLEN];    /* attach file name */
    char buf[BUFSIZ];		/* buffer */
    char *prstat, *actstat, *qstat, *rqstat;	/* Printer and queue status */
    char *st, *ps;
    char sbuf[BUFSIZ];		/* status buffer */
    char unattached_printer[MAXPARMLEN]; /* Name of the unattached printer */
    char servers[BUFSIZ];
    char *sp, *ep, *sr;		/* ACME Pointers, Inc. */

    att_mark = 0;
    if ((afp = fopen_daemon (Attach_file, "r"))) { /* Try to open attach file */
	if (fscanf (afp, "%s", afname) == 1) {
	    if (strsame (afname, Printer)) {
		fatal (XLOG_INFO, "Printer '%s' attached to itself", Printer);
	    }
	    (void)strcpy(unattached_printer, Printer);
	    Printer = afname;
	    att_mark = 1;
	}
	fclose(afp);
    }
    if (Debug > 3) {
	if (att_mark) {
	    log (XLOG_DEBUG, "C_status: printer %s attached to %s", unattached_printer, Printer);
	} else {
	    log (XLOG_DEBUG, "C_status: printer %s", Printer);
	}
    }

    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	fprintf (stdout,"Printer '%s' does not have printcap entry.\n", Printer);
	return (0);
    }
    /* we have a server here */
    if (SS && *SS) {
	ps = PS;
	st = ST;
	if (Set_pc_entry (SS, Status_pc_vars, Status_pc_len) == 0) {
	    log (XLOG_INFO, "Server '%s' queue '%s' does not have printcap entry.",
		 Printer, SS);
	    return (0);
	}
	PS = ps;
	ST = st;
	LO = Printer;
	SV = 0;
    }
    if (SD == 0 || *SD == 0) {
	log (XLOG_INFO, "Printer %s does not have spool directory", Printer);
	return (0);
    }
    chdir_SD ();
    /*
     * start by getting active server information
     */
    buf[0] = 0;
    active = Checklockfile (LO, &pid, buf, sizeof (buf), &LO_statb);

    /*
     * get numbers of jobs in queue
     */

    Jobcount = Getq ();

    /*
     * now format the info appropriately
     */
    qstat = (LO_statb.st_mode & DISABLE_QUEUE) ? "disabled " : "enabled ";
    prstat = (LO_statb.st_mode & DISABLE_PRINT) ? "disabled " : "enabled ";
    /*
     * get active server
     */
    if (Debug > 4)
	log (XLOG_DEBUG, "C_status: active '%d', Jobcount '%d', '%s'",
	     active, Jobcount, buf);
    if (SV == 0 || *SV == 0) {
	if (Jobcount == 0 && active == 0) {
	    *sbuf = '\0';
	} else if (Jobcount == 0 && active) {
	    (void) sprintf (sbuf, "(server %d)", (int) pid);
	} else if (Jobcount && active == 0) {
	    (void) strcpy (sbuf, "(no server)");
	} else {
	    (void) sprintf (sbuf, "(server %d, job %s)", (int) pid, buf);
	}
	actstat = sbuf;
    } else {
	(void) strcpy (servers, SV);
	(void) sprintf (sbuf, hdr_format, "", "", "X", "");
	if ((sp = (char *) strchr (sbuf, 'X')) == 0) {
	    fatal (XLOG_INFO, "C_status: header format bad");
	}
	while (sp - 1 > sbuf && isspace (sp[-1]))
	    --sp;
	*sp = 0;
	for (sp = servers; sp; sp = ep) {
	    ep = (char *) strchr (sp, ',');
	    if (ep) {
		*ep = 0;
		ep = ep + 1;
	    }
	    active = Checklockfile (sp, &pid, buf, sizeof (buf), &LO_statb);
	    if (LO_statb.st_mode & DISABLE_PRINT) {
		sr = "dis";
	    } else {
		sr = "ena";
	    }
	    if (active == 0) {
		(void) sprintf (sbuf + strlen (sbuf), "(%s: %s, no server)", sp, sr);
	    } else {
		(void) sprintf (sbuf + strlen (sbuf), "(%s: %s, %d, job %s)",
				sp, sr, (int) pid, buf);
	    }
	}
	actstat = sbuf;
    }
    if (LO_statb.st_mode & FORCE_REQUE) {
	rqstat = ", reorder requested";
    } else {
	rqstat = "";
    }
    /* displays heading if not already displayed */
    if (!op_init) {
	(void) fprintf (stdout, hdr_format,
			SS ? "Server" : "Queue", "Jobs", "Queueing", "Printing");
	op_init = 1;
    }
    /* prints the queue status */
    if (att_mark) {  /* Display also the attachement of a printer */
	(void) fprintf (stdout, attach_data_format, unattached_printer, " => ", 
			Printer, Jobcount, qstat, prstat, rqstat, actstat);
	att_mark = 0;
    }  else {
	(void) fprintf (stdout, data_format, Printer, Jobcount, qstat,
			prstat, rqstat, actstat);
    }
    /* prints the Printer status */
    printstatus ();
    (void) fflush (stdout);
    return (1);
}

/***********************************************************************
 * int C_topq (void)
 * Put the specified jobs at the top of Printer queue.
 ***********************************************************************/
static int
C_topq (void)
{
    int i;			/* ACME Integer, Inc. */
    int changed;		/* We changed the queue */
    char *cfname;		/* control file name */


    if ( m != 2) {
	if ((RM != NULL) && (*RM != '\0')) { /* It's a local queue */
	    (void) fprintf (stdout, "Local  queue  '%s':\n", Printer);
	} else {
	    (void) fprintf (stdout, "Local printer '%s':\n", Printer);
	}
    }
    Setuid_debug("top_q");

    /*
     * put the parameters in the list in the correct position
     */
    if (Parmcount == 0) {
	(void) fprintf (stdout, "error: topq missing job specification.\n");
	return (0);
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "C_topq: '%s'(%d)", Parms[0].str, Parms[0].num);

    /* get number of jobs in the queue */
    Jobcount = Getq ();
    /*
     * go through the queue and find the jobs to be promoted
     */
    changed = 0;
    for (i = 0; i < Jobcount; ++i) {
	if (Match_entry (&Queue[i])) {
	    /*
	     * Reposition the job by setting its priority to high level 
	     * and then fix the control file
	     */
	    if (promote (Queue[i].q_name, Queue[i].q_user)) {
		changed++;
		Queue[i].q_name[0] = 0;
	    }
	}
    }

    /*
     * Put the other high priority jobs lower
     */
    if (!changed) {
	return (1);
    }
    for (i = 0; i < Jobcount; i++) {
	cfname = Queue[i].q_name;
	if (*cfname && cfname[2] == 'A') {
	    touch (cfname);
	}
    }
    if ( m != 2) {
	if ((RM != NULL) && (*RM != '\0')) {  /* It's a local queue */
	    (void) fprintf (stdout, "Order changed for queue '%s'.\n", Printer);
	} else {
	    (void) fprintf (stdout, "Queue order changed for printer '%s'.\n", 
			    Printer);
	}
    }

    /*
     * Turn on the public execute bit of the lock file to get lpd to rebuild the queue
     * after the current job.
     */
    (void) Checklockfile (LO, (pid_t *) 0, (char *) 0, 0, &LO_statb);
    if (chmod_daemon (LO, (int) (LO_statb.st_mode & 0777) | FORCE_REQUE) < 0) {
	logerr (XLOG_INFO, "cannot force requeue on lockfile %s", LO);
	return (0);
    }
    (void) fflush (stdout);
    return (1);
}

/***************************************************************************
 * int promote( char *cfname, char *user )
 * promote a file to the A priority (top of queue), and head of the list.
 ***************************************************************************/
static int
promote (char *cfname, char *user) {
    char buf[BUFSIZ];         /* temporary cfname buffer */
    char c_old;               /* old priority */
    char c_new;               /* new priority */
    char c_job[4];            /* job number (as char's) */
    int num;                  /* real job number  */

    num = 0;
    (void) strcpy (buf, cfname);
    buf[STARTPR] = 'A';
    if (Debug > 0)
	log (XLOG_DEBUG, "renaming %s to %s", cfname, buf);
    user_to_daemon ();
    Setuid_debug("promote");
    if (strcmp (buf, cfname) && rename (cfname, buf) < 0) {
	logerr (XLOG_INFO, "cannot rename %s to %s", cfname, buf);
	daemon_to_user (); /* We are still daemon in this case, so switch back */
	return (0);
    }
    Setuid_debug("promote");
    daemon_to_user ();
    Setuid_debug("promote");
    (void)strncpy(c_job, &buf[3], 3);
    c_job[3] = '\0';
    c_old = cfname[2];
    c_new = buf[2];
    num = (int)atoi(c_job);
    
    if ( m != 2) {
	if ((RM != NULL) && (*RM != '\0')) { /* It's a local queue */
	    (void) fprintf (stdout, 
			    "Changing priority for job %d, owner %s, from %c to %c for queue '%s'.\n",
			    num, user, c_old, c_new, Printer);
	} else {
	    (void) fprintf (stdout, 
			    "Changing priority for job %d, owner %s, from %c to %c for printer '%s'.\n", 
			    num, user, c_old, c_new, Printer);
	}
    } else {
	if ((RM != NULL) && (*RM != '\0')) { /* It's a local queue */
	    (void) fprintf (stdout, 
			    "Changed priority to A for job %d, owner %s, after requeueing.\n",
			    num, user);
	} else {
	    (void) fprintf (stdout, 
			    "Changed priority to A for job %d, owner %s, after requeueing.\n", 
			    num, user);
	}
    }	
    (void) fflush (stdout);
    return (1);
}


/***************************************************************************
 * int changepriority (char *cfname, char *user)
 * change the priority of a file to the given priority and reorder queue.
 ***************************************************************************/
static int
changepriority (char *cfname, char *user) {
    char buf[BUFSIZ];         /* temporary cfname buffer */
    char c_old;               /* old priority */
    char c_new;               /* new priority */
    char c_job[4];            /* job number (as char's) */
    int num;                  /* real job number  */

    (void) strcpy (buf, cfname);
    num = 0;

    if (isdigit(*Parms[2].str)) /* 
				 * The given priority must 
				 * be an uppercase letter, not a digit
				 * We'll check that and convert it, if
				 * necessary (Stef)
				 */
	{
	    (void) fprintf(stdout, 
			   "Error: wrong priority specification; must be a single letter from A to Z, not %s.\n", 
			   Parms[2].str);
      return (0);
	}	    

    if ( islower(*Parms[2].str) ) /* 
				   * Lowercase letters will be
				   * "uppercased" (Stef)
				   */
	{
	    if (Debug > 2)
		{
		    log (XLOG_DEBUG, "changepriority: Parms[2].str (before conversion): %s",
			 Parms[2].str);  
		}
	    *Parms[2].str = toupper(*Parms[2].str);
	    if (Debug > 2)
		{
		    log (XLOG_DEBUG,"changepriority: Parms[2].str (after conversion): %s\n",
			 Parms[2].str); 
		}
	}
    
    if ( !(isupper(*Parms[2].str)) ) { /* 
					* Check if the given priority was
					* really an uppercase letter
					* (Stef)
					*/
	(void) fprintf (stdout, 
			"Error: wrong priority specification; must be a single letter from A to Z, not %s.\n",
			Parms[2].str);
	return (0);
    }	    
    
    buf[STARTPR] = *Parms[2].str;
    if (Debug > 0)
	log (XLOG_DEBUG, "changepriority: renaming %s to %s", cfname, buf);
    Setuid_debug("changepriority");
    user_to_daemon ();	
    if (strcmp (buf, cfname) && rename (cfname, buf) < 0)
	{
	    logerr (XLOG_INFO, "changepriority: cannot rename %s to %s", cfname, buf);
	    daemon_to_user (); /* We are still daemon in this case, so switch back */
	    return (0);
	}
    Setuid_debug("changepriority");
    daemon_to_user ();
    Setuid_debug("changepriority");
    (void)strncpy(c_job, &buf[3], 3);   /* Saving the jobnumber */
    c_job[3] = '\0';
    c_old = cfname[2];                  /* Saving the jobname */
    c_new = buf[2];
    num = (int)atoi(c_job);             /* Converting the job number */

    if ((RM != NULL) && (*RM != '\0')) {   /* It's a local queue */
	(void) fprintf (stdout, 
			"Changing priority for job %d, owner %s, from %c to %c for queue '%s'.\n", 
			num, user, c_old, c_new, Printer);
    } else {
	(void) fprintf (stdout, 
			"Changing priority for job %d, owner %s, from %c to %c for printer '%s'.\n", 
			num, user, c_old, c_new, Printer);
    }
    (void) fflush (stdout);
    return (1);
}


/***************************************************************************
 * demote a file to the Z priority (at the end of the queue), 
 * and head of the list.
 ***************************************************************************/
static int
demote (char *cfname, char *user) {
    char buf[BUFSIZ];         /* temporary cfname buffer */
    char c_old;               /* old priority */
    char c_new;               /* new priority */
    char c_job[4];            /* job number (as char's) */
    int num;                  /* real job number  */

    num = 0;
    (void) strcpy (buf, cfname);
    buf[STARTPR] = 'Z';
    if (Debug > 0)
	log (XLOG_DEBUG, "renaming %s to %s", cfname, buf);
    Setuid_debug("demote");
    user_to_daemon ();
    Setuid_debug("demote");
    if (strcmp (buf, cfname) && rename (cfname, buf) < 0)
	{
	    logerr (XLOG_INFO, "cannot rename %s to %s", cfname, buf);
	    daemon_to_user (); /* We are still daemon in this case, so switch back */
	    return (0);
	}
    Setuid_debug("demote");
    daemon_to_user ();
    Setuid_debug("demote");
    (void)strncpy(c_job, &buf[3], 3); /* Saving the jobnumber */
    c_job[3] = '\0';
    c_old = cfname[2];                /* Saving the jobname */
    c_new = buf[2];
    num = (int)atoi(c_job);           /* Converting the job number */

    if ((RM != NULL) && (*RM != '\0')) { /* It's a local queue */
	(void) fprintf (stdout, 
			"Changing priority for job %d, owner %s, from %c to %c for queue '%s'.\n", 
			num, user, c_old, c_new, Printer);
	} else {
	    (void) fprintf (stdout, 
			    "Changing priority for job %d, owner %s, from %c to %c for printer '%s'.\n", 
			    num, user, c_old, c_new, Printer);
	}
    (void) fflush (stdout);
    return (1);
}


/***************************************************************************
 * Change the modification time of the file. Returns boolean if successful.
 ***************************************************************************/
static void
touch (char *cfname) {
    FILE *fp;
    int i;

    if ((fp = Lockcf (cfname)) == NULL) {
	logerr (XLOG_INFO, "cannot open %s\n", cfname);
    } else if ((i = getc (fp)) == EOF) {
	logerr (XLOG_INFO, "cannot read %s\n", cfname);
    } else if (fseek (fp, 0L, 0) < 0) {;
	/* set pointer back to top of file */
	logerr (XLOG_INFO, "cannot seek %s\n", cfname);
    } else if (putc (i, fp) == EOF) {
	logerr (XLOG_INFO, "cannot write %s\n", cfname);
    } else if (fflush (fp) == EOF) {
	logerr (XLOG_INFO, "cannot flush %s\n", cfname);
    }
    if (fp != NULL) {
	(void) fclose (fp);
    }
}

/***************************************************************************
 * shows the values of the local options
 ***************************************************************************/
static void
Show_ops (void)
{
    int i;
    (void) fprintf (stdout, "From: %s, Host %s, Person %s\n", FQDN, Host, Person);
    (void) fprintf (stdout, "Is_local %d, Is_root %d\n", Is_local, Is_root);
    (void) fprintf (stdout, "Parmcount %d ", Parmcount);
    for (i = 0; i < Parmcount; ++i) {
	(void) fprintf (stdout, " '%s'(%d)", Parms[i].str, Parms[i].num);
    }
    (void) fprintf (stdout, "\n");
    (void) fflush (stdout);
}

/***************************************************************************
 * Change the priority of a specified job in the queue
****************************************************************************/
static int
C_chprio (void)
{
   int i;			/* ACME Integer, Inc. */
   int changed;			/* We changed the queue */
   char *cfname;       		/* control file name */

   if ((RM != NULL) && (*RM != '\0')) { /* It's a local queue */
       (void) fprintf (stdout, "Local  queue  '%s':\n", Printer);
   } else { /* local printer */
       (void) fprintf (stdout, "Local printer '%s':\n", Printer);
   }
   
   /*
    * put the parameters in the list in the correct position
    */
   if (Parmcount == 0) {
       (void) fprintf (stdout, "chprio: missing job specification.\n");
       return (0);
   }
   
   if (Debug > 4)
       log (XLOG_DEBUG, "C_chprio: '%s'(%d)", Parms[0].str, Parms[0].num);
   /* 
    * get number of jobs in the queue 
    */
   Jobcount = Getq ();
   /*
    * go through the queue and find the jobs to be changed in priority
    */
   changed = 0;
   for (i = 0; i < Jobcount; ++i)
       {
	   if (Match_entry (&Queue[i]))
	       {
		   /*
		    * Reposition the job by setting its priority to 
		    * the specified level and then fix the control file
		    */
		   if (changepriority (Queue[i].q_name, Queue[i].q_user) )
		       {
			   changed++;
			   Queue[i].q_name[0] = 0;
		       }
	       }
       }
   /*
    * Reorder the queue
    */
   if (!changed)
       {
	   return (1);
       }
   for (i = 0; i < Jobcount; i++)
       {
	   cfname = Queue[i].q_name;
	   if (Debug > 4)
	       {
		   log (XLOG_DEBUG, "C_chprio: Parms[2].str: %s\n", Parms[2].str);
	       }
	   if (*cfname && cfname[2] == *Parms[2].str)
	       {
		   touch (cfname);
	       }
       }
   
   if ((RM != NULL) && (*RM != '\0')) /* It's a local queue */
       {
	   (void) fprintf (stdout, "Order changed for queue '%s'.\n", Printer);
       }
   else /* local printer */
       {
	   (void) fprintf (stdout, "Queue order changed for printer '%s'.\n", Printer);
       }
   
   /*
    * Turn on the public execute bit of the lock file to
    * get lpd to rebuild the queue after the current job.
    */
   (void) Checklockfile (LO, (pid_t *) 0, (char *) 0, 0, &LO_statb);
   if (chmod_daemon (LO, (int) (LO_statb.st_mode & 0777) | FORCE_REQUE) < 0)
       {
	   logerr (XLOG_INFO, "cannot force requeue on lockfile %s", LO);
	   return (0);
       }
   return (1);
}

/***************************************************************************
 * Put the specified jobs at the end of Printer queue.
 ***************************************************************************/
static int
C_downq (void)
{
   int i;			/* ACME Integer, Inc. */
   int changed;			/* We changed the queue */
   char *cfname;		/* control file name */

   if ((RM != NULL) && (*RM != '\0')) /* It's a local queue */
       {
	   (void) fprintf (stdout, "Local  queue  '%s':\n", Printer);
       }
   else /* local printer */
       {
	   (void) fprintf (stdout, "Local printer '%s':\n", Printer);
       }

  /*
   * put the parameters in the list in the correct position
   */
   if (Parmcount == 0)
    {
	(void) fprintf (stdout, "downq: missing job specification.\n");
	return (0);
    }
   if (Debug > 4)
       log (XLOG_DEBUG, "C_downq: '%s'(%d)", Parms[0].str, Parms[0].num);
   /* 
    * get number of jobs in the queue 
    */
   Jobcount = Getq ();
   /*
    * go through the queue and find the jobs to be demoted
    */
   changed = 0;
   for (i = 0; i < Jobcount; ++i)
       {
	   if (Match_entry (&Queue[i]))
	       {
		   /*
		    * Reposition the job by setting its priority to low level
		    * and then fix the control file
		    */
		   if (demote (Queue[i].q_name, Queue[i].q_user) )
		       {
			   changed++;
			   Queue[i].q_name[0] = 0;
		       }
	       }
       }
   /*
    * Put the other priority jobs higher
    */
   if (!changed)
       {
	   return (1);
       }
   for (i = 0; i < Jobcount; i++)
       {
	   cfname = Queue[i].q_name;
	   if (*cfname && cfname[2] == 'Z')
	       {
		   touch (cfname);
	       }
       }
   
   if ((RM != NULL) && (*RM != '\0')) /* It's a local queue */
      {
	  (void) fprintf (stdout, "Order changed for queue '%s'.\n", Printer);
      }
   else /* local printer */
       {
	   (void) fprintf (stdout, "Queue order changed for printer '%s'.\n", Printer);
       }
   
   /*
    * Turn on the public execute bit of the lock file to
    * get lpd to rebuild the queue after the current job.
    */
   (void) Checklockfile (LO, (pid_t *) 0, (char *) 0, 0, &LO_statb);
   if (chmod_daemon (LO, (int) (LO_statb.st_mode & 0777) | FORCE_REQUE) < 0)
       {
	   logerr (XLOG_INFO, "cannot force requeue on lockfile %s", LO);
	   return (0);
       }
   return (1);
}


/***************************************************************************
 * C_lpq(void)
 * invoke Lpq with parameters.
 ***************************************************************************/
static int
C_lpq (void)
{
    char buf[BUFSIZ];
    char *bp, *ep;		/* ACME Pointers, Inc. */
    int i;

    ep = buf + sizeof (buf);
    bp = estrcp (buf, "lpq", ep);
    for (i = 0; i < Parmcount; ++i) {
	bp = estrcp (bp, " ", ep);
	bp = estrcp (bp, Parms[i].str, ep);
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "C_lpq: '%s'", buf);
    i = secure_system (buf);
    if (Debug > 4)
	log (XLOG_DEBUG, "C_lpq: status %d", i);
    return (i);
}

/***************************************************************************
 * C_lprm(void)
 * invoke Lprm with parameters.
 ***************************************************************************/
static int
C_lprm (void)
{
    char buf[BUFSIZ];
    char *bp, *ep;		/* ACME Pointers, Inc. */
    int i;

    ep = buf + sizeof (buf);
    bp = estrcp (buf, "lprm", ep);
    for (i = 0; i < Parmcount; ++i) {
	bp = estrcp (bp, " ", ep);
	bp = estrcp (bp, Parms[i].str, ep);
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "C_lprm: '%s'", buf);
    i = secure_system (buf);
    if (Debug > 4)
	log (XLOG_DEBUG, "C_lprm: status %d", i);
    return (i);
}

/***************************************************************************
 * C_remote(void)
 * do the indicated command on the remote host
 * command has the form: "remote op printer"
 * 1. get the command and see if it can be done remotely
 *    Parms[0] will be op, Parms[1] will be printer
 * 2. check the printer and make sure that it is a remote printer
 * 3. send the remote command to the far end, and get the information
 *    sent back.
 ***************************************************************************/

static int
C_remote (struct dispatch *cmd) {
    char **list;		/* list of printers */
    int i;			/* ACME Integers and Bearings, Inc. */
    int command;		/* command we are going to do */
    char param[BUFSIZ];		/* buffer for parameter list */

    if (Parmcount < 2) {
	(void) fprintf (stdout, "No or missing command or parameters.\n");
	return (0);
    }
    command = find_command (Parms[0].str);
    if (command == 0) {
	(void) fprintf (stdout, "Unknown command %s.\n", Parms[0].str);
	return (0);
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "C_remote: command %s, %d, %s",
	     dispatch[command].name, dispatch[command].flags, dispatch[command].summary);

    if (!IS_REMOTE (dispatch[command].flags)) {
	(void) fprintf (stdout, "Command %s cannot be done remotely.\n",
			Parms[0].str);
	return (0);
    }
    /*
     * set up parameter list
     */
    *param = '\0';
    if (IS_NO_PR_LIST (dispatch[command].flags)) {
	for (i = 2; i < Parmcount; ++i) {
	    strcat (param, " ");
	    strcat (param, Parms[i].str);
	}
    }
    /*
     * check on the printer
     */
    if (strsame ("all", Parms[1].str)) {
	if (Debug > 2)
	    log (XLOG_DEBUG, "'all' parameter");
	for (list = All_printers (); *list; ++list) {
	    Printer = *list;
	    if (remote_cando (dispatch[command].flags)) {
		Remote_control (dispatch[command].name, param);
	    }
	}
    } else {
	for (i = 1; i < Parmcount; ++i) {
	    Printer = Parms[i].str;
	    if (remote_cando (dispatch[command].flags)) {
		Remote_control (dispatch[command].name, param);
	    }
	    if (IS_NO_PR_LIST (dispatch[command].flags)) {
		break;
	    }
	}
    }
    return (1);
}

/***************************************************************************
 * returns 0 if not able to do to remote site, 1 otherwise
 ***************************************************************************/
static int
remote_cando (int flags) {
    int perms = 'C';		/* Permission Checking */

    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	printf ("printer '%s' does not have printcap entry\n", Printer);
	return (0);
    }
    if (RM == 0 || RP == 0) {
	printf ("printer '%s' not remote\n", Printer);
	return (0);
    }
    /*
     * check on the privileges needed You have to be local and root OR have C privs
     */
    if (IS_PRIV (flags) && !(Is_local && Is_root) &&
	Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) <= 0) {
	printf ("Sorry user %s, but you do not have printer control perms on '%s'\n",
		Person, Printer);
	return (0);
    }
    return (1);
}

/***************************************************************************
 * C_lpd(void)
 * 1. Check to see if there is an active lpd server by checking the
 *    lock file.
 * 2. Check to see if the /dev/printer is available as well.
 ***************************************************************************/

static int
C_lpd (void)
{
    pid_t f;

    (void) Checklockfile (Masterlock, &f, (char *) 0, 0, &LO_statb);
    if (f) {
	(void) fprintf (stdout, "Active LPD %ld\n", (long) f);
    } else {
	(void) fprintf (stdout, "LPD is not active.\n");
    }
    (void) fflush (stdout);
    return (1);
}

/***************************************************************************
 * C_up(void)
 * Calls C_enable and C_start and shows the queue
 ***************************************************************************/
static int
C_up (void)
{
    int s;
 
    j = 0;
    s = (C_start () | C_enable ());
    j = 1;
    Displayq (0);
    return (s);
}

/***************************************************************************
 * C_down(void)
 * 1. Call C_stop and C_disable
 * 2. Put message in Parms[] into .status file
 ***************************************************************************/
static int
C_down (void)
{
    int i;
    char statstr[BUFSIZ];

    *statstr = '\0';
    j = 0;
    if (C_stop () && C_disable ()) {
	for (i = 0; i < Parmcount; i++) {
	    if (i)
		strcat (statstr, " ");
	    strcat (statstr, Parms[i].str);
	}
	setstatus (statstr);
    }
    j = 1;
    Displayq (0);
    return (1);
}


/***************************************************************************
 * int C_attach (void)
 * attaches the specified printer to another printer 
 ***************************************************************************/
static int
C_attach (void)
{
    FILE *afp;			/* attach file pointer */
    int s;			/* success of abort/start */

    if (Debug > 3)
	log (XLOG_DEBUG, "C_attach: printer %s", Printer);
    if (Parmcount != 1) {
	(void) fprintf (stdout, "Missing destination specification.\n");
	return (0);
    }
    if (!*Attach_file) {
	(void) fprintf (stdout, "Warning: 'attach' feature not active.\n");
	return (0);
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "C_attach: '%s'(%d)", Parms[0].str, Parms[0].num);

    /* 
     * Check that a printer is not attached to itself (stupid, eh...?)
     */
    if (strsame(Printer, Parms[0].str))
	{
	    log (XLOG_INFO, "Printer '%s' cannot be attached to itself!", Printer);
	    return(0);
	}

    j = 0;
    m = 0;
    if ((s = C_abort ())) {
	if ((afp = fopen_daemon (Attach_file, "w"))) {
	    if (fprintf (afp, "%s\n", Parms[0].str) == EOF) {
		log (XLOG_INFO, "C_attach: can't attach printer %s", Printer);
		s = 0;
	    }
	    (void)fclose (afp);
	    (void)chmod (Attach_file, (S_IRUSR | S_IWUSR));
	    if (s) {
		s = C_start ();
	    }
	} else {
	    log (XLOG_INFO,
		 "C_attach: can't open file to attach printer %s", Printer);
	    s = 0;
	}
    }
    (void) fflush (stdout);
    j = 1;
    m = 1;
    return (s);
}

/***************************************************************************
 * unattaches the specified printer from another printer 
 ***************************************************************************/
static int
C_unattach (void)
{
    struct stat statb;		/* for statting files */
    FILE *afp;                  /* attach file pointer */
    char afname[MAXPARMLEN];    /* attach file name */
    int s;			/* return status */
    
    if (Debug > 3)
	log (XLOG_DEBUG, "C_unattach: printer %s", Printer);

    if (!*Attach_file) {
	(void) fprintf (stdout, "Warning: 'attach' feature not active.\n");
	return (0);
    }
    
    j = 0;
    m = 0;
    if ((s = C_abort ())) {
	if (stat (Attach_file, &statb) == 0) {
	    if ((afp = fopen_daemon (Attach_file, "r"))) {
		if (fscanf (afp, "%s", afname) != 1) {
		    fatal (XLOG_INFO, "attach file for printer %s corrupted!", 
			   Printer);
		}
	    }
	    (void)fclose(afp);
	    if (unlink_daemon (Attach_file) != 0) {
		log (XLOG_INFO,
		     "C_unattach: can't delete file to unattach printer %s",
		     Printer);
		s = 0;
	    }
	} else {
	    log (XLOG_INFO, "C_unattach: printer %s not attached", Printer);
	    s = 0;
	}

	if (s) {
	    if ((RM != NULL) && (*RM != '\0')) { /* It's a local queue */
		(void) fprintf (stdout, "Local  Queue  '%s': unattached from '%s'.\n",
				Printer, afname);
	    } else {
		(void) fprintf (stdout, "Local Printer '%s': unattached from '%s'.\n",
				Printer, afname);
	    }
	}
	s = C_start ();
    }
    
    (void) fflush (stdout);
    j = 1;
    m = 1;
    return (s);
}

#ifdef EUCS_ZOPTIONS
/***************************************************************************
 * int C_forms (void)
 * changes the forms from a printer to another form
 ***************************************************************************/
static int
C_forms (void)
{
    char *possible_options ();
    int active;
    pid_t pid;		/* active server and its pid */
    char buf[BUFSIZ];		/* buffer */
    char *st, *ps;
    char *possible_forms, *forms;
    FILE *fp;

    if (Debug > 3)
	log (XLOG_DEBUG, "C_forms: printer %s", Printer);
    if (Parmcount != 1) {
	(void) fprintf (stdout, "Missing forms specification.\n");
	return (0);
    }
    if (!Forms_file) {
	(void) fprintf (stdout, "Forms feature not active.\n");
	return (0);
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "C_forms: '%s'(%d)", Parms[0].str, Parms[0].num);
    forms = Parms[0].str;

    /* get number of jobs in the queue */
    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	fprintf (stdout, "Printer '%s' does not have printcap entry.\n", Printer);
	return (0);
    }
    /* If remote printer return error */
    if ((RM != NULL) && (*RM != '\0')) {
	fprintf (stdout, "Forms command only applies to local printers.\n");
	return (0);
    }
    /* we have a server here */
    if (SS && *SS) {
	ps = PS;
	st = ST;
	if (Set_pc_entry (SS, Status_pc_vars, Status_pc_len) == 0) {
	    log (XLOG_INFO, "Server '%s' queue '%s' does not have printcap entry.",
		 Printer, SS);
	    return (0);
	}
	PS = ps;
	ST = st;
	LO = Printer;
	SV = 0;
    }
    if (SD == 0 || *SD == 0) {
	log (XLOG_INFO, "Printer '%s' does not have spool directory.", Printer);
	return (0);
    }
    chdir_SD ();
    /*
     * start by getting active server information
     */
    buf[0] = 0;
    active = Checklockfile (LO, &pid, buf, sizeof (buf), &LO_statb);

    /*
     * get numbers of jobs in queue
     */

    Jobcount = Getq ();

    possible_forms = possible_options ();
    if (possible_forms == NULL || !valid_forms (forms, possible_forms)) {
	log (XLOG_INFO, "%s is not valid forms queue - choose from %s", forms,
	     possible_forms ? possible_forms : "(no forms)");
	return (0);
    } else {
	char *ip;
	char opt_file_name[MAXPATHLEN];

	strcpy (opt_file_name, Forms_file);
	if ((ip = (char *) strchr (Printer, '.')))
	    strcat (opt_file_name, ip);
	fp = fopen_daemon (opt_file_name, "w");
	if (fp == NULL) {
	    logerr (XLOG_INFO, "Couldn't open options file %s",
		    opt_file_name);
	    return (0);
	}
	fputs (forms, fp);
	fclose (fp);
	(void) utime (SD, NULL);
    }

    (void) fflush (stdout);
    return (1);
}
#endif /* EUCS_ZOPTIONS */

/***************************************************************************
 * holds the specified jobs in a printerqueue; even if the printer is
 * enabled these jobs will not be printed until they will explicitly
 * released with the release command
 ***************************************************************************/
static int
C_hold (void)
{
    if (Debug > 3)
	log (XLOG_DEBUG, "C_hold: printer %s", Printer);
    if (Parmcount != 1) {
	(void) fprintf (stdout, "Missing job specification.\n");
	return (0);
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "C_hold: '%s'(%d)", Parms[0].str, Parms[0].num);
    hold ();

    (void) fflush (stdout);
    return (1);
}

/***************************************************************************
 * releases specified jobs in a printer queue so that they get printed
 ***************************************************************************/
static int
C_release (void) {
    if (Debug > 3)
	log (XLOG_DEBUG, "C_release: printer %s", Printer);
    if (Parmcount != 1) {
	(void) fprintf (stdout, "Missing job specification.\n");
	return (0);
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "C_release: '%s'(%d)", Parms[0].str, Parms[0].num);

    release ();

    (void) fflush (stdout);
    return (1);
}

/***************************************************************************
 * moves the specified job from one printer queue to another printer
 * queue 
 ***************************************************************************/
static int
C_move (void) {
    if (Debug > 3)
	log (XLOG_DEBUG, "C_move: printer %s", Printer);
    if (Parmcount != 2) {
	(void) fprintf (stdout, "Missing job or destination specification.\n");
	return (0);
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "C_move: '%s'(%d) '%s'(%d)", Parms[0].str,
	     Parms[0].num, Parms[1].str, Parms[1].num);

    move ();

    (void) fflush (stdout);
    return (1);
}

/***************************************************************************
 * requeues the specified job from one printer queue to another printer
 * queue 
 ***************************************************************************/
static int
C_requeue (void)
{
    Parms[1].str = Printer;

    if (Debug > 3)
      log (XLOG_DEBUG, "C_requeue: printer %s", Printer);
    if (Parmcount != 1) {
      (void) fprintf (stdout, "Missing job or destination specification.\n");
      return (0);
    }
    if (Debug > 4)
      log (XLOG_DEBUG, "C_requeue: '%s'(%d) '%s'(%d)", Parms[0].str,
           Parms[0].num, Parms[1].str, Parms[1].num);

    requeue ();

    if (!mistake) {
	m = 2;
	(void) C_abort ();              /* Sets the queue to "no printing" */
	Parmcount = 2;
	Parms[0].num = (int)atoi (Newjobnum);
	Parms[1].str = "\0";
	(void) C_topq ();             /* Puts the job at the top of the queue */
	m = 1;
	Parmcount = 0;
	Displayq (0);
    }
      (void) fflush (stdout);
      return (1);
}
