#include "lp.h"
#include "library/errormsg.h"
#include "library/utils.h"
#include "common/hold.h"
#include "common/move.h"

static char *new_SD;		/* destination spool directory */
static int New_Job_number;      /* New job number, needed in move.c */

/* move_file: rename "from" to "to", or copy if they're on different
 * filesystems.
 */
static void
move_file (char *from, char *to) {
    user_to_daemon ();
    if (rename (from, to) != 0) {
	if (errno == EXDEV) {
	    if (copy_file (from, to, -1) < 0) {
		daemon_to_user ();	/* Switch back before exit */
		logerr_die (XLOG_INFO, "move_job: copy %s to %s failed", from, to);
	    }
	    if (unlink (from) < 0) {
		daemon_to_user ();	/* Switch back before exit */
		logerr (XLOG_INFO, "move_job: unlink %s failed", from);
	    }
	} else {
	    daemon_to_user ();		/* Switch back before exit */
	    logerr_die (XLOG_INFO, "move_job: rename %s to %s failed", from, to);
	}
    }
    daemon_to_user ();
}

/***************************************************************************
 * Move a job
 * ACTIONS:
 * 1. Move job's files to other spool directory
 ***************************************************************************/

static int
move_job (FILE *cfp, struct plp_queue *q, int Job_number) {
    struct stat stbuf;		/* for statting files */
    char l[BUFSIZ];		/* for reading the control file */
    char fullname[BUFSIZ];	/* for the full pathname */
    char newname[BUFSIZ];	/* full pathname of new name  */
    char *filepart;		/* end of the pathname */
    char *newfilepart;		/* end of the pathname */
    struct ch_newname           /* Struct for the new cf filename */
    {
	char anf[3];
	char anumber[3];
	char temp_shorthost[64];
    };
    struct ch_newname tmp_new_filepart; 
    char a_Job_number[4];       /* the new Jobnumber */
    char cf_buffer[BUFSIZ];     /* the new cf-file */
    int i;                      /* flag for removing junk from cf_buffer */

    assert(q!=(struct plp_queue*)0);
    assert(cfp!=(FILE*)0);
    i = 0;
    if (new_SD == 0 || *new_SD == 0) {
	fatal (XLOG_INFO, "no directory entry");
    }
    /* get the directory name and copy to buffer */
    if (strlen (new_SD) + sizeof (q->q_name) + 2 > sizeof (newname)) {
	logerr_die (XLOG_NOTICE, "INTERNAL: move_job, file name too big");
    }
    (void) strcpy (newname, new_SD);
    newfilepart = &newname[strlen (newname)];
    *newfilepart++ = '/';

    (void) strcpy (fullname, SD);
    filepart = &fullname[strlen (fullname)];
    *filepart++ = '/';
    (void) strcpy (filepart, q->q_name);

    /* check that the control file is there */
    if (stat (fullname, &stbuf) < 0) {
	return 1;		/* Doesn't exist */
    }
    while (fgets (l, sizeof (l), cfp)) {
	/* clobber the \n */
	l[strlen (l) - 1] = '\0';
	if (islower (l[0]) || ( l[0] =='U') ) {
	    if (islower (l[0])) {
		(void) strcpy (filepart, &l[1]);
		if (stat (fullname, &stbuf) < 0) {
		    (void) fclose (cfp);
		    if (Debug > 5)
			log (XLOG_DEBUG,
			     "move_job: data file %s does not exist", l);
		} else {
		    /* Move file */
		    (void) strcpy ((char *) &(tmp_new_filepart.anf[0]), &(l[1]));
		    sprintf(a_Job_number , "%03d", Job_number);
		    tmp_new_filepart.anumber [0] = a_Job_number [0];
		    tmp_new_filepart.anumber [1] = a_Job_number [1];
		    tmp_new_filepart.anumber [2] = a_Job_number [2];
		    (void) strcpy (newfilepart, (char *) &(tmp_new_filepart.anf[0]));
		    
		    strncat(cf_buffer, &l[0], 1);
		    strcat(cf_buffer, (char *) &tmp_new_filepart);
		    strcat(cf_buffer, "\n");
		    
		    if (Debug > 4)
			log (XLOG_DEBUG, "move_job: renaming %s to %s ", fullname, newname);
		    move_file (fullname, newname);
	    Setuid_debug("move_job");
		}
	    } else {
		/* hopefully 'd' preceedes 'U'  */
		strncat(cf_buffer, &l[0], 1);
		strcat(cf_buffer, (char *) &tmp_new_filepart);
		strcat(cf_buffer, "\n");
	    }
	} else {
	    if ( i == 0 ) /* First run; erase all the junk in cf_buffer */
		{
		    (void)strcpy(cf_buffer, &l[0]); 
		    strcat(cf_buffer, "\n");
		    i++; 
		} else {
		    strcat(cf_buffer, &l[0]);
		    strcat(cf_buffer, "\n"); 
		}
	}
    }

    if (Debug > 5)
	log (XLOG_DEBUG, "move_job: control file %s is ok", q->q_name);
    /* rewrite the control-file and close it */
    (void) rewind (cfp);
    cf_buffer[strlen(cf_buffer)-1] = '\0' ;
    fputs ( cf_buffer , cfp ) ;
    (void) fclose (cfp);

    (void) strcpy (&(tmp_new_filepart.anf[0]), q->q_name);
    tmp_new_filepart.anumber [0] = a_Job_number [0] ;
    tmp_new_filepart.anumber [1] = a_Job_number [1] ;
    tmp_new_filepart.anumber [2] = a_Job_number [2] ;
    (void) strcpy (newfilepart, &(tmp_new_filepart.anf[0]));
    (void) strcpy (filepart, q->q_name);
    if (Debug > 4) {
	log (XLOG_DEBUG, "move_job: renaming %s to %s ", fullname, newname);
    }
    move_file (fullname, newname);
    New_Job_number = Job_number;
    return (0);
}

/***************************************************************************
 * domove(struct plp_queue *q, int Job_number)
 * move the job
 * 1. Lock the control file.
 * 2. If unsuccessful, find the server PID and kill it off.
 * 3. move the job.
 ***************************************************************************/

static void
domove (struct plp_queue *q, int Job_number) {
    FILE *cfp;

    assert(q!=(struct plp_queue*)0);
    if ((cfp = Lockcf (q->q_name)) == NULL) {
	/* hmmm... looks like an active server */
	if ((cfp = fopen_daemon (q->q_name, "rw")) == NULL) {
	    /* nope, the file has really gone */
	    logerr (XLOG_INFO, "control file %s not read-writable", q->q_name);
	    return;
	}
	/* well, we will just have to kill off the server */
	if (q->q_daemon == 0) {
	    /*
	     * Hmmm... we have this fellow running the file, and it is locked.  That
	     * means that it just started running this guy. Better check again.
	     */
	    (void) Checkactive ();
	}
	if (q->q_daemon) {
	    (void) fprintf (stdout, "Killing off %s server %d.",
			    q->q_server, q->q_daemon);

	    if (killpg (q->q_daemon, SIGINT) < 0) {
		if (Debug > 2)
		    log (XLOG_DEBUG, "server %s (%d) was not alive",
			 q->q_server, q->q_daemon);
	    }
	}
    }
    if (Debug > 3)
	log (XLOG_DEBUG, "domove: moving files for job %s", q->q_name);
    move_job (cfp, q, Job_number);
}

void
move (void) {
    int i;
    struct plp_queue *q;	/* job entry */
    int perms;			/* hold perms values */
    int perms_move;             /* for move permissions */
    int control_perms;		/* has control perms */
    int control_perms_move;     /* for checking if the user can move his job */
    short job_is_there;         /* flag, if the job was found in the queue */
    char *src_Printer;		/* source queue name */
    char buf[MAXPATHLEN];	/* holds the pathname */
    FILE *fp;                   /* for sequence number */
    int j;                      /* command waiting time */
    int Job_number;             /* jobnumber of the job to be moved */
    char *parmnum;              /* pointer on paranumber */
    char paranumber[4];         /* charstring containing the job number */
    char *mem_q_user;           /* pointer on memuser */
    char memuser[4];            /* holds the user name from the controlfile*/

    mem_q_user = memuser;
    parmnum = paranumber;
    control_perms_move = 0;
    job_is_there = 0;

    /*
     * If explicitly asked for debug then echo messages to stdout otherwise (db flag in
     * printcap) just log messages in file.
     */
    Echo_on_stdout = Debug;
    src_Printer = Printer;
    /*
     * get the printcap entry
     */
    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	(void) fprintf (stdout, "Printer or Queue %s does not exist!\n", Parms[1].str);
	(void) fflush (stdout);
	return;
    }
    if (SD == 0 || *SD == 0) {
	if (Debug > 2)
	    log (XLOG_DEBUG, "not a Printer");
	return;
    }
    
    /* Searching the username in the control file */
    Jobcount = Getq ();
    for (i = 0; i < Jobcount; ++i) {
      q = &Queue[i];  /* searching thru the queue */
        sprintf(parmnum, "%d", q->q_num);
      /* searching for the right job number */
      if (!strcmp (parmnum, Parms[0].str)) {
              mem_q_user = q->q_user;
	      job_is_there = 1;
      } 
    }
    /* Flush out error message */
    if (!job_is_there) {
	(void) fprintf (stdout,"Warning: job %d does not exist on printer %s.\n", 
			Parms[0].num, Printer);
	(void) fflush (stdout);
	return;
    }

    /* get the printcap entry of destPrinter*/
    Printer = Parms[1].str;
    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
        (void) fprintf (stdout, "Printer or Queue %s does not exist!\n", Parms[1].str);
        (void) fflush (stdout);
        return;
    }

    new_SD = (char *) malloc (strlen (SD) + 1);
    (void) strcpy (new_SD, SD);

    /*
     * set the flags needed
     */
    Is_local = hostcmp (FQDN, Host) == 0;
    Is_root = strsame (Person, "root");
    
    perms = 'C';                
    perms_move = 'M';           /* check for control or move perms 
				 * on destination queue 
				 */
    if (! (Is_root || 
	   Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) > 0 || 
	   Checkperm (FQDN, Person, First_name, &perms_move, (int *) 0, 0) > 0) ) {
        (void) fprintf (stdout, "Sorry user %s, but you do not have move permission to printer '%s'!\n", Person, Printer);
	(void) fflush (stdout);
        return;
    }

	/* check if the jobowner and the mover are the same - root has 
	   permission too */
    if (! (Is_root || 
	   (! (strcmp (mem_q_user, Person)) && 
	      Checkperm (FQDN, Person, First_name, &perms_move, (int *) 0, 0) > 0 ))) { 
        (void) fprintf (stdout, "Sorry user %s, but you cannot move job %d, owner %s, to printer '%s'!\n", Person, Parms[0].num, mem_q_user, Printer);
	(void) fflush (stdout);
        return;
      }
    control_perms_move = 1;
    Printer = src_Printer;

    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	(void) fprintf (stdout, "Printer or Queue %s does not exist!\n", Printer);
	(void) fflush (stdout);
	return;
    }
    if (SD == 0 || *SD == 0) {
	if (Debug > 2)
	    log (XLOG_DEBUG, "not a Printer");
	return;
    }
    /* chdir to source spool directory */
    chdir_SD ();
    /*
     * check to see that the user has RMJOB privs on this machine
     */
    perms = 'R';		/* must be able to at least use the Printer */
    if (!(Is_root || 
	  Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) > 0) ) {
	(void) fprintf (stdout,
		"Sorry user %s, but you do not have move permission on printer '%s'!\n",
		Person, First_name);
	control_perms = 0;
	return;
    }
    perms = 'C';		/* check for control perms */
    control_perms = 1;
    if (!(Is_root || 
	  control_perms_move == 1 || 
	  Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) > 0)) {
	     control_perms = 0;
    }
    if (Debug > 4) {
	log (XLOG_DEBUG,
	     "move: Is_root %d, Is_local %d, control_perms %d",
	     Is_root, Is_local, control_perms);
    }
    /*
     * get the job queue
     */
    Jobcount = Getq ();
    (void) Checkactive ();
    /*
     * run down list
     */
    (void) fprintf (stdout, "Printer '%s' (%s):\n", Printer, Host);
    (void) fflush (stdout);
    if (Debug > 5)
	log (XLOG_DEBUG, "move: Jobcount = %d", Jobcount);
    /*
     * get the sequence file name; this is needed when the job will be
     * moved in a spool directory where alread a job with this number
     * exists. Better increase the number of the move job up by 1.
     */
    (void) sprintf (buf, "%s/.seq.%s", new_SD, ShortHost);
    if (Debug > 3)
	log (XLOG_DEBUG, "move: sequence file name '%s'", buf);
    /*
     * lock the sequence file and get a new number
     */
    {
	pid_t jobnum_pid_t;

	for (j = 0;
	   (fp = Getlockfile (buf, &jobnum_pid_t, (char *) 0, 0, &LO_statb)) == NULL
	     && j < 3; ++j) {
	    sleep ((unsigned) (j + (getpid () & 1)));
	}
	if (fp == NULL) {
	    Diemsg ("move: cannot lock sequence file %s, try later (%s)", buf,
		    Errormsg (errno));
	}
	Job_number = (int) jobnum_pid_t;
    }
    /*
     * set the sequence number to the new value mod 1000;
     */
    Job_number = (Job_number + 1) % 1000;
    Setlockfile (buf, fp, Job_number, Time_str ());
    (void) fclose (fp);
    if (Debug > 4) 
	log (XLOG_DEBUG, "Move: number %d", Job_number);
    
    for (i = 0; i < Jobcount; ++i) {
	q = &Queue[i];
	if (Debug > 3)
	    log (XLOG_DEBUG, "move: checking %s, %d, %s",
		 q->q_name, q->q_num, q->q_user);
	if (shouldhold (q, control_perms)) {
	    domove (q, Job_number);
/* *** Old style with cf filename displayed: ****
 *	    (void) fprintf (stdout, "move %s, job %d owner %s\n",
 *		    q->q_name, q->q_num, q->q_user);
 */

/*** New style with only the old and the new job number displayed ***/
	    (void) fprintf (stdout,
		    "Moving job %d, owner %s, from printer '%s' to '%s', job %d.\n",
		    q->q_num, q->q_user, Printer, Parms[1].str, New_Job_number);
	    (void) fflush (stdout);
	}
    }

    /*
     * give the server a kick
     */
    (void) Startserver ();

    /* and kick the server that we moved the job to */
    Printer = Parms[1].str;
    (void) Startserver ();
    Printer = src_Printer;
}
