/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: Checkperm.c
 * printer permission checking
 ***************************************************************************
 * Checkperm - determine if a user has permissions to act on a printer queue
 *
 * Input:
 *  permfile - name of the permissions file,
 *    each line of which has the form:
 *		hostname userid printer_name op pr max current
 * 		hostname - the host name of the invoker
 * 		userid - the user name of the invoker
 * 		printer_name - the printer queue to be checked
 *		op -the maximum operation level, A is 'Highest', Z is 'lowest'
 *			Note: R is for LPR, M is for move job,
 *                            C is for LPC, * allows all
 *		pr - the maximum priority level, A is 'Highest', Z is 'lowest'
 *		max - max number of pages allowed
 *		current - current number of pages
 *
 *	# -- comment
 *      #hostname username groupname printername op pr maxpages currentpages
 *      attila    papowell *         imagen      C  A  1000     100
 *      !*        cs9*     *         lpr         *  *
 *      !*        *        cs4       imagen      *  *
 *      *         jsmith   *         imagen      *  *  0        0
 *      *         root     *         *           *  *
 *
 *	Note: * is the wildcard indicator and matches all characters
 *	Note: if maxpages == 0, then page count is unlimited.
 *  Note: if a ! is in the first column  and there is a match
 *    (i.e.- we should grant permission), then permission is refused.
 *    In the example above, this prevents users whose names start with
 *    cs9 will be forbidden access to the lpr queue.
 *
 * Output:
 * 0 - if userid does not have permissions to act on printer queue
 * nonzero - permission granted
 * <0 - permission barred
 *
 * If any of arguments is not specified (i.e.- 0 or NULL value),
 * then checking is not done for the field.
 *****************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "library/filestack.h"
#include "library/filecache.h"
#include "library/utils.h"

#ifdef NIS
#include <rpc/rpc.h>
#include <rpcsvc/ypclnt.h>
#include <rpcsvc/yp_prot.h>
extern int innetgr ();
#endif

/*
 * match( str, pattern )	   returns: 1 = matches, 0 = match failed
 * 
 * null pattern is equivalent to '*'.
 * 
 * -- str is null, cannot match -- ignore (ie. pass) (match returns 1)
 * -- pattern is <fixed>*, str is <fixed><anything> (match returns 1)
 * -- pattern is !<fixed>*, str is <fixed><anything> (match returns 0)
 * -- if not above, match returns 0
 */

static int
match (char *str, char *pattern) {
    int c;

    if (DbgPerms > 9)
	log (XLOG_DEBUG, "match str '%s', pattern '%s'\n", str, pattern);
    if (str == 0) {
	if (DbgPerms > 8)
	    log (XLOG_DEBUG, "match success");
	return (1);
    } else {
	if ((pattern == 0) || strsame (pattern, "*")) {
	    if (DbgPerms > 8)
		log (XLOG_DEBUG, "match success");
	    return (1);
	}
	/* else fall through... */
    }

    while ((c = *pattern) != '\0') {
	if (DbgPerms > 9)
	    log (XLOG_DEBUG, "match partial str '%s', pattern '%s'\n",
		 str, pattern);
	switch (c) {
	    /*
	     * match 0 or more characters in the string
	     */
	case '*':
	    if (pattern[1] == '\0') {
		return (1);	/* AJCD */
	    }
	    if (match (str, pattern + 1)) {
		return (1);
	    }
	    if (*str && match (str + 1, pattern)) {
		return (1);
	    }
	    return (0);
	case '?':
	    if (*str == 0) {
		return (0);
	    }
	    break;
	default:
	    if (c != *str) {
		return (0);
	    }
	    break;
	}
	++pattern;
	++str;
    }
    return (*pattern == *str);
}

static inline void
strcpy_nocase (char *to, char *from) {
    char *x, *y;
    for (x = from, y = to; *x; x++, y++) {
	if (isalpha (*x) && isupper (*x)) {
	    *y = *x + 040;
	} else {
	    *y = *x;
	}
    }
    *y = '\0';
}

/* this is needed for hostnames. */

static int
match_nocase (char *str, char *pattern) {
    char str_lc[MAXPARMLEN + 1];
    char pattern_lc[MAXPARMLEN + 1];
    char *strp, *patternp;

    if (str) {
	strcpy_nocase (str_lc, str); strp = str_lc;
    } else {
	strp = NULL;
    }
    if (pattern) {
	strcpy_nocase (pattern_lc, pattern); patternp = pattern_lc;
    } else {
	patternp = NULL;
    }
    return (match (strp, patternp));
}

static int host_arg_is_unqualified;

/*
 * scan_perms(char *buf) scans a line of permissions and returns status:
 * -1 deny	 	1 allow			0 neither
 */
static int
scan_perms (char *buf, char *host, char *user, char *groups, char *printerq,
	int *op, int *pr_level, int pages)
{
    char hostname[MAXPARMLEN + 1];
    char username[MAXPARMLEN + 1];
    char groupname[MAXPARMLEN + 1];
    char printerqname[MAXPARMLEN + 1];
    char operation[MAXPARMLEN + 1];
    char priority[MAXPARMLEN + 1];
    int max, current;
    char *str, *fq_host;
    int i;
    int invert, must_match;
    int grp_found;

    must_match = invert = 0;
    current = max = 0;
    priority[0] = 0;
    operation[0] = 0;

    str = buf;
    switch (buf[0]) {
    case '!':
	invert = 1;
	++str;
	break;
    case '~':
	must_match = 1;
	++str;
	break;
    }

    /* sscanf returns number of fields converted */
    i = sscanf (str, "%s%s%s%s%s%s%d%d", hostname, username, groupname,
		printerqname, operation, priority, &max, &current);
    if (i == 0) {
	return 0;
    } else if (i < 4) {
	fatal (XLOG_CRIT, "Error in perms file! line='%s'", buf);
	return 0;
    }
    if (DbgPerms > 6)
	log (XLOG_DEBUG,
	     "host=%s,user=%s,group=%s,printer=%s,oper=%s,priority=%s,max%d,current%d",
	hostname, username, groupname, printerqname, operation, priority, max, current);

    /*
     * if the hostname field is '*', this means any host can do it otherwise we should
     * check to see if a valid host name. If the first character of the hostname is '@',
     * we check if the host is in the netgroup specified.
     */
#ifdef NIS
    if (hostname[0] == '@') {	/* look up machine in netgroup */
	if (!innetgr (hostname + 1, host, NULL, NULL)) {
	    if (must_match) {
		return -1;
	    }
	    return 0;
	}
    } else
#endif /* NIS */

    fq_host = NULL;
    if (host_arg_is_unqualified && host && *host) {
	char *domain;

	/* if there's a domain in the hostname field of the perms
	 * file, and we've got an unqualified hostname in the control
	 * file, try appending the domain to the unqualified hostname
	 * and testing it with gethostbyname(). If ghbn works, we'll
	 * append the domain and pass it on to the matching routine;
	 * otherwise, we'll treat it as a failure.
	 */

        if ((domain = strchr (hostname, '.')) != NULL) {
	    if (fq_host = host_in_domain (host, domain)) {
		host = fq_host;

	    } else {
		/* the perms file has a domain, but the unqualified
		 * hostname is not in this domain -> not a match.
		 */
		free (fq_host);
		if (DbgPerms > 6)
		    log (XLOG_DEBUG, "scan_perms: host bad, %s, %s",
			 host ? host : "<NONE>", hostname);
		if (must_match) {
		    return -1;
		}
		return 0;
	    }
	}
    }

    if (!match_nocase (host, hostname)) {
	if (DbgPerms > 6)
	    log (XLOG_DEBUG, "scan_perms: host bad, %s, %s",
		 host ? host : "<NONE>", hostname);
	if (must_match) {
	    return -1;
	}
	return 0;
    }
    if (DbgPerms > 6)
	log (XLOG_DEBUG, "scan_perms: host ok, %s, %s",
	     host ? host : "<NONE>", hostname);

    if (fq_host) {
	free (fq_host);
	/* note that we may not be able to use host from here on in. */
    }

    /*
     * compare the user names. If first character of username is '@', test if in
     * netgroup.
     */
#ifdef NIS
    if (username[0] == '@') {	/* look up user in netgroup */
	if (!innetgr (username + 1, NULL, user, NULL)) {
	    if (must_match) {
		return -1;
	    }
	    return 0;
	}
    } else
#endif /* NIS */
    if (!match (user, username)) {
	if (DbgPerms > 6)
	    log (XLOG_DEBUG, "scan_perms: user bad, %s, %s",
		    user ? user : "<NONE>", username);
	if (must_match) {
	    return -1;
	}
	return 0;
    }
    if (DbgPerms > 6)
	log (XLOG_DEBUG, "scan_perms: user ok, %s, %s",
	     user ? user : "<NONE>", username);

    /* compare the group names */
    if (groups[0]) {
	grp_found = 0;
	while (*groups && !grp_found) {
	    grp_found += match (groups, groupname);
	    if (!grp_found) {
		groups += strlen (groups) + 1;
	    }
	}
    } else
	grp_found = match (0, groupname);
    if (!grp_found) {
	if (DbgPerms > 6)
	    log (XLOG_DEBUG, "scan_perms: group bad, %s, %s",
		 groups[0] ? groups : "<NONE>", groupname);
	if (must_match) {
	    return -1;
	}
	return 0;
    }
    if (DbgPerms > 6)
	log (XLOG_DEBUG, "scan_perms: group ok, %s, %s",
	     groups[0] ? groups : "<NONE>", groupname);

    /* compare the printer queue names */
    if (!match (printerq, printerqname)) {
	if (DbgPerms > 6)
	    log (XLOG_DEBUG, "scan_perms: printerq bad, %s, %s",
		 printerq ? printerq : "<NONE>", printerqname);
	if (must_match) {
	    return -1;
	}
	return 0;
    }
    if (DbgPerms > 6)
	log (XLOG_DEBUG, "scan_perms: printerq ok, %s, %s",
	     printerq ? printerq : "<NONE>", printerqname);

    /* compare the operations */
    if (op && operation[0] != '*' && *op < operation[0]) {
	if (DbgPerms > 6)
	    log (XLOG_DEBUG, "scan_perms: op bad, %c, %c",
		 op ? *op : '?', operation[0]);
	if (must_match) {
	    return -1;
	}
	return 0;
    }
    if (DbgPerms > 6)
	log (XLOG_DEBUG,
	     "scan_perms: OK, must_match %d, wanted %c(%d), have %c(%d), invert %d",
	     must_match, op ? *op : '?', operation[0], op ? *op : '?', operation[0], invert);

    /*
     * Well, you had to have a match, and you did.  So this means that you passed, but
     * you do not use this entry.
     */
    if (must_match) {
	return 0;
    }
    if (DbgPerms > 6)
	log (XLOG_DEBUG,
	     "scan_perms: pages- %d, max %d, current %d", pages, max, current);
    /* So far, so good. Now a couple of final checks before allowing */
    if (pages && max && current > max) {
	/* page limit exceeded in first matching entry: bad news */
	return -1;
    }
    if (pr_level && priority[0] && priority[0] != '*') {
	/* we want to check priory and there are limits */
	if (priority[0] > *pr_level) {
	    /* sorry, this is too high */
	    *pr_level = priority[0];
	}
    }
    /* the final hurdle; did it allow or deny you? */
    if (invert) {
	return -1;
    }
    return 1;
}

#ifdef NIS
static int
perms_nis (char *outval, char *host, char *user, char *group,
	char *printerq, int *op, int *pr_level, int pages)
{
    char *np;
    int permission;

    permission = 0;

    if (outval == 0) {
	logerr (XLOG_INFO, "perms_nis: no NIS match");
	return 0;
    }
    if (DbgPerms > 5)
	log (XLOG_DEBUG,
	     "perms_nis host=%s,user=%s,group=%s,printerq=%s,op=%c,pr_level=%c,pages=%d",
	     host ? host : "NONE", user ? user : "NONE",
	     group[0] ? group : "NONE", printerq ? printerq : "NONE",
	     op ? *op : '?', pr_level ? *pr_level : '?', pages);

    if (strchr (host, '.') == NULL) {
        host_arg_is_unqualified = 1;
    } else {
        host_arg_is_unqualified = 0;
    }

    /*
     * scan through the list looking for a match outval is a string consisting of
     * newline-separated entries
     */
    permission = 0;
    while (permission == 0 && outval && *outval) {
	if ((np = strchr (outval, '\n')))
	    *np++ = '\0';	/* terminate current entry */
	if (*outval && (*outval == '\n' || *outval == '#')) {
	    outval = np;
	    continue;
	}
	permission = scan_perms (outval, host, user, group, printerq,
				 op, pr_level, pages);
	outval = np;
    }

    if (DbgPerms > 4)
	log (XLOG_DEBUG, "perms_nis %d, printerq %s for %s@%s",
	     permission,
	     printerq ? printerq : "NONE", user ? user : "NONE", host ? host : "NONE");

    return (permission);
}

/*
 * Checkperm_NIS checks the NIS database for permissions entries The keys
 * "printer_perms.printerq", "printer_perms.host" and "printer_perms" are tried until a
 * match is found.
 */
static int
Checkperm_NIS (char *host, char *user, char *group, char *printerq,
	int *op, int *pr_level, int pages) {
    int res = 0;		/* result */
    char *np;			/* next character pointer */
    char *outval;
    int outvallen;
    char *stddomain;
    char buf[YPMAXMAP];		/* NIS map */

    if (yp_get_default_domain (&stddomain))
	return 0;

    /* Try NIS lookup for permissions */
    (void) strcpy (buf, "printer_perms");
    np = buf + strlen (buf);

    /* tag on queue name */
    *np = PNAME_SEPCH;
    (void) strcpy (np + 1, printerq);

    if (!NIS_printcap_byname || !*NIS_printcap_byname)
	logerr_die (XLOG_CRIT, "No NIS printcap byname map!");

    if (yp_match (stddomain, NIS_printcap_byname, buf, strlen (buf),
		  &outval, &outvallen) == 0) {
	res = perms_nis (outval, host, user, group, printerq, op,
			 pr_level, pages);
	free (outval);
    }
    if (!res) {
	/* tag on host name */
	(void) strcpy (np + 1, ShortHost);

	if (yp_match (stddomain, NIS_printcap_byname, buf, strlen (buf),
		      &outval, &outvallen) == 0) {
	    res = perms_nis (outval, host, user, group, printerq,
			     op, pr_level, pages);
	    free (outval);
	}
    }
    if (!res) {
	/* try just printer_perms */
	*np = '\0';

	if (yp_match (stddomain, NIS_printcap_byname, buf, strlen (buf),
		      &outval, &outvallen) == 0) {
	    res = perms_nis (outval, host, user, group, printerq,
			     op, pr_level, pages);
	    free (outval);
	}
    }
#ifndef YP_GET_DOMAIN_BROKEN
    free (stddomain);
#endif /* NOT defined YP_GET_DOMAIN_BROKEN */
    return (res);
}

#endif /* NIS */

#ifdef HESIOD
static int
perms_hes (char **hesptr, char *host, char *user, char *group, char *printerq,
	int *op, int *pr_level, int pages) {
    int permission;

    permission = 0;

    if (hesptr == 0) {
	logerr (XLOG_INFO, "perms_hes: no Hesiod match");
	return 0;
    }
    if (DbgPerms > 5)
	log (XLOG_DEBUG,
	     "perms_hes host=%s,user=%s,group=%s,printerq=%s,op=%c,pr_level=%c,pages=%d",
	     host ? host : "NONE", user ? user : "NONE",
	     group[0] ? group : "NONE", printerq ? printerq : "NONE",
	     op ? *op : '?', pr_level ? *pr_level : '?', pages);

    if (strchr (host, '.') == NULL) {
        host_arg_is_unqualified = 1;
    } else {
        host_arg_is_unqualified = 0;
    }

    /*
     * scan through the entries looking for a match
     */
    permission = 0;
    while (permission == 0 && *hesptr != NULL) {
	if (*hesptr[0] == '\n' || *hesptr[0] == '#') {
#ifdef HESIOD_FREE
	    (void) free (*hesptr++);
#else
	    hesptr++;
#endif /* HESIOD_FREE  */
	    continue;
	}
	permission = scan_perms (*hesptr, host, user, group, printerq,
				 op, pr_level, pages);
#ifdef HESIOD_FREE
	(void) free (*hesptr++);
#else
	hesptr++;
#endif /* HESIOD_FREE */
    }
#ifdef HESIOD_FREE
    while (*hesptr)		/* free remaining pointers */
	(void) free (*hesptr++);
#endif /* HESIOD_FREE */

    if (DbgPerms > 4)
	log (XLOG_DEBUG, "perms_hes %d, printerq %s for %s@%s",
	     permission,
	     printerq ? printerq : "NONE", user ? user : "NONE", host ? host : "NONE");

    return (permission);
}

/*
 * Checkperm_hes checks the Hesiod database for permissions entries The keys
 * "printer_perms.printerq", "printer_perms.host" and "printer_perms" are tried until a
 * match is found
 */
static int
Checkperm_hes (char *host, char *user, char *group, char *printerq,
	int *op, int *pr_level, int pages) {
    int res = 0;		/* result */
    char **hp;			/* hesiod match pointer */
    char *np;			/* next character pointer */
    char buf[HES_MAXKEYLEN];	/* Hesiod key */

    /* Try Hesiod lookup for permissions */
    (void) strcpy (buf, "printer_perms");
    np = buf + strlen (buf);

    /* tag on queue name */
    *np = PNAME_SEPCH;
    (void) strcpy (np + 1, printerq);

    if (!Hesiod_printcap_key || !*Hesiod_printcap_key) {
	logerr_die (XLOG_CRIT, "hesiod-printcap-key not set!");
    }
    if ((hp = hesiod_match (buf, Hesiod_printcap_key, NULL))) {
	res = perms_hes (hp, host, user, group, printerq, op,
			 pr_level, pages);
    }
    if (!res) {
	/* tag on host name */
	(void) strcpy (np + 1, ShortHost);

	if ((hp = hesiod_match (buf, Hesiod_printcap_key, NULL))) {
	    res = perms_hes (hp, host, user, group, printerq, op,
			     pr_level, pages);
	}
    }
    if (!res) {
	/* try just permissions */
	*np = '\0';

	if ((hp = hesiod_match (buf, Hesiod_printcap_key, NULL))) {
	    res = perms_hes (hp, host, user, group, printerq, op,
			     pr_level, pages);
	}
    }
    return (res);
}

#endif /* HESIOD */

#ifdef LOCAL

static struct file_cache perm_cache;
static int reading_perm_from_cache;

static char *
perm_entry_file (char *buf, int size, FILE *permfile) {
    if (!Use_perm_cache) {
        if (DbgPerms > 6)
            log (XLOG_DEBUG, "perms caching disabled");
	assert (permfile != NULL);
	return (Gets (buf, size, &permfile, Permfile_include_path));

    } else if (!reading_perm_from_cache) {
	/* a per-printer permfile -- we don't cache these */
	assert (permfile != NULL);
	return (Gets (buf, size, &permfile, Permfile_include_path));
    }

    if (perm_cache.size == 0) {
        fatal (XLOG_NOTICE, "perm cache is unset!");
    }
    if (perm_cache.entries[perm_cache.ind] == NULL) {
        /* there's nothing more to read */
        rewind_cache (&perm_cache);
        return NULL;
    }

    (void) strncpy (buf, perm_cache.entries[perm_cache.ind], size - 1);
    if (DbgPerms > 7) {
        log (XLOG_DEBUG, "in cache: got entry %d [%s]", perm_cache.ind, buf);
    }
    perm_cache.ind++;

    return buf;
}

static int
perms_file (FILE *permfile, char *host, char *user, char *group,
	char *printerq, int *op, int *pr_level, int pages) {
    int permission;
    char buf[BUFSIZ];

    permission = 0;

    if (DbgPerms > 5)
	log (XLOG_DEBUG,
	   "perms_file host=%s,user=%s,group=%s,printerq=%s,op=%c,pr_level=%c,pages=%d",
	     host ? host : "NONE", user ? user : "NONE",
	     group[0] ? group : "NONE", printerq ? printerq : "NONE",
	     op ? *op : '?', pr_level ? *pr_level : '?', pages);

    if (host && (strchr (host, '.') == NULL)) {
        host_arg_is_unqualified = 1;
    } else {
        host_arg_is_unqualified = 0;
    }

    /*
     * scan through the file looking for a match
     */
    permission = 0;
    while (permission == 0 &&
	perm_entry_file (buf, sizeof (buf), permfile) != NULL)
    {
	if (buf[0] == '\n' || buf[0] == '#') {
	    continue;		/* skip comments, blank lines */
	}
	permission = scan_perms (buf, host, user, group, printerq,
				 op, pr_level, pages);
    }

    if (DbgPerms > 4)
	log (XLOG_DEBUG, "perms_file %d, printerq %s for %s@%s",
			permission, printerq ? printerq : "NONE",
			user ? user : "NONE", host ? host : "NONE");
    if (permfile != NULL) {
	(void) fclose (permfile);
    }

    return (permission);
}

/*
 * Checkperm_file checks the XU variable and then the permfile for permissions
 */
static int
Checkperm_file (char *host, char *user, char *group, char *printerq,
	int *op, int *pr_level, int pages) {
    int res = 0;		/* result of first check */
    char *pf = XU;
    FILE *permfile;

    reading_perm_from_cache = 0;
    if (pf && *pf) {
	char buf[MAXPATHLEN];

	if (*pf != '/') {
	    (void) sprintf (buf, "%s/%s", SD, pf);
	    pf = buf;
	}
	if ((permfile = fopen (pf, "r")) != NULL) {
	    res = perms_file (permfile, host, user, group, printerq,
			      op, pr_level, pages);
	} else {
	    log (XLOG_INFO, "cannot open XU file %s: %s", pf, Errormsg(errno));
	}
    }
    if (!res) {
	if (Use_perm_cache && perm_cache.entries[0]) {
	    /* we have a cache, use it. */
	    reading_perm_from_cache = 1;
	    rewind_cache (&perm_cache);
	    if (DbgPerms > 2)
		log (XLOG_DEBUG, "checkperm: reading perms from cache");
	    permfile = NULL;

	} else {
	    if (DbgPerms > 2)
		log (XLOG_DEBUG, "checkperm: reading perms from file");
	    if ((permfile = fopen_path (Permfile_path, "r")) == NULL) {
		fatal (XLOG_INFO, "failed to open permissions file: %s", Errormsg(errno));
	    }
	}
	res = perms_file (permfile, host, user, group, printerq,
			  op, pr_level, pages);
    }
    return (res);
}

void build_perm_cache (void) {
    char buf[BUFSIZ];
    FILE *permfile;
    struct file_cache *perm_cache_ref;

    if (!Use_perm_cache) {
	if (DbgPerms > 6)
	    log (XLOG_DEBUG, "perms caching disabled");
	return;
    }

    perm_cache_ref = &perm_cache; chkandrealloc_cache (&perm_cache_ref);
    flush_cache (&perm_cache);

    if ((permfile = fopen_path (Permfile_path, "r")) == NULL) {
	fatal (XLOG_INFO, "failed to open permissions file: %s",
		Errormsg(errno));
    }

    while (Gets (buf, sizeof (buf), &permfile, Permfile_include_path) != NULL)
    {
	if (buf[0] == '\n' || buf[0] == '#') {
	    continue;
	}
	if (DbgPerms > 7)
	    log (XLOG_DEBUG, "caching perm entry %d [%s]", perm_cache.ind, buf);

	perm_cache_ref = &perm_cache; chkandrealloc_cache (&perm_cache_ref);
	cache_addline (&perm_cache, buf);
    }
    cache_addeof (&perm_cache);
    rewind_cache (&perm_cache);
    fclose (permfile);
}

void
create_perm_cache (void) {
    if (!Use_perm_cache) {
        if (DbgPerms > 6)
            log (XLOG_DEBUG, "perms caching disabled");
        return;
    }
    perm_cache.size = 0;
    build_perm_cache ();
}

void
flush_perm_cache (void) {
    if (!Use_perm_cache) {
	return;
    }
    if (DbgPerms > 2)
	log (XLOG_DEBUG, "flushing perm cache");

    build_perm_cache ();
}

#endif /* LOCAL */

/*
 * Checkperm checks the permissions according to the current configuration. Files,
 * Hesiod, and NIS are consulted until a deny or allow is found. Only local domains are
 * searched for permissions.
 */
int
Checkperm (char *host, char *user, char *printerq, int *op,
	int *pr_level, int pages) {
    int res = 0;
    int i;
    char groups[NGROUPS_MAX * GRP_NAM_MAX];	/* groupnames to check for */

#if (defined(NIS) || defined(HESIOD)) && defined(CROSS_DOMAIN)
    char qname[MAXPARMLEN];	/* printer name plus domain */

    if (out_domain && printerq) {
	(void) strcpy (qname, printerq);	/* add "@domain" to printer name */
	(void) strcat (qname, "@");
	(void) strcat (qname, out_domain);
	printerq = qname;
    }
#endif /* (defined(NIS) || defined(HESIOD)) && defined(CROSS_DOMAIN) */

    groups[0] = 0;
    if (user) {			/* AJCD: get user's group for use in perms file */
	struct passwd *pwent;
	struct group *grent;
	char *gp = groups;

	if (((pwent = getpwnam (user)) != NULL) &&
	    ((grent = getgrgid (pwent->pw_gid)) != NULL)) {
	    (void) strcpy (gp, grent->gr_name);
	    gp += strlen (grent->gr_name) + 1;	/* primary group */
	    while ((grent = getgrent ())) {
		char **mem;
		for (mem = grent->gr_mem; mem && *mem; mem++) {
		    if (strsame (*mem, user) &&
				((gp - groups) < (NGROUPS_MAX - 1) * GRP_NAM_MAX))
		    {
			(void) strcpy (gp, grent->gr_name);
			gp += strlen (grent->gr_name) + 1;
		    }
		}
	    }
	    (void) endgrent ();
	    *gp = 0;
	}
    }
    if (DbgPerms > 5)
	log (XLOG_DEBUG, "Checkperm: checking permissions for user %s, primary group %s, printer %s, host %s",
	     user ? user : "(none)", groups[0] ? groups : "(none)",
	     printerq ? printerq : "(none)",
	     host ? host : "(none)");

    for (i = 0; i < num_confs && res == 0; i++)
	if (conf[i].domain == NULL)	/* only check in local domain */
	    switch (conf[i].type) {
#ifdef LOCAL
	    case LOCAL_TYPE:
		res = Checkperm_file (host, user, groups,
				      printerq, op, pr_level, pages);
		break;
#endif /* LOCAL */
#ifdef NIS
	    case NIS_TYPE:
		res = Checkperm_NIS (host, user, groups,
				     printerq, op, pr_level, pages);
		break;
#endif /* NIS */
#ifdef HESIOD
	    case HESIOD_TYPE:
		res = Checkperm_hes (host, user, groups,
				     printerq, op, pr_level, pages);
		break;
#endif /* HESIOD */
	    default:
		log (XLOG_INFO, "Cannot lookup permissions of type %d.\n",
			 conf[i].type);
	    }

    return (res);
}
