/*
 * setuid.c: routines to manipulate user-ids securely (and hopefully, portably). The
 * internals of this are very hairy, because (a) there's lots of sanity checking (if -D
 * is active, at least); and (b) there's at least three different setuid-swapping
 * semantics to support :(
 * 
 * Note that the main function saves the errno, then restores it afterwards; this means it's
 * safe to do "root_to_user();some_syscall();user_to_root();" and use the errno.
 * 
 * "root" is the user who owns the setuid executable (privileged). "user"
 * is the user who runs it. "daemon" is an arbitrary user who owns data files used
 * internally by the PLP utilities (spool directories, etc).
 * 
 * root_to_user();	-- set euid from root to user
 * user_to_root();	-- set euid from user to root
 * root_to_daemon();	-- set euid from root to daemon
 * daemon_to_root();	-- set euid from daemon to root
 * full_user_perms()	-- set euid *and uid* to user -- lose all privileges
 * (also full_daemon_perms(), full_root_perms())
 * 
 * - Justin Mason <jmason@iona.ie> May '94.
 */

#include "lp.h"
#include "library/errormsg.h"

#ifdef HAVE_VFORK_H
#include <vfork.h>
#endif

uid_t UserUID;			/* the real uid */
/* there's also DaemonUID, uid for "daemon" user, set by Get_Daemon() */

#define I_AM_UNSET              0
#define I_AM_ROOT               1
#define I_AM_USER               2
#define I_AM_DAEMON             3

static char *i_am_str[] = {"(unset)", "root", "user", "daemon"};
static int am_i_root = I_AM_UNSET;	/* a flag for the *_daemon functions */

#ifdef HAVE_SETEUID
#ifdef HAVE_SETREUID
#undef HAVE_SETEUID	/* probably emulated by setreuid() (mj) */
#endif
#endif

/* yuck -- HP/UX cpp doesn't seem to support #elif. duh. */
#ifdef HAVE_SETEUID
# define setup_for_changes() (0)		/* no-op */
# define change_euid(to) (seteuid (to))
#else
# ifdef HAVE_SETRESUID
#  define setup_for_changes() (0)		/* no-op */
#  define change_euid(to) (setresuid (-1, to, -1))
# else
#  ifdef HAVE_SETREUID
#   define setup_for_changes() (setreuid (0, 0)) /* argh */
#   define change_euid(to) (setreuid (-1, to))
#  else
    XXXXX I do not know how to change the EUID!!
#  endif
# endif
#endif

/* make sure we get the correct behaviour; don't rely on setuid(). */
#ifdef HAVE_SETRESUID
# define setuid(to) (setresuid (to,to,to))
#else
# ifdef HAVE_SETREUID
#  define setuid(to) (setreuid (to,to))
/* else just use setuid */
# endif
#endif

/**************************************************************/

static void 
seteuid_wrapper (uid_t from, uid_t to, int new_state) {
    int tmp_errno;

#ifdef NON_SETUID
    if (Debug > 6) log (XLOG_DEBUG, "set-euid: setuid mode is disabled for testing.");
    return;

#else
    if (from == to) {
	am_i_root = new_state;
	return;
    }
    tmp_errno = errno;		/* save it */

    if (Debug > 6) {
	log (XLOG_DEBUG, "set-euid: from (%d, %s) to (%d, %s)",
	     from, i_am_str[am_i_root], to, i_am_str[new_state]);

	if (from != geteuid ()) {	/* do this sanity check if we're debugging */
	    log (XLOG_DEBUG, "set-euid: from (%d, %s) is not euid! (euid=%d)",
		 from, i_am_str[am_i_root], geteuid ());
	}
    }
    if (Debug > 7)
	log (XLOG_DEBUG, "before euid change: (%d, %d)", getuid (), geteuid ());

    if (from != 0) {
	if (change_euid (0) < 0) {
	    logerr_die (XLOG_ERR, "change-euid failed (fs mounted with -nosuid option?)");
	}
	if (Debug > 7)
	    log (XLOG_DEBUG, "from->0 result: (%d, %d)", getuid (), geteuid ());
    }
    if (to != 0) {
	if (change_euid (to) < 0) {
	    logerr_die (XLOG_INFO, "change-euid failed");
	}
	if (Debug > 7)
	    log (XLOG_DEBUG, "0->to result: (%d, %d)", getuid (), geteuid ());
    }
    am_i_root = new_state;
    errno = tmp_errno;		/* restore it */
#endif
}

inline void root_to_user ()   { seteuid_wrapper (0, UserUID, I_AM_USER); }
inline void root_to_daemon () { seteuid_wrapper (0, DaemonUID, I_AM_DAEMON); }
inline void user_to_root ()   { seteuid_wrapper (UserUID, 0, I_AM_ROOT); }
inline void user_to_daemon () { seteuid_wrapper (UserUID, DaemonUID, I_AM_DAEMON); }
inline void daemon_to_root () { seteuid_wrapper (DaemonUID, 0, I_AM_ROOT); }
inline void daemon_to_user () { seteuid_wrapper (DaemonUID, UserUID, I_AM_USER); }

/*************************************************************
 * set both uid and euid to the same value, using setuid().
 */

static void 
setuid_wrapper (uid_t to) {
#ifdef NON_SETUID
    if (Debug > 6) log (XLOG_DEBUG, "set-uid: setuid mode is disabled for testing.");
    return;

#else
    if (Debug > 6) {
	log (XLOG_DEBUG, "set-uid: from (%d, %s) to uid=%d, euid=%d (no setuid privs)",
	     geteuid(), i_am_str[am_i_root], to, to);
    }
    if (change_euid (0) < 0) {
    	/* we may need to have root perms to setuid(), on SVR4 at least */
	logerr_die (XLOG_INFO, "seteuid to root failed");
    }
    if (setuid (to) < 0) {
	logerr_die (XLOG_INFO, "setuid failed");
    }
    /* okay, uid and euid are both set now. We can exec safely */
    if (Debug > 6) {
        log (XLOG_DEBUG, "after setuid: (%d, %d)", getuid (), geteuid ());
    }
#endif
}

inline void full_user_perms () { setuid_wrapper (UserUID); }
inline void full_daemon_perms () { setuid_wrapper (DaemonUID); }
inline void full_root_perms () { setuid_wrapper ((uid_t) 0); }

/****************************************************************************
 * Get the _real_ real-user-id (since setreuid() needs ruid to be 0 for our
 * euid acrobatics). Save it, then set ruid to be zero, if that's necessary
 * for later use. This is safe, as the euid is used by all the functions
 * we use for access checking and so on, and we reset both euid and ruid
 * before execing anything else.
 ***************************************************************************/

void 
set_user_uid (int is_this_lpd) {
#ifdef IS_AUX
    /* A/UX needs this to be more BSD-compatible. */
    setcompat (COMPAT_BSDPROT);
    set42sig();
#endif

#ifdef NON_SETUID
    /* testing mode -- no setuid changes */
    UserUID = DaemonUID = getuid ();

    /* use unprivileged ports in the socket code */
    Maxportno = 60000;
    Minportno = IPPORT_RESERVED;

#else
    if (geteuid () == 0) {
	/* our effective user id == 0, therefore: (a) run as root, or
	 * (b) run as another user (but the executable has the setuid bit set)
	 */
	if (is_this_lpd) {
	    UserUID = DaemonUID;
	} else {
	    UserUID = getuid ();
	}
	if (setup_for_changes () < 0) {
	    logerr_die (XLOG_INFO, "set ruid and euid failed (fs mounted -nosuid?)");
	}
	/* the application code must call root_to_user () after we return */

    } else {
	/* we weren't installed setuid, or the setuid perms on the executable
	 * haven't worked (due to -nosuid), so we can't change our euid. */
	fatal (XLOG_CRIT,
	    "executable's setuid bit unset or fs is mounted -nosuid!");
    }
#endif
}

void
Setuid_debug (char *progname) {
    /* this is necessary because Debug isn't set 'til AFTER the EUID
     * changes (minimise the "I am root" area just after main()).
     */
    if (Debug > 6) {
	log (XLOG_DEBUG, "%s: setup for EUID changes: User=%d, Daemon=%d, Root=0; current=%d",
		progname, UserUID, DaemonUID, geteuid());
    }
}

/****************************************************************************
 * set up a safe environment for filters.
 * consists of:
 *  LD_LIBRARY_PATH (safe value), PATH (safe value), TZ (from env), 
 *  USER, LOGNAME, HOME, LOGDIR, SHELL, IFS (safe value).
 ***************************************************************************/

static char *safe_env[MAXUSERENVIRON];

static void
setup_env (void) {
    struct passwd *pw;
    char **ep;
    int i;

    i = 0;

    if ((pw = getpwuid (getuid ())) == 0) {
	logerr_die (XLOG_INFO, "setup_env: getpwuid(%d) failed", getuid ());
    }
    safe_env[i] = (char *) malloc (6 + strlen (pw->pw_name));
    (void) sprintf (safe_env[i], "USER=%s", pw->pw_name);
    i++;

    safe_env[i] = (char *) malloc (9 + strlen (pw->pw_name));
    (void) sprintf (safe_env[i], "LOGNAME=%s", pw->pw_name);
    i++;

    safe_env[i] = (char *) malloc (6 + strlen (pw->pw_dir));
    (void) sprintf (safe_env[i], "HOME=%s", pw->pw_dir);
    i++;

    safe_env[i] = (char *) malloc (8 + strlen (pw->pw_dir));
    (void) sprintf (safe_env[i], "LOGDIR=%s", pw->pw_dir);
    i++;

    safe_env[i] = (char *) malloc (6 + strlen (Filter_path));
    (void) sprintf (safe_env[i], "PATH=%s", Filter_path);
    i++;

    safe_env[i] = (char *) malloc (18 + strlen (Filter_LD_path));
    (void) sprintf (safe_env[i], "LD_LIBRARY_PATH=%s", Filter_LD_path);
    i++;

    safe_env[i] = (char *) malloc (14);
    (void) strcpy (safe_env[i], "SHELL=/bin/sh");
    i++;

    safe_env[i] = (char *) malloc (7);
    (void) sprintf (safe_env[i], "IFS= \t");
    i++;

    /* copy in what was saved before -- this has already been checked. */
    for (ep = SafeEnviron; *ep != NULL; ep++) {
	safe_env[i++] = *ep;
    }
    safe_env[i++] = NULL;
    if (Debug > 5) {
	for (i = 0; safe_env[i]; i++) {
	    log (XLOG_DEBUG, "dump_envs: %s", safe_env[i]);
	}
    }
}

/***************************************************************************/

static char metachars[] = "<>|`;";	/* whinge if these crop up */
static char blanks[] = " \t\r\n";
static char meta_or_quotes_or_blanks[] = " \t\r\n<>|`;'\"";

/* strndup - make copy of string */

static char *strndup(char *str, int len) {
    char   *dst;

    if ((dst = (char *) malloc(len + 1)) == 0)
        logerr_die (XLOG_NOTICE, Malloc_failed_msg);
    dst[len] = 0;
    return ((char *) strncpy(dst, str, len));
}

static char *find_executable (char *execname) {
    static char exec_path[MAXPATHLEN];
    char *tmppath, *step;
    struct stat stbuf;

    if ((*execname == '/') || (!Filter_path || !*Filter_path)) {
	(void) strcpy (exec_path, execname);
        return (execname);
    }

    tmppath = strdup (Filter_path);
    /* we can now rip tmppath to shreds with impunity */

    step = (char *) strtok (tmppath, ":");
    do {
	(void) sprintf (exec_path, "%s/%s", step, execname);

        if (stat (exec_path, &stbuf) < 0)
	    continue;

	if (Debug > 4) {
	    log (XLOG_DEBUG, "%s exists", exec_path);
	}
	 
	/* check to see if we can execute it. */
	if (((stbuf.st_uid == geteuid()) && (stbuf.st_mode & S_IXUSR)) ||
	    ((stbuf.st_gid == getegid()) && (stbuf.st_mode & S_IXGRP)) ||
	    (stbuf.st_mode & S_IXOTH))
	{
	    if (Debug > 4) {
		log (XLOG_DEBUG, "%s is executable", exec_path);
	    }
	    free (tmppath);
	    return (exec_path);
	}

    } while ((step = (char *) strtok (NULL, ":")));

    (void) strcpy (exec_path, execname);
    free (tmppath);
    return (execname);
}

static int 
ss_fork_and_or_exec (char **args, int fork_flag) {
    char *executable;		/* the path to argv[0] */
    plp_status_t stat;		/* daughter status */
    pid_t i, pid;

    if (fork_flag && (pid = vfork ())) {
	if (pid < 0) {
	    logerr_die (XLOG_INFO, "secure_system: fork failed");
	}
	i = plp_waitpid (pid, &stat, WUNTRACED);

	if (Debug > 3) {
	    log (XLOG_DEBUG, "secure_system: %d finished (%s)",
		 i, Decode_status (&stat));
	}

	return (PLP_WTERMSIG(stat) | PLP_WCOREDUMP(stat)
		| (PLP_WEXITSTATUS(stat) << 8));
    }
    setup_env ();		/* set up the environment variables. */
    executable = find_executable (args[0]);	/* find the command's executable. */

    execve (executable, args, safe_env);

    /* if execve succeeds, we'll never get this far. */
    logerr (XLOG_INFO, "couldn't run '%s'!", executable);

#ifdef HAVE__EXIT
    /* use _exit instead of exit 'cos of vfork. */
    _exit (2);
#else
    exit (2);
#endif
    return (0);		/* we don't actually get this far */
}

static char *
word_in_quotes (char **string, char ch, int *len) {
    char *ind, *word;

    (*string)++;
    if ((ind = strchr(*string, ch)) == NULL) {
	logerr (XLOG_DEBUG, "unmatched quote char (%c) in command line", ch);
	return (NULL);
    }
    *len = ind - *string;
    word = strndup (*string, *len);
    *string = ind + 1;
    return (word);
}

#define C_TICKS	'\''
#define C_QUOTE	'"'

static int 
secure_system_backend (char *cmd, int fork_flag) {
    char *args[BUFSIZ];
    int argno = 0;
    int len;
    char *tmpword;

    if (Debug > 4) {
	log (XLOG_DEBUG, "secure-system: about to exec '%s'", cmd);
    }

    while (*cmd) {
        if (argno >= BUFSIZ - 1) {		/* Sanity check. */
            logerr_die (XLOG_NOTICE, "too many arguments");
            exit (1);
        }

        cmd += strspn(cmd, blanks);	/* Skip leading whitespace. */

        /* Process next token. */

	if (*cmd == '\0') {
	    break;

	} else if (strchr(metachars, *cmd)) {	/* log it and don't exec the command. */
	    log (XLOG_INFO, "secure_system: attempted shell metachars: '%c'(%d)",
	   	 *cmd, *cmd);
	    return (1);

	} else if (*cmd == C_TICKS) {	/* word(s)+space in 'ticks' */
            if (!(args[argno++] = word_in_quotes (&cmd, C_TICKS, &len))) {
		return (1);
	    }

	} else if (*cmd == C_QUOTE) {	/* word(s)+space in "quotes" */
            if (!(args[argno++] = word_in_quotes (&cmd, C_QUOTE, &len))) {
		return (1);
	    }

        } else {				/* regular word */
	    long argsize = sizeof(char);

	    /* note: argsize includes trailing null. */
	    args[argno] = (char *) malloc (argsize);
	    *args[argno] = '\0';
	    
was_quote:
	    if ((len = strcspn(cmd, meta_or_quotes_or_blanks))) {
		argsize += len;
		args[argno] = (char *) realloc (args[argno], argsize);
	    }

	    /* (void) strcat (args[argno], strndup(cmd, len));
	     * - unnecessary, removed by Greg Wohletz <greg@duke.CS.UNLV.EDU>
	     */
	    (void) strncat (args[argno], cmd, len);
	    *((args[argno] + argsize) - 1) = '\0';
	    cmd += len;

	    /* do we have a situation like: arg'ume'nt? */
	    if (*cmd == C_TICKS) {
		if (!(tmpword = word_in_quotes (&cmd, C_TICKS, &len))) {
		    return (1);
		}
		argsize += len;
		args[argno] = (char *) realloc (args[argno], argsize);
		(void) strcat (args[argno], tmpword);
		free (tmpword);
		goto was_quote;
	    }
	    if (*cmd == C_QUOTE) {
		if (!(tmpword = word_in_quotes (&cmd, C_QUOTE, &len))) {
		    return (1);
		}
		argsize += len;
		args[argno] = (char *) realloc (args[argno], argsize);
		(void) strcat (args[argno], tmpword);
		free (tmpword);
		goto was_quote;
	    }
	    argno++;
        }
    }
    args[argno] = NULL;

    if (Debug > 6) {
	char **i;
	for (i = args; *i; i++) {
	    log (XLOG_DEBUG, "secure-system: arg = '%s'", *i);
	}
    }

    return (ss_fork_and_or_exec (args, fork_flag));
}

inline int secure_system (char *cmd) {
    return (secure_system_backend (cmd, 1));
}

inline void secure_exec (char *cmd) {
    (void) secure_system_backend (cmd, 0);
}
