/*
 * tcp-lp: a PLP LP-pipe command for communicating over TCP sockets.
 *
 * You are safe using the first few arguments to communicate LP-pipe-
 * specific configuration (such as the remote hostname and remote
 * port); PLP will append it's arguments to those specified in the
 * printcap entry.
 *                              -- Justin Mason 26 Aug 94.
 *
 * lppipename arguments \	(these two are in the printcap entry)
 *
 * -PPrinter -wwidth -llength -xwidth -ylength [-c] [-iindent] \
 * [-Zoptions] [-Cclass] [-Jjob] [-Raccntname] -nlogin -hHost  \
 * -Fformat -Ddebug [affile]
 *				(all added by lpd)
 */

#include "lp-pipe.h"

#define OPEN_RETRY_DELAY         15	/* delay for open attempts */
#define WAIT_RETRY_DELAY         5	/* delay while waiting for print */

int sockfd = -1, wait_til_ready;
char *Host;
unsigned short Port;

void usage (void) {
    fprintf (stderr, "usage: tcp-lp [-W] host port [... args]\n");
    exit (LP_FAILED);
}

void open_sock (int waiting_til_ready) {
    struct hostent *host;       /* host entry pointer */
    struct sockaddr_in sin;

    if ((*Host >= '0') && (*Host <= '9')) {
	unsigned long a;

	a = inet_addr (Host);

	/* ffffetc. is unsigned-long-ese for -1. */
	if (a == 0xffffffff) {
	    fprintf (stderr, "illegal ip-address: %s\n", Host);
	    exit (LP_FAILED);
	}
	if ((host = gethostbyaddr ((char *) &a, IP_ADDRESS_LEN, AF_INET)) == NULL)
	{
	    fprintf (stderr, "unknown ip-address %s: %s\n", Host, ERRSTR);
	    exit (LP_FAILED);
	}
    } else if ((host = gethostbyname (Host)) == NULL) {
	fprintf (stderr, "unknown host %s: %s\n", Host, ERRSTR);
	exit (LP_FAILED);
    }

    bzero ((char *) &sin, sizeof (sin));
    sin.sin_family = AF_INET;
    bcopy (host->h_addr, (caddr_t) & sin.sin_addr, host->h_length);
    sin.sin_family = host->h_addrtype;
    sin.sin_port = htons (Port);

    do {	/* open the socket. */
	if (waiting_til_ready) {
	    setstatus ("Connecting to printer...", Host);
	}

        if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
            perror("can't open socket");
        }

        if (connect (sockfd, (struct sockaddr *) & sin, sizeof (sin)) == 0) {
            break;

        } else {
	    close (sockfd);
	    if (waiting_til_ready) {
		sleep (WAIT_RETRY_DELAY);
	    } else {
		setstatus ("Retrying connection - offline? (%s)", ERRSTR);
		sleep (OPEN_RETRY_DELAY);
	    }
        }
    } while (1);                /* block forever */
}

int
do_write (int fd, char *buf, int n) {
    int i;
    int N;

    N = n;
    while (n > 0) {
	if ((i = write (fd, buf, n)) < 0)
	    return i;
	n -= i;
	buf += i;
    }
    return N;
}

void
tcp_pipe (void) {
    int in, out;
    int eoi = 0;
    int do_exit = 0;
    fd_set rmask;
    char buf[BUFSIZ];

    open_sock (0);
    setstatus ("Printing.");
    while (!eoi) {
	FD_ZERO (&rmask);
	FD_SET (0, &rmask);	/* stdin */
	FD_SET (sockfd, &rmask);	/* socket */
	/* Find out where input is available */
	select (sockfd + 1, &rmask, 0, 0, 0);
	if (FD_ISSET (0, &rmask)) {
	    in = read (0, buf, BUFSIZ);
	    switch (in) {
		case -1:	/* error */
		    perror ("read error (std input)");
		    do_exit = LP_RETRY;
		    eoi = 1;
		    break;
		case 0:		/* EOF */
		    eoi = 1;
		    break;
		default:
		    out = do_write (sockfd, buf, in);
		    if (in != out) {
			perror ("write error (socket)");
			do_exit = LP_RETRY;
			eoi = 1;
		    }
		    break;
	    }
	}
	if (FD_ISSET (sockfd, &rmask)) {
	    in = read (sockfd, buf, BUFSIZ);
	    switch (in) {
		case -1:	/* error */
		    perror ("read error (socket)");
		    do_exit = LP_RETRY;
		    eoi = 1;
		    break;
		case 0:		/* EOF */
		    eoi = 1;
		    break;
		default:
		    out = do_write (0, buf, in); /* yes, we write to stdin! */
		    if (in != out) {
			perror ("write error (std input)");
			do_exit = LP_RETRY;
			eoi = 1;
		    }
		    break;
	    }
	}
    }
    close (sockfd);

    if (wait_til_ready) {
	open_sock (1);
	close (sockfd);
    }
    if (do_exit)
	exit (do_exit);
}

plp_signal_t
kill_job (int signal)	 /* the current job has been lprm'ed */
{
    if (sockfd > 0) {
	/* send an interrupt character to stop the current job */
        (void) write (sockfd, "\003", 1);
	(void) close (sockfd);
    }
    /* exit with a success code; we don't want the queue to stop */
    exit (LP_SUCCESS);
}

int main (int argc, char **argv) {
    int i;

    /* first two args are the host/port pair.
     * 
     */
    if (argc < 3) usage ();
    i = 1;
    if (!strcmp (argv[1], "-W")) {
	wait_til_ready = 1;
	i++;
    } else {
	wait_til_ready = 0;
    }

    Host = argv[i];
    i++;

    if ((Port = atoi (argv[i])) <= 0)
        usage ();
    i++;

    while (i < argc) {
	if (*argv[i] == '-') {
	    /* tcp-lp doesn't need to use any of the filter options,
	     * so just ignore them.
	     */

	} else {
	    /* it's the status filename. */
	    status_fname = argv[i];
	}
	i++;
    }

    (void) signal (SIGINT, kill_job);

    tcp_pipe ();
    exit (LP_SUCCESS);
}
