/* ----------------------------------------------------------------------------
Linux PCMCIA ethernet adapter driver for the New Media Ethernet LAN.

  The Ethernet LAN uses the Advanced Micro Devices (AMD) Am79C940 Media
  Access Controller for Ethernet (MACE).  It is essentially the Am2150
  PCMCIA Ethernet card contained in the the Am2150 Demo Kit.

Written by Roger C. Pao <rpao@paonet.org>

  This software may be used and distributed according to the terms of
  the GNU Public License.

References

  Am2150 Technical Reference Manual
  Am79C940 (MACE) Data Sheet
  Am79C90 (C-LANCE) Data Sheet

  Eric Mears, New Media Corporation
  Tom Pollard, New Media Corporation
  Ken Lesniak, Silicon Graphics, Inc. <lesniak@boston.sgi.com>
  Donald Becker <becker@cesdis1.gsfc.nasa.gov>
  David Hinds <dhinds@allegro.stanford.edu>

  The Linux client driver is based on the 3Com 3c589 client driver by
  David Hinds.

  The Linux network driver outline is based on the 3Com 3c589 driver and
  the example skeleton.c kernel code, both of which are by Donald
  Becker.

  The Am2150 network driver hardware interface code is based on the
  OS/9000 driver for the New Media Ethernet LAN by Eric Mears.

  Special thanks for testing and help in debugging this driver goes
  to Ken Lesniak.


-------------------------------------------------------------------------------
Driver Notes and Issues
-------------------------------------------------------------------------------

1. Developed on a Dell 320SLi
   PCMCIA Card Services 2.5.6
   Linux dell 1.2.5 #6 Sat Apr 15 18:51:01 PDT 1995 i386

2. rc.pcmcia may require loading pcmcia_core with io_speed=300:
   'insmod pcmcia_core.o io_speed=300'.
   This will avoid problems with fast systems thinking the card is an
   anonymous memory card.

3. Future: Allow hot extraction.  In the meantime, use 'ifconfig eth0 down'
   before extraction.

4. Future: Multiple transmit request processing.

5. Future: Multicast processing.


-------------------------------------------------------------------------------
History
-------------------------------------------------------------------------------
95/05/10 rpao	V0.09 Bug fix: Must use IO_DATA_PATH_WIDTH_AUTO.
95/05/10 rpao	V0.08
		Bug fix: Make all non-exported functions private by using
		static keyword.
		Bug fix: Test IntrCnt _before_ reading MACE_IR.
95/05/10 rpao	V0.07 Statistics.
95/05/09 rpao	V0.06 Fix rx_framecnt problem by addition of PCIC wait states.

rpao=Roger C. Pao <rpao@paonet.org>
---------------------------------------------------------------------------- */


/* ----------------------------------------------------------------------------
Conditional Compilation
---------------------------------------------------------------------------- */

#define FORCE_I365_WAIT			0


/* ----------------------------------------------------------------------------
Include Files
---------------------------------------------------------------------------- */

#include <linux/config.h>

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/bitops.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>

#include "cs_types.h"
#include "cs.h"
#include "cistpl.h"
#include "ds.h"


/* ----------------------------------------------------------------------------
Defines
---------------------------------------------------------------------------- */

#define ETHER_ADDR_LEN			ETH_ALEN
					/* 6 bytes in an Ethernet Address */

/*
The Am2150 has a Xilinx XC3042 field programmable gate array (FPGA)
which manages the interface between the MACE and the PCMCIA bus.  It
also includes buffer management for the 32K x 8 SRAM to control up to
four transmit and 12 receive frames at a time.
*/

/* Am2150 Ethernet Card I/O Mapping */
#define AM2150_RCV			0x00
#define AM2150_XMT			0x04
#define AM2150_XMT_SKIP			0x09
#define AM2150_RCV_NEXT			0x0A
#define AM2150_RCV_FRAME_COUNT		0x0B
#define AM2150_MACE_BANK		0x0C
#define AM2150_MACE_BASE		0x10

/* MACE Registers */
#define MACE_RCVFIFO			0
#define MACE_XMTFIFO			1
#define MACE_XMTFC			2
#define MACE_XMTFS			3
#define MACE_XMTRC			4
#define MACE_RCVFC			5
#define MACE_RCVFS			6
#define MACE_FIFOFC			7
#define MACE_IR				8
#define MACE_IMR			9
#define MACE_PR				10
#define MACE_BIUCC			11
#define MACE_FIFOCC			12
#define MACE_MACCC			13
#define MACE_PLSCC			14
#define MACE_PHYCC			15
#define MACE_CHIPIDL			16
#define MACE_CHIPIDH			17
#define MACE_IAC			18
/* Reserved */
#define MACE_LADRF			20
#define MACE_PADR			21
/* Reserved */
/* Reserved */
#define MACE_MPC			24
/* Reserved */
#define MACE_RNTPC			26
#define MACE_RCVCC			27
/* Reserved */
#define MACE_UTR			29
#define MACE_RTR1			30
#define MACE_RTR2			31

/* MACE Bit Masks */
#define MACE_XMTFS_XMTSV		0x80
#define MACE_XMTFS_UFLO			0x40
#define MACE_XMTFS_LCOL			0x20
#define MACE_XMTFS_MORE			0x10
#define MACE_XMTFS_ONE			0x08
#define MACE_XMTFS_DEFER		0x04
#define MACE_XMTFS_LCAR			0x02
#define MACE_XMTFS_RTRY			0x01

#define MACE_RCVFS_RCVSTS		0xF000
#define MACE_RCVFS_OFLO			0x8000
#define MACE_RCVFS_CLSN			0x4000
#define MACE_RCVFS_FRAM			0x2000
#define MACE_RCVFS_FCS			0x1000

#define MACE_FIFOFC_RCVFC		0xF0
#define MACE_FIFOFC_XMTFC		0x0F

#define MACE_IR_JAB			0x80
#define MACE_IR_BABL			0x40
#define MACE_IR_CERR			0x20
#define MACE_IR_RCVCCO			0x10
#define MACE_IR_RNTPCO			0x08
#define MACE_IR_MPCO			0x04
#define MACE_IR_RCVINT			0x02
#define MACE_IR_XMTINT			0x01

#define MACE_MACCC_PROM			0x80
#define MACE_MACCC_DXMT2PD		0x40
#define MACE_MACCC_EMBA			0x20
#define MACE_MACCC_RESERVED		0x10
#define MACE_MACCC_DRCVPA		0x08
#define MACE_MACCC_DRCVBC		0x04
#define MACE_MACCC_ENXMT		0x02
#define MACE_MACCC_ENRCV		0x01

#define MACE_IAC_ADDRCHG		0x80
#define MACE_IAC_PHYADDR		0x04
#define MACE_IAC_LOGADDR		0x02

#define MACE_UTR_RTRE			0x80
#define MACE_UTR_RTRD			0x40
#define MACE_UTR_RPA			0x20
#define MACE_UTR_FCOLL			0x10
#define MACE_UTR_RCVFCSE		0x08
#define MACE_UTR_LOOP_INCL_MENDEC	0x06
#define MACE_UTR_LOOP_NO_MENDEC		0x04
#define MACE_UTR_LOOP_EXTERNAL		0x02
#define MACE_UTR_LOOP_NONE		0x00
#define MACE_UTR_RESERVED		0x01

/* Switch MACE register bank (only 0 and 1 are valid) */
#define MACEBANK(win_num) outb((win_num), ioaddr + AM2150_MACE_BANK)


#define MACE_IMR_DEFAULT \
  (0xFF-(MACE_IR_CERR | MACE_IR_MPCO | MACE_IR_RCVINT | MACE_IR_XMTINT))


/* ----------------------------------------------------------------------------
Type Definitions
---------------------------------------------------------------------------- */

typedef struct _mace_private {
    struct enet_statistics stats;
} mace_private;


/* ----------------------------------------------------------------------------
Private Global Variables
---------------------------------------------------------------------------- */

static char *version="nmclan_cs 0.09 1995/05/11 (Roger C. Pao)\n";

static dev_info_t dev_info="nmclan_cs";
static dev_link_t *dev_list=NULL;

static char *if_names[]={
  "10baseT",
  "undefined",
  "undefined",
  "BNC",
  "Auto"
};

#ifdef PCMCIA_DEBUG
static int pc_debug=PCMCIA_DEBUG;
#endif

#ifdef NMCLAN_DEBUG
static int nmclan_debug=NMCLAN_DEBUG;
#else
static int nmclan_debug=3;
#endif


/* ----------------------------------------------------------------------------
Parameters
	These are the parameters that can be set during loading with
	'insmod'.
---------------------------------------------------------------------------- */

static int if_port=4; /* default=auto */
  /*
   * 0=10base-T (twisted pair)
   * 1=undefined
   * 2=undefined
   * 3=10base-2 (BNC)
   * 4=auto
   */

/* Bit map of interrupts to choose from */
static u_long irq_mask=0xdeb8; /* 3, 4, 5, 7, 9, 10, 11, 12, 14, 15 */


/* ----------------------------------------------------------------------------
Function Prototypes
---------------------------------------------------------------------------- */

int nmclan_probe(struct device *dev);

static void nmclan_config(dev_link_t *link);
static void nmclan_release(u_long arg);
static int nmclan_event(event_t event, int priority,
		       event_callback_args_t *args);

static void nmclan_reset(struct device *dev);
static int mace_config(struct device *dev, struct ifmap *map);
static int mace_open(struct device *dev);
static int mace_close(struct device *dev);
static int mace_start_xmit(struct sk_buff *skb, struct device *dev);
static void mace_interrupt(int irq, struct pt_regs *regs);
static struct enet_statistics *mace_get_stats(struct device *dev);
static int mace_rx(struct device *dev, unsigned char RxCnt);
static void set_multicast_list(struct device *dev, int num_addrs, void *addrs);

static dev_link_t *nmclan_attach(void);
static void nmclan_detach(dev_link_t *);


/* ----------------------------------------------------------------------------
cs_error
	Report a Card Services related error.
---------------------------------------------------------------------------- */
static void cs_error(int func, int ret)
{
  CardServices(ReportError, dev_info, (void *)func, (void *)ret);
} /* cs_error */


/* ----------------------------------------------------------------------------
nmclan_probe
	If we're linked into the kernel, this gets called from the net
	driver initialization code in Space.c.  It returns failure,
	since all nmclan devices are dynamically allocated.
---------------------------------------------------------------------------- */
#ifndef MODULE
int nmclan_probe(struct device *dev)
{
#ifdef PCMCIA_DEBUG
  if (pc_debug)
    printk(version);
#endif
  register_pcmcia_driver(&dev_info, &nmclan_attach, &nmclan_detach);
  return -1;
} /* nmclan_probe */
#endif


/* ----------------------------------------------------------------------------
nmclan_init
	We never need to do anything when a nmclan device is "initialized"
	by the net software, because we only register already-found cards.
---------------------------------------------------------------------------- */
static int nmclan_init(struct device *dev)
{
  return 0;
} /* nmclan_init */


/* ----------------------------------------------------------------------------
nmclan_attach
	Creates an "instance" of the driver, allocating local data
	structures for one device.  The device is registered with Card
	Services.
---------------------------------------------------------------------------- */
static dev_link_t *nmclan_attach(void)
{
  client_reg_t client_reg;
  dev_link_t *link;
  struct device *dev;
  int ret;


#ifdef PCMCIA_DEBUG
  if (pc_debug)
    printk("nmclan_attach()\n");
#endif

  /* Create new ethernet device */
  link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
  memset(link, 0, sizeof(struct dev_link_t));
  link->release.function = &nmclan_release;
  link->release.data = (u_long)link;
  link->io.NumPorts1 = 32;
  link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
  link->io.IOAddrLines = 5;
  link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
  link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
  link->irq.IRQInfo2 = irq_mask;
  link->conf.Attributes = CONF_ENABLE_IRQ;
  link->conf.Vcc = 50;
  link->conf.IntType = INT_MEMORY_AND_IO;
  link->conf.ConfigIndex = 1;
  link->conf.Present = PRESENT_OPTION;

  dev = kmalloc(sizeof(struct device), GFP_KERNEL);
  memset(dev, 0, sizeof(struct device));

  /* Allocate private data area for this device. */
  dev->priv = kmalloc(sizeof(mace_private), GFP_KERNEL);
  memset(dev->priv, 0, sizeof(mace_private));

  dev->hard_start_xmit = &mace_start_xmit;
  dev->set_config = &mace_config;
  dev->get_stats = &mace_get_stats;
  dev->set_multicast_list = &set_multicast_list;
  ether_setup(dev);
  dev->name = link->dev_name;
  dev->init = &nmclan_init;
  dev->open = &mace_open;
  dev->stop = &mace_close;
  dev->tbusy = 1;
  link->priv = dev;

  /* Register with Card Services */
  link->next = dev_list;
  dev_list = link;
  client_reg.dev_info = &dev_info;
  client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
  client_reg.EventMask = CS_EVENT_CARD_INSERTION |
    CS_EVENT_CARD_REMOVAL | CS_EVENT_RESET_REQUEST |
    CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
  client_reg.event_handler = &nmclan_event;
  client_reg.Version = 0x0210;
  client_reg.event_callback_args.client_data = link;
  ret = CardServices(RegisterClient, &link->handle, &client_reg);
  if (ret != 0) {
    cs_error(RegisterClient, ret);
    nmclan_detach(link);
    return NULL;
  }

  return link;
} /* nmclan_attach */


/* ----------------------------------------------------------------------------
nmclan_detach
	This deletes a driver "instance".  The device is de-registered
	with Card Services.  If it has been released, all local data
	structures are freed.  Otherwise, the structures will be freed
	when the device is released.
---------------------------------------------------------------------------- */
static void nmclan_detach(dev_link_t *link)
{
  dev_link_t **linkp;


#ifdef PCMCIA_DEBUG
  if (pc_debug)
    printk("nmclan_detach(0x%p)\n", link);
#endif

  /* Locate device structure */
  for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
    if (*linkp == link) break;
  if (*linkp == NULL)
    return;

  if (link->state & DEV_CONFIG) {
    nmclan_release((u_long)link);
    if (link->state & DEV_STALE_CONFIG) {
      link->state |= DEV_STALE_LINK;
      return;
    }
  }

  if (link->handle)
    CardServices(DeregisterClient, link->handle);

  /* Unlink device structure, free bits */
  *linkp = link->next;
  if (link->priv) {
    struct device *dev = link->priv;

    if (dev->priv)
      kfree_s(dev->priv, sizeof(mace_private));
    kfree_s(link->priv, sizeof(struct device));
  }
  kfree_s(link, sizeof(struct dev_link_t));

} /* nmclan_detach */


/* ----------------------------------------------------------------------------
mace_read
	Reads a MACE register.  This is bank independent; however, the
	caller must ensure that this call is not interruptable.  We are
	assuming that during normal operation, the MACE is always in
	bank 0.
---------------------------------------------------------------------------- */
static int mace_read(int ioaddr, int reg)
{
  int data = 0xFF;
  unsigned long flags;


  switch (reg >> 4) {
    case 0: /* register 0-15 */
      data = inb(ioaddr + AM2150_MACE_BASE + reg);
      break;
    case 1: /* register 16-31 */
      save_flags(flags);
      cli();
      MACEBANK(1);
      data = inb(ioaddr + AM2150_MACE_BASE + (reg & 0x0F));
      MACEBANK(0);
      restore_flags(flags);
      break;
  }
  return (data & 0xFF);
} /* mace_read */


/* ----------------------------------------------------------------------------
mace_write
	Writes to a MACE register.  This is bank independent; however,
	the caller must ensure that this call is not interruptable.  We
	are assuming that during normal operation, the MACE is always in
	bank 0.
---------------------------------------------------------------------------- */
static void mace_write(int ioaddr, int reg, int data)
{
  unsigned long flags;


  switch (reg >> 4) {
    case 0: /* register 0-15 */
      outb(data & 0xFF, ioaddr + AM2150_MACE_BASE + reg);
      break;
    case 1: /* register 16-31 */
      save_flags(flags);
      cli();
      MACEBANK(1);
      outb(data & 0xFF, ioaddr + AM2150_MACE_BASE + (reg & 0x0F));
      MACEBANK(0);
      restore_flags(flags);
      break;
  }
} /* mace_write */


/* ----------------------------------------------------------------------------
mace_init
	Resets the MACE chip.
---------------------------------------------------------------------------- */
static void mace_init(int ioaddr, char *enet_addr)
{
  int i;


  /* MACE Software reset */
  mace_write(ioaddr, MACE_BIUCC, 1);
  while (mace_read(ioaddr, MACE_BIUCC) & 0x01) {
    /* Wait for reset bit to be cleared automatically after <= 200ns */;
  }
  mace_write(ioaddr, MACE_BIUCC, 0);

  /* The Am2150 requires that the MACE FIFOs operate in burst mode. */
  mace_write(ioaddr, MACE_FIFOCC, 0x0F);

  mace_write(ioaddr, MACE_RCVFC, 0); /* Disable Auto Strip Receive */
  mace_write(ioaddr, MACE_IMR, 0xFF); /* Disable all interrupts until _open */

  /*
   * Bit 2-1 PORTSEL[1-0] Port Select.
   * 00 AUI/10Base-2
   * 01 10Base-T
   * 10 DAI Port (reserved in Am2150)
   * 11 GPSI
   * For this card, only the first two are valid.
   * So, PLSCC should be set to
   * 0x00 for 10Base-2
   * 0x02 for 10Base-T
   * Or just set ASEL in PHYCC below!
   */
  switch (if_port) {
    case 0:
      mace_write(ioaddr, MACE_PLSCC, 0x02);
      break;
    case 3:
      mace_write(ioaddr, MACE_PLSCC, 0x00);
      break;
    default:
      mace_write(ioaddr, MACE_PHYCC, /* ASEL */ 4);
      /* ASEL Auto Select.  When set, the PORTSEL[1-0] bits are overridden,
	 and the MACE device will automatically select the operating media
	 interface port. */
      break;
  }

  mace_write(ioaddr, MACE_IAC, MACE_IAC_ADDRCHG | MACE_IAC_PHYADDR);
  /* Poll ADDRCHG bit */
  while (mace_read(ioaddr, MACE_IAC) & MACE_IAC_ADDRCHG)
    ;
  /* Set PADR register */
  for (i = 0; i < ETHER_ADDR_LEN; i++)
    mace_write(ioaddr, MACE_PADR, enet_addr[i]);

  /* MAC Configuration Control Register should be written last */
  /* Let set_multicast_list set this. */
  /* mace_write(ioaddr, MACE_MACCC, MACE_MACCC_ENXMT | MACE_MACCC_ENRCV); */
} /* mace_init */


/* ----------------------------------------------------------------------------
nmclan_config
	This routine is scheduled to run after a CARD_INSERTION event
	is received, to configure the PCMCIA socket, and to make the
	ethernet device available to the system.
---------------------------------------------------------------------------- */
static void nmclan_config(dev_link_t *link)
{
  client_handle_t handle;
  struct device *dev;
  tuple_t tuple;
  cisparse_t parse;
  u_char buf[64];
  int i;
  short ioaddr, *phys_addr;


  handle = link->handle;
  dev = link->priv;
  phys_addr = (short *)dev->dev_addr;

#ifdef PCMCIA_DEBUG
  if (pc_debug)
    printk("nmclan_config(0x%p)\n", link);
#endif

  do {
    tuple.Attributes = 0;
    tuple.DesiredTuple = CISTPL_CONFIG;
    i = CardServices(GetFirstTuple, handle, &tuple);
    if (i != CS_SUCCESS) break;
    tuple.TupleData = buf;
    tuple.TupleDataMax = 64;
    tuple.TupleOffset = 0;
    i = CardServices(GetTupleData, handle, &tuple);
    if (i != CS_SUCCESS) break;
    i = CardServices(ParseTuple, handle, &tuple, &parse);
    if (i != CS_SUCCESS) break;
    link->conf.ConfigBase = (caddr_t)parse.config.base;
  } while (0);
  if (i != CS_SUCCESS) {
    printk("nmclan_cs: unable to parse CIS\n");
    link->state &= ~DEV_CONFIG_PENDING;
    return;
  }

  /* Configure card */
  link->state |= DEV_CONFIG;

  do {
    i = CardServices(RequestIO, link->handle, &link->io);
    if (i != CS_SUCCESS) {
      cs_error(RequestIO, i);
      break;
    }
    i = CardServices(RequestIRQ, link->handle, &link->irq);
    if (i != CS_SUCCESS) {
      cs_error(RequestIRQ, i);
      break;
    }
    i = CardServices(RequestConfiguration, link->handle, &link->conf);
    if (i != CS_SUCCESS) {
      cs_error(RequestConfiguration, i);
      break;
    }
    i = request_irq(link->irq.AssignedIRQ, &mace_interrupt,
		    0, "nmclan_cs");
    if (i != CS_SUCCESS) {
      printk("nmclan_cs: unable to allocate irq %ld\n", link->irq.AssignedIRQ);
      break;
    }
    dev->irq = link->irq.AssignedIRQ;
    dev->base_addr = link->io.BasePort1;
    dev->tbusy = 0;
    i = register_netdev(dev);
    if (i != 0) {
      printk("nmclan_cs: register_netdev() failed\n");
      break;
    }
  } while (0);

  ioaddr = dev->base_addr;


  /* Read the ethernet address from the CIS. */
  do {
    tuple.DesiredTuple = 0x80 /* CISTPL_CFTABLE_ENTRY_MISC */;
    i = CardServices(GetFirstTuple, handle, &tuple);
    if (i != CS_SUCCESS) break;
    tuple.TupleData = buf;
    tuple.TupleDataMax = 64;
    tuple.TupleOffset = 0;
    i = CardServices(GetTupleData, handle, &tuple);
    if (i != CS_SUCCESS) break;
    memcpy(dev->dev_addr, tuple.TupleData, ETHER_ADDR_LEN);
  } while (0);
  if (i != CS_SUCCESS) {
    printk("nmclan_cs: unable to parse Ethernet address from CIS\n");
    link->state &= ~DEV_CONFIG_PENDING;
    return;
  }

  link->state &= ~DEV_CONFIG_PENDING;
  if (i != 0) {
    nmclan_release((u_long)link);
    return;
  }


  /* Verify configuration by reading the MACE ID. */
  {
    char sig[2];

    sig[0] = mace_read(ioaddr, MACE_CHIPIDL);
    sig[1] = mace_read(ioaddr, MACE_CHIPIDH);
    if ((sig[0] == 0x40) && ((sig[1] == 0x09) || (sig[1] == 0x19))) {
#ifdef PCMCIA_DEBUG
      if (pc_debug)
	printk("nmclan_cs configured: mace id=%x %x\n", sig[0], sig[1]);
#endif
    } else {
      printk("nmclan_cs: mace id not found: %x %x should be 0x40 (0x09 or 0x19)\n", sig[0], sig[1]);
      link->state &= ~DEV_CONFIG_PENDING;
      return;
    }
  }

  mace_init(ioaddr, dev->dev_addr);


  /* The if_port symbol can be set when the module is loaded */
  switch (if_port) {
    case 0:
    case 3:
      dev->if_port = if_port;
      break;
    default:
      dev->if_port = 4;
      break;
  }

#if 0
  /* Determine which port we are using if auto is selected */
  if (if_port==4) {
    if (mace_read(ioaddr, MACE_PHYCC) & /* LNKFL */ 0x80)
      /* 10base-T receiver is in link fail, MACE is using AUI port. */
      dev->if_port = 3;
    else
      dev->if_port = 0;
  }
  /* Unfortunately, this doesn't seem to work.  LNKFL is always set.
     LNKFL is supposed to be opposite the green LED on the edge of the card.
     It doesn't work if it is checked and printed in _open() either.
     It does work if check in _start_xmit(), but that's not a good place
     to printk. */
#endif
  if (nmclan_debug > 3)
    printk("%s: mace_phycc 0x%X.\n", dev->name, mace_read(ioaddr, MACE_PHYCC));

  printk(version);
  printk("%s: nmclan, io %#3lx, irq %d, %s port,", dev->name,
	 dev->base_addr, dev->irq, if_names[dev->if_port]);
  for (i = 0; i < ETHER_ADDR_LEN; i++)
      printk(" %02X", dev->dev_addr[i]);
  printk("\n");

} /* nmclan_config */


/* ----------------------------------------------------------------------------
nmclan_release
	After a card is removed, nmclan_release() will unregister the
	net device, and release the PCMCIA configuration.  If the device
	is still open, this will be postponed until it is closed.
---------------------------------------------------------------------------- */
static void nmclan_release(u_long arg)
{
  dev_link_t *link = (dev_link_t *)arg;
  struct device *dev = link->priv;


#ifdef PCMCIA_DEBUG
  if (pc_debug)
    printk("nmclan_release(0x%p)\n", link);
#endif

  if (link->open) {
    printk("nmclan_cs: release postponed, '%s' still open\n",
	   link->dev_name);
    link->state |= DEV_STALE_CONFIG;
    return;
  }

  if (link->dev_name[0] != '\0')
    unregister_netdev(dev);
  CardServices(ReleaseConfiguration, link->handle);
  CardServices(ReleaseIO, link->handle, &link->io);
  CardServices(ReleaseIRQ, link->handle, &link->irq);
  if (dev->irq != 0) {
    free_irq(dev->irq);
    irq2dev_map[dev->irq] = NULL;
  }

  link->state &= ~DEV_CONFIG;
  if (link->state & DEV_STALE_LINK)
    nmclan_detach(link);

} /* nmclan_release */


/* ----------------------------------------------------------------------------
nmclan_event
	The card status event handler.  Mostly, this schedules other
	stuff to run after an event is received.  A CARD_REMOVAL event
	also sets some flags to discourage the net drivers from trying
	to talk to the card any more.
---------------------------------------------------------------------------- */
static int nmclan_event(event_t event, int priority,
		       event_callback_args_t *args)
{
  dev_link_t *link = args->client_data;
  struct device *dev = link->priv;

    
#ifdef PCMCIA_DEBUG
  if (pc_debug)
    printk("nmclan_event()\n");
#endif
    
  switch (event) {
#ifdef PCMCIA_DEBUG
    case CS_EVENT_REGISTRATION_COMPLETE:
      if (pc_debug)
	printk("nmclan_cs: registration complete\n");
      break;
#endif
    case CS_EVENT_CARD_REMOVAL:
      link->state &= ~DEV_PRESENT;
      if (link->state & DEV_CONFIG) {
	dev->tbusy = 1; dev->start = 0;
	link->release.expires = 5;
	add_timer(&link->release);
      }
      break;
    case CS_EVENT_CARD_INSERTION:
      link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
      nmclan_config(link);
      break;
    case CS_EVENT_PM_SUSPEND:
      if (link->state & DEV_CONFIG) {
	link->state |= DEV_SUSPEND;
	if (link->open) {
	  dev->tbusy = 1; dev->start = 0;
	}
	CardServices(ReleaseConfiguration, link->handle);
      }
      break;
    case CS_EVENT_PM_RESUME:
      if (link->state & DEV_SUSPEND) {
	link->state &= ~DEV_SUSPEND;
	CardServices(RequestConfiguration, link->handle, &link->conf);
	if (link->open) {
	  nmclan_reset(dev);
	  dev->tbusy = 0; dev->start = 1;
	}
      }
      break;
    case CS_EVENT_RESET_REQUEST:
      return 1;
      break;
  }
  return 0;
} /* nmclan_event */

/* ------------------------------------------------------------------------- */


/* ----------------------------------------------------------------------------
nmclan_reset
	Reset and restore all of the nmclan registers.
---------------------------------------------------------------------------- */
static void nmclan_reset(struct device *dev)
{
  mace_init(dev->base_addr, dev->dev_addr);
  mace_write(dev->base_addr, MACE_IMR, MACE_IMR_DEFAULT);
} /* nmclan_reset */


/* ----------------------------------------------------------------------------
mace_config
	[Someone tell me what this is supposed to do?]
---------------------------------------------------------------------------- */
static int mace_config(struct device *dev, struct ifmap *map)
{
  if (map->port != (u_char)(-1)) {
    dev->if_port = map->port;
    printk("%s: switched to %s port\n", dev->name, if_names[dev->if_port]);
  }
  return 0;
} /* mace_config */


/* ----------------------------------------------------------------------------
mace_open
	Open device driver.  set_multicast_list _must_ be called after this.
---------------------------------------------------------------------------- */
static int mace_open(struct device *dev)
{
  int ioaddr = dev->base_addr;
  dev_link_t *link;


  for (link = dev_list; link; link = link->next)
    if (link->priv == dev) break;
  if (!DEV_OK(link))
    return -ENODEV;

  link->open++;
#ifdef MODULE
  MOD_INC_USE_COUNT;
#endif

  MACEBANK(0);

  irq2dev_map[dev->irq] = dev;

  dev->interrupt = 0;
  dev->tbusy = 0;
  dev->start = 1;

  nmclan_reset(dev);

  return 0; /* Always succeed */
} /* mace_open */


/* ----------------------------------------------------------------------------
mace_close
	Closes device driver.
---------------------------------------------------------------------------- */
static int mace_close(struct device *dev)
{
  int ioaddr = dev->base_addr;
  dev_link_t *link;


  for (link = dev_list; link; link = link->next)
    if (link->priv == dev) break;
  if (link == NULL)
    return -ENODEV;

  if (nmclan_debug > 2)
    printk("%s: shutting down ethercard.\n", dev->name);
  
#if 0
  /* TBD: It would be more useful to just power down the entire socket. */
  if (dev->if_port == 3)
    /* Turn off thinnet power.  Green! */
    outw(StopCoax, ioaddr + EL3_CMD);
#endif

  /* Mask off all interrupts from the MACE chip. */
  outb(0xFF, ioaddr + AM2150_MACE_BASE + MACE_IMR);

  link->open--;
  dev->start = 0;
  if (link->state & DEV_STALE_CONFIG)
    nmclan_release((u_long)link);

#ifdef MODULE
  MOD_DEC_USE_COUNT;
#endif

  return 0;
} /* mace_close */


/* ----------------------------------------------------------------------------
mace_start_xmit
	This routine begins the packet transmit function.  When completed,
	it will generate a transmit interrupt.
---------------------------------------------------------------------------- */
static int mace_start_xmit(struct sk_buff *skb, struct device *dev)
{
  int ioaddr = dev->base_addr;


  /* Transmitter timeout, serious problems. */
  if (dev->tbusy) {
    int tickssofar = jiffies - dev->trans_start;

    if (tickssofar < 40)
      return 1;
    printk("%s: transmit timed out\n", dev->name);
    dev->trans_start = jiffies;
    nmclan_reset(dev);
    dev->tbusy = 0;
  }

  if (skb == NULL) {
    dev_tint(dev);
    return 0;
  }

  if (skb->len <= 0)
    return 0;
 
  if (nmclan_debug > 4) {
    printk("%s: mace_start_xmit(length = %ld) called.\n",
      dev->name, skb->len);
  }
 
  /* Avoid timer-based retransmission conflicts. */
  if (set_bit(0, (void*)&dev->tbusy) != 0)
    printk("%s: transmitter access conflict.\n", dev->name);
  else {
    /* WARNING: Do not write any more than what is reported! */
    /* Put out the word header... */
    outw(skb->len, ioaddr + AM2150_XMT);
    /* ... and the packet */
    outsw(ioaddr + AM2150_XMT, skb->data, skb->len >> 1);
    if (skb->len & 1) {
      /* Odd byte transfer */
      outb(skb->data[skb->len-1], ioaddr + AM2150_XMT);
    }

    dev->trans_start = jiffies;
    /* dev->tbusy = 0; We are single request right now.
		       mace_interrupt() will clear it for us. */
  }

  dev_kfree_skb(skb, FREE_WRITE);
    
  return 0;
} /* mace_start_xmit */


/* ----------------------------------------------------------------------------
mace_interrupt
	The interrupt handler.
---------------------------------------------------------------------------- */
static void mace_interrupt(int irq, struct pt_regs *regs)
{
  struct device *dev = (struct device *)(irq2dev_map[irq]);
  mace_private *lp = (mace_private *)dev->priv;
  int ioaddr, status;
  int IntrCnt = 3;


  if (dev == NULL) {
    printk ("mace_interrupt(): irq 0x%X for unknown device.\n", irq);
    return;
  }

  if (dev->interrupt) {
    printk("%s: re-entering the interrupt handler.\n", dev->name);
    return;
  }
  dev->interrupt = 1;

  ioaddr = dev->base_addr;

  if (dev->start == 0) {
    printk("%s: interrupt from dead card\n", dev->name);
    goto exception;
  }

  /* WARNING: MACE_IR is a READ/CLEAR port! */
  while (
    (IntrCnt--) &&
    ((status = inb(ioaddr + AM2150_MACE_BASE + MACE_IR)) &
      (MACE_IR_XMTINT | MACE_IR_RCVINT))
  ) {
    if (nmclan_debug > 4) {
      printk("mace_interrupt: irq 0x%X status 0x%X.\n", irq, status);
    }

    if (status & MACE_IR_RCVINT) {
      mace_rx(dev, 5);
    }

    if (status & MACE_IR_XMTINT) {
      unsigned char fifofc;
      unsigned char xmtfs;

      fifofc = inb(ioaddr + AM2150_MACE_BASE + MACE_FIFOFC);
      if ((fifofc & MACE_FIFOFC_XMTFC)==0) {
	lp->stats.tx_errors++;
	outb(0xFF, ioaddr + AM2150_XMT_SKIP);
      }

      /* Transmit Retry Count (XMTRC, reg 4), if needed,
	 must be read here. */

      if (
        (xmtfs = inb(ioaddr + AM2150_MACE_BASE + MACE_XMTFS)) &
        MACE_XMTFS_XMTSV /* Transmit Status Valid */
      ) {
	if (xmtfs & MACE_XMTFS_UFLO) lp->stats.tx_fifo_errors++;
	  /* Underflow.  Indicates that the Transmit FIFO emptied before
	     the end of frame was reached. */
	if (xmtfs & MACE_XMTFS_LCOL) lp->stats.collisions++;
	  /* Late Collision */
	/* if (xmtfs & MACE_XMTFS_MORE) More; */
	  /* MORE than one retry was needed */
	/* if (xmtfs & MACE_XMTFS_ONE) One; */
	  /* Exactly ONE retry occurred */
	/* if (xmtfs & MACE_XMTFS_DEFER) Defer; */
	  /* Transmission was defered */
	if (xmtfs & MACE_XMTFS_LCAR) lp->stats.tx_carrier_errors++;
	  /* Loss of carrier */
	if (xmtfs & MACE_XMTFS_RTRY) lp->stats.tx_aborted_errors++;
	  /* Retry error: transmit aborted after 16 attempts */
      }

      lp->stats.tx_packets++;
      dev->tbusy = 0;
      mark_bh(NET_BH);
    }

    if (status & 0x7A) {
      /* if (status & MACE_IR_JAB) Jabber Error; */
        /* Jabber Error--Excessive transmit duration (20-150ms) */
      /* if (status & MACE_IR_BABL) Babble Error; */
        /* Babble Error-->1518 bytes transmitted */
      if (status & MACE_IR_CERR) lp->stats.tx_heartbeat_errors++;
	/* Collision Error.  CERR indicates the absence of the
	   Signal Quality Error Test message after a packet
	   transmission. */
      /* if (status & MACE_IR_RCVCCO) Receive Collision Count Overflow; */
        /* Receive collision count overflow */
      /* if (status & MACE_IR_RNTPCO) Runt Packet Count Overflow; */
        /* Runt Packet Count Overflow */
      if (status & MACE_IR_MPCO) lp->stats.rx_missed_errors = 255;
        /* Missed Packet Count Overflow */
    }
  }

exception:
  dev->interrupt = 0;
  return;
} /* mace_interrupt */


/* ----------------------------------------------------------------------------
mace_rx
	Receives packets.
---------------------------------------------------------------------------- */
static int mace_rx(struct device *dev, unsigned char RxCnt)
{
  mace_private *lp = (mace_private *)dev->priv;
  int ioaddr = dev->base_addr;
  unsigned char rx_framecnt;
  unsigned short rx_status;


  while (
    ((rx_framecnt = inb(ioaddr + AM2150_RCV_FRAME_COUNT)) > 0) &&
    (rx_framecnt <= 12) &&
    (RxCnt--)
  ) {
    rx_status = inw(ioaddr + AM2150_RCV);

    if (nmclan_debug > 5) {
      printk("%s: in mace_rx(), framecnt 0x%X, rx_status 0x%X.\n",
        dev->name, rx_framecnt, rx_status);
    }

    if (rx_status & MACE_RCVFS_RCVSTS) { /* Error, update stats. */
      lp->stats.rx_errors++;
      if (rx_status & MACE_RCVFS_OFLO) lp->stats.rx_over_errors++;
      /* if (rx_status & MACE_RCVFS_CLSN) lp->stats.collisions++; */
      if (rx_status & MACE_RCVFS_FRAM) lp->stats.rx_frame_errors++;
      if (rx_status & MACE_RCVFS_FCS) lp->stats.rx_crc_errors++;
    } else {
      short pkt_len = (rx_status & ~MACE_RCVFS_RCVSTS) - 4;
        /* Auto Strip is off, always subtract 4 */
      struct sk_buff *skb;

      lp->stats.rx_length_errors += inb(ioaddr + AM2150_RCV);
	/* runt packet count */
      lp->stats.collisions += inb(ioaddr + AM2150_RCV);
	/* rcv collision count */

      skb = alloc_skb(pkt_len, GFP_ATOMIC);
      if (nmclan_debug > 4)
	printk("    receiving packet size 0x%X rx_status 0x%X.\n",
	  pkt_len, rx_status);
      if (skb != NULL) {
	skb->len = pkt_len;
	skb->dev = dev;

	/* WARNING: Do not read any more than what is reported! */
	insw(ioaddr + AM2150_RCV, skb->data, pkt_len >> 1);
	if (pkt_len & 1) {
	    /* Odd byte transfer */
	    skb->data[pkt_len-1] = inb(ioaddr + AM2150_RCV);
	}

	netif_rx(skb);

	lp->stats.rx_packets++;
	outb(0xFF, ioaddr + AM2150_RCV_NEXT); /* skip to next frame */

	if (nmclan_debug > 5) {
	  rx_framecnt = inb(ioaddr + AM2150_RCV_FRAME_COUNT);
	  printk("    1 framecnt 0x%X.\n", rx_framecnt);
	}
	continue;
      } else {
        if (nmclan_debug)
	  printk("%s: couldn't allocate a sk_buff of size %d.\n",
	    dev->name, pkt_len);
	lp->stats.rx_dropped++;
      }
    }
    outb(0xFF, ioaddr + AM2150_RCV_NEXT); /* skip to next frame */

    if (nmclan_debug > 5) {
      rx_framecnt = inb(ioaddr + AM2150_RCV_FRAME_COUNT);
      printk("    2 framecnt 0x%X.\n", rx_framecnt);
    }
  }

  if (nmclan_debug > 5) {
    rx_framecnt = inb(ioaddr + AM2150_RCV_FRAME_COUNT);
    printk("    3 framecnt 0x%X.\n", rx_framecnt);
  }

  return 0;
} /* mace_rx */


/* ----------------------------------------------------------------------------
update_stats
	Update statistics.  We change to register window 1, so this
	should be run single-threaded if the device is active. This is
	expected to be a rare operation, and it's simpler for the rest
	of the driver to assume that window 0 is always valid rather
	than use a special window-state variable.
---------------------------------------------------------------------------- */
static void update_stats(int ioaddr, struct device *dev)
{
  mace_private *lp = (mace_private *)dev->priv;
    

  if (nmclan_debug > 5)
    printk("%s: updating the statistics.\n", dev->name);

  lp->stats.rx_missed_errors += mace_read(ioaddr, MACE_MPC);

  return;
} /* update_stats */


/* ----------------------------------------------------------------------------
mace_get_stats
	Gathers ethernet statistics from the MACE chip.
---------------------------------------------------------------------------- */
static struct enet_statistics *mace_get_stats(struct device *dev)
{
  mace_private *lp = (mace_private *)dev->priv;
  unsigned long flags;


  save_flags(flags);
  cli();
  update_stats(dev->base_addr, dev);
  restore_flags(flags);
  return &lp->stats;
} /* enet_statistics */


/* ----------------------------------------------------------------------------
updateCRC
	Modified from Am79C90 data sheet.
---------------------------------------------------------------------------- */
static void updateCRC(int *CRC, int bit)
{
  int poly[]={
    1,1,1,0, 1,1,0,1,
    1,0,1,1, 1,0,0,0,
    1,0,0,0, 0,0,1,1,
    0,0,1,0, 0,0,0,0
  }; /* CRC polynomial.  poly[n] = coefficient of the x**n term of the
	CRC generator polynomial. */

  int j;


  /* shift CRC and control bit (CRC[32]) */
  for (j = 32; j > 0; j--)
    CRC[j] = CRC[j-1];
  CRC[0] = 0;

  /* If bit XOR(control bit) = 1, set CRC = CRC XOR polynomial. */
  if (bit ^ CRC[32])
    for (j = 0; j < 32; j++)
      CRC[j] ^= poly[j];
} /* updateCRC */


/* ----------------------------------------------------------------------------
BuildLAF
	Build logical address filter.
	Modified from Am79C90 data sheet.

Input
	ladrf: logical address filter (contents initialized to 0)
	adr: ethernet address
---------------------------------------------------------------------------- */
static void BuildLAF(int *ladrf, int *adr)
{
  int CRC[33]={1}; /* CRC register, 1 word/bit + extra control bit */

  int i, byte; /* temporary array indices */
  int hashcode; /* the output object */


  CRC[32]=0;

  for (byte = 0; byte < 6; byte++)
    for (i = 0; i < 8; i++)
      updateCRC(CRC, (adr[byte] >> i) & 1);

  hashcode = 0;
  for (i = 0; i < 6; i++)
    hashcode = (hashcode << 1) + CRC[i];

  byte = hashcode >> 3;
  ladrf[byte] |= (1 << (hashcode & 7));

  if (nmclan_debug > 3) {
    printk("    adr =");
    for (i = 0; i < 6; i++)
      printk(" %02X", adr[i]);
    printk("\n    hashcode = %d(decimal), ladrf[0:63] =", hashcode);
    for (i = 0; i < 8; i++)
      printk(" %02X", ladrf[i]);
    printk("\n");
  }
} /* BuildLAF */


/* ----------------------------------------------------------------------------
set_multicast_list
	Set or clear the multicast filter for this adaptor.

Input
	num_addrs == -1	Promiscuous mode, receive all packets
	num_addrs == 0	Normal mode, clear multicast list
	num_addrs > 0	Multicast mode, receive normal and MC packets, and do
			best-effort filtering.
---------------------------------------------------------------------------- */
static void set_multicast_list(struct device *dev, int num_addrs, void *addrs)
{
  short ioaddr = dev->base_addr;
  int adr[6] = {0}; /* Ethernet address */
  int ladrf[8] = {0}; /* Logical address filter */
  int i;


  if (nmclan_debug > 1)
    printk("%s: setting Rx mode to %d addresses.\n", dev->name, num_addrs);

  if (num_addrs > 0) {

    printk("Attempt to set multicast list detected.\n");

    /* Calculate multicast logical address filter */
    for (i = 0; i < num_addrs; i++) {
      /* TBD: Set adr[] to the i'th address */
      BuildLAF(ladrf, adr);
    }
    mace_write(ioaddr, MACE_IAC, MACE_IAC_ADDRCHG | MACE_IAC_LOGADDR);
    /* Poll ADDRCHG bit */
    while (mace_read(ioaddr, MACE_IAC) & MACE_IAC_ADDRCHG)
      ;
    /* Set LADRF register */
    for (i = 0; i < ETHER_ADDR_LEN; i++)
      mace_write(ioaddr, MACE_LADRF, ladrf[i]);

    mace_write(ioaddr, MACE_UTR, MACE_UTR_RCVFCSE | MACE_UTR_LOOP_EXTERNAL);
    mace_write(ioaddr, MACE_MACCC, MACE_MACCC_ENXMT | MACE_MACCC_ENRCV);

  } else if (num_addrs < 0) {

    /* Promiscuous mode: receive all packets */
    mace_write(ioaddr, MACE_UTR, MACE_UTR_LOOP_EXTERNAL);
    mace_write(ioaddr, MACE_MACCC,
      MACE_MACCC_PROM | MACE_MACCC_ENXMT | MACE_MACCC_ENRCV
    );

  } else {

    /* Normal mode */
    mace_write(ioaddr, MACE_UTR, MACE_UTR_LOOP_EXTERNAL);
    mace_write(ioaddr, MACE_MACCC, MACE_MACCC_ENXMT | MACE_MACCC_ENRCV);

  }
} /* set_multicast_list */


/* ------------------------------------------------------------------------- */

#ifdef MODULE
char kernel_version[] = UTS_RELEASE;

/* ----------------------------------------------------------------------------
init_module
---------------------------------------------------------------------------- */
int init_module(void)
{
#ifdef PCMCIA_DEBUG
  if (pc_debug)
    printk(version);
#endif
  register_pcmcia_driver(&dev_info, &nmclan_attach, &nmclan_detach);
  return 0;
} /* init_module */


/* ----------------------------------------------------------------------------
cleanup_module
---------------------------------------------------------------------------- */
void cleanup_module(void)
{
  printk("nmclan_cs: unloading\n");
  unregister_pcmcia_driver(&dev_info);
  while (dev_list != NULL)
    nmclan_detach(dev_list);
} /* cleanup_module */

#endif /* MODULE */
