------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUNTIME COMPONENTS                          --
--                                                                          --
--   A D A . T E X T _ I O . W I D E _ T E X T _ I O . M O D U L A R _ I O  --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                                                                          --
--     Copyright (C) 1992,1993,1994,1995 Free Software Foundation, Inc.     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Text_IO.Modular_Aux;

with System.Unsigned_Types; use System.Unsigned_Types;
with System.WCh_Con;        use System.WCh_Con;
with System.WCh_WtS;        use System.WCh_WtS;

package body Ada.Text_IO.Wide_Text_IO.Modular_IO is

   subtype TFT is Ada.Text_IO.File_Type;
   --  File type required for calls to routines in Aux

   package Aux renames Ada.Text_IO.Modular_Aux;
   --  We can share the normal Text_IO circuits for the non-string cases
   --  since numeric values involve no wide character values, and the first
   --  character of a wide character value (ESC or an upper half character)
   --  always looks non-numeric for the Get case). For the Wide_String cases
   --  we can share the normal Text_IO circuits by converting to String.

   ---------
   -- Get --
   ---------

   procedure Get
     (File  : in File_Type;
      Item  : out Num;
      Width : in Field := 0)
   is
   begin
      if Num'Size > Unsigned'Size then
         Aux.Get_LLU (TFT (File), Long_Long_Unsigned (Item), Width);
      else
         Aux.Get_Uns (TFT (File), Unsigned (Item), Width);
      end if;

   exception
      when Constraint_Error => raise Data_Error;
   end Get;

   procedure Get
     (Item  : out Num;
      Width : in Field := 0)
   is
   begin
      Get (Current_Input, Item, Width);
   end Get;

   procedure Get
     (From : in Wide_String;
      Item : out Num;
      Last : out Positive)
   is
      S : constant String := Wide_String_To_String (From, WCEM_Upper);
      --  String on which we do the actual conversion. Note that the method
      --  used for wide character encoding is irrelevant, since if there is
      --  a character outside the Standard.Character range then the call to
      --  Aux.Gets will raise Data_Error in any case.

   begin
      if Num'Size > Unsigned'Size then
         Aux.Gets_LLU (S, Long_Long_Unsigned (Item), Last);
      else
         Aux.Gets_Uns (S, Unsigned (Item), Last);
      end if;

   exception
      when Constraint_Error => raise Data_Error;
   end Get;

   ---------
   -- Put --
   ---------

   procedure Put
     (File  : in File_Type;
      Item  : in Num;
      Width : in Field := Default_Width;
      Base  : in Number_Base := Default_Base)
   is
   begin
      if Num'Size > Unsigned'Size then
         Aux.Put_LLU (TFT (File), Long_Long_Unsigned (Item), Width, Base);
      else
         Aux.Put_Uns (TFT (File), Unsigned (Item), Width, Base);
      end if;
   end Put;

   procedure Put
     (Item  : in Num;
      Width : in Field := Default_Width;
      Base  : in Number_Base := Default_Base)
   is
   begin
      Put (Current_Output, Item, Width, Base);
   end Put;

   procedure Put
     (To   : out Wide_String;
      Item : in Num;
      Base : in Number_Base := Default_Base)
   is
      S : String (To'First .. To'Last);

   begin
      if Num'Size > Unsigned'Size then
         Aux.Puts_LLU (S, Long_Long_Unsigned (Item), Base);
      else
         Aux.Puts_Uns (S, Unsigned (Item), Base);
      end if;

      for J in S'Range loop
         To (J) := Wide_Character'Val (Character'Pos (S (J)));
      end loop;
   end Put;

end Ada.Text_IO.Wide_Text_IO.Modular_IO;
