------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                 I N T E R F A C E S . C . P T H R E A D S                --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                             $Revision: 1.2 $                             --
--                                                                          --
--   Copyright (C) 1991,1992,1993,1994,1995,1996 Florida State University   --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is the Linux version of this package.

--  This package interfaces with Pthreads. It is not a complete interface;
--  it only includes what is needed to implement the Ada runtime.

with System;

with Interfaces.C.System_Constants;
--  Used for: Add_Prio
--            pthread_attr_t_size
--            pthread_mutexattr_t_size
--            pthread_mutex_t_size
--            pthread_condattr_t_size
--            pthread_cond_t_size
--            NO_PRIO_INHERIT
--            PRIO_INHERIT
--            PRIO_PROTECT

with Interfaces.C.Posix_Error;
--  Used for: Return_Code

with Interfaces.C.POSIX_RTE;
--  Used for: Signal,
--            Signal_Set

with Interfaces.C.POSIX_Timers;
--  Used for: timespec

package Interfaces.C.Pthreads is

   package C renames Interfaces.C;

   subtype Return_Code is C.Posix_Error.Return_Code;

   subtype void_ptr is System.Address;

   Alignment : constant := Natural'Min (Standard'Maximum_Alignment, 8);

   type pthread_t       is private;
   type pthread_mutex_t is private;
   type pthread_cond_t  is private;

   type pthread_attr_t      is private;
   type pthread_mutexattr_t is private;
   type pthread_condattr_t  is private;
   type pthread_key_t       is private;

   type pthread_sched_param_t is record
      prio    : C.int;
      no_data : System.Address;
   end record;

   --
   --  Pthread Attribute Initialize / Destroy
   --
   function pthread_attr_init
     (attr : access pthread_attr_t) return Return_Code;
   pragma Import (C,
                  pthread_attr_init,
                  "pthread_attr_init",
                  "pthread_attr_init");

   function pthread_attr_destroy
     (attr : access pthread_attr_t) return Return_Code;
   pragma Import (C,
                  pthread_attr_destroy,
                  "pthread_attr_destroy",
                  "pthread_attr_destroy");

   --
   --  Thread Attributes
   --
   function pthread_attr_setstacksize
     (attr : access pthread_attr_t; stacksize : size_t) return Return_Code;
   pragma Import (C,
                  pthread_attr_setstacksize,
                  "pthread_attr_setstacksize",
                  "pthread_attr_setstacksize");

   function pthread_attr_setdetachstate
     (attr : access pthread_attr_t; detachstate : C.int) return Return_Code;
   pragma Import (C,
                  pthread_attr_setdetachstate,
                  "pthread_attr_setdetachstate",
                  "pthread_attr_setdetachstate");

   function pthread_attr_setschedparam
     (attr   : access pthread_attr_t;
      param  : access pthread_sched_param_t) return Return_Code;
   pragma Import (C,
                  pthread_attr_setschedparam,
                  "pthread_attr_setschedparam",
                  "pthread_attr_setschedparam");

   function pthread_attr_getschedparam
     (attr   : access pthread_attr_t;
      param  : access pthread_sched_param_t) return Return_Code;
   pragma Import (C,
                  pthread_attr_getschedparam,
                  "pthread_attr_getschedparam",
                  "pthread_attr_getschedparam");

   --
   --  Thread Creation & Management
   --

   type void_ptr_void_ptr_Ptr is
      access function (arg : void_ptr) return void_ptr;

   subtype start_addr is void_ptr_void_ptr_Ptr;

   function pthread_create
     (thread        : access pthread_t;
      attr          : access pthread_attr_t;
      start_routine :        start_addr;
      arg           :        void_ptr) return Return_Code;
   pragma Import (C, pthread_create, "pthread_create", "pthread_create");

   function pthread_join
     (thread : pthread_t; status : access void_ptr) return Return_Code;
   pragma Import (C, pthread_join, "pthread_join", "pthread_join");

   function pthread_detach (thread : access pthread_t) return Return_Code;
   pragma Import (C, pthread_detach, "pthread_detach", "pthread_detach");

   procedure pthread_exit (status : void_ptr);
   pragma Import (C, pthread_exit, "pthread_exit", "pthread_exit");

   procedure pthread_yield (arg : void_ptr);
   pragma Import (C, pthread_yield, "pthread_yield", "pthread_yield");

   function pthread_self return pthread_t;
   pragma Import (C, pthread_self, "pthread_self", "pthread_self");

   function pthread_kill (thread : pthread_t; sig : C.int) return Return_Code;
   pragma Import (C, pthread_kill, "pthread_kill", "pthread_kill");

   function pthread_setschedparam
     (attr   : pthread_t;
      policy : C.int;
      param  : access pthread_sched_param_t) return Return_Code;
   pragma Import (C,
                  pthread_setschedparam,
                  "pthread_setschedparam",
                  "pthread_setschedparam");

   function pthread_getschedparam
     (attr   : pthread_t;
      policy : access C.int;
      param  : access pthread_sched_param_t) return Return_Code;
   pragma Import (C,
                  pthread_getschedparam,
                  "pthread_getschedparam",
                  "pthread_getschedparam");

   --
   --  Mutex Initialization / Destruction
   --

   function pthread_mutexattr_init
     (attr : access pthread_mutexattr_t) return Return_Code;
   pragma Import (C,
                  pthread_mutexattr_init,
                  "pthread_mutexattr_init",
     "pthread_mutexattr_init");

   function pthread_mutexattr_destroy
     (attr : access pthread_mutexattr_t) return Return_Code;
   pragma Import (C,
     pthread_mutexattr_destroy,
                  "pthread_mutexattr_destroy",
                  "pthread_mutexattr_destroy");

   --  Mutex Operations

   function pthread_mutex_init
     (mutex : access pthread_mutex_t;
      attr  : access pthread_mutexattr_t) return Return_Code;
   pragma Import (C,
                  pthread_mutex_init,
                  "pthread_mutex_init",
                  "pthread_mutex_init");

   function pthread_mutex_destroy (mutex : access pthread_mutex_t)
   return Return_Code;
   pragma Import (C,
     pthread_mutex_destroy,
     "pthread_mutex_destroy",
                  "pthread_mutex_destroy");

   function pthread_mutex_lock
     (mutex : access pthread_mutex_t) return Return_Code;
   pragma Import (C,
                  pthread_mutex_lock,
                  "pthread_mutex_lock",
     "pthread_mutex_lock");

   function pthread_mutex_trylock (mutex : access pthread_mutex_t)
   return Return_Code;
   pragma Import (C,
                  pthread_mutex_trylock,
                  "pthread_mutex_trylock",
                  "pthread_mutex_trylock");

   function pthread_mutex_unlock
     (mutex : access pthread_mutex_t) return Return_Code;
   pragma Import (C,
                  pthread_mutex_unlock,
                  "pthread_mutex_unlock",
                  "pthread_mutex_unlock");

   --  Condition Initialization / Destruction

   function pthread_condattr_init
     (attr : access pthread_condattr_t) return Return_Code;
   pragma Import (C,
                  pthread_condattr_init,
                  "pthread_condattr_init",
                  "pthread_condattr_init");

   function pthread_condattr_destroy
     (attr : access pthread_condattr_t) return Return_Code;
   pragma Import (C,
                  pthread_condattr_destroy,
                  "pthread_condattr_destroy",
                  "pthread_condattr_destroy");

   function pthread_condattr_settype
     (attr : access pthread_condattr_t; cond : C.int) return Return_Code;
   pragma Import (C,
                  pthread_condattr_settype,
                  "pthread_condattr_settype",
                  "pthread_condattr_settype");

   function pthread_condattr_gettype
     (attr : access pthread_condattr_t;
      cond : access C.int) return Return_Code;
   pragma Import (C,
                  pthread_condattr_gettype,
                  "pthread_condattr_gettype",
                  "pthread_condattr_gettype");

   --  Condition Operations

   function pthread_cond_init
     (cond : access pthread_cond_t;
      attr : access pthread_condattr_t) return Return_Code;
   pragma Import (C,
                  pthread_cond_init,
                  "pthread_cond_init",
                  "pthread_cond_init");

   function pthread_cond_destroy
     (cond : access pthread_cond_t) return Return_Code;
   pragma Import (C,
                  pthread_cond_destroy,
                  "pthread_cond_destroy",
                  "pthread_cond_destroy");

   function pthread_cond_signal
     (cond : access pthread_cond_t) return Return_Code;
   pragma Import (C,
                  pthread_cond_signal,
                  "pthread_cond_signal",
                  "pthread_cond_signal");

   function pthread_cond_broadcast
     (cond : access pthread_cond_t) return Return_Code;
   pragma Import (C,
                  pthread_cond_broadcast,
                  "pthread_cond_broadcast",
                  "pthread_cond_broadcast");

   function pthread_cond_wait
     (cond  : access pthread_cond_t;
      mutex : access pthread_mutex_t) return Return_Code;
   pragma Import (C,
     pthread_cond_wait,
     "pthread_cond_wait",
                  "pthread_cond_wait");

   function pthread_cond_timedwait
     (cond    : access pthread_cond_t;
      mutex   : access pthread_mutex_t;
      abstime : access C.POSIX_Timers.timespec) return Return_Code;
   pragma Import (C,
                  pthread_cond_timedwait,
                  "pthread_cond_timedwait",
                  "pthread_cond_timedwait");

   --  Thread-Specific Data

   type Proc_void_ptr_Ptr is access procedure (value : void_ptr);

   function pthread_key_create
     (key : access pthread_key_t; destructor : Proc_void_ptr_Ptr)
      return Return_Code;
   pragma Import (C,
                  pthread_key_create,
                  "pthread_key_create",
                  "pthread_key_create");

   function pthread_setspecific
     (key : pthread_key_t; value : void_ptr) return Return_Code;
   pragma Import (C,
                  pthread_setspecific,
                  "pthread_setspecific",
                  "pthread_setspecific");

   function pthread_getspecific (key : pthread_key_t) return void_ptr;
   pragma Import (C,
                  pthread_getspecific,
                  "pthread_getspecific",
                  "pthread_getspecific");

   --
   --  Miscelaneous routines.
   --
   function sigwait
     (set : access C.POSIX_RTE.Signal_Set) return C.POSIX_RTE.Signal;
   pragma Import (C, sigwait, "sigwait", "sigwait");

   function pthread_sigmask
     (how    : C.int;
      set    : access C.POSIX_RTE.Signal_Set;
      oset   : access C.POSIX_RTE.Signal_Set) return Return_Code;
   pragma Import (C,
                  pthread_sigmask,
                  "pthread_sigmask",
                  "pthread_sigmask");

   procedure pthread_cleanup_push
     (routine : System.Address;
      arg     : System.Address);
   pragma Import (C,
                  pthread_cleanup_push,
                  "pthread_cleanup_push",
                  "pthread_cleanup_push");

   procedure pthread_cleanup_pop (execute : C.int);
   pragma Import (C,
                  pthread_cleanup_pop,
                  "pthread_cleanup_pop",
                  "pthread_cleanup_pop");

private

   --  The use of longword alignment for the following C types is
   --  a stopgap measure which is not generally portable.  A portable
   --  solution will require some means of getting alignment information
   --  from the C compiler.

   type pthread_attr_t is
     array (1 .. System_Constants.pthread_attr_t_size) of unsigned_char;
   for pthread_attr_t'Alignment use Alignment;

   type pthread_mutexattr_t is
     array (1 .. System_Constants.pthread_mutexattr_t_size) of unsigned_char;
   for pthread_mutexattr_t'Alignment use Alignment;

   type pthread_mutex_t is
     array (1 .. System_Constants.pthread_mutex_t_size) of unsigned_char;
   for pthread_mutex_t'Alignment use Alignment;

   type pthread_condattr_t is
     array (1 .. System_Constants.pthread_condattr_t_size) of unsigned_char;
   for pthread_condattr_t'Alignment use Alignment;

   type pthread_cond_t is
     array (1 .. System_Constants.pthread_cond_t_size) of unsigned_char;
   for pthread_cond_t'Alignment use Alignment;

   type pthread_t is new void_ptr;

   type pthread_key_t is new void_ptr;

end Interfaces.C.Pthreads;
