# =============================================================================
#
# File:		Editor.tcl
# Project:	TkDesk
#
# Started:	21.11.94
# Changed:	28.03.96
# Author:	cb
#
# Description:	Implements a class for multi-buffer editor windows plus
#               supporting procs.
#
# Copyright (C) 1996  Christian Bolik
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# See the file "COPYING" in the base directory of this distribution
# for more.
#
# -----------------------------------------------------------------------------
#
# Sections:
#
# =============================================================================

#
# =============================================================================
#
# Class:	dsk_Editor
# Desc:		Implements a class for multi-buffer editor windows.
#
# Methods:	buffer create <file> - create a new buffer for $file
#               buffer delete <name> - delete buffer $name
#               buffer display <name> - switch to buffer $name
# Procs:	id - used internally to name objects of this class
# Publics:
#

itcl_class dsk_Editor {

    constructor {args} {
	global tkdesk [set this]

	#
	# Create a toplevel with this object's name
	# (later accessible as $this-top):
	#
        set class [$this info class]
        ::rename $this $this-tmp-
        ::toplevel $this -class $class
	wm withdraw $this
	catch "rename $this-top {}"
        ::rename $this $this-top
        ::rename $this-tmp- $this

	frame $this.fMenu -bd 2 -relief raised
	pack $this.fMenu -fill x

	# ---- File Menu

	menubutton $this.fMenu.mbFile -text "File" -underline 0 \
		-menu $this.fMenu.mbFile.menu
 	pack $this.fMenu.mbFile -side left

	menu [set m $this.fMenu.mbFile.menu]
	$m add command -label "New  " -underline 0 \
		-command "$this buffer new {}"
	$m add command -label "Load...  " -underline 0 \
		-command "$this load"
	$m add command -label "Reload " -underline 0 \
		-command "$this buffer reload"
	$m add command -label "Print... " -underline 0 \
		-command "$this print"
	$m add separator
	$m add command -label "Save " -underline 0 \
		-command "$this save" -accelerator "Meta-s"
	$m add command -label "Save as...  " -underline 5 \
		-command "$this save as"
	$m add separator
	$m add command -label "Close File " -underline 0 \
		-command "$this buffer delete *current*" -accelerator "Meta-c"
	$m add command -label "Close Window " -underline 6 \
		-command "$this close_win"
	$m add command -label "Close All " \
		-command "dsk_editor delall"

	# ---- Edit Menu
	
	menubutton $this.fMenu.mbEdit -text "Edit" -underline 0 \
		-menu $this.fMenu.mbEdit.menu
	pack $this.fMenu.mbEdit -side left

	menu [set m $this.fMenu.mbEdit.menu]
	$m add command -label "Undo" -underline 0 -state disabled
	$m add separator
	$m add command -label "Cut" -underline 0 -command "$this cut" \
		-accelerator "Ctrl-x"
	$m add command -label "Copy" -underline 1 -command "$this copy" \
		-accelerator "Ctrl-c"
	$m add command -label "Paste" -underline 0 -command "$this paste" \
		-accelerator "Ctrl-v"
	$m add command -label "Select all" -underline 7 \
		-command "$this.ft.text tag add sel 1.0 end"
	$m add separator
	$m add command -label "Goto Line... " -underline 0 \
		-command "$this gotoline" -accelerator "Ctrl-g"
	$m add command -label "Search/Replace... " -underline 0 \
		-command "$this search" -accelerator "Ctrl-s"
	$m add command -label "HyperSearch... " -underline 0 \
		-command "$this hypersearch" -accelerator "Ctrl-h"
	$m add command -label "Find Selection" -underline 0 \
		-command "$this findsel" -accelerator "Meta-Spc"

	# ---- Options Menu

	menubutton $this.fMenu.mbOptions -text "Options" -underline 0 \
		-menu $this.fMenu.mbOptions.menu
	pack $this.fMenu.mbOptions -side left

	menu [set m $this.fMenu.mbOptions.menu]
	::set [set this](auto_indent) $tkdesk(editor,auto_indent)
	$m add checkbutton -label " Auto Indent" -underline 1 \
		-variable [set this](auto_indent) \
		-command "$this set_auto_indent"
	$m add separator
	$m add command -label "Font..." -underline 0 \
		-command "$this setfont"
	$m add command -label "Default Font  " -underline 0 \
		-command "$this setfont default"
	
	# ---- Buffer Menu

	menubutton $this.fMenu.mbBuffer -text "Buffers" -underline 0 \
		-menu $this.fMenu.mbBuffer.menu
	pack $this.fMenu.mbBuffer -side left

	menu $this.fMenu.mbBuffer.menu \
		-postcommand "$this _buffer_menu $this.fMenu.mbBuffer.menu"
	# add dummy entry to work around bug in pre Tk 4.0p2:
	$this.fMenu.mbBuffer.menu add command -label "dummy"
	
	tk_menuBar $this.fMenu $this.fMenu.mbFile $this.fMenu.mbEdit \
		$this.fMenu.mbOptions $this.fMenu.mbBuffer 
	tk_bindForTraversal $this

	# ---- Text Widget

	frame $this.f1 -bd 1 -relief raised
	pack $this.f1 -fill both -expand yes

	cb_text $this.ft -vscroll 1 -pad $tkdesk(pad) -width 20 -height 5 \
		-bd 2 -relief sunken -lborder 1 -setgrid 1 \
		-wrap char -font $tkdesk(editor,font)
	pack $this.ft -in $this.f1 -fill both -expand yes -pady $tkdesk(pad)
	blt_drag&drop target $this.ft.text handler \
		file "$this _dd_drophandler"

	#
	# Bindings
	#
	bind $this <Any-Enter> "focus $this.ft.text"
	bind $this.ft.text <Alt-s> "$this save; break"
	bind $this.ft.text <Alt-c> "$this buffer delete *current*; break"
	bind $this.ft.text <Control-c> "$this copy; break"
	bind $this.ft.text <Control-x> "$this cut; break"
	bind $this.ft.text <Control-v> "$this paste; break"	
	bind $this.ft.text <Control-g> "$this gotoline; break"
	bind $this.ft.text <Control-s> "$this search; break"
	bind $this.ft.text <Control-h> "$this hypersearch; break"
	bind $this.ft.text <Alt-space> "$this findsel; break"
	global cb_Text
	set cb_Text(change_callback,$this.ft.text) "$this _changing"
	set_auto_indent		

	#
	# Window manager settings
	#
	wm title $this "no file"
	wm minsize $this 20 5
	wm geometry $this 80x25
	wm protocol $this WM_DELETE_WINDOW "$this close_win"
	wm iconbitmap $this @$tkdesk(library)/images/pencil3.xbm

	# Other inits
	set current(buffer) ""
	::set [set this](case) 0

	eval config $args
	catch "wm deiconify $this"
	update
    }

    destructor {
	global [set this]

	::unset [set this]
        #after 10 rename $this-top {}	;# delete this name
        catch {destroy $this}	;# destroy associated window
	catch {destroy ${this}-search}
	catch {destroy ${this}-hsearch}
    }

    #
    # ----- Methods and Procs -------------------------------------------------
    #

    method config {config} {
    }

    method load {} {
	global tkdesk

	set curfile $buffer($currentid,file)
	if {[string match "New *" $curfile] || \
		[string first " " $curfile] > -1} {
	    set filter [$tkdesk(active_viewer) curdir]*
	} else {
	    if [file readable [file dirname $curfile]] {
		set filter [string trimright [file dirname $curfile] "/"]/*
	    } else {
		set filter [$tkdesk(active_viewer) curdir]*
	    }
	}
	set file [cb_fileSelector -filter $filter \
		-label "Select a file to edit:" -showall 1]

	if {$file != ""} {
	    $this buffer create $file
	}   
    }

    method save {{as ""}} {
	global tkdesk
	
	set id $currentid
	set curfile $buffer($currentid,file)
	if {[string match "New *" $curfile] || \
		[string match "* (Output)" $curfile]} {
	    set filter [$tkdesk(active_viewer) curdir]*
	} else {
	    if [file readable [file dirname $curfile]] {
		set filter [string trimright [file dirname $curfile] "/"]/*
	    } else {
		set filter [$tkdesk(active_viewer) curdir]*
	    }
	}
	if {$as != "" || [string match "New *" $buffer($id,file)] || \
		[string match "* (Output)" $curfile]} {
	    set fname [cb_fileSelector -filter $filter \
		    -label "Save file as:"]
	    if {$fname == ""} {
		return
	    } else {
		set buffer($id,file) $fname
	    }
	}

	set file [_make_fname_safe $buffer($id,file)]
	set ext [file extension $file]
	if {$ext == ".gz" || $ext == ".z"} {
	    set err [catch  "set fd \[open \"|gzip >$file\" w\]"]
	} elseif {$ext == ".Z"} {
	    set err [catch  "set fd \[open \"|compress >$file\" w\]"]
	} else {
	    set err [catch "set fd \[open \"$file\" w\]"]
	}
	if $err {
	    dsk_errbell
	    cb_error "Couldn't open $file for writing!"
	    return
	}
	dsk_busy
	puts -nonewline $fd [$this.ft.text get 1.0 end]
	close $fd
	dsk_lazy
	set changed($id) 0
	set file [subst $file]
	wm title $this "[cb_tilde $file]"
	wm iconname $this "[file tail $file]"
    }

    method print {} {
	global tkdesk tmppcmd

	set tmppcmd $tkdesk(cmd,print)
	if {[cb_readString "Print command (pipe):" tmppcmd "Print File"] \
		!= ""} {
	    set tmppcmd [string trimleft $tmppcmd "|"]
	    set tkdesk(editor,print_command) $tmppcmd
	    set fd [open "|$tmppcmd" "w"]
	    dsk_busy
	    puts -nonewline $fd [$this.ft.text get 1.0 end]
	    dsk_lazy
	    close $fd
	    set tkdesk(cmd,print) $tmppcmd
	}
	unset tmppcmd
    }
	
    method buffer {cmd args} {
	switch $cmd {
	    new {return [eval _buffer_new $args]}
	    create {return [eval _buffer_create $args]}
	    delete {return [eval _buffer_delete $args]}
	    display {return [eval _buffer_display $args]}
	    reload {return [eval _buffer_reload $args]}	    
	}
    }

    method _buffer_new {file} {
	if {$file == ""} {
	    set file "New File"
	}
	set id [incr bufcount]
	set buffer($id,file) $file
	set buffer($id,text) ""
	set buffer($id,vpos) 0
	set buffer($id,cursor) 1.0
	set buffer($id,win) $this
	set changed($id) 0
	$this buffer display $id
    }
	
    
    method _buffer_create {file} {
	global tkdesk
	
	#set file [subst $file]
	if ![file exists $file] {
	    catch "cb_info \"$file does not exist. Creating new file.\""
	    $this buffer new $file
	} else {
	    set ext [file extension $file]
	    #set file [_make_fname_safe $file]
	    if {$ext == ".gz" || $ext == ".z"} {
	        set err [catch  {set fd [open "|gzip -cd $file"]}]
	    } elseif {$ext == ".Z"} {
	        set err [catch  {set fd [open "|zcat $file"]}]
	    } else {
	        set err [catch {set fd [open $file]}]
	    }
	    if $err {
		dsk_errbell
		catch "cb_error \"Error: Couldn't open $file for reading!\""
		catch "$this delete"
		return
	    }
	    dsk_busy
	    set id [incr bufcount]
	    set buffer($id,file) $file
	    set buffer($id,text) [read $fd]
	    set buffer($id,vpos) 0
	    set buffer($id,cursor) 1.0
	    set buffer($id,win) $this
	    set changed($id) 0
	    close $fd
	    dsk_lazy
	    $this buffer display $id
	}
	if {[string first $tkdesk(configdir) $file] == 0} {
	    if ![winfo exists $this.fMenu.mbTkDesk] {
		menubutton $this.fMenu.mbTkDesk -text "TkDesk" \
			-menu [set m $this.fMenu.mbTkDesk.m] \
			-underline 0
		pack $this.fMenu.mbTkDesk -side left
		
		menu $m
		$m add command -label "Save and Reload into TkDesk " \
			-command "$this _tkdesk reload" \
			-accelerator F5
		$m add command -label "Save, Reload and Close" \
			-command "$this _tkdesk reload_and_close" \
			-accelerator F6
		bind $this.ft.text <F5> "$this _tkdesk reload"
		bind $this.ft.text <F6> "$this _tkdesk reload_and_close"
	    }
	}
    }

    method _tkdesk {cmd} {
	global tkdesk
	
	set id $currentid
	set curfile $buffer($currentid,file)
	if {[string first $tkdesk(configdir) $curfile] < 0} {
	    dsk_errbell
	    cb_error "This is not one of TkDesk's current configuration files!"
	    return
	}
	
	switch $cmd {
	    "reload" {
		save
		dsk_reread_config [file tail $curfile]
	    }
	    "reload_and_close" {
		save
		dsk_reread_config [file tail $curfile]
		buffer delete *current*
	    }
	}
    }

    method _buffer_display {id} {
	if {$currentid != -1} {
	    set buffer($currentid,vpos) [lindex \
		    [cb_old_sb_get $this.ft.vscroll] 2]
	    set buffer($currentid,cursor) [$this.ft.text index insert]
	    set buffer($currentid,text) [$this.ft.text get 1.0 end]
	}
	$this.ft.text delete 1.0 end
	$this.ft.text insert end $buffer($id,text)
	$this.ft.text yview $buffer($id,vpos)
	$this.ft.text mark set insert $buffer($id,cursor)
	if {[wm state $this] == "iconic"} {
	    wm deiconify $this
	} else {
	    raise $this
	}
	set currentid $id
	set name $buffer($id,file)
	if $changed($id) {
	    wm title $this "* [cb_tilde $name]"
	    wm iconname $this "*[file tail $name]"
	} else {
	    wm title $this [cb_tilde $name]
	    wm iconname $this [file tail $name]
	}
    }

    method _buffer_delete {id} {
	if {$id == "*current*"} {
	    set id $currentid
	}

	if $changed($id) {
	    if {$currentid != $id} {
		$this buffer display $id
	    }

	    cb_raise $this
	    catch {destroy $this.tqmod}
	    set ans [tk_dialog $this.tqmod "File modified" \
		    "This file has been modified. Save it?" \
		    questhead 0 "Yes" "No" "Cancel"]
	    if {$ans == 0} {
		save
	    } elseif {$ans == 2} {
		return "cancel"
	    }
	}
	
	$this.ft.text delete 1.0 end
	set currentid -1
	unset buffer($id,file)
	unset buffer($id,text)
	unset buffer($id,vpos)
	unset buffer($id,cursor)
	unset buffer($id,win)
	unset changed($id)	

	set id -1
	for {set i 1} {$i <= $bufcount} {incr i} {
	    if [info exists buffer($i,file)] {
		if {$buffer($i,win) == $this} {
		    set id $i
		    break
		}
	    }
	}
	if {$id != -1} {
	    $this buffer display $id
	    return ""
	} else {
	    $this delete
	}
    }

    method _buffer_reload {} {
	
	set id $currentid

	if $changed($id) {
	    if {$currentid != $id} {
		$this buffer display $id
	    }
	    cb_raise $this
	    catch {destroy $this.tqmod}
	    set ans [tk_dialog $this.tqmod "File modified" \
		    "This file has been modified. Save it?" \
		    questhead 0 "Yes" "No" "Cancel"]
	    if {$ans == 0} {
		save
	    } elseif {$ans == 2} {
		return "cancel"
	    }
	}
	set file $buffer($id,file)
	set ext [file extension $file]
	if {$ext == ".gz" || $ext == ".z"} {
	    set err [catch  "set fd \[open \"|gzip -cd $file\"\]"]
	} elseif {$ext == ".Z"} {
	    set err [catch  "set fd \[open \"|zcat $file\"\]"]
	} else {
	    set err [catch "set fd \[open $file\]"]
	}
	if $err {
	    dsk_errbell
	    cb_error "Error: Couldn't open $file for reading!"
	    return
	}
	dsk_busy
	set buffer($id,text) [read $fd]
	set changed($id) 0
	close $fd
	wm title $this [cb_tilde $file]
	wm iconname $this "[file tail $file]"
	dsk_lazy
	set buffer($id,vpos) [lindex [cb_old_sb_get $this.ft.vscroll] 2]
	set buffer($id,cursor) [$this.ft.text index insert]
	$this.ft.text delete 1.0 end
	$this.ft.text insert end $buffer($id,text)
	$this.ft.text yview $buffer($id,vpos)
	$this.ft.text mark set insert $buffer($id,cursor)
    }	

    method _buffer_menu {menu} {
	global tkdesk
	
	catch "$menu delete 0 last"
	for {set id 1} {$id <= $bufcount} {incr id} {
	    if ![info exists buffer($id,file)] continue
	    if {$id == $currentid} {
		if $changed($id) {
		    set l "@[file tail $buffer($id,file)]"
		} else {
		    set l ">[file tail $buffer($id,file)]"		    
		}
	    } else {
		if $changed($id) {
		    set l "*[file tail $buffer($id,file)]"
		} else {
		    set l " [file tail $buffer($id,file)]"		    
		}
	    }
	    $menu add command -label $l \
		    -font $tkdesk(font,file_lbs) \
		    -command "$buffer($id,win) buffer display $id"
	}
    }

    method _changing {} {
	
	if !$changed($currentid) {
	    set changed($currentid) 1
	    set file $buffer($currentid,file)
	    wm title $this "* [cb_tilde $file]"
	    wm iconname $this "*[file tail $file]"
	}
    }

    method close_win {} {
	for {set id 1} {$id <= $bufcount} {incr id} {
	    if [info exists buffer($id,file)] {
		if {$buffer($id,win) == $this} {
		    if {[$this buffer delete $id] == "cancel"} {
			return cancel
		    }
		}
	    }
	}
	return ""
    }

    method copy {} {
	global cb_Text
	
	set sel [$this.ft.text tag ranges sel]
	if {$sel != ""} {
	    set cutbuffer [eval $this.ft.text get $sel]
	    selection clear $this.ft.text
	    catch "unset cb_Text(selstart)"
	}
    }

    method cut {} {
	global cb_Text

	set sel [$this.ft.text tag ranges sel]
	if {$sel != ""} {
	    set cutbuffer [eval $this.ft.text get $sel]
	    eval $this.ft.text delete $sel
	    cb_Text_change_callback $this.ft.text
	    selection clear $this.ft.text
	    catch "unset cb_Text(selstart)"
	}
    }

    method paste {} {
	global cb_Text

	if {$cutbuffer != ""} {
	    $this.ft.text insert insert $cutbuffer
	    $this.ft.text yview -pickplace insert
	    cb_Text_change_callback $this.ft.text
	    selection clear $this.ft.text
	    catch "unset cb_Text(selstart)"
	} else {
	    dsk_bell
	}
    }

    method gotoline {} {
	global tmplnr
	
	set curline [lindex [split [$this.ft.text index insert] "."] 0]
	set tmplnr ""
	cb_readString "Goto line (current: $curline):" tmplnr "Goto Line" 10
	if {$tmplnr != ""} {
	    # test if $tmplnr contains a number:
	    set err [catch {$this.ft.text mark set insert $tmplnr.0}]
	    if !$err {
		$this.ft.text yview -pickplace insert
	    } else {
		dsk_errbell
		cb_error "Invalid line number!"
	    }
	}
	unset tmplnr
    }

    method search {} {
	global tkdesk [set this]
	
	set t "$this-search"
	if [winfo exists $t] {
	    cb_raise $t
	    focus $t.es
	    return
	}

	toplevel $t
	wm withdraw $t
	
	frame $t.fs -bd 1 -relief raised
	pack $t.fs -fill both -expand yes
	frame $t.fsf
	pack $t.fsf -in $t.fs -fill both -expand yes \
		-padx $tkdesk(pad) -pady $tkdesk(pad)
	frame $t.fslf
	pack $t.fslf -in $t.fsf -fill x
	label $t.ls -text "Search for (regexp):" -anchor w
	pack $t.ls -in $t.fslf -side left
	checkbutton $t.cbCase -text "Case sensitive" -relief flat \
		-variable [set this](case)
	pack $t.cbCase -in $t.fslf -side right
	entry $t.es -bd 2 -relief sunken -width 40
	pack $t.es -in $t.fsf -fill x -expand yes -ipady 2 -pady $tkdesk(pad)
	bind $t.es <1> \
		"focus %W ;\
		$t.bReplace config -relief flat ;\
		$t.bSearch config -relief sunken"
	bind $t.es <Tab> \
		"$this _do_search ;\
		focus $t.er ;\
		$t.bSearch config -relief flat ;\
		$t.bReplace config -relief sunken"
	bind $t.es <Escape> "destroy $t"
	bind $t.es <Return> "$this _do_search"

	frame $t.fr -bd 1 -relief raised
	pack $t.fr -fill both -expand yes
	frame $t.frf
	pack $t.frf -in $t.fr -fill both -expand yes \
		-padx $tkdesk(pad) -pady $tkdesk(pad)
	label $t.lr -text "Replace with:" -anchor w
	pack $t.lr -in $t.frf -anchor w
	entry $t.er -bd 2 -relief sunken -width 40
	pack $t.er -in $t.frf -fill x -expand yes -ipady 2 -pady $tkdesk(pad)
	bind $t.er <1> \
		"$this _do_search ;\
		focus %W ;\
		$t.bSearch config -relief flat ;\
		$t.bReplace config -relief sunken"
	bind $t.er <Tab> \
		"focus $t.es ;\
		$t.bReplace config -relief flat ;\
		$t.bSearch config -relief sunken"
	bind $t.er <Escape> "destroy $t"
	bind $t.er <Return> "$this _do_replace"

	frame $t.fb -bd 1 -relief raised
	pack $t.fb -fill x
	cb_button $t.bSearch -text  "Search" -default 1 \
		-command "$this _do_search"
	cb_button $t.bReplace -text "Replace" \
		-command "$this _do_replace"
	cb_button $t.bRepAll -text "Replace all" \
		-command "$this _do_replace all"
	cb_button $t.bClose -text " Close " -command "destroy $t"
	pack $t.bSearch $t.bReplace $t.bRepAll $t.bClose \
		-in $t.fb -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad)

	bind $t <Any-Enter> \
		"if \{\[focus\] != \"$t.er\"\} \{ \
		focus $t.es ;\
		$t.bReplace config -relief flat ;\
		$t.bSearch config -relief sunken ;\
		\}"
	
	
	wm title $t "Search/Replace"
	wm minsize $t 311 170
	update idletasks

	# determine position of search window:
	set rw [winfo reqwidth $t]
	set rh [winfo reqheight $t]
	set ww [winfo width $this]
	set wh [winfo height $this]
	set sw [winfo screenwidth $t]
	set sh [winfo screenheight $t]
	set wx [winfo x $this]
	set wy [winfo y $this]
	set dy 34 ;# offset for window decoration
	set x $wx
	if {$x < 0} {set x 0}
	if {$x + $rw > $sw} {set x [expr $sw - $rw]}
	set h1 $wy
	set h2 [expr $sh - $wy - $wh]
	if {$h1 > $h2} {
	    set y [expr $wy - $rh]
	    if {$y < 0} {set y 0}
	} else {
	    set y [expr $wy + $wh]
	    if {$y + $rh + $dy > $sh} {
		set y [expr $sh - $rh - $dy]
	    }
	}
	#puts "$wx $wy $ww $wh, $rw $rh, $h1 $h2, $x $y"
	wm geometry $t +$x+$y
	wm deiconify $t

	focus $t.es
    }

    method _do_search {} {
	global [set this]

	set exp [$this-search.es get]
	if {$exp == ""} return

	set match_range ""
	set tw $this.ft.text
	catch "$tw tag remove sel 1.0 end"
	dsk_busy
	set stidx "insert"
	if ![set [set this](case)] {
	    set success [regexp -nocase -indices -- \
		    $exp [$tw get insert end] mrange]
	} else {
	    set success [regexp -indices -- \
		    $exp [$tw get insert end] mrange]
	}
       	dsk_lazy
	if !$success {
	    set restart ![cb_yesno "No more matches. Restart at top?"]
	    if $restart {
		dsk_busy
		set stidx "1.0"
		if ![set [set this](case)] {
		    set success [regexp -nocase -indices -- \
			    $exp [$tw get 1.0 insert] mrange]
		} else {
		    set success [regexp -indices -- \
			    $exp [$tw get 1.0 insert] mrange]
		}
		dsk_lazy
	    } else {
		return
	    }
	}
	if $success {
	    set mstart [lindex $mrange 0]
	    set mend [expr [lindex $mrange 1] + 1]
	    lappend match_range [$tw index "$stidx + $mstart chars"]
	    lappend match_range [$tw index "$stidx + $mend chars"]
	    $tw tag add sel "$stidx + $mstart chars" "$stidx + $mend chars"
	    $tw yview -pickplace "$stidx + $mstart chars"
	    $tw mark set insert "$stidx + $mend chars"
	    set search_regexp $exp
	} else {
	    dsk_bell
	}
    }

    method _do_replace {{mode "single"}} {
	global [set this] cb_Text
	
	if {$match_range == ""} {
	    $this _do_search
	}
	if {$match_range == ""} return	
	set mstart [lindex $match_range 0]
	if {$mode == "single"} {
	    set mend [lindex $match_range 1]
	} else {
	    set mend end
	}
	set subexp [$this-search.er get]
	if {$subexp == ""} return

	dsk_busy
	set exp $search_regexp
	set tw $this.ft.text
	set otext [$tw get $mstart $mend]
	if ![set [set this](case)] {
	    if {$mode == "single"} {
		regsub -nocase -- $exp $otext $subexp stext
	    } else {
		regsub -all -nocase -- $exp $otext $subexp stext
	    }
	} else {
	    if {$mode == "single"} {
		regsub -- $exp $otext $subexp stext
	    } else {
		regsub -all -- $exp $otext $subexp stext
	    }
	}
	
	$tw delete $mstart $mend
	$tw insert $mstart $stext
	cb_Text_change_callback $tw
	selection clear $tw
	catch "unset cb_Text(selstart)"
	dsk_lazy

	if {$mode == "single"} {
	    $this _do_search
	}
    }

    # ----------------------
    # hypersearch:
    # Creates a toplevel to enter a regular expression. Than grep is run on
    # the current buffer and the listbox is filled with matching lines.
    # Pressing MB 1 on one listbox entry makes the respective line the first
    # visible in the buffer.
    #
    method hypersearch {} {
	global tkdesk [set this]
	
	set t "$this-hsearch"
	if [winfo exists $t] {
	    cb_raise $t
	    return
	}

	toplevel $t
	wm withdraw $t
	
	frame $t.fs -bd 1 -relief raised
	pack $t.fs -fill x
	frame $t.fsf
	pack $t.fsf -in $t.fs -fill x \
		-padx $tkdesk(pad) -pady $tkdesk(pad)
	frame $t.fslf
	pack $t.fslf -in $t.fsf -fill x
	label $t.ls -text "Search for (regexp):" -anchor w
	pack $t.ls -in $t.fslf -side left
	checkbutton $t.cbCase -text "Case sensitive" -relief flat \
		-variable [set this](case)
	pack $t.cbCase -in $t.fslf -side right
	entry $t.es -bd 2 -relief sunken -width 40
    	menubutton $t.mbHist -bd 2 -relief raised \
		-bitmap @$tkdesk(library)/cb_tools/bitmaps/combo.xbm \
		-menu $t.mbHist.menu
    	menu $t.mbHist.menu \
		-postcommand "hsearch_history buildmenu $t.mbHist.menu"
	# add dummy entry to work around bug in pre Tk 4.0p2:
	$t.mbHist.menu add command -label "dummy"
	pack $t.es \
		-in $t.fsf -side left -fill x -expand yes \
		-ipady 2 -pady $tkdesk(pad)
	pack $t.mbHist \
		-in $t.fsf -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad) -ipadx 2 -ipady 2
	bind $t.es <Escape> "destroy $t"
	bind $t.es <Return> "
	    $this _do_hsearch
	    hsearch_history add \[list \[$t.es get\]\]
	"

	frame $t.fm -bd 1 -relief raised
	pack $t.fm -fill both -expand yes
	frame $t.fmf
	pack $t.fmf -in $t.fm -fill both -expand yes \
		-padx $tkdesk(pad) -pady $tkdesk(pad)
	label $t.lMatches -text "Matches:" -anchor w
	pack $t.lMatches -in $t.fmf -fill x
	cb_listbox $t.flb -vscroll 1 -hscroll 1 -lborder 0 -uborder 1 \
		-width 10 -height 4 \
		-font $tkdesk(editor,font) -setgrid 1 \
		-selectmode single
	# $t.flb config -bd 1 -relief raised
	pack $t.flb -in $t.fmf -fill both -expand yes
	bind $t.flb.lbox <1> "
	    %W selection clear 0 end
	    set tmplbi \[%W  nearest %y\]
	    %W selection set \$tmplbi
	    $this _hsearch_callback \$tmplbi
	    unset tmplbi
	"

	frame $t.fb -bd 1 -relief raised
	pack $t.fb -fill x
	cb_button $t.bSearch -text " Search " \
		-command "$this _do_hsearch ; \
		          hsearch_history add \[list \[$t.es get\]\]" \
		-default 1
	cb_button $t.bClose -text "  Close  " -command "destroy $t"
	pack $t.bSearch $t.bClose \
		-in $t.fb -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad)

	bind $t <Any-Enter> "focus $t.es"
	
	wm title $t {"HyperSearch"}
	wm minsize $t 6 2
	wm geometry $t 30x6
	wm deiconify $t
    }

    method _hsearch_callback {line_nr} {
	$this.ft.text yview [expr [lindex $hsearch_lnr $line_nr] - 1]
    }

    method _do_hsearch {} {
	global [set this]

	set exp [$this-hsearch.es get]
	if {$exp == ""} return

	set t "$this-hsearch"
	set lb "$t.flb.lbox"
	$lb delete 0 end

	dsk_busy
	if ![set [set this](case)] {
	    set err [catch "set grep_output \
		    \[exec grep -ni \$exp << \[$this.ft.text get 1.0 end\]\]"]
	} else {
	    set err [catch "set grep_output \
		    \[exec grep -n \$exp << \[$this.ft.text get 1.0 end\]\]"]
	}
	if $err {
	    dsk_lazy
	    dsk_bell
	    return
	}
	set grep_output [split $grep_output \n]

	set hsearch_lnr ""
	foreach match $grep_output {
	    set lnr [string range $match 0 [expr [string first : $match] - 1]]
	    lappend hsearch_lnr $lnr
	    $lb insert end [string range $match \
		    [expr [string first : $match] + 1] end]
	}
	dsk_lazy
    }

    
    method findsel {} {
	global [set this] cb_Text
	
	set tw $this.ft.text
	set selr [$tw tag ranges sel]
	catch "unset cb_Text(selstart)"
	if {$selr != ""} {
	    set exp [$tw get [lindex $selr 0] [lindex $selr 1]]
	    dsk_busy
	    if ![set [set this](case)] {
		set success [regexp -nocase -indices -- \
			$exp [$tw get insert end] mrange]
	    } else {
		set success [regexp -indices -- \
			$exp [$tw get insert end] mrange]
	    }
	    dsk_lazy
	    if $success {
		selection clear $tw
		set mstart [lindex $mrange 0]
		set mend [expr [lindex $mrange 1] + 1]
		$tw tag add sel "insert + $mstart chars" "insert + $mend chars"
		$tw yview -pickplace "insert + $mstart chars"
		$tw mark set insert "insert + $mend chars"
	    } else {
		dsk_bell
	    }
	} else {
	    dsk_bell
	}
    }

    method setfont {{what ""}} {
	global tkdesk
	
	if {$what == "default"} {
	    $this.ft.text config -font $tkdesk(editor,font)
	} else {
	    $this.ft.text config -font [cb_fontSel]
	}
    }

    method set_auto_indent {} {
	global [set this]

	if [set [set this](auto_indent)] {
	    bind $this.ft.text <Return> {
		set tmpline [%W get "insert linestart" "insert lineend"]
		regexp {^[	 ]*} $tmpline tmpmatch
		%W insert insert "\n$tmpmatch"
		%W yview -pickplace insert
		cb_Text_change_callback %W
		unset tmpline tmpmatch
		break
	    }
	} else {
	    bind $this.ft.text <Return> {
		%W insert insert \n
		%W yview -pickplace insert
		cb_Text_change_callback %W
		break
	    }
	}
    }

    method _dd_drophandler {} {
	global DragDrop

	foreach file $DragDrop(file) {
	    $this buffer create $file
	}
    }
	
    proc id {{cmd ""}} {
	if {$cmd == ""} {
	    set i $object_id
	    incr object_id
	    return $i
	} elseif {$cmd == "reset"} {
	    set object_id 0
	}
    }

    #
    # ----- Variables ---------------------------------------------------------
    #

    public files "" {
	foreach f $files {
	    $this buffer create $f
	}
    }

    public name "" {
	
	set buffer($currentid,file) $name
	wm title $this [cb_tilde $name]
	wm iconname $this "[file tail $name]"
    }

    protected currentid -1
    protected match_range ""
    protected search_regexp ""
    protected hsearch_lnr ""

    common buffer
    common bufcount 0
    common changed
    common object_id 0
    common cutbuffer ""
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_editor
# Args:		cmd	command to invoke
#		args	other arguments
# Returns: 	""
# Desc:		Meta function to access the built-in editor.
# Side-FX:	none
#

set dsk_editor(cnt) 0
proc dsk_editor {cmd args} {
    global dsk_editor

    switch $cmd {
	new {
    	    set w .de[dsk_Editor :: id]
	    dsk_Editor $w
	    $w buffer new {}
	}
	load {
	    set w .de[dsk_Editor :: id]
	    dsk_Editor $w
	    foreach file $args {
		$w buffer create $file
	    }
	}
	string {
	    set w .de[dsk_Editor :: id]
	    dsk_Editor $w
	    $w buffer new {}
	    eval $w.ft.text insert end $args
	    # $w _changing
	}	    
	delall {
	    foreach obj [itcl_info objects -class dsk_Editor] {
		$obj close_win
	    }
	}
	cmd {
	    set cnt [incr dsk_editor(cnt)]
	    set cmd $args
	    blt_bgexec dsk_editor(stat$cnt) -output dsk_editor(result$cnt) \
		    /bin/sh -c "$cmd" &
	    dsk_status "Launched:  $cmd"
	    tkwait variable dsk_editor(stat$cnt)
	    dsk_status "Exit:  $cmd"
	    set w .de[dsk_Editor :: id]
	    dsk_Editor $w
	    $w buffer new {}
	    $w.ft.text insert end $dsk_editor(result$cnt)
	    $w.ft.text mark set insert 1.0
	    $w config -name "$cmd (Output)"
	}
    }
}


proc dsk_editor_hsearch_cb {t exp} {
    $t.es delete 0 end
    $t.es insert end [string trimright [string trimleft $exp \{] \}]
}



