/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include "gtk.h"

typedef enum {
  SEL_TYPE_NONE,
  APPLE_PICT,
  ATOM,
  ATOM_PAIR,
  BITMAP,
  C_STRING,
  COLORMAP,
  COMPOUND_TEXT,
  DRAWABLE,
  INTEGER,
  PIXEL,
  PIXMAP,
  SPAN,
  STRING,
  TEXT,
  WINDOW,
  LAST_SEL_TYPE
} SelType;

GdkAtom seltypes[LAST_SEL_TYPE];

typedef struct _Target {
  gchar *target_name;
  SelType type;
  GdkAtom target;
  gint format;
  GtkSelectionFunction *handler;
} Target;

/* The following is a list of all the selection targets defined
   in the ICCCM */

static Target targets[] = {
  { "ADOBE_PORTABLE_DOCUMENT_FORMAT",	    STRING, 	   0, 8,  NULL },
  { "APPLE_PICT", 			    APPLE_PICT,    0, 8,  NULL },
  { "BACKGROUND",			    PIXEL,         0, 32, NULL },
  { "BITMAP", 				    BITMAP,        0, 32, NULL },
  { "CHARACTER_POSITION",                   SPAN, 	   0, 32, NULL },
  { "CLASS", 				    TEXT, 	   0, 8,  NULL },
  { "CLIENT_WINDOW", 			    WINDOW, 	   0, 32, NULL },
  { "COLORMAP", 			    COLORMAP,      0, 32, NULL },
  { "COLUMN_NUMBER", 			    SPAN, 	   0, 32, NULL },
  { "COMPOUND_TEXT", 			    COMPOUND_TEXT, 0, 8,  NULL },
  /*  { "DELETE", "NULL", 0, ?, NULL }, */
  { "DRAWABLE", 			    DRAWABLE,      0, 32, NULL },
  { "ENCAPSULATED_POSTSCRIPT", 		    STRING, 	   0, 8,  NULL },
  { "ENCAPSULATED_POSTSCRIPT_INTERCHANGE",  STRING, 	   0, 8,  NULL },
  { "FILE_NAME", 			    TEXT, 	   0, 8,  NULL },
  { "FOREGROUND", 			    PIXEL, 	   0, 32, NULL },
  { "HOST_NAME", 			    TEXT, 	   0, 8,  NULL },
  /*  { "INSERT_PROPERTY", "NULL", 0, ? NULL }, */
  /*  { "INSERT_SELECTION", "NULL", 0, ? NULL }, */
  { "LENGTH", 				    INTEGER, 	   0, 32, NULL },
  { "LINE_NUMBER", 			    SPAN, 	   0, 32, NULL },
  { "LIST_LENGTH", 			    INTEGER,       0, 32, NULL },
  { "MODULE", 				    TEXT, 	   0, 8,  NULL },
  /*  { "MULTIPLE", "ATOM_PAIR", 0, 32, NULL }, */
  { "NAME", 				    TEXT, 	   0, 8,  NULL },
  { "ODIF", 				    TEXT,          0, 8,  NULL },
  { "OWNER_OS", 			    TEXT, 	   0, 8,  NULL },
  { "PIXMAP", 				    PIXMAP,        0, 32, NULL },
  { "POSTSCRIPT", 			    STRING,        0, 8,  NULL },
  { "PROCEDURE", 			    TEXT,          0, 8,  NULL },
  { "PROCESS",				    INTEGER,       0, 32, NULL },
  { "STRING", 				    STRING,        0, 8,  NULL },
  { "TARGETS", 				    ATOM, 	   0, 32, NULL },
  { "TASK", 				    INTEGER,       0, 32, NULL },
  { "TEXT", 				    TEXT,          0, 8 , NULL },
  { "TIMESTAMP", 			    INTEGER,       0, 32, NULL },
  { "USER", 				    TEXT, 	   0, 8,  NULL },
};

static int num_targets = sizeof(targets)/sizeof(Target);

static int have_selection = FALSE;

GtkWidget *selection_text;
GtkWidget *selection_button;
GString *selection_string = NULL;

static void
init_atoms ()
{
  int i;

  seltypes[SEL_TYPE_NONE] = GDK_NONE;
  seltypes[APPLE_PICT] = gdk_atom_intern ("APPLE_PICT",FALSE);
  seltypes[ATOM]       = gdk_atom_intern ("ATOM",FALSE);
  seltypes[ATOM_PAIR]  = gdk_atom_intern ("ATOM_PAIR",FALSE);
  seltypes[BITMAP]     = gdk_atom_intern ("BITMAP",FALSE);
  seltypes[C_STRING]   = gdk_atom_intern ("C_STRING",FALSE);
  seltypes[COLORMAP]   = gdk_atom_intern ("COLORMAP",FALSE);
  seltypes[COMPOUND_TEXT] = gdk_atom_intern ("COMPOUND_TEXT",FALSE);
  seltypes[DRAWABLE]   = gdk_atom_intern ("DRAWABLE",FALSE);
  seltypes[INTEGER]    = gdk_atom_intern ("INTEGER",FALSE);
  seltypes[PIXEL]      = gdk_atom_intern ("PIXEL",FALSE);
  seltypes[PIXMAP]     = gdk_atom_intern ("PIXMAP",FALSE);
  seltypes[SPAN]       = gdk_atom_intern ("SPAN",FALSE);
  seltypes[STRING]     = gdk_atom_intern ("STRING",FALSE);
  seltypes[TEXT]       = gdk_atom_intern ("TEXT",FALSE);
  seltypes[WINDOW]     = gdk_atom_intern ("WINDOW",FALSE);

  for (i=0; i<num_targets; i++)
    targets[i].target = gdk_atom_intern (targets[i].target_name, FALSE);
}

void
selection_toggled (GtkWidget *widget)
{
  if (GTK_TOGGLE_BUTTON(widget)->active)
    {
      have_selection = gtk_selection_owner_set (widget,
						GDK_SELECTION_PRIMARY,
						GDK_CURRENT_TIME);
      if (!have_selection)
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(widget), FALSE);
    }
  else
    {
      if (have_selection)
	{
	  if (gdk_selection_owner_get (GDK_SELECTION_PRIMARY) == widget->window)
	    gtk_selection_owner_set (NULL, GDK_SELECTION_PRIMARY,
				     GDK_CURRENT_TIME);
	  have_selection = FALSE;
	}
    }
}

void
selection_handle (GtkWidget *widget, 
		  GtkSelectionData *selection_data, gpointer data)
{
  guchar *buffer;
  gint len;

  if (!selection_string)
    {
      buffer = NULL;
      len = 0;
    }      
  else
    {
      buffer = (guchar *)selection_string->str;
      len = selection_string->len;
    }
  
  gtk_selection_data_set (selection_data,
			  selection_data->target == seltypes[COMPOUND_TEXT] ?
			          seltypes[COMPOUND_TEXT] : seltypes[STRING],
			  8, buffer, len);
}

gint
selection_clear (GtkWidget *widget, GdkEventSelection *event)
{
  have_selection = FALSE;
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(widget), FALSE);

  return TRUE;
}

gchar *
stringify_atom (guchar *data, gint *position)
{
  gchar *str = gdk_atom_name (*(GdkAtom *)(data+*position));
  *position += sizeof(GdkAtom);
    
  return str;
}

gchar *
stringify_text (guchar *data, gint *position)
{
  gchar *str = g_strdup ((gchar *)(data+*position));
  *position += strlen (str) + 1;
    
  return str;
}

gchar *
stringify_xid (guchar *data, gint *position)
{
  gchar buffer[20];
  gchar *str;

  sprintf(buffer,"0x%x",*(guint32 *)(data+*position));
  str = g_strdup (buffer);

  *position += sizeof(guint32);
    
  return str;
}

gchar *
stringify_integer (guchar *data, gint *position)
{
  gchar buffer[20];
  gchar *str;

  sprintf(buffer,"%d",*(int *)(data+*position));
  str = g_strdup (buffer);

  *position += sizeof(int);
    
  return str;
}

gchar *
stringify_span (guchar *data, gint *position)
{
  gchar buffer[42];
  gchar *str;

  sprintf(buffer,"%d - %d",((int *)(data+*position))[0],
	  ((int *)(data+*position))[1]);
  str = g_strdup (buffer);

  *position += 2*sizeof(int);
    
  return str;
}

void
selection_received (GtkWidget *widget, GtkSelectionData *data)
{
  int position;
  int i;
  SelType seltype;
  char *str;
  
  if (data->length < 0)
    {
      g_print("Error retrieving selection\n");
      return;
    }

  seltype = SEL_TYPE_NONE;
  for (i=0; i<LAST_SEL_TYPE; i++)
    {
      if (seltypes[i] == data->type)
	{
	  seltype = i;
	  break;
	}
    }

  if (seltype == SEL_TYPE_NONE)
    {
      char *name = gdk_atom_name (data->type);
      g_print("Don't know how to handle type: %s (%ld)\n",
	      name?name:"<unknown>",
	      data->type);
      return;
    }

  if (selection_string != NULL)
    g_string_free (selection_string, TRUE);

  selection_string = g_string_new (NULL);

  gtk_text_freeze (GTK_TEXT (selection_text));
  gtk_text_set_point (GTK_TEXT (selection_text), 0);
  gtk_text_forward_delete (GTK_TEXT (selection_text), 
			   gtk_text_get_length (GTK_TEXT (selection_text)));

  position = 0;
  while (position < data->length)
    {
      switch (seltype)
	{
	case ATOM:
	  str = stringify_atom (data->data, &position);
	  break;
	case COMPOUND_TEXT:
	case STRING:
	case TEXT:
	  str = stringify_text (data->data, &position);
	  break;
	case BITMAP:
	case DRAWABLE:
	case PIXMAP:
	case WINDOW:
	case COLORMAP:
	  str = stringify_xid (data->data, &position);
	  break;
	case INTEGER:
	case PIXEL:
	  str = stringify_integer (data->data, &position);
	  break;
	case SPAN:
	  str = stringify_span (data->data, &position);
	  break;
	default:
	  {
	    char *name = gdk_atom_name (data->type);
	    g_print("Can't convert type %s (%ld) to string\n",
		    name?name:"<unknown>",
		    data->type);
	    position = data->length;
	    continue;
	  }
	}
      gtk_text_insert (GTK_TEXT (selection_text), NULL, 
		       &selection_text->style->black, 
		       NULL, str, -1);
      gtk_text_insert (GTK_TEXT (selection_text), NULL, 
		       &selection_text->style->black, 
		       NULL, "\n", -1);
      g_string_append (selection_string, str);
      g_free (str);
    }
  gtk_text_thaw (GTK_TEXT (selection_text));
}

void
paste (GtkWidget *widget, GtkWidget *entry)
{
  char *name;
  GdkAtom atom;

  name = gtk_entry_get_text (GTK_ENTRY(entry));
  atom = gdk_atom_intern (name, FALSE);

  if (atom == GDK_NONE)
    {
      g_print("Could not create atom: \"%s\"\n",name);
      return;
    }

  gtk_selection_convert (selection_button, GDK_SELECTION_PRIMARY, atom, 
			 GDK_CURRENT_TIME);
}

void
quit ()
{
  gtk_exit (0);
}

int
main (int argc, char *argv[])
{
  GtkWidget *dialog;
  GtkWidget *button;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *entry;
  GtkWidget *hscrollbar;
  GtkWidget *vscrollbar;
  GtkWidget *hbox;
  
  gtk_init (&argc, &argv);

  init_atoms();

  dialog = gtk_dialog_new ();
  gtk_widget_set_name (dialog, "Test Input");
  gtk_container_border_width (GTK_CONTAINER(dialog), 0);

  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
		      GTK_SIGNAL_FUNC (quit), NULL);

  table = gtk_table_new (4, 2, FALSE);
  gtk_container_border_width (GTK_CONTAINER(table), 10);

  gtk_table_set_row_spacing (GTK_TABLE (table), 0, 5);
  gtk_table_set_row_spacing (GTK_TABLE (table), 1, 2);
  gtk_table_set_row_spacing (GTK_TABLE (table), 2, 2);
  gtk_table_set_col_spacing (GTK_TABLE (table), 0, 2);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->vbox), 
		      table, TRUE, TRUE, 0);
  gtk_widget_show (table);
  
  selection_button = gtk_toggle_button_new_with_label ("Claim Selection");
  gtk_table_attach (GTK_TABLE (table), selection_button, 0, 2, 0, 1,
		    GTK_EXPAND | GTK_FILL, 0, 0, 0);
  gtk_widget_show (selection_button);

  gtk_signal_connect (GTK_OBJECT(selection_button), "toggled",
		      GTK_SIGNAL_FUNC (selection_toggled), NULL);
  gtk_signal_connect (GTK_OBJECT(selection_button), "selection_clear_event",
		      GTK_SIGNAL_FUNC (selection_clear), NULL);
  gtk_signal_connect (GTK_OBJECT(selection_button), "selection_received",
		      GTK_SIGNAL_FUNC (selection_received), NULL);

  gtk_selection_add_handler (selection_button, GDK_SELECTION_PRIMARY,
			     seltypes[STRING], selection_handle, NULL);

  gtk_selection_add_handler (selection_button, GDK_SELECTION_PRIMARY,
			     seltypes[TEXT], selection_handle, NULL);

  gtk_selection_add_handler (selection_button, GDK_SELECTION_PRIMARY,
			     seltypes[COMPOUND_TEXT],
			     selection_handle, NULL);

  selection_text = gtk_text_new (NULL, NULL);
  gtk_table_attach_defaults (GTK_TABLE (table), selection_text, 0, 1, 1, 2);
  gtk_widget_show (selection_text);
  
  hscrollbar = gtk_hscrollbar_new (GTK_TEXT (selection_text)->hadj);
  gtk_table_attach (GTK_TABLE (table), hscrollbar, 0, 1, 2, 3,
		    GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (hscrollbar);
  
  vscrollbar = gtk_vscrollbar_new (GTK_TEXT (selection_text)->vadj);
  gtk_table_attach (GTK_TABLE (table), vscrollbar, 1, 2, 1, 2,
		    GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_widget_show (vscrollbar);

  hbox = gtk_hbox_new (FALSE, 2);
  gtk_table_attach (GTK_TABLE (table), hbox, 0, 2, 3, 4,
		    GTK_EXPAND | GTK_FILL, 0, 0, 0);
  gtk_widget_show (hbox);

  label = gtk_label_new ("Target:");
  gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  entry = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX(hbox), entry, TRUE, TRUE, 0);
  gtk_widget_show (entry);

  /* .. And create some buttons */
  button = gtk_button_new_with_label ("Paste");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), 
		      button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (paste), entry);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Quit");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), 
		      button, TRUE, TRUE, 0);

  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), 
			     GTK_OBJECT (dialog));
  gtk_widget_show (button);

  gtk_widget_show (dialog);

  gtk_main ();

  return 0;
}
