/*
 *  Apple II emulator by Alexander Jean-Claude Bottema (C) 1994
 *
 *  $Id: defs.H,v 1.15 1998/02/07 21:21:37 chernabog Exp $
 *
 * MODIFICATION HISTORY
 *   v0.3 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jan 1997.
 *   v0.4 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jun 1997.
 *   v0.5 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Feb 1998.
 *     This code has nothing to do with my employer, GTE Internetworking,
 *     BBN Technologies.  It was written completely on my own time and on
 *     my own machine.
 *
 *  global #defines, typedefs, enums, etc for assembly and C source
 *  files.
 **/


#ifdef __ASSEMBLY__

#define FLAG(a,b) a ## b

#else

#define FLAG(a,b) b

#define	K *1024				/* kilobytes */
typedef void (*Function)();
typedef enum { False, True } Tr;	/* --- Domain of Truth values
					       (using the notation by
					       John Allen - "The
					       Anatomy of LISP") --- */

#define IIE_MODE 2
#define IIU_MODE 1
#define II_MODE  0
#define LAZY_COLOR 1
#define COLOR 2
#define LAZY_INTERP 3
#define INTERP 4
typedef enum {JOY_OFF, JOY_KYBD, JOY_DIGITAL, JOY_PCJOY} joystick_mode;


#define SW_TEXT 0xC050
#define SW_MIXED 0xC052
#define SW_PAGE2 0xC054
#define SW_HIRES 0xC056
#ifdef APPLE_IIE
#define SW_80STORE 0xC000
#define SW_RAMRD 0xC002
#define SW_RAMWRT 0xC004
#define SW_ALTZP 0xC008
#define SW_80COL 0xC00C
#define SW_ALTCHAR 0xC00E
#define SW_SLOTC3ROM 0xC00B	/* anomaly */
#define SW_SLOTCXROM 0xC006
#define SW_DHIRES 0xC05E
#define SW_IOUDIS 0xC07E
#endif



#ifdef DEBUGGER
#define BUF_X		39
#define BUF_Y		22
/* debugger commands */
enum token_type { MEM, DIS, REGS, SETMEM, STEP, FINISH, UNTIL, GO, VM,
		  BREAK, WATCH, CLEAR, IGNORE, STATUS, OPCODES, LC, DRIVE,
		  SEARCH, HELP, LOG, BSAVE, BLOAD, SAVE, UNKNOWN };
#define MAX_BRKPTS	16

#endif

#endif


/* -------------------------------------------------------------------------
    Macros and equates
   ------------------------------------------------------------------------- */

#define BANK2		FLAG($,0x10000)
#define RebootSig	FLAG($,0x01)
#define ResetSig	FLAG($,0x02)
#define DebugStepSig	FLAG($,0x04)
#define EnterDebugSig	FLAG($,0x08)

#define C_Flag		FLAG($,0x1)		/* 6502 Carry	 	   */
#define X_Flag		FLAG($,0x2)		/* 6502 Xtra		   */
#define I_Flag		FLAG($,0x4)		/* 6502 Interrupt disable  */
#define V_Flag		FLAG($,0x8)		/* 6502 Overflow	   */
#define B_Flag		FLAG($,0x10)		/* 6502 Break		   */
#define D_Flag		FLAG($,0x20)		/* 6502 Decimal mode	   */
#define Z_Flag		FLAG($,0x40)		/* 6502 Zero		   */
#define N_Flag		FLAG($,0x80)		/* 6502 Neg		   */

#define NZ_Flag		FLAG($,0xC0)
#define NZC_Flag	FLAG($,0xC1)
#define NVZC_Flag	FLAG($,0xC9)
#define NV_Flag		FLAG($,0x88)

#define C_Flag_Not	FLAG($,0xFE)
#define Z_Flag_Not      FLAG($,0xBF)
#define NZ_Flag_Not	FLAG($,0x3F)
#define NZC_Flag_Not	FLAG($,0x3E)
#define NVZC_Flag_Not	FLAG($,0x36)
#define NVZ_Flag_Not	FLAG($,0x37)
#define NV_Flag_Not	FLAG($,0x77)

#define C_Flag_Bit	FLAG($,8)		/* 6502 Carry		   */
#define X_Flag_Bit	FLAG($,9)		/* 6502 Xtra		   */
#define I_Flag_Bit	FLAG($,10)		/* 6502 Interrupt disable  */
#define V_Flag_Bit	FLAG($,11)		/* 6502 Overflow	   */
#define B_Flag_Bit	FLAG($,12)		/* 6502 Break		   */
#define D_Flag_Bit	FLAG($,13)		/* 6502 Decimal mode	   */
#define Z_Flag_Bit	FLAG($,14)		/* 6502 Zero		   */
#define N_Flag_Bit	FLAG($,15)		/* 6502 Neg		   */

#define X_Reg		%bl			/* 6502 X register in %bl  */
#define Y_Reg		%bh			/* 6502 Y register in %bh  */
#define A_Reg		%cl			/* 6502 A register in %cl  */
#define F_Reg		%ch			/* 6502 flags in %ch	   */
#define FF_Reg		%ecx			/* 6502 flags for bt	   */
#define SP_Reg		%edx			/* 6502 Stack pointer	   */
#define SP_Reg_L	%dl			/* 6502 Stack pointer low  */
#define SP_Reg_H	%dh			/* 6502 Stack pointer high */
#define PC_Reg		%si			/* 6502 Program Counter    */
#define PC_Reg_E	%esi			/* 6502 Program Counter    */
#define EffectiveAddr	%di			/* Effective address	   */
#define EffectiveAddr_E	%edi			/* Effective address	   */

/* #define Offset_Reg	%ebp */

/* -------------------------------------------------------------------------
    Video
   ------------------------------------------------------------------------- */

#define Video_Reg	%ebx			/* Video mode register	   */
#define Video_Text_Bit	FLAG($,24)		/* Text mode bit	   */
#define Video_Mixed_Bit FLAG($,25)		/* Mixed mode bit	   */
#define Video_Page2_Bit FLAG($,26)		/* Page2 mode bit	   */
#define Video_Hires_Bit FLAG($,27)		/* Hires mode bit	   */
#define SVGA_Page_Bit	FLAG($,31)		/* SVGA page bit	   */

/* -------------------------------------------------------------------------
    Disk drive
   ------------------------------------------------------------------------- */

#define Disk_Old		FLAG($,0x1)
#define	Disk_Write_Protect	FLAG($,0x2)
#define Disk_Present		FLAG($,0x4)
#define Disk_Motor_On		FLAG($,0x8)
#define Disk_Output_Mode	FLAG($,0x10)
#define Disk_Track_Modified	FLAG($,0x20)
#define Disk_Buffer_Valid	FLAG($,0x40)

#define Disk_Not_Output_Mode	FLAG($,0xEF)
#define Disk_Not_Buffer_Valid	FLAG($,0xBF)
#define Disk_Not_Motor_On	FLAG($,0xF7)

/* -------------------------------------------------------------------------
    CPU (6502) Routines
   ------------------------------------------------------------------------- */

#define GetFromPC_B	xorl	%eax, %eax;			\
			movw	PC_Reg, EffectiveAddr;		\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			incw	PC_Reg;

#define GetFromPC_W	xorl	%eax, %eax;			\
			movw	PC_Reg, EffectiveAddr;		\
			incw	EffectiveAddr;			\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			xchgb	%al, %ah;			\
			decw	EffectiveAddr;			\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			addw	$2, PC_Reg;

#define GetFromEA_B	call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);

#define GetFromMem_B(x)						\
			movl	x, EffectiveAddr_E;		\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);

#define GetFromMem_W(x)						\
			movl	x, EffectiveAddr_E;		\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			xchgb	%al, %ah;			\
			incw	EffectiveAddr;			\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			xchgb	%al, %ah;


#define Wait		movw	SYMBOL_NAME(apple_speed), %ax;	\
		   0:						\
			decw	%ax;				\
			jnz	0b;

#ifdef DEBUGGER

#define SaveState	movl	PC_Reg, SYMBOL_NAME(debug_PC);		\
			movl	EffectiveAddr, SYMBOL_NAME(debug_EA);	\
			movl	SP_Reg, SYMBOL_NAME(debug_SP);		\
			movb	X_Reg, SYMBOL_NAME(debug_X);		\
			movb	Y_Reg, SYMBOL_NAME(debug_Y);		\
			movb	A_Reg, SYMBOL_NAME(debug_A);		\
			movb	F_Reg, SYMBOL_NAME(debug_F);

#define ReplaceState	movl	SYMBOL_NAME(debug_PC), PC_Reg;		\
			movl	SYMBOL_NAME(debug_EA), EffectiveAddr;	\
			movl	SYMBOL_NAME(debug_SP), SP_Reg;		\
			movb	SYMBOL_NAME(debug_X), X_Reg;		\
			movb	SYMBOL_NAME(debug_Y), Y_Reg;		\
			movb	SYMBOL_NAME(debug_A), A_Reg;		\
			movb	SYMBOL_NAME(debug_F), F_Reg;
#endif

/* DEBUGGER version of continue.  (we have other exception_flag
   values) */
#ifdef DEBUGGER
#define Continue	Wait						   \
			testb   $0xFF, SYMBOL_NAME(exception_flag);	   \
			jnz     1f;					   \
			GetFromPC_B					   \
/* 			movl	PC_Reg_E, SYMBOL_NAME(debug_PC); */\
/* 			movb	%al, SYMBOL_NAME(current_opcode); */\
			jmp	*table_opcodes(,%eax,4);		   \
									   \
/* exception */	1:	cmpb	RebootSig, SYMBOL_NAME(exception_flag);	   \
			jz	4f;/*reboot*/				   \
			cmpb	ResetSig, SYMBOL_NAME(exception_flag);	   \
			jz	5f;/*reset*/				   \
			cmpb	DebugStepSig, SYMBOL_NAME(exception_flag); \
			jz	2f;/*debug step*/			   \
			jmp	3f;/*enter debugger*/			   \
									   \
/* step */	2:	SaveState;					   \
			ret;						   \
									   \
/* enter */	3:	SaveState;					   \
			pushal;						   \
/* do debugging */	call	SYMBOL_NAME(c_update_keyboard);		   \
			popal;						   \
			ReplaceState					   \
			GetFromPC_B					   \
			jmp	*table_opcodes(,%eax,4);		   \
									   \
/* reboot */	4:	movb	$0, SYMBOL_NAME(exception_flag);	   \
/* 			popl	%ebp; */\
			ret;						   \
/* reset */	5:	movb	$0, SYMBOL_NAME(exception_flag);	   \
			movw	$0xfffc, EffectiveAddr;			   \
			call	read_memory_word;			   \
			movw    %ax, PC_Reg;				   \
			GetFromPC_B					   \
			jmp	*table_opcodes(,%eax,4);
#else
#define Continue	Wait						\
			testb   $0xFF, SYMBOL_NAME(exception_flag);	\
			jnz     1f;					\
			GetFromPC_B					\
			jmp	*table_opcodes(,%eax,4);		\
/* exception */	1:	cmpb	RebootSig, SYMBOL_NAME(exception_flag);	\
			jz	2f;/*reboot*/				\
			movb	$0, SYMBOL_NAME(exception_flag);	\
			movw	$0xfffc, EffectiveAddr;			\
			call	read_memory_word;			\
			movw    %ax, PC_Reg;				\
			GetFromPC_B					\
			jmp	*table_opcodes(,%eax,4);		\
/* reboot */	2:	movb	$0, SYMBOL_NAME(exception_flag);	\
			ret;
#endif


#define FlagC		lahf;				\
  			andb	C_Flag, %ah;		\
  			andb	C_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagZ		lahf;				\
  			andb	Z_Flag, %ah;		\
  			andb	Z_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagN		lahf;				\
  			andb	N_Flag, %ah;		\
  			andb	N_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagNV		lahf;				\
  			andb	NV_Flag, %ah;		\
  			andb	NV_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagNZ		lahf;				\
  			andb	NZ_Flag, %ah;		\
  			andb	NZ_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagNZC		lahf;				\
			andb	NZC_Flag, %ah;		\
			andb	NZC_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagNVZC	lahf;				\
			seto	%al;			\
			shlb    $3, %al;		\
			andb	NZC_Flag, %ah;		\
			andb	NVZC_Flag_Not, F_Reg;	\
			orb	%al, %ah;		\
			orb	%ah, F_Reg;

#define Push(x)		movb	x, SYMBOL_NAME(apple_ii_64k)(,SP_Reg,1); \
			decb	SP_Reg_L;

#define Pop(x)		incb	SP_Reg_L;				 \
			movb	SYMBOL_NAME(apple_ii_64k)(,SP_Reg,1), x;

/* Immediate Addressing - the operand is contained in the second byte of the
   instruction. */
#define GetImm		movw	PC_Reg, EffectiveAddr;	\
			incw	PC_Reg;

/* Absolute Addressing - the second byte of the instruction is the low
   order address, and the third byte is the high order byte. */
#define GetAbs		GetFromPC_W;			\
			movw	%ax, EffectiveAddr;

/* Zero Page Addressing - the second byte of the instruction is an
   address on the zero page */
#define GetZPage	GetFromPC_B;			\
			movw	%ax, EffectiveAddr;

/* Zero Page Indexed Addressing - The effective address is calculated by
   adding the second byte to the contents of the index register.  Due
   to the zero page addressing nature of this mode, no carry is added
   to the high address byte, and the crossing of page boundaries does
   not occur. */
#define GetZPage_X	GetFromPC_B;			\
			addb	X_Reg, %al;		\
			movw	%ax, EffectiveAddr;

#define GetZPage_Y	GetFromPC_B;			\
			addb	Y_Reg, %al;		\
			movw	%ax, EffectiveAddr;

/* Absolute Indexed Addressing - The effective address is formed by
   adding the contents of X or Y to the address contained in the
   second and third bytes of the instruction. */
#define GetAbs_X	GetFromPC_W;			\
			addb	X_Reg, %al;		\
			adcb	$0, %ah;		\
			movw	%ax, EffectiveAddr;

#define GetAbs_Y	GetFromPC_W;			\
			addb	Y_Reg, %al;		\
			adcb	$0, %ah;		\
			movw	%ax, EffectiveAddr;

/* Absolute Indirect Addressing - The second and third bytes of the
   instruction are the low and high bytes of an address, respectively.
   The contents of the fully specified memory location is the
   low-order byte of the effective address.  The next memory location
   contains the high order byte of the effective address. */
/*HACK - used?*/
#define GetInd		GetFromPC_W;		\
			GetFromMem_W(%eax)

/* Zero Page Indirect Addressing (65c02) - The second byte of the
   instruction points to a memory location on page zero containing the
   low order byte of the effective address.  The next location on page
   zero contains the high order byte of the address. */
#define GetIndZPage	GetFromPC_B;			\
			movw	%ax, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			incw	EffectiveAddr;		\
			andw	$0xFF, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			movw	%ax, EffectiveAddr;

/* Zero Page Indexed Indirect Addressing - The second byte is added to
   the contents of the X index register; the carry is discarded.  The
   result of this addition points to a memory location on page zero
   whose contents is the low order byte of the effective address.  The
   next memory location in page zero contains the high-order byte of
   the effective address.  Both memory locations specifying the high
   and low-order bytes must be in page zero. */
#define GetIndZPage_X	GetFromPC_B;			\
			addb	X_Reg, %al;		\
			movw	%ax, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			incw	EffectiveAddr;		\
			andw	$0xFF, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			movw	%ax, EffectiveAddr;

/* Indirect Indexed Addressing - The second byte of the instruction
   points to a memory location in page zero.  The contents of this
   memory location are added to the contents of the Y index register,
   the result being the low order byte of the effective address.  The
   carry from this addition is added to the contents of the next page
   zero memory location, the result being the high order byte of the
   effective address. */
#define GetIndZPage_Y   GetFromPC_B;			\
			movw	%ax, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			incw	EffectiveAddr;		\
			andw	$0xFF, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			addb	Y_Reg, %al;		\
			adcb	$0, %ah;		\
			movw	%ax, EffectiveAddr;

#define DoADC_b		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			adcb	%al, A_Reg;		\
			FlagNVZC

#define	DoADC_d		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			adcb	A_Reg, %al;		\
			daa;				\
			movb	%al, A_Reg;		\
			FlagNVZC

#define DoAND		call	read_memory;	\
			andb	%al, A_Reg;	\
			FlagNZ

#define DoASL		call	read_memory;	\
			shlb	$1, %al;	\
			FlagNZC			\
			call	write_memory;

#define DoBIT		andb	NVZ_Flag_Not, F_Reg;	\
			call	read_memory;		\
			testb	%al, A_Reg;		\
			FlagZ				\
			movb    %al, %ah;		\
			andb	$0x40, %al;		\
			shrb	$3, %al;		\
			orb	%al, F_Reg;		\
			andb    $0x80, %ah;		\
			orb	%ah, F_Reg;

#define DoCMP		call	read_memory;	\
			cmpb	%al, A_Reg;	\
			cmc;			\
			FlagNZC

#define DoCPX		call	read_memory;	\
			cmpb	%al, X_Reg;	\
			cmc;			\
			FlagNZC

#define DoCPY		call	read_memory;	\
			cmpb	%al, Y_Reg;	\
			cmc;			\
			FlagNZC

#define DoDEC		call	read_memory;	\
			decb	%al;		\
			FlagNZ			\
			call	write_memory;

#define DoEOR		call	read_memory;	\
			xorb	%al, A_Reg;	\
			FlagNZ

#define DoINC		call	read_memory;	\
			incb	%al;		\
			FlagNZ			\
			call	write_memory;

#define DoJMP		movw	EffectiveAddr, PC_Reg;

#define DoJSR		movw	PC_Reg, %ax;		\
			decw	%ax;			\
			Push(%ah)			\
			Push(%al)			\
			movw	EffectiveAddr, PC_Reg;

#define DoLDA		call	read_memory;	\
			movb	%al, A_Reg;	\
			orb	%al, %al;	\
			FlagNZ

#define DoLDX		call	read_memory;	\
			movb	%al, X_Reg;	\
			orb	%al, %al;	\
			FlagNZ

#define DoLDY		call	read_memory;	\
			movb	%al, Y_Reg;	\
			orb	%al, %al;	\
			FlagNZ

#define DoLSR		call	read_memory;	\
			shrb	$1, %al;	\
			FlagNZC			\
			call	write_memory;

#define DoORA		call	read_memory;	\
			orb	%al, A_Reg;	\
			FlagNZ

#define DoROL(L,CONT)	call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rclb	$1, %al;		\
			jc	L;			\
			orb	%al, %al;		\
			clc;				\
			FlagNZC				\
			call	write_memory;		\
			CONT				\
	  L##:		orb	%al, %al;		\
			stc;				\
			FlagNZC				\
			call	write_memory;		\
			CONT

#define DoROR(L,CONT)	call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rcrb	$1, %al;		\
			jc	L;			\
			orb	%al, %al;		\
			clc;				\
			FlagNZC				\
			call	write_memory;		\
			CONT				\
	  L##:		orb	%al, %al;		\
			stc;				\
			FlagNZC				\
			call	write_memory;		\
			CONT

#define DoSBC_b		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			sbbb	%al, A_Reg;		\
			cmc;				\
			FlagNVZC

#define DoSBC_d		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			xchgb	A_Reg, %al;		\
			sbbb	A_Reg, %al;		\
			das;				\
			movb	%al, A_Reg;		\
			cmc;				\
			FlagNVZC

#define DoSTA		movb	A_Reg, %al;	\
			call	write_memory;

#define DoSTX		movb	X_Reg, %al;	\
			call	write_memory;

#define DoSTY		movb	Y_Reg, %al;	\
			call	write_memory;

/* -------------------------------------------------------------------------
    65c02 instructions
   ------------------------------------------------------------------------- */

#define DoSTZ		movb	$0x0, %al;	\
			call	write_memory;

#define DoTRB		call	read_memory;	\
			andb	$0x3f, %al;	\
			FlagNV			\
			orb	%ah, %al;	\
			notb	A_Reg;		\
			andb	A_Reg, %al;	\
			FlagZ			\
			call	write_memory;	\
			notb	A_Reg;

#define DoTSB		call	read_memory;	\
			andb	$0x3f, %al;	\
			FlagNV			\
			orb	%ah, %al;	\
			orb	A_Reg, %al;	\
			FlagZ			\
			call	write_memory;

/* -------------------------------------------------------------------------
    Undocumented 6502 (Illegal instructions)
   ------------------------------------------------------------------------- */

        /* AAX = A AND X -> M */
#define DoAAX		movb	A_Reg, %al;	\
			andb	X_Reg, %al;	\
			FlagNZ			\
			call	write_memory;

	/* AMA = ORA 238, AND M, TAX */
#define DoAMA		orb	$238, A_Reg;	\
			call	read_memory;	\
			andb	%al, A_Reg;	\
			movb	A_Reg, X_Reg;	\
			FlagNZ

	/* ANA = AND M, Carry = BIT 7 */
#define DoANA(CONT)	call	read_memory;	\
			andb	%al, A_Reg;	\
			js	1f;		\
			clc;			\
			FlagNZC			\
			CONT			\
		1:	stc;			\
			FlagNZC			\
			CONT

	/* ANB = same as ANA */
#define DoANB(CONT)	call	read_memory;	\
			andb	%al, A_Reg;	\
			js	1f;		\
			clc;			\
			FlagNZC			\
			CONT			\
		1:	stc;			\
			FlagNZC			\
			CONT

	/* AXM = (A AND X) - M -> X */
#define DoAXM		call	read_memory;		\
			andb	A_Reg, X_Reg;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			sbbb	%al, X_Reg;		\
			cmc;				\
			FlagNVZC

	/* AXS = (A AND X) -> S, A AND X AND 17 -> M */
			/* HACK!!!!!!!!!!!!!!! */
#define DoAXS		movb	A_Reg, SP_Reg_L;	\
			andb	X_Reg, SP_Reg_L;	\
			movb	SP_Reg_L, %al;		\
			andb	$17, %al;		\
			FlagNZ	/* \ wasn't here */	\
			call	write_memory; 

	/* DCP = DEC M, CMP M */
#define	DoDCP		call	read_memory;	\
			decb	%al;		\
			cmpb	%al, A_Reg;	\
			cmc;			\
			FlagNZC			\
			call	write_memory;

	/* ISB = INC M, SBC M */
#define DoISB_b		call	read_memory;		\
			incb	%al;			\
			call	write_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			sbbb	%al, A_Reg;		\
			cmc;				\
			FlagNVZC

#define DoISB_d		call	read_memory;		\
			incb	%al;			\
			call	write_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			xchgb	A_Reg, %al;		\
			sbbb	A_Reg, %al;		\
			das;				\
			movb	%al, A_Reg;		\
			cmc;				\
			FlagNVZC

	/* LAN = ROL M, AND M */
#define DoLAN(CONT)	call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rclb	$1, %al;		\
			jc	1f;			\
			call	write_memory;		\
			andb	%al, A_Reg;		\
			clc;				\
			FlagNZC				\
			CONT				\
		   1:	call	write_memory;		\
			andb	%al, A_Reg;		\
			stc;				\
			FlagNZC				\
			CONT

	/* LAS = LDA M, TAX, TXS */
#define DoLAS		call	read_memory;	\
			movb	%al, A_Reg;	\
			movb	%al, X_Reg;	\
			movb	%al, SP_Reg_L;	\
			orb	%al, %al;	\
			FlagNZ

	/* LAX = LDA M, TAX */
#define DoLAX		call	read_memory;	\
			movb	%al, A_Reg;	\
			movb	%al, X_Reg;	\
			orb	%al, %al;	\
			FlagNZ

	/* LOR = ASL M, ORA M */
#define DoLOR		call	read_memory;	\
			shlb	$1, %al;	\
			FlagC			\
			call	write_memory;	\
			orb	%al, A_Reg;	\
			FlagNZ

	/* RAD = ROR M, ADC M */
#define DoRAD_b		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rcrb	$1, %al;		\
			adcb	%al, A_Reg;		\
			pushl	%eax;			\
			FlagNVZC			\
			popl	%eax;			\
			call	write_memory;

#define DoRAD_d		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rcrb	$1, %al;		\
			pushfl;				\
			call	write_memory;		\
			popfl;				\
			adcb	A_Reg, %al;		\
			daa;				\
			movb	%al, A_Reg;		\
			FlagNVZC

	/* RAM = AND M, LSR A */
#define DoRAM		call	read_memory;	\
			andb	%al, A_Reg;	\
			shrb	$1, A_Reg;	\
			FlagNZC

	/* RBM = same as RAM */
#define DoRBM		DoRAM

	/* REO = LSR M, EOR M */
#define DoREO		call	read_memory;	\
			shrb	$1, %al;	\
			xorb	%al, A_Reg;	\
			FlagNZC			\
			call	write_memory;

	/* DoZBC = same as SBC */
#define DoZBC_b		DoSBC_b
#define DoZBC_d		DoSBC_d

	/* TEA = (A AND X AND (OP+2)+1) -> M */
#define	DoTEA		pushl	EffectiveAddr_E;	\
			movw	PC_Reg, EffectiveAddr;	\
			decw	EffectiveAddr;		\
			call	read_memory;		\
			popl	EffectiveAddr_E;	\
			incb	%al;			\
			andb	A_Reg, %al;		\
			andb	X_Reg, %al;		\
			FlagNZ				\
			call	write_memory;

	/* TEX = (X AND (OP+2)+1) -> M */
#define DoTEX		pushl	EffectiveAddr_E;	\
			movw	PC_Reg, EffectiveAddr;	\
			decw	EffectiveAddr;		\
			call	read_memory;		\
			popl	EffectiveAddr_E;	\
			incb	%al;			\
			andb	X_Reg, %al;		\
			FlagNZ				\
			call	write_memory;

	/* TEY = (Y AND 1) -> M */
#define DoTEY		movb	Y_Reg, %al;	\
			andb	$1, %al;	\
			FlagNZ			\
			call	write_memory;

	/* XMA = (X AND M) AND (A OR 238) -> A */
			/* HACK!!!!!!!!!!!!!!! */
#define DoXMA	/* the \ wasn't here before */	\
			call	read_memory;	\
			andb	X_Reg, %al;	\
			orb	$238, A_Reg;	\
			andb	%al, A_Reg;	\
			FlagNZ

/**********************************************************************/
#ifdef DEBUGGER
#define checkDebuggerNoGraphics(EXIT)				\
	/* no graphics in debugger console */			\
		testb	$0xFF, SYMBOL_NAME(debug_no_graphics);	\
		jnz	EXIT;
#else
#define checkDebuggerNoGraphics(EXIT)
#endif		

/* PlotByte - macro to plot a black/white byte into graphics
   memory. TABLE = expanded_col_hires_even,
   expanded_col_hires_odd. BASE = 0x2000,0x4000. */
#define PlotByte(BASE,X,TABLE,OPP_TABLE,INTERP_COLOR,ALT_INTERP_COLOR)	      \
		pushl	%eax;				/* save regs */	      \
		pushl	%ebx;						      \
		pushl	%ecx;						      \
/* 	pushl	EffectiveAddr_E;\ */\
									      \
		xorb	%ah, %ah;			/* clear noise */     \
		leal	SYMBOL_NAME(expanded_col_hires_##TABLE)		      \
			(,%eax,8), %ebx;		/* ebx = mask addrs*/ \
									      \
		movl	EffectiveAddr_E, %ecx;		/* ecx = mem addrs */ \
		subw	BASE, EffectiveAddr;		/* - graphics base */ \
		movw	SYMBOL_NAME(hires_page_offset)			      \
			(,EffectiveAddr_E,2), %ax;	/* eax = GM offset */ \
		movl	%ecx, EffectiveAddr_E;		/* + graphics base */ \
		addl	SYMBOL_NAME(GM), %eax;		/* eax += GM base */  \
									      \
		movl	(%ebx), %ecx;			/* long -> GM */      \
		movl	%ecx, (%eax);					      \
		addl	$4, %eax;			/* inc pointers */    \
		addl	$4, %ebx;					      \
		movw	(%ebx), %cx;			/* word -> GM */      \
		movw	%cx, (%eax);					      \
		addl	$2, %eax;			/* inc pointers */    \
		addl	$2, %ebx;					      \
		movb	(%ebx), %cl;			/* byte -> GM */      \
		movb	%cl, (%eax);					      \
									      \
		/* dynamic test color on byte edge for strict colors */	      \
		testb	$0xFF, SYMBOL_NAME(strict_color);		      \
		jz	PB_exit##X;	/* no strict color mode exit */	      \
									      \
		movw	(%eax), %cx;					      \
		testb	$0xFF, %ch;	/* check right color */		      \
		jz	PB_next0##X;	/* right black, do other end */	      \
/* 	andw	SYMBOL_NAME(read_hires_page_offset),\ */\
/* 		EffectiveAddr_E;\ */\
	movw	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %cx;\
		andb	$1, %ch;					      \
		jz	PB_black0##X;	/* right black */		      \
		andb	$0x40, %cl;					      \
		jz	PB_black0##X;	/* inside black, right colored */     \
		movw	$0x3737, (%eax);/* edge is white (#55) */	      \
		jmp	PB_next0##X;					      \
PB_black0##X:								      \
	movzwl	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %ecx;\
		movb	%ch, %cl;					      \
		xorb	%ch, %ch;					      \
		leal	SYMBOL_NAME(expanded_col_hires_##OPP_TABLE)	      \
			(,%ecx,8), %ebx;				      \
		incl	%eax;						      \
		movb	(%ebx), %cl;					      \
		movb	%cl, (%eax);					      \
		decl	%eax;						      \
PB_next0##X:								      \
		decw	EffectiveAddr;		/* previous byte */	      \
		subl	$7, %eax;	/* left edge of byte */		      \
		movb	(%eax), %cl;					      \
		testb	$0xFF, %cl;	/* check left color */		      \
		jz	PB_next1##X;	/* left black, done */		      \
	movw	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %cx;\
		andb	$0x40, %cl;					      \
		jz	PB_black1##X;	/* left black */		      \
		andb	$0x1, %ch;					      \
		jz	PB_black1##X;	/* left colored, inside black */      \
		movw	$0x3737, (%eax);/* edge is white (#55) */	      \
		jmp	PB_next1##X;					      \
PB_black1##X:								      \
	movzbl	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %ecx;\
		leal	SYMBOL_NAME(expanded_col_hires_##OPP_TABLE)	      \
			(,%ecx,8), %ebx;				      \
		addb	$6, %ebx;					      \
		movb	(%ebx), %cl;					      \
		movb	%cl, (%eax);					      \
PB_next1##X:								      \
		incw	EffectiveAddr;					      \
		/* do extra calculation for interpolated colors */	      \
		cmpb	$2, SYMBOL_NAME(strict_color);			      \
		jne	PB_exit##X;					      \
									      \
		decw	EffectiveAddr;					      \
		CalculateInterpColor(X,2,ALT_INTERP_COLOR);		      \
PB_next2##X:								      \
		incw	EffectiveAddr;					      \
		incl	%eax;						      \
		CalculateInterpColor(X,3,INTERP_COLOR);			      \
PB_next3##X:								      \
		addl	$6, %eax;					      \
		CalculateInterpColor(X,4,INTERP_COLOR);			      \
PB_next4##X:								      \
		incw	EffectiveAddr;					      \
		incl	%eax;						      \
		CalculateInterpColor(X,5,ALT_INTERP_COLOR);		      \
PB_next5##X:								      \
	decw	EffectiveAddr;\
PB_exit##X:								      \
/* 	popl	EffectiveAddr_E;\ */\
		popl	%ecx;				/* restore regs */    \
		popl	%ebx;						      \
		popl	%eax;

/* calculates the color at the edge of interpolated bytes */
#define CalculateInterpColor(X,Y,INTERP_COLOR)				\
		testb	$0xFF, (%eax);					\
		jnz	PB_next##Y##X;	/* not black, next */		\
		movw	(%eax), %cx;	/* off+1 in %ch */		\
		testb	$0xFF, %ch;					\
		jz	PB_next##Y##X;	/* off+1 is black, next */	\
/*		decl	%eax;*/						\
		movb	-1(%eax), %cl;	/* off-1 in %cl */		\
/*		incl	%eax;*/						\
		testb	$0xFF, %cl;					\
		jz	PB_next##Y##X;	/* off-1 is black, next */	\
		cmpb	$55, %cl;	/* off-1 is white? */		\
		je	PB_white0##X##Y;				\
		movb	%cl, (%eax);	/* store non-white */		\
		jmp	PB_next##Y##X;	/* next */			\
PB_white0##X##Y:							\
		cmpb	$55, %ch;	/* off+1 is white? */		\
		je	PB_white##X##Y;					\
		movb	%ch, (%eax);	/* store non-white */		\
		jmp	PB_next##Y##X;	/* next */			\
PB_white##X##Y:				/* both sides are white */	\
	movzbl	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %ecx;\
		shrb	$7, %cl;					\
 		movb	SYMBOL_NAME(INTERP_COLOR)(,%ecx,1), %bl;	\
		movb	%bl, (%eax);


/* UpdateHiresRows() X=0,1 OFFSET=0x2027,0x4027 */
#define UpdateHiresRows(PRE,X,OFFSET)					 \
update_hires_rows_##X:							 \
		movl	$20, %ecx;	/* ECX: 40 column counter */	 \
		movl	%ebx, %edx;	/* EBX: pixel row counter */	 \
		shrb	$3, %dl;	/* normalize to 0 - 23 */	 \
					/* EDI: row offset */		 \
		movw	SYMBOL_NAME(video_line_offset)(,%edx,2),	 \
			EffectiveAddr;					 \
		movl	%ebx, %edx;					 \
		andb	$0x7, %dl;					 \
		shlw	$10, %dx;	/* EDX: offset range 0 - 1C00 */ \
		addw	OFFSET, %dx;	/* add base end-row offset */	 \
		addw	%dx, EffectiveAddr;	/* EDI: mem address */	 \
update_hires_columns_##X:						 \
/*	call	SYMBOL_NAME(read_memory); */\
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		cmpw	$159, %bx;	/* mixed mode boundary */	 \
		jg	update_hires_mixed_##X;				 \
		call	SYMBOL_NAME(PRE##ram_hires_page##X##_odd);	 \
		decw	EffectiveAddr;	/* previous address */		 \
/*	call	SYMBOL_NAME(read_memory); */\
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		call	SYMBOL_NAME(PRE##ram_hires_page##X##_even);	 \
		jmp	update_hires_cont_##X;				 \
update_hires_mixed_##X:							 \
		call	SYMBOL_NAME(PRE##ram_hires_mixed##X##_odd);	 \
		decw	EffectiveAddr;	/* previous address */		 \
/*	call	SYMBOL_NAME(read_memory); */\
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		call	SYMBOL_NAME(PRE##ram_hires_mixed##X##_even);	 \
update_hires_cont_##X:							 \
		decw	EffectiveAddr;	/* previous address */		 \
		decb	%cl;		/* dec column counter */	 \
		jnz	update_hires_columns_##X;			 \
		decw	%bx;		/* dec row */			 \
		jns	update_hires_rows_##X;


/* UpdateRows X=0,1 OFF=0x427,0x827 */
#define UpdateRows(PRE,X,OFFSET,EBX,EDI)				\
update_rows_##X:							\
		movl	$39, %ecx;					\
		movw	SYMBOL_NAME(video_line_offset)(,%ebx,2),	\
			EffectiveAddr;					\
		addw	OFFSET, EffectiveAddr;				\
update_columns_##X:							\
/*	call	SYMBOL_NAME(read_memory); */\
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		cmpb	$19, EBX;					\
		jg	update_mixed_##X;				\
		call	SYMBOL_NAME(PRE##ram_text_page##X);		\
		jmp	update_cont_##X;				\
update_mixed_##X:							\
		call	SYMBOL_NAME(PRE##ram_text_mixed##X);		\
update_cont_##X:							\
		decw	EDI;						\
		decb	%cl;						\
		jns	update_columns_##X;				\
		decb	%bl;						\
		jns	update_rows_##X;


/* PlotCharacterRow() */
#define PlotCharacterRow			\
		movl	(PC_Reg_E), %ecx;	\
		movl	%ecx, (%eax);		\
		addl	$4, PC_Reg_E;		\
		addl	$4, %eax;		\
		movw	(PC_Reg_E), %cx;	\
		movw	%cx, (%eax);		\
		addl	$2, PC_Reg_E;		\
		addl	$2, %eax;		\
		movb	(PC_Reg_E), %cl;	\
		movb	%cl, (%eax);		\
		addl	$2, PC_Reg_E;

#define PlotBlockRow				\
		movl	%edx, (%eax);		\
		addl	$4, %eax;		\
		movw	%dx, (%eax);		\
		addl	$2, %eax;		\
		movb	%dl, (%eax);


#if 0
/* UpdateDHiresRows - update entire double hires graphics memory. */
#define UpdateDHiresRows(X,OFF)						 \
update_dhires_rows##X:							 \
		movl	$40, %ecx;	/* ECX: 40 column counter */	 \
		movl	%ebx, %edx;	/* EBX: pixel row counter */	 \
		shrb	$3, %dl;	/* normalize to 0 - 23 */	 \
					/* EDI: row offset */		 \
		movw	SYMBOL_NAME(video_line_offset)(,%edx,2),	 \
			EffectiveAddr;					 \
		movl	%ebx, %edx;					 \
		andb	$0x7, %dl;					 \
		shlw	$10, %dx;	/* EDX: offset range 0 - 1C00 */ \
		addw	OFF, %dx;	/* add base end-row offset */	 \
		addw	%dx, EffectiveAddr;	/* EDI: mem address */	 \
update_dhires_columns##X:						 \
/*	call	SYMBOL_NAME(read_memory); */\
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		call	SYMBOL_NAME(iie_plot_dhires##X);		 \
		subw	$2, EffectiveAddr;	/* previous address */	 \
		decb	%cl;		/* dec column counter */	 \
		jnz	update_dhires_columns##X;			 \
		decw	%bx;		/* dec row */			 \
		jns	update_dhires_rows##X;
#endif

/* PlotDHiresByte - plots a normalized byte of dhires color directly
   into graphics memory */
#define PlotDHiresByte							\
		movb	SYMBOL_NAME(dhires_colors1)(,%ebx,1), %dl;	\
		movb	%dl, (%eax);					\
		incl	%eax;						\
		movb	SYMBOL_NAME(dhires_colors2)(,%ebx,1), %dl;	\
		movb	%dl, (%eax);					\
		incl	%eax;

#define PlotDHires(OFF,X)						      \
		pushl	%eax;				/* save regs */	      \
		pushl	%ebx;						      \
		pushl	%ecx;						      \
		pushl	%edx;						      \
		pushl	EffectiveAddr_E;				      \
									      \
		andw	$0xFFFF, EffectiveAddr;		/* erase offset */    \
		btr	$0, EffectiveAddr_E;		/* normalize */	      \
		movl	EffectiveAddr_E, %ecx;		/* ecx = mem addrs */ \
		subw	OFF, EffectiveAddr;		/* - graphics base */ \
		movw	SYMBOL_NAME(hires_page_offset)			      \
			(,EffectiveAddr_E,2), %ax;	/* eax = GM offset */ \
		movb	SYMBOL_NAME(hires_col_offset)			      \
			(,EffectiveAddr_E,1), %bl;			      \
		addl	SYMBOL_NAME(GM), %eax;		/* eax += GM base */  \
									      \
	leal	SYMBOL_NAME(apple_ii_64k), EffectiveAddr_E;\
		addl	%ecx, EffectiveAddr_E;				      \
		movl	EffectiveAddr_E, %ecx;				      \
		addl	BANK2, %ecx;					      \
									      \
		testb	$0xFF, %bl;					      \
		jz	plot_dhires##X##_cont;				      \
		subl	$2, %eax;					      \
		movzbl	-1(EffectiveAddr_E), %ebx;			      \
		movb	0(%ecx), %bh;					      \
		btr	$7, %bl;					      \
		shrb	$3, %bl;					      \
		shlb	$4, %bh;					      \
		orb	%bh, %bl;					      \
		xorb	%bh, %bh;					      \
		PlotDHiresByte						      \
									      \
plot_dhires##X##_cont:							      \
		movl	%ecx, %edx;					      \
		movb	2(%edx), %cl;					      \
		shll	$28, %ecx;					      \
									      \
		movzbl	1(EffectiveAddr_E), %ebx;			      \
		btr	$7, %bl;		/* erase msb */		      \
		shll	$21, %ebx;					      \
		orl	%ebx, %ecx;					      \
									      \
		movzbl	1(%edx), %ebx;					      \
		btr	$7, %bl;		/* erase msb */		      \
		shll	$14, %ebx;					      \
		orl	%ebx, %ecx;					      \
									      \
		movzbl	0(EffectiveAddr_E), %ebx;			      \
		btr	$7, %bl;		/* erase msb */		      \
		shll	$7, %ebx;					      \
		orl	%ebx, %ecx;					      \
									      \
		movzbl	0(%edx), %ebx;					      \
		btr	$7, %bl;		/* erase msb */		      \
		orl	%ebx, %ecx;					      \
		/* 00000001 11111122 22222333 3333xxxx */		      \
									      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		popl	EffectiveAddr_E;				      \
		andl	$0xFFFF, EffectiveAddr_E;/* for safety */	      \
		popl	%edx;						      \
		popl	%ecx;			 /* restore regs */	      \
		popl	%ebx;						      \
		popl	%eax;						      \
		ret;

#define PlotDHiresNibble						\
		leal	SYMBOL_NAME(dhires_colors)(,%ecx,8), %ecx;	\
		movl	(%ecx), %edx;					\
		movl	%edx, (%eax);

#define offset_RAMRD_RAMWRT						\
		movl	SYMBOL_NAME(ramrd_offset), %eax;		\
		movl	%eax, SYMBOL_NAME(read_text_page_offset);	\
		movl	%eax, SYMBOL_NAME(read_hires_page_offset);	\
		movl	SYMBOL_NAME(ramwrt_offset), %eax;		\
		movl	%eax, SYMBOL_NAME(write_text_page_offset);	\
		movl	%eax, SYMBOL_NAME(write_hires_page_offset);

/* don't show write to aux memory bank */
#define CheckVidPageWrite					\
		testb	$0xFF, SYMBOL_NAME(ramwrt_flag);	\
		jnz	ram_nop;
