/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#undef USE_GNOME
#include "userlist.h"
#include "style.h"
#include "xchat.h"
#include "dcc.h"
#include "draw.h"
#include "menu.h"
#include "plugin.h"
#include <sys/time.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <ctype.h>
#include <netdb.h>
#include <time.h>

int handle_command(char *cmd, struct session *sess, int history);


extern GSList *sess_list;
extern GSList *serv_list;
extern GSList *dcc_list;
extern GSList *command_list;
extern GSList *button_list;
extern struct xchatprefs prefs;

extern struct session* tab_msg_session(char* target, struct server* serv);
extern struct session *find_session_from_channel(char *chan, struct server *serv);
extern void make_non_channel_window(struct session *sess);
extern void maingui_updatebuttons(struct session *);
extern int list_delentry(GSList **list, char *name);
extern void list_addentry(GSList **list, char *cmd, char *name);
extern void zvt_clear(GtkWidget *wid);
extern void check_homedir(char *file);
extern struct DCC *find_dcc(char *nick, char *file, int type);
extern void dcc_close(struct DCC *dcc, int stat, int destroy);
extern struct dialog *new_dialog(struct server *serv, char *nick);
extern struct dialog *find_dialog(struct server *serv, char *nick);
extern struct sockaddr_in *dcc_write_chat(char *nick, char *text);
extern void PrintText(struct session *sess, char *text);
extern void connect_server(struct session *sess, char *server, int port, int quiet);
extern void channel_action(struct session *sess, char *tbuf, char *chan, char *from, char *text, char fromme);
extern void user_new_nick(struct server *serv, char *outbuf, char *nick, char *newnick, int quiet);
extern void channel_msg(struct server *serv, char *outbuf, char *chan, char *from, char *text, char fromme);
extern void dcc_show_list(struct session *sess, char *outbuf);
extern void dcc_get_nick(struct session *sess, char *nick);
extern void dcc_send(struct session *sess, char *tbuf, char *to, char *file);
extern void dcc_chat(struct session *sess, char *nick);
extern void disconnect_server(struct session *sess, int sendquit);
extern void dcc_draw(struct session *sess, char *nick);
extern void notify_showlist(struct session *sess);
extern void notify_adduser(char *name);
extern int notify_deluser(char *name);
extern void show_lastlog_window(void);
extern void search_lastlog(struct session *_sess, char *search);
extern char *file_part(char *file);
extern void dcc_send_filereq(struct session *sess, char *nick);
#ifdef USE_PERL
extern int perl_load_file(char *script_name);
extern int perl_command(char *cmd, struct session *sess);
#endif
extern int module_command (char *cmd, struct session *sess, char *tbuf, char *word[], char *word_eol[]);
extern int module_load (char *name, struct session *sess);
extern int module_list(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
extern int module_unload (char *name, struct session *sess);


void notj_msg(struct session *sess)
{  
   PrintText(sess, "No channel joined. Try /join #<channel>\n");
}

void notc_msg(struct session *sess)
{
   PrintText(sess, "Not connected. Try /server <host> [<port>]\n");
}

void process_data_init(unsigned char *buf, char *cmd, char *word[], char *word_eol[])
{
   int wordcount = 2;
   int space = FALSE;
   int quote = FALSE;
   int j = 0;

   word[1] = cmd;
   word_eol[1] = buf;

   while(1)
   {
     switch(*cmd)
     {
      case 0:
	jump:
	buf[j] = 0;
	for(j=wordcount; j<32; j++)
	{
	   word[j] = "\000\000";
	   word_eol[j] = "\000\000";
	}
	return;
      case '\042':
	if(quote) quote = FALSE; else quote = TRUE;
	break;
      case ' ':
	if(!quote)
	{
	   if(!space)
	   {
	      buf[j] = 0;
	      j++;

	      word[wordcount] = &buf[j];
	      word_eol[wordcount] = cmd+1;
	      wordcount++;

	      if(wordcount==31) goto jump;

	      space = TRUE;
	   }
	   break;
	}
      default:
	space = FALSE;
	buf[j] = *cmd;
	j++;
     }
     cmd++;
   }
}

int cmd_addbutton(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_away(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_ban(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_clear(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_ctcp(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_dcc(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_debug(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_delbutton(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_deop(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_devoice(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_discon(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_exec(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_gate(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_help(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_hidever(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_invite(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_join(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_kick(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_kickban(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_lastlog(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_load(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_loaddll(struct session *sess, char *tbuf, char *word[], char *word_eol[]);		       
int cmd_me(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_msg(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_mode(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_names(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_nctcp(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_nick(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_notice(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_notify(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_op(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_part(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_ping(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_query(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_quit(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_quote(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_rmdll(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_scpinfo(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_server(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_topic(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_unban(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_unloadall(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_wallchop(struct session *sess, char *tbuf, char *word[], char *word_eol[]);
int cmd_voice(struct session *sess, char *tbuf, char *word[], char *word_eol[]);

struct commands cmds[] =
{
     {"ADDBUTTON",cmd_addbutton,0,0,"/ADDBUTTON <name> <action>\n"},
     {"AWAY",   cmd_away,   1, 0, "/AWAY [<reason>]\n"},
     {"BAN",    cmd_ban,    1, 1, "/BAN <mask>\n"},
     {"CLEAR",  cmd_clear,  0, 0, 0},
     {"CTCP",   cmd_ctcp,   1, 0, "/CTCP <nick> <message>\n"},
     {"DCC",    cmd_dcc,    0, 0, "\n"
	 "/DCC GET <nick>          - receive an offered file\n"
         "/DCC SEND <nick> <file>  - send a file to someone\n"
         "/DCC LIST                - show DCC list\n"
	 "/DCC CHAT <nick>         - offer DCC CHAT to someone\n"
	 "/DCC DRAW <nick>         - offer DCC DRAW to another X-Chat user\n"
	 "/DCC CLOSE <type> <nick> <file>         example:\n"
	 "         /dcc close send johnsmith file.tar.gz\n"
	  },
     {"DEBUG",  cmd_debug,  0, 0, 0},
     {"DELBUTTON",cmd_delbutton,0,0,"/DELBUTTON <name>\n"},
     {"DEOP",   cmd_deop,   1, 1, "/DEOP <nick>\n"},
     {"DEVOICE",cmd_devoice,1, 1, "/DEVOICE <nick>\n"},
     {"DISCON", cmd_discon, 0, 0, "/DISCON\n\nDisconnects from server\n"},
     {"EXEC",   cmd_exec,   0, 0, 0},
     {"GATE",   cmd_gate,   0, 0, "/GATE <host> [<port>]\n"},
     {"HELP",   cmd_help,   0, 0, 0},
     {"HIDEVER",cmd_hidever,0, 0, 0},
     {"INVITE", cmd_invite, 1, 0, "/INVITE <nick> <channel>\n"},
     {"JOIN",   cmd_join,   1, 0, "/JOIN <channel>\n"},
     {"KICK",   cmd_kick,   1, 1, "/KICK <nick>\n"},
     {"KICKBAN",cmd_kickban,1, 1, "/KICKBAN <nick>\n"},
     {"LASTLOG",cmd_lastlog,0, 0, "/LASTLOG <search>\n"},
#ifdef USE_PLUGIN
     {"LISTDLL",module_list,0, 0, 0},
#endif
     {"LOAD",   cmd_load,   0, 0, "LOAD <file>\n"},
#ifdef USE_PLUGIN
     {"LOADDLL",cmd_loaddll,0, 0, "LOADDLL <file>\n"},
#endif
     {"ME",     cmd_me,     1, 1, "/ME <action>\n"},
     {"MSG",    cmd_msg,    1, 0, "/MSG <nick> <message>\n"},
     {"MODE",   cmd_mode,   1, 0, "/MODE [<channel>] <mode>\n"},
     {"NAMES",  cmd_names,  1, 1, 0},
     {"NCTCP",  cmd_nctcp,  1, 0, 0},
     {"NICK",   cmd_nick,   0, 0, 0},
     {"NOTICE", cmd_notice, 1, 0, "/NOTICE <nick/channel> <message>\n"},
     {"NOTIFY", cmd_notify, 0, 0, 0},
     {"OP",     cmd_op,     1, 1, "/OP <nick>\n"},
     {"PART",   cmd_part,   1, 1, 0},
     {"PING",   cmd_ping,   1, 0, 0},
     {"QUERY",  cmd_query,  0, 0, 0},
     {"QUIT",   cmd_quit,   0, 0, 0},
     {"QUOTE",  cmd_quote,  1, 0, 0},
#ifdef USE_PLUGIN
     {"RMDLL",  cmd_rmdll,  0, 0, "RMDLL <dll name>\n"},
#endif
#ifdef USE_PERL
     {"SCPINFO",cmd_scpinfo,0, 0, 0},
#endif
     {"SERVER", cmd_server, 0, 0, "/SERVER <host> [<port>]\n"},
     {"TOPIC",  cmd_topic,  1, 1, 0},
     {"UNBAN",  cmd_unban,  1, 1, 0},
#ifdef USE_PERL
     {"UNLOADALL",cmd_unloadall,0,0,0},
#endif
     {"WALLCHOP",cmd_wallchop,1, 1, 0},
     {"VOICE",  cmd_voice,  1, 1, 0},
     {0,0,0,0,0}
};

void help(struct session *sess, char *helpcmd, int quiet)
{
   int i = 0;
      while(1)
      {
	 if(!cmds[i].name) break;
	 if(!strcasecmp(helpcmd, cmds[i].name))
	 {
	    if(cmds[i].help)
	    {
	       PrintText(sess, "Usage: ");
	       PrintText(sess, cmds[i].help);
	       return;
	    } else {
	       if(!quiet) PrintText(sess, "\nNo help available on that command.\n");
	       return;
	    }
	 }
	 i++;
      }
   if(!quiet) PrintText(sess, "No such command.\n");
}

#define find_word_to_end(a, b) word_eol[b]
#define find_word(a, b) word[b]


int cmd_addbutton(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word[2] && *word_eol[3])
   {
      list_addentry(&button_list, word_eol[3], word[2]);
      maingui_updatebuttons(sess);
      return TRUE;
   }
   return FALSE;
}

int cmd_away(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *reason = find_word_to_end(cmd, 2);
   if(*reason)
   {
     sprintf(tbuf, "AWAY :%s\r\n", reason);
     send(sess->server->sok, tbuf, strlen(tbuf), 0);
   } else
     send(sess->server->sok, "AWAY\r\n", 6, 0);
   return TRUE;
}

int cmd_ban(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *mask = find_word(pdibuf, 2);
   if(*mask)
     sprintf(tbuf, "MODE %s +b %s\r\n", sess->channel, mask);
   else
     sprintf(tbuf, "MODE %s +b\r\n", sess->channel);
   send(sess->server->sok, tbuf, strlen(tbuf), 0);
   return TRUE;
}

int cmd_clear(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(!sess->zvt)
     gtk_text_backward_delete((GtkText *)sess->textgad,
			      gtk_text_get_length((GtkText *)sess->textgad));
   else
     zvt_clear(sess->textgad);
   return TRUE;
}

int cmd_ctcp(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *to = find_word(pdibuf, 2);
   if(*to)
   {
      char *msg = find_word_to_end(cmd, 3);
      if(*msg)
      {
	 char *cmd = msg;

	 while(1)
	 {
	    if(*cmd == ' ' || *cmd == 0) break;
	    *cmd = toupper(*cmd);
	    cmd++;
	 }

	 sprintf(tbuf, "PRIVMSG %s :\001%s\001\r\n", to, msg);
	 send(sess->server->sok, tbuf, strlen(tbuf), 0);
	 sprintf(tbuf, "\0033>\003 %s\0033< \003 CTCP %s\n", to, msg);
	 PrintText(sess, tbuf);
	 return TRUE;
      }
   }
   return FALSE;
}

int cmd_dcc(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   int goodtype;
   struct DCC *dcc = 0;
   char *type = find_word(pdibuf, 2);
   if(*type)
   {
      if(!strcasecmp(type, "HELP"))
	return FALSE;
      if(!strcasecmp(type, "CLOSE"))
      {
	 if(*word[3] && *word[4])
	 {
	    goodtype = 0;
	    if(!strcasecmp(word[3], "SEND"))
	    {
	       dcc = find_dcc(word[4], word[5], TYPE_SEND);
	       if(dcc) dcc_close(dcc, 0, TRUE);
	       goodtype = TRUE;
	    }
	    if(!strcasecmp(word[3], "GET"))
	    {
	       dcc = find_dcc(word[4], word[5], TYPE_RECV);
	       if(dcc) dcc_close(dcc, 0, TRUE);
	       goodtype = TRUE;
	    }
	    if(!strcasecmp(word[3], "DRAW"))
	    {
	       dcc = find_dcc(word[4], "", TYPE_DRAWRECV);
	       if(!dcc) dcc = find_dcc(word[4], "", TYPE_DRAWSEND);
	       if(dcc)
	       {
		  if(dcc->dccdraw)
		    gtk_widget_destroy(dcc->dccdraw->window);
		  else
		    dcc_close(dcc, 0, TRUE);
	       }
	       goodtype = TRUE;
	    }
	    if(!strcasecmp(word[3], "CHAT"))
	    {
	       dcc = find_dcc(word[4], "", TYPE_CHATRECV);
	       if(!dcc) dcc = find_dcc(word[4], "", TYPE_CHATSEND);
	       if(dcc) dcc_close(dcc, 0, TRUE);
	       goodtype = TRUE;
	    }

	    if(goodtype)
	    {
	       if(!dcc) PrintText(sess, STARTON" No such DCC.\n");
	    } else
	       return FALSE;
       	    return TRUE;

	 }
	 return FALSE;
      }
      if(!strcasecmp(type, "CHAT"))
      {
	 char *nick = find_word(pdibuf, 3);
	 if(*nick) dcc_chat(sess, nick);
	 return TRUE;
      }
      if(!strcasecmp(type, "LIST"))
      {
	 dcc_show_list(sess, tbuf);
	 return TRUE;
      }
      if(!strcasecmp(type, "DRAW"))
      {
	 if(*word[3])
	 {
	    dcc_draw(sess, word[3]);
	    return TRUE;
	 } else
	    return FALSE;
      }
      if(!strcasecmp(type, "GET"))
      {
	 char *nick = find_word(pdibuf, 3);
	 if(*nick) dcc_get_nick(sess, nick);
	 return TRUE;
      }
      if(!strcasecmp(type, "SEND"))
      {
	 char *nick = find_word(pdibuf, 3);
	 if(*nick) 
	 {
	    int i = 4;
	    char *file;
	    while(1)
	    {
	       file = find_word(pdibuf, i);
	       if(!*file && i == 4)
	       {
		  dcc_send_filereq(sess, nick);
		  return TRUE;
	       }
	       if(!*file) break;
	       dcc_send(sess, tbuf, nick, file);
	       i++;
	    }
	 }
	 return TRUE;
      }
   } else
     dcc_show_list(sess, tbuf);
   return TRUE;
}

int cmd_debug(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   struct session *s;
   struct server *v;
   struct DCC *d;
   GSList *list = sess_list;

   PrintText(sess, "Session   Channel    Server\n");
   while(list)
   {
      s = (struct session *)list->data;
      sprintf(tbuf, "0x%lx %-10.10s 0x%lx\n",
	      (unsigned long)s, s->channel, (unsigned long)s->server);
      PrintText(sess, tbuf);
      list = list->next;
   }

   list = serv_list;
   PrintText(sess, "Server    Sock  Name\n");
   while(list)
   {
      v = (struct server *)list->data;
      sprintf(tbuf, "0x%lx %-5ld %s\n",
	      (unsigned long)v, v->sok, v->servername);
      PrintText(sess, tbuf);
      list = list->next;
   }

   list = dcc_list;
   PrintText(sess, "DCC       Sock  File      Type\n");
   while(list)
   {
      d = (struct DCC *)list->data;
      sprintf(tbuf, "0x%lx %-5ld %-9.9s %d\n",
	      (unsigned long)d, d->sok, file_part(d->file), (int)d->type);
      PrintText(sess, tbuf);
      list = list->next;
   }

   return TRUE;
}

int cmd_delbutton(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word[2])
   {
      if(list_delentry(&button_list, word[2]))
	maingui_updatebuttons(sess);
      /*else
	PrintText(sess, STARTON" No such button\n");*/
      return TRUE;
   }
   return FALSE;
}

int cmd_deop(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   int i = 2;
   while(1)
   {
      char *name = find_word(pdibuf, i);
      if(!*name) break;
      sprintf(tbuf, "MODE %s -o %s\r\n", sess->channel, name);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      i++;
   }
   return TRUE;
}

int cmd_devoice(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   int i = 2;
   while(1)
   {
      char *name = find_word(pdibuf, i);
      if(!*name) break;
      sprintf(tbuf, "MODE %s -v %s\r\n", sess->channel, name);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      i++;
   }
   return TRUE;
}

int cmd_discon(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   disconnect_server(sess, TRUE);
   return TRUE;
}

int cmd_exec(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word_eol[2])
   {
      FILE *fp = popen(word_eol[2], "r");
      if(fp)
      {
	 while(1)
	 {
	    memset(tbuf, 0, 1025);
	    fread(tbuf, 1024, 1, fp);
	    PrintText(sess, tbuf);
	    if(feof(fp)) break;
	 }
      }
      return TRUE;
   }
   return FALSE;
}

int cmd_quit(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word_eol[2])
     sess->quitreason = word_eol[2];
   else
     sess->quitreason = prefs.quitreason;
   gtk_widget_destroy(sess->window);
   return 2;
}

int cmd_gate(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *server = find_word(pdibuf, 2);
   if(*server)
   {
      char *port = find_word(pdibuf, 3);
      if(*port)
	connect_server(sess, server, atoi(port), TRUE);
      else
	connect_server(sess, server, 23, TRUE);
      return TRUE;
   }
   return FALSE;
}

int cmd_help(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   int i = 0;
   char *helpcmd = "";
   
   if(tbuf) helpcmd = find_word(pdibuf, 2);
   if(*helpcmd)
   {
      help(sess, helpcmd, FALSE);
   } else {
      struct popup *pop;
      GSList *list = command_list;
      char *buf = malloc(4096);
      int t = 1, j;
      strcpy(buf, "\nCommands Available:\n\n  ");
      while(1)
      {
	 if(!cmds[i].name) break;
	 strcat(buf, cmds[i].name);
	 t++;
	 if(t == 6)
	 {
	    t = 1;
	    strcat(buf, "\n  ");
	 } else
	   for(j=0; j<(10-strlen(cmds[i].name)); j++) strcat(buf, " ");
	 i++;
      } 
      strcat(buf, "\n\nType /HELP <command> for more information\n\n");
      strcat(buf, "User defined commands:\n\n  ");
      t = 1;
      while(list)
      {
	 pop = (struct popup *)list->data;
	 strcat(buf, pop->name);
	 t++;
	 if(t == 6)
	 {
	    t = 1;
	    strcat(buf, "\n  ");
	 } else
	    for(j=0; j<(10-strlen(pop->name)); j++) strcat(buf, " ");
	 list = list -> next;
      }
      strcat(buf, "\n\n");
      PrintText(sess, buf);
      free(buf);
   }
   return TRUE;
}

int hidever = 0;

int cmd_hidever(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(hidever)
   {
      hidever = FALSE;
      PrintText(sess, STARTON" HIDE-VER off\n");
   } else {
      hidever = TRUE;
      PrintText(sess, STARTON" HIDE-VER on\n");
   }
   return TRUE;
}

int cmd_invite(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word[2] && *word[3])
   {
      sprintf(tbuf, "INVITE %s %s\r\n", word[2], word[3]);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      return TRUE;
   }
   return FALSE;
}

int cmd_join(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *chan = find_word(pdibuf, 2);
   if(*chan)
   {
      char *po, *pass = find_word(pdibuf, 3);
      if(*pass)
	sprintf(tbuf, "JOIN %s %s\r\n", chan, pass);
      else
	sprintf(tbuf, "JOIN %s\r\n", chan);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      if(sess->channel[0] == 0)
      {
	 po = strchr(chan, ',');
	 if(po) *po = 0;
	 strncpy(sess->waitchannel, chan, 200);
      }
      return TRUE;
   }
   return FALSE;
}

int cmd_kick(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *nick = find_word(pdibuf, 2);
   if(*nick)
   {
      char *reason = find_word_to_end(cmd, 3);
      if(*reason)
	sprintf(tbuf, "KICK %s %s :%s\r\n", sess->channel, nick, reason);
      else
	sprintf(tbuf, "KICK %s %s\r\n", sess->channel, nick);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      return TRUE;
   }
   return FALSE;
}

int cmd_kickban(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *nick = find_word(pdibuf, 2);
   if(*nick)
   {
      sprintf(tbuf, "MODE %s +b %s!*@*\r\n", sess->channel, nick);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      {
	 char *reason = find_word_to_end(cmd, 3);
	 if(*reason)
	   sprintf(tbuf, "KICK %s %s :%s\r\n", sess->channel, nick, reason);
	 else
	   sprintf(tbuf, "KICK %s %s\r\n", sess->channel, nick);
	 send(sess->server->sok, tbuf, strlen(tbuf), 0);
	 return TRUE;
      }
   }
   return FALSE;
}

int cmd_lastlog(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *search = find_word(pdibuf, 2);

   show_lastlog_window();
   search_lastlog(sess, search);

   return TRUE;
}

int cmd_load(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
#ifdef USE_PERL
    char *file = find_word(pdi_buf, 2);
    check_homedir(file);
    switch (perl_load_file(file))
    {
    case 0:
       return TRUE;
    case 1:
       PrintText(sess, "Error compiling script\n");
       return FALSE;
    case 2:
       PrintText(sess, "Error Loading file\n");
       return FALSE;
    }
    return FALSE;
#else
   PrintText(sess, "Perl scripting not available in this compilation.\n");
   return TRUE;
#endif
}

#ifdef USE_PLUGIN
int cmd_loaddll(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char    *file = find_word(pdi_buf, 2);

   check_homedir(file);
   if (module_load(file, sess) == 0)
     return TRUE;
   return FALSE;
}
#endif

int cmd_me(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *act = find_word_to_end(cmd, 2);
   if(*act)
   {
      channel_action(sess, tbuf, sess->channel, sess->server->nick, act, TRUE);
      sprintf(tbuf, "PRIVMSG %s :\001ACTION %s\001\r\n", sess->channel, act);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      return TRUE;
   }
   return FALSE;
}

int cmd_msg(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *nick = find_word(pdibuf, 2);
   if(*nick)
   {
      char *msg = find_word_to_end(cmd, 3);
      if(*msg)
      {
	 if(strcmp(nick, ".") == 0) { /* /msg the last nick /msg'ed */
	    if(sess->lastnick[0])
	      nick = sess->lastnick;
	 } else
	   strcpy(sess->lastnick, nick); /* prime the last nick memory */
	 
	 sprintf(tbuf, "\0033>\003 %s\0033< \003 %s\n", nick, msg);
	 PrintText(sess, tbuf);
	 if(!dcc_write_chat(nick, msg))
	 {
	    sprintf(tbuf, "PRIVMSG %s :%s\r\n", nick, msg);
	    send(sess->server->sok, tbuf, strlen(tbuf), 0);
	 }
	 return TRUE;
      }
   }
   return FALSE;
}

int cmd_mode(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word_eol[2])
   {
      sprintf(tbuf, "MODE %s\r\n", word_eol[2]);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      return TRUE;
   }
   return FALSE;
}

int cmd_names(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word[2])
     sprintf(tbuf, "NAMES %s\r\n", word[2]);
   else
     sprintf(tbuf, "NAMES %s\r\n", sess->channel);
   send(sess->server->sok, tbuf, strlen(tbuf), 0);
   return TRUE;
}

int cmd_nctcp(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word[2] && *word_eol[3])
   {
      sprintf(tbuf, "NOTICE %s :\001%s\001\r\n", word[2], word_eol[3]);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      return TRUE;
   }
   return FALSE;
}

int cmd_nick(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *nick = find_word(pdibuf, 2);
   if(*nick)
   {
     if(sess->server->connected)
     {
	sprintf(tbuf, "NICK %s\r\n", nick);
	send(sess->server->sok, tbuf, strlen(tbuf), 0);
     } else
	user_new_nick(sess->server, tbuf, sess->server->nick, nick, TRUE);
      return TRUE;
   }
   return FALSE;
}

int cmd_notice(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word[2] && *word_eol[3])
   {
      sprintf(tbuf, "NOTICE %s :%s\r\n", word[2], word_eol[3]);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      sprintf(tbuf, "\0033>\003 %s\0033< \003 %s\n", word[2], word_eol[3]);
      PrintText(sess, tbuf);
      return TRUE;
   }
   return FALSE;
}

int cmd_notify(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word[2])
   {
      int i = 1;
      while(1)
      {
	 i++;
	 if(!*word[i]) break;
	 if(notify_deluser(word[i]))
	 {
	    sprintf(tbuf, STARTON" %s deleted from notify list.\n", word[i]);
	    PrintText(sess, tbuf);
	    return TRUE;
	 }
	 notify_adduser(word[i]);
	 sprintf(tbuf, STARTON" %s added to notify list.\n", word[i]);
	 PrintText(sess, tbuf);
      }
   } else
      notify_showlist(sess);
   return TRUE;
}

int cmd_op(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   int i = 2;
   while(1)
   {
      char *name = find_word(pdibuf, i);
      if(!*name) break;
      sprintf(tbuf, "MODE %s +o %s\r\n", sess->channel, name);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      i++;
   }
   return TRUE;
}

int cmd_part(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *chan = find_word(pdibuf, 2);
   if(!*chan) chan = sess->channel;
   if ((*chan) && ((chan[0] == '#') || (chan[0] == '&')))
   {
      sprintf(tbuf, "PART %s\r\n", chan);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      return TRUE;
   }
   return FALSE;
}

int cmd_ping(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   unsigned long tim;
   struct timeval timev;
   char *to = word[2];

   gettimeofday(&timev, 0);
   tim = (timev.tv_sec - 50000) * 1000000 + timev.tv_usec;

   if(*to)
      sprintf(tbuf, "PRIVMSG %s :\001PING %lu\001\r\n", to, tim);
   else
      sprintf(tbuf, "PING %lu :%s\r\n", tim, sess->server->servername);
   send(sess->server->sok, tbuf, strlen(tbuf), 0);
   return TRUE;
}

int cmd_query(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *nick = find_word(pdibuf, 2);
   if(*nick)
   {
      if(prefs.privmsgtab)
      {
	 if(!find_session_from_channel(nick, sess->server))
	 {
	    sess = tab_msg_session(nick, sess->server);
	    make_non_channel_window(sess);
	 }
      } else {
	 if(!find_dialog(sess->server, nick))
	   new_dialog(sess->server, nick);
      }
      return TRUE;
   }
   return FALSE;
}

int cmd_quote(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *raw = find_word_to_end(cmd, 2);
   if(*raw)
   {
      send(sess->server->sok, raw, strlen(raw), 0);
      send(sess->server->sok, "\r\n", 2, 0);
      return TRUE;
   }
   return FALSE;
}

#ifdef USE_PLUGIN
int cmd_rmdll(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char    *dllname = find_word(pdi_buf, 2);

   if (module_unload(dllname, sess) == 0)
     return TRUE;
   return FALSE;
}
#endif

int cmd_server(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *server = find_word(pdibuf, 2);
   if(*server)
   {
      char *port = find_word(pdibuf, 3);
      if(*port)
      {
	 char *pass = find_word(pdibuf, 4);
	 if(pass) strcpy(sess->server->password, pass);
	 connect_server(sess, server, atoi(port), FALSE);
      } else
	 connect_server(sess, server, 6667, FALSE);
      return TRUE;
   }
   return FALSE;
}

int cmd_topic(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *topic = find_word_to_end(cmd, 2);
   if(*topic)
     sprintf(tbuf, "TOPIC %s :%s\r\n", sess->channel, topic);
   else
     sprintf(tbuf, "TOPIC %s\r\n", sess->channel);  
   send(sess->server->sok, tbuf, strlen(tbuf), 0);
   return TRUE;
}

int cmd_unban(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   char *mask = find_word(pdibuf, 2);
   if(*mask)
   {
      sprintf(tbuf, "MODE %s -b %s\r\n", sess->channel, mask);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      return TRUE;
   }
   return FALSE;
}

int cmd_wallchop(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   if(*word_eol[2])
   {
      struct user *user = sess->userlist;
      char *ttbuf = malloc(4096);
      if(!ttbuf) return TRUE;
      sprintf(tbuf, "NOTICE %%s :[wallchop/@%s] %s\r\n", sess->channel, word_eol[2]);
      while(user)
      {
	 if(user->op)
	 {
	    sprintf(ttbuf, tbuf, user->user);
	    send(sess->server->sok, ttbuf, strlen(ttbuf), 0);
	 }
	 user = user->next;
      }
      free(ttbuf);
      return TRUE;
   }
   return FALSE;
}

int cmd_voice(struct session *sess, char *tbuf, char *word[], char *word_eol[])
{
   int i = 2;
   while(1)
   {
      char *name = find_word(pdibuf, i);
      if(!*name) break;
      sprintf(tbuf, "MODE %s +v %s\r\n", sess->channel, name);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
      i++;
   }
   return TRUE;
}

void check_special_chars(char *cmd) /* check for ^X */
{
   int len = strlen(cmd);
   char *buf = malloc(len + 1);
   if(buf)
   {
      int i = 0, j = 0;
      while(cmd[j])
      {
	 switch(cmd[j])
	 {
	  case '^':
	    if(j+3 < len && (isdigit(cmd[j+1]) && isdigit(cmd[j+2]) && isdigit(cmd[j+3])))
	    {
	       char tbuf[4];
	       tbuf[0] = cmd[j+1];
	       tbuf[1] = cmd[j+2];
	       tbuf[2] = cmd[j+3];
	       tbuf[3] = 0;
	       buf[i] = atoi(tbuf);
	       j+=3;
	    } else {
	       switch(cmd[j+1])
	       {
		case 'R':
		  buf[i] = '\026'; break;
		case 'U':
		  buf[i] = '\037'; break;
		case 'B':
		  buf[i] = '\002'; break;
		case 'C':
		  buf[i] = '\003'; break;
		case '^':
		  buf[i] = '^'; break;
		default:
		  buf[i] = '^'; j--; break;
	       }
	       j++;
	    }
	    break;
	  default:
	    buf[i] = cmd[j];
	 }
	 j++;
	 i++;
      }
      buf[i] = 0;
      strcpy(cmd, buf);
      free(buf);
   }
}

void check_nick_completion(struct session *sess, char *cmd, char *tbuf)
{
   char *space = strchr(cmd, ' ');
   if(space && space != cmd)
   {
      if(space[-1] == ':'  &&  space-1 != cmd)
      {
	 long len = (long)space - (long)cmd - 1;
	 if(len < 64)
	 {
	    struct user *user = sess->userlist;
	    char nick[64];
	    memcpy(nick, cmd, len);
	    nick[len] = 0;
	    while(user)
	    {
	       if(!strncasecmp(user->user, nick, len))
	       {
		  sprintf(tbuf, "%s:%s", user->user, space);
		  return;
	       }
	       user = user->next;
	    }
	 }
      }
   }
   tbuf[0] = 0;
}

void user_command(struct session *sess, char *cmd, char *word[], char *word_eol[])
{
   char tbuf[4096];
   char num[2];
   int i = 0, j = 0, n;

   num[1] = 0;

   while(1)
   {
      switch(cmd[i])
      {
       case 0:
	 tbuf[j] = 0;
	 handle_command(tbuf, sess, FALSE);
	 return;
       case '&':
       case '%':
	 if(isdigit(cmd[i+1]))
	 {
	    num[0] = cmd[i+1];
	    n = atoi(num);
	    if(*word[n] == 0)
	    {
	       PrintText(sess, "Bad arguments for user command.\n");
	       return;
	    }
	    if(cmd[i] == '%')
	      strcpy(&tbuf[j], word[n]);
	    else
	      strcpy(&tbuf[j], word_eol[n]);
	    j = strlen(tbuf);
	    i++;
	 } else {
	    switch(cmd[i+1])
	    {
	     case 'c':
	       if(!sess->channel[0])
	       {
		  notj_msg(sess);
		  return;
	       } else {
		  strcpy(&tbuf[j], sess->channel);
		  j = strlen(tbuf);
		  i++;
	       }
	       break;
	    }
	 }
	 break;
       default:
	 tbuf[j] = cmd[i];
	 j++;
      }
      i++;
   }
}

int handle_command(char *cmd, struct session *sess, int history)
{
   struct popup *pop;
   int user_cmd = FALSE, i;
   GSList *list = command_list;
   unsigned char pdibuf[2048];
   unsigned char tbuf[4096];
   char *word[32];
   char *word_eol[32];

   if(history) history_add(&sess->history, cmd);

   if(cmd[0] == '/')
   {
      cmd++;
#ifdef USE_PERL
      if (perl_command(cmd, sess)) return TRUE;
#endif

      process_data_init(pdibuf, cmd, word, word_eol);

      if (EMIT_SIGNAL(XP_USERCOMMAND, sess, pdibuf, word, word_eol, NULL, 0) == 1)
      	return TRUE;

#ifdef USE_PLUGIN
      if (module_command(pdibuf, sess, tbuf, word, word_eol) == 0)
	return TRUE;
#endif

      while(list)
      {
	 pop = (struct popup *)list->data;
	 if(!strcasecmp(pop->name, pdibuf))
	 {
	    user_command(sess, pop->cmd, word, word_eol);
	    user_cmd = TRUE;
	 }
	 list = list -> next;
      }

      if(user_cmd) return TRUE;

      check_special_chars(cmd);      

      i = 0;
      while(1)
      {
	 if(!cmds[i].name) break;
	 if(!strcasecmp(pdibuf, cmds[i].name))
	 {
	    if(cmds[i].needserver && !sess->server->connected)
	    {
	       notc_msg(sess);
	       return TRUE;
	    }
	    if(cmds[i].needchannel && sess->channel[0]==0)
	    {
	       notj_msg(sess);
	       return TRUE;
	    }
	    switch(cmds[i].callback(sess, tbuf, word, word_eol))
	    {
	     case FALSE:
	       help(sess, cmds[i].name, TRUE); break;
	     case 2:
	       return FALSE;
	    }
	    return TRUE;
	 }
	 i++;
      }
      PrintText(sess, "Unknown Command. Try /help\n");

   } else {
      
      check_special_chars(cmd);
      
      if(sess->server->connected)
      {
	 if(sess->channel[0])
	 {
	    char newcmd[4096];
	    if(prefs.nickcompletion)
	      check_nick_completion(sess, cmd, newcmd);
	    else
	      newcmd[0] = 0;
	    if(!newcmd[0])
	    {
	       channel_msg(sess->server, tbuf, sess->channel, sess->server->nick, cmd, TRUE);
	       sprintf(tbuf, "PRIVMSG %s :%s\r\n", sess->channel, cmd);

	       if(sess->channel[0] != '#' && sess->channel[0] != '&')
	       {
		  struct sockaddr_in *ip;
		  ip = dcc_write_chat(sess->channel, cmd);
		  if(ip)
		  {
		     gtk_entry_set_text(GTK_ENTRY(sess->topicgad),
					inet_ntoa(ip->sin_addr));
		     return TRUE;
		  }
	       }

	    } else {
	       channel_msg(sess->server, tbuf, sess->channel, sess->server->nick, newcmd, TRUE);
               sprintf(tbuf, "PRIVMSG %s :%s\r\n", sess->channel, newcmd);
	    }
            send(sess->server->sok, tbuf, strlen(tbuf), 0);
	 } else
	   notj_msg(sess);
      } else
	notc_msg(sess);
   }
   return TRUE;
}

void handle_inputgad(GtkWidget *igad, struct session *sess)
{
   char *cr, *cmd = gtk_entry_get_text(GTK_ENTRY(igad));

   if(cmd[0] == 0) return;

   cr = strchr(cmd, '\n');
   if(cr)
   {
      while(1)
      {
	 if(cr) *cr = 0;
	 if(!handle_command(cmd, sess, TRUE)) return;
	 if(!cr) break;
	 cmd = cr + 1;
	 if(*cmd == 0) break;
	 cr = strchr(cmd, '\n');
      }
   } else {
     if(!handle_command(cmd, sess, TRUE)) return;
   }

   gtk_entry_set_text(GTK_ENTRY(igad), "");
}
