/* update.c   Common update routines for normal and shadow passwd file

   Copyright 1997, 1998 Thorsten Kukuk, <kukuk@uni-paderborn.de>
   Complete rewritten for better shadow support.

   Copyright 1994, 1995 Olaf Kirch, <okir@monad.swb.de>

   This program are free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this program; see the file COPYING.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA. */

#ifndef LINT
static const char rcsid[] = "$Id: update.c,v 1.13 1998/10/10 08:16:54 kukuk Exp $";
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "system.h"

#include <pwd.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include "yppasswd.h"
#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif
#ifdef HAVE_SHADOW_H
#include <shadow.h>
#endif

#ifndef HAVE_LCKPWDF
#include <compat/lckpwdf.c>
#endif

#ifndef CHECKROOT
/* Set to 1 if you want to chack against the root password. */
#define CHECKROOT   0
#endif

#ifndef _PATH_PASSWD
#define _PATH_PASSWD		"/etc/passwd"
#endif
#ifdef HAVE_GETSPNAM
#ifndef _PATH_SHADOW
#define _PATH_SHADOW		"/etc/shadow"
#endif
#endif
#ifndef _PATH_SHELLS
#define _PATH_SHELLS		"/etc/shells"
#endif

/* How often to retry locking the passwd file...
 */
#define MAX_RETRIES 5

char *path_passwd = _PATH_PASSWD;
char *path_passwd_tmp = NULL;
char *path_passwd_old = NULL;
#ifdef HAVE_GETSPNAM
char *path_shadow = _PATH_SHADOW;
char *path_shadow_tmp = NULL;
char *path_shadow_old = NULL;
#endif

#ifndef svc_getcaller
/* This is a cheat. */
#define svc_getcaller(x) 	((struct sockaddr_in *) (x)->xp_rtaddr.buf)
#endif
#define xprt_addr(xprt)		(svc_getcaller(xprt)->sin_addr)
#define xprt_port(xprt)		ntohs(svc_getcaller(xprt)->sin_port)
void reaper (int sig);

/*===============================================================*
 * Argument validation. Avoid \n... (ouch).
 * We can't use isprint, because people may use 8bit chars which
 * aren't recognized as printable in the default locale.
 *===============================================================*/
static int
validate_string (char *what, char *str)
{
  while (*str && *str != ':' && *str >= 32)
    str++;
  if (*str == '\0')
    return 1;
  syslog (LOG_ALERT, "Invalid characters in %s argument: \"%s\"",
	  what, str);
  return 0;
}

static int
validate_args (struct xpasswd *pw)
{
  if (pw->pw_name[0] == '-' || pw->pw_name[0] == '+')
    {
      syslog (LOG_ALERT, "attempt to modify NIS passwd entry \"%s\"",
	      pw->pw_name);
      return 0;
    }

  return validate_string ("password", pw->pw_passwd)
    && validate_string ("shell", pw->pw_shell)
    && validate_string ("gecos", pw->pw_gecos);
}

static int
shell_ok (char *shell)
{
  char buffer[256];
  FILE *fp;

  if ((fp = fopen (_PATH_SHELLS, "r")) == NULL)
    {
      syslog (LOG_WARNING, "can't open %s", _PATH_SHELLS);
      return 0;
    }
  while (fgets (buffer, 255, fp) != NULL)
    {
      buffer[255] = '\0';
      if (!strncmp (buffer, shell, strcspn (buffer, " \t\n")))
	{
	  fclose (fp);
	  return 1;
	}
    }

  fclose (fp);
  return 0;
}

#ifdef HAVE_GETSPNAM
static int
isexpired(const struct spwd *sp)
{
  long now = time ((time_t *) 0) / (24L*3600L);

  if (sp->sp_expire > 0 && now >= sp->sp_expire)
    return 3;

  if (sp->sp_lstchg > 0 && sp->sp_max >= 0 && sp->sp_inact >= 0 &&
      now >= sp->sp_lstchg + sp->sp_max + sp->sp_inact)
    return 2;
  if (sp->sp_lstchg == -1 ||
      sp->sp_max == -1 || sp->sp_max >= (10000L))
    return 0;
  if (now >= sp->sp_lstchg + sp->sp_max)
    return 1;
  return 0;
}
#endif

#if CHECKROOT
static int
password_ok(char *plain, char *crypted, char *root)
{
  if (crypted[0] == '\0')
    return 1;
  if (strcmp(crypt(plain, crypted), crypted) == 0)
    return 1;
  if (strcmp(crypt(plain, root), root) == 0)
    return 1;

  return 0;
}
#endif

/*===============================================================*
 * The /etc/passwd update handler
 *===============================================================*/
int *
yppasswdproc_pwupdate_1 (yppasswd *yppw, struct svc_req *rqstp)
{
  struct xpasswd *newpw;	/* passwd struct passed by the client */
  struct passwd *pw;		/* passwd struct obtained from fgetpwent() */
#ifdef HAVE_GETSPNAM
  struct spwd *spw;
  int have_shadow = 0, shadow_changed = 0;
  FILE *oldsf, *newsf;
#endif
  int retries, gotit, c;
  int chsh = 0, chfn = 0;
  FILE *oldpf, *newpf;
  static int res;
  char logbuf[255];
#if CHECKROOT
  char       rootpass[32];

  if ((pw = getpwnam ("root")) != NULL)
    {
#ifdef HAVE_GETSPNAM
      if (strcmp (pw->pw_passwd, "x") == 0)
	{
	  if ((spw = getspnam ("root")) != NULL)
	    strcpy (rootpass, spw->sp_pwdp);
	  else
	    strcpy (rootpass, "x");
	}
      else
#endif
	strcpy(rootpass, pw->pw_passwd);
    }
  else
    strcpy(rootpass, "x");
#endif

  newpw = &yppw->newpw;
  res = 1;

#ifdef HAVE_SNPRINTF
  snprintf (logbuf, sizeof (logbuf) -1, "update %.12s (uid=%d) from host %s",
	    yppw->newpw.pw_name,
	    yppw->newpw.pw_uid,
	    inet_ntoa (xprt_addr (rqstp->rq_xprt)));
#else
  if (sizeof (logbuf)-1 < 75 + strlen (yppw->newpw.pw_name))
    strcpy (logbuf, "logbuf too small!");
  else
    sprintf (logbuf, "update %.12s (uid=%d) from host %s",
	     yppw->newpw.pw_name,
	     yppw->newpw.pw_uid,
	     inet_ntoa (xprt_addr (rqstp->rq_xprt)));
#endif

  if (!validate_args (newpw))
    {
      syslog (LOG_ALERT, "%s failed", logbuf);
      return &res;
    }

  /* Lock the passwd file. We retry several times. Maybe it would be
   * better to just return an error condition and have the client reattempt
   * instead? This procedure is already slow enough...
   */
  retries = 0;
  while (lckpwdf () && retries < MAX_RETRIES)
    {
      sleep (1);
      retries++;
    }

  if (retries == MAX_RETRIES)
    {
      syslog (LOG_NOTICE, "%s failed", logbuf);
      syslog (LOG_NOTICE, "password file locked");
      goto error2;
    }

  /* Open the passwd file for reading. We can't use getpwent and friends
     here, because they go through the YP maps, too. */
  if ((oldpf = fopen (path_passwd, "r")) == NULL)
    {
      syslog (LOG_ERR, "%s failed", logbuf);
      syslog (LOG_ERR, "Can't open %s: %m", path_passwd);
      goto error2;
    }

  /* Open a temp passwd file */
  if ((newpf = fopen (path_passwd_tmp, "w+")) == NULL)
    {
      syslog (LOG_ERR, "%s failed", logbuf);
      syslog (LOG_ERR, "Can't open %s: %m", path_passwd_tmp);
      fclose (oldpf);
      oldpf=0;
      goto error2;
    }
  chmod (path_passwd_tmp, S_IRUSR+S_IWUSR+S_IRGRP+S_IROTH);

#ifdef HAVE_GETSPNAM
  /* Open the shadow file for reading. */
  if ((oldsf = fopen (path_shadow, "r")) != NULL)
    {
      /* We have a shadow file */
      have_shadow = 1;
      shadow_changed = 0;

      if ((newsf = fopen (path_shadow_tmp, "w+")) == NULL)
	{
	  syslog (LOG_ERR, "%s failed", logbuf);
	  syslog (LOG_ERR, "Can't open %s.tmp: %m", path_passwd);
	  fclose (oldsf);
	  fclose (newpf);
	  fclose (oldpf);
	  oldsf=newpf=oldpf=0;
	  goto error2;
	}
      chmod (path_shadow_tmp, S_IRUSR);
    }
  else
    {
      have_shadow = 0;
      shadow_changed = 0;
      newsf = NULL;
    }
#endif

  gotit = 0;

  /* Loop over all passwd entries */
  while ((pw = fgetpwent (oldpf)) != NULL)
    {
      /* check if this is the uid we want to change. A few
	 sanity checks added for consistency. */
      if ((uid_t)newpw->pw_uid == pw->pw_uid &&
	  (uid_t)newpw->pw_gid == pw->pw_gid &&
	  !strcmp (newpw->pw_name, pw->pw_name) && !gotit)
	{
	  /* Check the password. */
	  if (newpw->pw_passwd[0] != 'x' && newpw->pw_passwd[0] != '*')
	    {
#ifdef HAVE_GETSPNAM
	      if (have_shadow && pw->pw_passwd[0] == 'x' &&
		  pw->pw_passwd[1] == '\0')
		{
		  while ((spw = fgetspent (oldsf)) != NULL)
		    {
		      if (strcmp (newpw->pw_name, spw->sp_namp) == 0)
			{
#if CHECKROOT
			  if (!password_ok(yppw->oldpass, spw->sp_pwdp,
					   rootpass))
#else
			  if (spw->sp_namp[0] != '\0' &&
			      strcmp (crypt (yppw->oldpass, spw->sp_pwdp),
				      spw->sp_pwdp))
#endif
			    {
			      syslog (LOG_WARNING, "%s rejected", logbuf);
			      syslog (LOG_WARNING, "Invalid password.");
			      goto error;
			    }
			  /* test if password is expired */
			  if (spw->sp_pwdp[0] != '!' && !isexpired (spw) &&
			      !(spw->sp_max >= 0 && spw->sp_min > spw->sp_max))
			    {
			      time_t now, ok;

			      time(&now);
			      /* test if it is allowed to change the password*/
			      ok = (spw->sp_lstchg * (24L*3600L)) +
				(spw->sp_min > 0 ?
				 spw->sp_min * (24L*3600L) : 0);
			      if (now < ok)
				{
				  syslog (LOG_WARNING, "%s rejected", logbuf);
				  syslog (LOG_WARNING,
					  "now < minimum age for `%s'",
					  spw->sp_namp);
				  goto error;
				}
			      else
				{
				  /* set the new passwd */
				  spw->sp_pwdp = newpw->pw_passwd;
				  spw->sp_lstchg = (long int)now / (24L*3600L);
				  shadow_changed = 1;
				}
			    }
			}
		      if (putspent (spw, newsf) < 0)
			{
			  syslog (LOG_ERR, "%s failed", logbuf);
			  syslog (LOG_ERR,
				  "Error while writing new shadow file: %m");
			  shadow_changed = 0;
			  goto error;
			}
		    }
		  fclose (newsf);
		  fclose (oldsf);
		  newsf=oldsf=0;
		  if (shadow_changed)
		    {
		      unlink (path_shadow_old);
		      link (path_shadow, path_shadow_old);
		      rename (path_shadow_tmp, path_shadow);
		    }
		  else
		    unlink (path_shadow_tmp);
		}
	      else
#endif
		{
		  /* This user doesn't have shadow files */
#if CHECKROOT
		  if (!password_ok(yppw->oldpass, pw->pw_passwd,
				   rootpass))
#else
		  if (pw->pw_passwd[0] != '\0' &&
		      strcmp (crypt (yppw->oldpass, pw->pw_passwd),
			      pw->pw_passwd))
#endif
		    {
		      syslog (LOG_WARNING, "%s rejected", logbuf);
		      syslog (LOG_WARNING, "Invalid password.");
		      break;
		    }
		  /* set the new passwd */
		  pw->pw_passwd = newpw->pw_passwd;
		}
	    }
	  /* set the new shell, and full name */
	  chsh = (strcmp (pw->pw_shell, newpw->pw_shell) != 0);
	  if (chsh)
	    {
	      if (!allow_chsh)
		{
		  syslog (LOG_WARNING, "%s rejected", logbuf);
		  syslog (LOG_WARNING, "chsh not permitted");
		  break;
		}
	      if (!shell_ok (newpw->pw_shell))
		{
		  syslog (LOG_WARNING, "%s rejected", logbuf);
		  syslog (LOG_WARNING, "invalid shell: %s",
			  newpw->pw_shell);
		  break;
		}
	      pw->pw_shell = newpw->pw_shell;
	    }
	  chfn = (strcmp (pw->pw_gecos, newpw->pw_gecos) != 0);
	  if (chfn)
	    {
	      if (!allow_chfn)
		{
		  syslog (LOG_WARNING, "%s rejected", logbuf);
		  syslog (LOG_WARNING, "chfn not permitted");
		  break;
		}
	      pw->pw_gecos = newpw->pw_gecos;
	    }
	  gotit++;
	}

      /* write the passwd entry to tmp file
       */
      if (putpwent (pw, newpf) < 0)
	{
	  syslog (LOG_ERR, "%s failed", logbuf);
	  syslog (LOG_ERR, "Error while writing new password file: %m");
	  break;
	}
      /* fflush (newpf); */
    }
 error:
  if(newpf) fclose (newpf);
  if(oldpf) fclose (oldpf);
#ifdef HAVE_GETSPNAM
  if (have_shadow && !shadow_changed)
    {
      if(newsf) fclose (newsf);
      if(oldsf) fclose (oldsf);
    }
#endif

  /* Check if we dropped out of the loop because of an error.
   * If so, return an error to the client.
   */
  if (pw != NULL)
    {
      unlink (path_passwd_tmp);
#ifdef HAVE_GETSPNAM
      if (have_shadow)
	unlink (path_shadow_tmp);
#endif
    error2:
      ulckpwdf ();
      return (&res);
    }

  /* Check whether we actually changed anything
   */
  if (!gotit)
    {
      syslog (LOG_WARNING, "%s failed", logbuf);
      syslog (LOG_WARNING, "User not in password file.");
      unlink (path_passwd_tmp);
#ifdef HAVE_GETSPNAM
      if (have_shadow)
	unlink (path_shadow_tmp);
#endif
      ulckpwdf ();
      return (&res);
    }

  unlink (path_passwd_old);
  link (path_passwd, path_passwd_old);
  rename (path_passwd_tmp, path_passwd);	/* atomic */
#ifdef HAVE_GETSPNAM
  if (have_shadow && !shadow_changed)
    {
      unlink (path_shadow_tmp);
    }
#endif
  ulckpwdf ();

  /* Fork off process to rebuild NIS passwd.* maps.
   */
  if ((c = fork ()) < 0)
    {
      /* Do NOT restore old password file. Someone else may already
       * be using the new one. */
      syslog (LOG_ERR, "%s failed", logbuf);
      syslog (LOG_ERR, "Couldn't fork map update process: %m");
      return (&res);
    }
  if (c == 0)
    {
#ifdef HAVE_GETSPNAM
      if (shadow_changed)
	execlp (MAP_UPDATE_PATH, MAP_UPDATE, "shadow", NULL);
      else
#endif
	execlp (MAP_UPDATE_PATH, MAP_UPDATE, "passwd", NULL);
      syslog (LOG_ERR, "Error: couldn't exec map update process: %m");
      exit (1);
    }

  syslog (LOG_INFO, "%s successful. Password changed.", logbuf);
  if (chsh || chfn)
    {
      syslog (LOG_INFO, "Shell %schanged (%s), GECOS %schanged (%s).",
	      chsh ? "" : "un", newpw->pw_shell,
	      chfn ? "" : "un", newpw->pw_gecos);
    }
  res = 0;

  return (&res);
}
