/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 *
 * TaskBar
 */

#include "icewm.h"

TaskBarApp::TaskBarApp(YFrameWindow *frame, YWindow *aParent, Window win): YWindow(aParent, win) {
    fFrame = frame;
    fPrev = fNext = 0;
    selected = 0;
    fShown = 1;
}

TaskBarApp::~TaskBarApp() {
}

void TaskBarApp::setShown(int ashow) {
    if (ashow != fShown) {
        fShown = ashow;
    }
}

void TaskBarApp::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    YColor *bg = normalTaskBarAppBg;
    YColor *fg = normalTaskBarAppFg;
    int p;

    if (frame()->focused() || selected == 2) {
        p = 2;
        if (frame()->focused()) {
            bg = activeTaskBarAppBg;
            fg = activeTaskBarAppFg;
        }
        g.setColor(bg);
        g.drawBorderW(0, 0, width() - 1, height() - 1, false);
    } else {
        p = 1;
        g.setColor(bg);
        g.drawBorderW(0, 0, width() - 1, height() - 1, true);
    }
    g.fillRect(p, p, width() - 3, height() - 3);
    if (frame()->clientIcon() && frame()->clientIcon()->small())
        g.drawMaskPixmap(frame()->clientIcon()->small(), p + 1, p + 1);

    char *str = (char *)frame()->client()->iconTitle();
    if (str == 0 || str[0] == 0)
        str = (char *)frame()->client()->windowTitle();
    if (str) {
        g.setColor(fg);
        if (frame()->focused())
            g.setFont(activeTaskBarFont);
        else
            g.setFont(normalTaskBarFont);
        g.drawChars((char *)str, 0, strlen(str),
                    p + 3 + 16,
                    p + height() - titleFont->descent() - 4);
    }
}

void TaskBarApp::handleButton(const XButtonEvent &button) {
    YWindow::handleButton(button);
    if (button.button == 1 || button.button == 2) {
        if (button.type == ButtonPress) {
            selected = 2;
            repaint();
        } else if (button.type == ButtonRelease) {
            if (selected == 2) {
                if (button.button == 1) {
                    if (button.state & Mod1Mask)
                        frame()->wmMinimize();
                    else
                        frame()->activate();
                } else if (button.button == 2) {
                    if (button.state & Mod1Mask) {
                        frame()->client()->setWorkspaces(frame()->client()->workspaces() | (1 << app->activeWorkspace()));
                        frame()->activate();
                    } else {
                        frame()->wmOccupyOnlyWorkspace(app->activeWorkspace());
                        frame()->activate();
                    }
                }
            }
            selected = 0;
            repaint();
        }
    }
}

void TaskBarApp::handleCrossing(const XCrossingEvent &crossing) {
    if (selected > 0) {
        if (crossing.type == EnterNotify) {
            selected = 2;
            repaint();
        } else if (crossing.type == LeaveNotify) {
            selected = 1;
            repaint();
        }
    }
}

void TaskBarApp::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &up, int /*count*/) {
    //if (up.button == 2)
    //    frame()->wmMinimize();
    //else
    if (up.button == 3) {
        frame()->updateMenu();
        frame()->windowMenu()->popup(0, frame(),
                                     up.x_root, up.y_root, -1, -1,
                                     YPopupWindow::pfCanFlipVertical |
                                     YPopupWindow::pfCanFlipHorizontal |
                                     YPopupWindow::pfPopupMenu);
    }
}

YClock::YClock(YWindow *aParent, Window win): YWindow(aParent, win) {
    selected = 0;
    setSize(96, 20);
}

YClock::~YClock() {
}

void YClock::handleButton(const XButtonEvent &button) {
    if (button.type == ButtonPress) {
        selected = 2;
        repaint();
    } else if (button.type == ButtonRelease) {
        if (selected == 2) {
            selected = 0;
            repaint();
        } else
            selected = 0;
    }
}

void YClock::handleCrossing(const XCrossingEvent &crossing) {
    if (crossing.type == EnterNotify) {
        if (selected == 1) {
            selected = 2;
            repaint();
        }
    } else if (crossing.type == LeaveNotify) {
        if (selected == 2) {
            selected = 1;
            repaint();
        }
    }
}

void YClock::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    /*XFillRectangle(app->display(),
                   handle(),
                   blackGC,
                   0, 0, width(), height());*/

    int x = 0;
    time_t newTime = time(NULL);
    YPixmap *pix[6];
    YPixmap *deli;

    struct tm *t = localtime(&newTime);

    if (selected == 2) {
        deli = PixNoColon;
        pix[0] = PixNum[(t->tm_year / 10) % 10];
        pix[1] = PixNum[t->tm_year % 10];
        pix[2] = PixNum[((t->tm_mon + 1) / 10) % 10];
        pix[3] = PixNum[(t->tm_mon + 1) % 10];
        pix[4] = PixNum[(t->tm_mday / 10) % 10];
        pix[5] = PixNum[t->tm_mday % 10];
    } else {
        deli = PixColon;
        pix[0] = PixNum[(t->tm_hour / 10) % 10];
        pix[1] = PixNum[t->tm_hour % 10];
        pix[2] = PixNum[(t->tm_min / 10) % 10];
        pix[3] = PixNum[t->tm_min % 10];
        pix[4] = PixNum[(t->tm_sec / 10) % 10];
        pix[5] = PixNum[t->tm_sec % 10];
    }
    g.drawPixmap(pix[0], x, 0); x += pix[0]->width();
    g.drawPixmap(pix[1], x, 0); x += pix[1]->width();
    g.drawPixmap(deli  , x, 0); x += deli->width();
    g.drawPixmap(pix[2], x, 0); x += pix[2]->width();
    g.drawPixmap(pix[3], x, 0); x += pix[3]->width();
    g.drawPixmap(deli  , x, 0); x += deli->width();
    g.drawPixmap(pix[4], x, 0); x += pix[4]->width();
    g.drawPixmap(pix[5], x, 0); x += pix[5]->width();
}

TaskBar::TaskBar(YWindow *aParent, Window win): YWindow(aParent, win) {
    unsigned int ht = 26;

#define BASE1 2
#define ADD1 3
#define BASE2 3
#define ADD2 5

    fWorkspaceButton = 0;
    if (taskBarShowWorkspaces) {
        fWorkspaceButton = (YButton **)malloc(sizeof(YButton *) * workspaceCount);
        assert(fWorkspaceButton != 0);
    }
    
    fFirst = fLast = 0;
    fCount = 0;
    setStyle(wsOverrideRedirect);

    fClock = new YClock(this);
    if (fClock->height() + ADD1 > ht) ht = fClock->height() + ADD1;
    if (showMailBoxStatus)
        mailBoxStatus = new MailBoxStatus(mailBoxPath, mailCommand, this);
    fApplications = new YButton(this, rootMenu);
    fApplications->setPixmap(startPixmap);
    if (fApplications->height() + ADD1 > ht) ht = fApplications->height() + ADD1;
    fWinList = new YButton(this, windowListMenu);
    fWinList->setPixmap(windowsPixmap);
    if (fWinList->height() + ADD1 > ht) ht = fWinList->height() + ADD1;

    if (taskBarShowWorkspaces) {
        for (int w = 0; w < workspaceCount; w++) {
            YButton *wk = new YButton(this, cmdActivateWorkspace, (void *)w);
            wk->setText(workspaceNames[w]);
            if (wk->height() + ADD2 > ht) ht = wk->height() + ADD2;
            fWorkspaceButton[w] = wk;
        }
    }
    setGeometry(-1,
                taskBarAtTop ? -1 : int(app->root()->height() - ht),
                app->root()->width() + 2, ht + 1);

    leftX = 2;
    rightX = width() - 4;
    if (showTaskBarClock) {
        fClock->setPosition(rightX - fClock->width(),
                            BASE1 + (ht - ADD1 - fClock->height()) / 2);
        fClock->show();
        rightX -= fClock->width() + 2;
    }
    if (mailBoxStatus) {
        mailBoxStatus->setPosition(rightX - mailBoxStatus->width() - 1,
                                   BASE2 + (ht - ADD2 - mailBoxStatus->height()) / 2);
	mailBoxStatus->show();
        rightX -= mailBoxStatus->width() + 3;
    }
    leftX += 2;
    fApplications->setPosition(leftX,
                               BASE1 + (ht - ADD1 - fApplications->height()) / 2);
    fApplications->show();
    leftX += fApplications->width();
    //leftX += 2;
    fWinList->setPosition(leftX,
                         BASE1 + (ht - ADD1 - fWinList->height()) / 2);
    fWinList->show();
    leftX += fWinList->width() + 2;

    if (taskBarShowWorkspaces) {
        leftX += 2;
        for (int w = 0; w < workspaceCount; w++) {
            YButton *wk = fWorkspaceButton[w];
            //leftX += 2;
            wk->setSize(wk->width(), ht - ADD2);
            wk->setPosition(leftX, BASE2); // + (ht - ADD - wk->height()) / 2);
            wk->show();
            leftX += wk->width();
        }
        leftX += 4;
    }
}

TaskBar::~TaskBar() {
    delete fClock; fClock = 0;
    delete mailBoxStatus; mailBoxStatus = 0;
    for (int w = 0; w < workspaceCount; w++)
        delete fWorkspaceButton[w];
}

void TaskBar::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    g.setColor(taskBarBg);
    g.draw3DRect(0, 0, width() - 1, height() - 1, true);
    g.fillRect(1, 1, width() - 2, height() - 2);
}

void TaskBar::handleButton(const XButtonEvent &button) {
    if (button.type == ButtonPress) {
        if (button.button == 1) {
            if (button.state & Mod1Mask)
                lower();
            else if (!(button.state & ControlMask))
                raise();
        }
    }
    YWindow::handleButton(button);
}

void TaskBar::handleClick(const XButtonEvent &down, const XButtonEvent &up, int count) {
    if (up.button == 1) {
    } else if (up.button == 2) {
    } else if (up.button == 3 && count == 1) {
        rootMenu->popup(0, 0, up.x_root, up.y_root, -1, -1,
                        YPopupWindow::pfCanFlipVertical |
                        YPopupWindow::pfCanFlipHorizontal);
    }
    YWindow::handleClick(down, up, count);
}

void TaskBar::handleDrag(const XButtonEvent &/*down*/, const XMotionEvent &motion) {
#ifndef NO_CONFIGURE
    int newPosition = 0;
    
    if (motion.y_root < int(app->root()->height() / 2))
        newPosition = 1;
    
    if (taskBarAtTop != newPosition) {
        taskBarAtTop = newPosition;
        setPosition(x(), taskBarAtTop ? -1 : int(app->root()->height() - height() + 1));

        app->relocateWindows(0, taskBarAtTop ? height() : - height());
    }
#endif
}

void TaskBar::insert(TaskBarApp *tapp) {
    fCount++;
    tapp->setNext(0);
    tapp->setPrev(fLast);
    if (fLast)
        fLast->setNext(tapp);
    else
        fFirst = tapp;
    fLast = tapp;
}

void TaskBar::remove(TaskBarApp *tapp) {
    fCount--;

    if (tapp->getPrev())
        tapp->getPrev()->setNext(tapp->getNext());
    else
        fFirst = tapp->getNext();

    if (tapp->getNext())
        tapp->getNext()->setPrev(tapp->getPrev());
    else
        fLast = tapp->getPrev();
}

TaskBarApp *TaskBar::addApp(YFrameWindow *frame) {
    if (frame->client() == windowList)
        return 0;

    TaskBarApp *tapp = new TaskBarApp(frame, this);

    if (tapp != 0) {
        insert(tapp);
        tapp->show();
        if (!frame->visibleOn(app->activeWorkspace()))
            tapp->setShown(0);
        relayout();
    }
    return tapp;
}

void TaskBar::removeApp(YFrameWindow *frame) {
    TaskBarApp *f = fFirst;

    while (f) {
        if (f->frame() == frame) {
            f->hide();
            remove(f);
            delete f;
            relayout();
            return ;
        }
        f = f->getNext();
    }
}

void TaskBar::relayout() {
    int x, y, w, h;
    int tc = 0;

    TaskBarApp *a = fFirst;
    
    while (a) {
        if (a->getShown())
            tc++;
        a = a->getNext();
    }

    if (tc < 3) tc = 3;

    h = height() - ADD2 - 1;
    w = (rightX - leftX - 2) / tc - 2;
    x = leftX;
    y = BASE2 + (height() - ADD2 - 1 - h) / 2;

    TaskBarApp *f = fFirst;
        
    while (f) {
        if (f->getShown()) {
            f->setGeometry(x, y, w, h);
            f->show();
            x += w;
            x += 2;
        } else
            f->hide();
        f = f->getNext();
    }
}
