/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "xchat.h"
#include "ignore.h"
#include "plugin.h"
#include "gtkutil.h"
#include "cfgfiles.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


extern GSList *ignore_list;

extern int match (char *mask, char *name);  /* util.c */
extern void PrintText (struct session *sess, unsigned char *text);  /* text.c */


int ignored_ctcp = 0;           /* keep a count of all we ignore */
int ignored_priv = 0;
int ignored_chan = 0;
int ignored_noti = 0;
int ignored_invi = 0;
int ignored_total = 0;

GtkWidget *ignorewin = 0;
GtkWidget *ignorelist;
GtkWidget *entry_mask;
GtkWidget *entry_ctcp;
GtkWidget *entry_private;
GtkWidget *entry_channel;
GtkWidget *entry_notice;
GtkWidget *entry_invi;
GtkWidget *entry_unignore;      /* these are toggles, not really entrys */

GtkWidget *num_ctcp;
GtkWidget *num_priv;
GtkWidget *num_chan;
GtkWidget *num_noti;
GtkWidget *num_invi;

/* ignore_add(...)

 * returns:
 *            0 fail
 *            1 success
 *            2 success (old ignore has been changed)
 */

int
ignore_add (char *mask, int priv, int noti, int chan, int ctcp, int invi, int unignore)
{
   struct ignore *ig = 0;
   GSList *list;
   int change_only = FALSE;

   /* first check if it's already ignored */
   list = ignore_list;
   while (list)
   {
      ig = (struct ignore *) list->data;
      if (!strcasecmp (ig->mask, mask))
      {
         /* already ignored, change the flags */
         change_only = TRUE;
         break;
      }
      list = list->next;
   }

   if (!change_only)
      ig = malloc (sizeof (struct ignore));

   if (!ig)
      return 0;

   ig->mask = strdup (mask);
   ig->priv = priv;
   ig->noti = noti;
   ig->chan = chan;
   ig->ctcp = ctcp;
   ig->invi = invi;
   ig->unignore = unignore;

   if (!change_only)
      ignore_list = g_slist_append (ignore_list, ig);
   ignore_gui_update (1);

   if (change_only)
      return 2;

   return 1;
}

void
ignore_showlist (struct session *sess)
{
   struct ignore *ig;
   GSList *list = ignore_list;
   char tbuf[256];
   int i = 0;

   EMIT_SIGNAL (XP_TE_IGNOREHEADER, sess, 0, 0, 0, 0, 0);

   while (list)
   {
      ig = (struct ignore *) list->data;
      i++;

      sprintf (tbuf, " %-20s ", ig->mask);
      if (ig->priv)
         strcat (tbuf, "YES  ");
      else
         strcat (tbuf, "NO   ");
      if (ig->noti)
         strcat (tbuf, "YES  ");
      else
         strcat (tbuf, "NO   ");
      if (ig->chan)
         strcat (tbuf, "YES  ");
      else
         strcat (tbuf, "NO   ");
      if (ig->ctcp)
         strcat (tbuf, "YES  ");
      else
         strcat (tbuf, "NO   ");
      if (ig->invi)
         strcat (tbuf, "YES  ");
      else
         strcat (tbuf, "NO   ");
      if (ig->unignore)
         strcat (tbuf, "YES  ");
      else
         strcat (tbuf, "NO   ");
      strcat (tbuf, "\n");
      PrintText (sess, tbuf);
      /*EMIT_SIGNAL (XP_TE_IGNORELIST, sess, ig->mask, 0, 0, 0, 0); */
      /* use this later, when TE's support 7 args */
      list = list->next;
   }

   if (!i)
      EMIT_SIGNAL (XP_TE_IGNOREEMPTY, sess, 0, 0, 0, 0, 0);

   EMIT_SIGNAL (XP_TE_IGNOREFOOTER, sess, 0, 0, 0, 0, 0);
}

/* ignore_del()

 * one of the args must be NULL, use mask OR *ig, not both
 *
 */

int
ignore_del (char *mask, struct ignore *ig)
{
   if (!ig)
   {
      GSList *list = ignore_list;

      while (list)
      {
         ig = (struct ignore *) list->data;
         if (!strcasecmp (ig->mask, mask))
            break;
         list = list->next;
         ig = 0;
      }
   }
   if (ig)
   {
      ignore_list = g_slist_remove (ignore_list, ig);
      free (ig->mask);
      free (ig);
      ignore_gui_update (1);
      return TRUE;
   }
   return FALSE;
}

/* check if a msg should be ignored by browsing our ignore list */

int
ignore_check (char *host, int priv, int noti, int chan, int ctcp, int invi)
{
   struct ignore *ig;
   GSList *list = ignore_list;

   /* check if there's an UNIGNORE first, they take precendance. */
   while (list)
   {
      ig = (struct ignore *) list->data;
      if (ig->unignore)
      {
         if (
               (ig->priv && priv) ||
               (ig->noti && noti) ||
               (ig->chan && chan) ||
               (ig->ctcp && ctcp) ||
               (ig->invi && invi)
            )
         {
            if (match (ig->mask, host))
               return FALSE;
         }
      }
      list = list->next;
   }

   list = ignore_list;
   while (list)
   {
      ig = (struct ignore *) list->data;

      if (
            (ig->priv && priv) ||
            (ig->noti && noti) ||
            (ig->chan && chan) ||
            (ig->ctcp && ctcp) ||
            (ig->invi && invi)
         )
      {
         if (match (ig->mask, host))
         {
            ignored_total++;
            if (priv)
               ignored_priv++;
            if (noti)
               ignored_noti++;
            if (chan)
               ignored_chan++;
            if (ctcp)
               ignored_ctcp++;
            if (invi)
               ignored_invi++;
            ignore_gui_update (2);
            return TRUE;
         }
      }
      list = list->next;
   }

   return FALSE;
}

char *
ignore_read_next_entry (char *my_cfg, struct ignore *ignore)
{
   char tbuf[1024];
   char mask[256];

   /* Casting to char * done below just to satisfy compiler */

   if (my_cfg)
   {
      my_cfg = cfg_get_str (my_cfg, "mask ", mask);
      ignore->mask = strdup (mask);
   }
   if (my_cfg)
   {
      my_cfg = cfg_get_str (my_cfg, "ctcp ", (char *) tbuf);
      ignore->ctcp = atoi ((char *) tbuf);
   }
   if (my_cfg)
   {
      my_cfg = cfg_get_str (my_cfg, "private ", (char *) tbuf);
      ignore->priv = atoi ((char *) tbuf);
   }
   if (my_cfg)
   {
      my_cfg = cfg_get_str (my_cfg, "channel ", (char *) tbuf);
      ignore->chan = atoi ((char *) tbuf);
   }
   if (my_cfg)
   {
      my_cfg = cfg_get_str (my_cfg, "notice ", (char *) tbuf);
      ignore->noti = atoi ((char *) tbuf);
   }
   if (my_cfg)
   {
      my_cfg = cfg_get_str (my_cfg, "invite ", (char *) tbuf);
      ignore->invi = atoi ((char *) tbuf);
   }
   if (my_cfg)
   {
      my_cfg = cfg_get_str (my_cfg, "unignore ", (char *) tbuf);
      ignore->unignore = atoi ((char *) tbuf);
   }
   return my_cfg;
}

void
ignore_load ()
{
   struct ignore *ignore;
   struct stat st;
   char *cfg, *my_cfg;
   char file[256];
   int fh;

   snprintf (file, sizeof file, "%s/ignore.conf", get_xdir ());
   fh = open (file, O_RDONLY);
   if (fh != -1)
   {
      fstat (fh, &st);
      if (st.st_size)
	 cfg = malloc (st.st_size);
      else
	 cfg = NULL;
      if (cfg)
      {
         read (fh, cfg, st.st_size);
         my_cfg = cfg;
         while (my_cfg)
         {
            ignore = malloc (sizeof (struct ignore));
            if ((my_cfg = ignore_read_next_entry (my_cfg, ignore)))
               ignore_list = g_slist_append (ignore_list, ignore);
            else
               free (ignore);
         }
         free (cfg);
      }
      close (fh);
   }
}

void
ignore_save ()
{
   char buf[1024];
   int fh;
   GSList *temp = ignore_list;

   snprintf (buf, sizeof buf, "%s/ignore.conf", get_xdir ());
   fh = open (buf, O_TRUNC | O_WRONLY | O_CREAT, 0600);
   if (fh != -1)
   {
      while (temp)
      {
         snprintf (buf, sizeof buf,
                   "mask = %s\n"
                   "ctcp = %d\n"
                   "private = %d\n"
                   "channel = %d\n"
                   "notice = %d\n"
                   "invite = %d\n"
                   "unignore = %d\n\n",
                   ((struct ignore *) temp->data)->mask,
                   ((struct ignore *) temp->data)->ctcp,
                   ((struct ignore *) temp->data)->priv,
                   ((struct ignore *) temp->data)->chan,
                   ((struct ignore *) temp->data)->noti,
                   ((struct ignore *) temp->data)->invi,
                   ((struct ignore *) temp->data)->unignore);
         write (fh, buf, strlen (buf));
         temp = temp->next;
      }
      close (fh);
   }
}

void
ignore_save_clist_tomem (void)
{
   struct ignore *ignore;
   char *tmp;
   int i = 0;
   GSList *list;

   list = ignore_list;
   while (list)
   {
      ignore = (struct ignore *) list->data;
      ignore_list = g_slist_remove (ignore_list, ignore);
      free (ignore->mask);
      free (ignore);
      list = ignore_list;
   }

   while (1)
   {
      if (!gtk_clist_get_text (GTK_CLIST (ignorelist), i, 0, &tmp))
      {
         break;
      }
      ignore = malloc (sizeof (struct ignore));
      memset (ignore, 0, sizeof (struct ignore));
      ignore->mask = strdup (tmp);
      gtk_clist_get_text (GTK_CLIST (ignorelist), i, 1, &tmp);
      if (!strcmp (tmp, "Yes"))
         ignore->ctcp = 1;
      gtk_clist_get_text (GTK_CLIST (ignorelist), i, 2, &tmp);
      if (!strcmp (tmp, "Yes"))
         ignore->priv = 1;
      gtk_clist_get_text (GTK_CLIST (ignorelist), i, 3, &tmp);
      if (!strcmp (tmp, "Yes"))
         ignore->chan = 1;
      gtk_clist_get_text (GTK_CLIST (ignorelist), i, 4, &tmp);
      if (!strcmp (tmp, "Yes"))
         ignore->noti = 1;
      gtk_clist_get_text (GTK_CLIST (ignorelist), i, 5, &tmp);
      if (!strcmp (tmp, "Yes"))
         ignore->invi = 1;
      gtk_clist_get_text (GTK_CLIST (ignorelist), i, 6, &tmp);
      if (!strcmp (tmp, "Yes"))
         ignore->unignore = 1;
      ignore_list = g_slist_append (ignore_list, ignore);
      i++;
   }
}

void
ignore_add_clist_entry (struct ignore *ignore)
{
   char *nnew[7];

   nnew[0] = ignore->mask;
   if (ignore->ctcp)
      nnew[1] = "Yes";
   else
      nnew[1] = "No";
   if (ignore->priv)
      nnew[2] = "Yes";
   else
      nnew[2] = "No";
   if (ignore->chan)
      nnew[3] = "Yes";
   else
      nnew[3] = "No";
   if (ignore->noti)
      nnew[4] = "Yes";
   else
      nnew[4] = "No";
   if (ignore->invi)
      nnew[5] = "Yes";
   else
      nnew[5] = "No";
   if (ignore->unignore)
      nnew[6] = "Yes";
   else
      nnew[6] = "No";

   gtk_clist_append (GTK_CLIST (ignorelist), nnew);
}

void
ignore_sort_clicked (void)
{
   gtk_clist_sort (GTK_CLIST (ignorelist));
}

void
ignore_delete_entry_clicked (GtkWidget * wid, struct session *sess)
{
   int row;

   row = gtkutil_clist_selection (ignorelist);
   if (row != -1)
   {
      gtk_clist_unselect_all (GTK_CLIST (ignorelist));
      gtk_clist_remove ((GtkCList *) ignorelist, row);
   }
}

void
ignore_new_entry_clicked (GtkWidget * wid, struct session *sess)
{
   int i, row;
   gchar *nnew[7];
   /*
      Serverlist copies an existing entry, but not much point to do so here
    */
   nnew[0] = "new!new@new.com";
   nnew[1] = "No";
   nnew[2] = "No";
   nnew[3] = "No";
   nnew[4] = "No";
   nnew[5] = "No";
   nnew[6] = "No";
   row = gtkutil_clist_selection (ignorelist);
   i = gtk_clist_insert (GTK_CLIST (ignorelist), row + 1, nnew);
   gtk_clist_select_row (GTK_CLIST (ignorelist), i, 0);
   gtk_clist_moveto (GTK_CLIST (ignorelist), i, 0, 0.5, 0);
}

void
ignore_check_state_toggled (GtkWidget * wid, int row, int col)
{
   char *text;
   int state;

   gtk_clist_get_text (GTK_CLIST (ignorelist), row, col, &text);
   if (!strcmp (text, "Yes"))
      state = 1;
   else
      state = 0;

   gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wid), state);
}

void
ignore_row_unselected (GtkWidget * clist, int row, int column,
                GdkEventButton * even)
{
   gtk_entry_set_text (GTK_ENTRY (entry_mask), "");
}

void
ignore_row_selected (GtkWidget * clist, int row, int column,
 GdkEventButton * even, gpointer sess)
{
   char *mask;
   row = gtkutil_clist_selection (ignorelist);
   if (row != -1)
   {
      gtk_clist_get_text (GTK_CLIST (ignorelist), row, 0, &mask);
      gtk_entry_set_text (GTK_ENTRY (entry_mask), mask);
      ignore_check_state_toggled (entry_ctcp, row, 1);
      ignore_check_state_toggled (entry_private, row, 2);
      ignore_check_state_toggled (entry_channel, row, 3);
      ignore_check_state_toggled (entry_notice, row, 4);
      ignore_check_state_toggled (entry_invi, row, 5);
      ignore_check_state_toggled (entry_unignore, row, 6);
   } else
      ignore_row_unselected (0, 0, 0, 0);
}

void
ignore_handle_mask (GtkWidget * igad)
{
   int row;

   row = gtkutil_clist_selection (ignorelist);
   if (row != -1)
   {
      char *mask = gtk_entry_get_text ((GtkEntry *) igad);
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 0, mask);
   }
}

void
ignore_ctcp_toggled (GtkWidget * igad, gpointer serv)
{
   int row;
   row = gtkutil_clist_selection (ignorelist);
   if (GTK_TOGGLE_BUTTON (igad)->active)
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 1, "Yes");
   else
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 1, "No");
}

void
ignore_private_toggled (GtkWidget * igad, gpointer serv)
{
   int row;
   row = gtkutil_clist_selection (ignorelist);
   if (GTK_TOGGLE_BUTTON (igad)->active)
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 2, "Yes");
   else
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 2, "No");
}

void
ignore_channel_toggled (GtkWidget * igad, gpointer serv)
{
   int row;
   row = gtkutil_clist_selection (ignorelist);
   if (GTK_TOGGLE_BUTTON (igad)->active)
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 3, "Yes");
   else
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 3, "No");
}

void
ignore_notice_toggled (GtkWidget * igad, gpointer serv)
{
   int row;
   row = gtkutil_clist_selection (ignorelist);
   if (GTK_TOGGLE_BUTTON (igad)->active)
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 4, "Yes");
   else
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 4, "No");
}

void
ignore_invi_toggled (GtkWidget * igad, gpointer serv)
{
   int row;
   row = gtkutil_clist_selection (ignorelist);
   if (GTK_TOGGLE_BUTTON (igad)->active)
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 5, "Yes");
   else
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 5, "No");
}

void
ignore_unignore_toggled (GtkWidget * igad, gpointer serv)
{
   int row;
   row = gtkutil_clist_selection (ignorelist);
   if (GTK_TOGGLE_BUTTON (igad)->active)
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 6, "Yes");
   else
      gtk_clist_set_text ((GtkCList *) ignorelist, row, 6, "No");
}

int
close_ignore_list ()
{
   ignore_save_clist_tomem ();
   ignore_save ();
   gtk_widget_destroy (ignorewin);
   return 0;
}

int
close_ignore_gui_callback ()
{
   ignorewin = 0;
   return 0;
}

GtkWidget *
ignore_stats_entry (GtkWidget * box, char *label, int value)
{
   GtkWidget *wid;
   char buf[16];

   sprintf (buf, "%d", value);
   gtkutil_label_new (label, box);
   wid = gtkutil_entry_new (16, box, 0, 0);
   gtk_widget_set_usize (wid, 30, 0);
   gtk_entry_set_editable (GTK_ENTRY (wid), FALSE);
   gtk_entry_set_text (GTK_ENTRY (wid), buf);

   return wid;
}

void
ignore_gui_open ()
{
   static gchar *titles[] =
   {"Mask", "CTCP", "Private", "Chan", "Notice", "Invite", "Unignore"};
   GtkWidget *vbox, *box, *wid, *stat_box,
            *frame;
   GSList *temp = ignore_list;

   if (ignorewin)
   {
      gdk_window_show (ignorewin->window);
      return;
   }
   ignorewin = gtkutil_window_new ("X-Chat: Ignore list", "ignorelist",
   0, 0, close_ignore_gui_callback, 0);
   gtk_window_set_policy (GTK_WINDOW (ignorewin), TRUE, TRUE, FALSE);

   vbox = gtk_vbox_new (0, 2);
   gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);
   gtk_container_add (GTK_CONTAINER (ignorewin), vbox);
   gtk_widget_show (vbox);

   ignorelist = gtkutil_clist_new (7, titles, vbox, GTK_POLICY_ALWAYS,
               ignore_row_selected, 0,
                                   ignore_row_unselected, 0, GTK_SELECTION_BROWSE);

   gtk_widget_set_usize (ignorelist, 500, 180);

   gtk_clist_set_column_width (GTK_CLIST (ignorelist), 0, 196);
   gtk_clist_set_column_width (GTK_CLIST (ignorelist), 1, 40);
   gtk_clist_set_column_width (GTK_CLIST (ignorelist), 2, 40);
   gtk_clist_set_column_width (GTK_CLIST (ignorelist), 3, 40);
   gtk_clist_set_column_width (GTK_CLIST (ignorelist), 4, 40);
   gtk_clist_set_column_width (GTK_CLIST (ignorelist), 5, 40);

   box = gtk_hbox_new (0, 0);
   gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, TRUE, 5);
   gtk_widget_show (box);

   gtkutil_label_new ("Ignore Mask:", box);
   entry_mask = gtkutil_entry_new (99, box, ignore_handle_mask, 0);

   box = gtk_hbox_new (0, 0);
   gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, TRUE, 5);
   gtk_widget_show (box);

   entry_ctcp = gtk_check_button_new_with_label ("CTCP");
   gtk_signal_connect (GTK_OBJECT (entry_ctcp), "toggled",
                       GTK_SIGNAL_FUNC (ignore_ctcp_toggled), 0);
   gtk_container_add (GTK_CONTAINER (box), entry_ctcp);
   gtk_widget_show (entry_ctcp);

   entry_private = gtk_check_button_new_with_label ("Private");
   gtk_signal_connect (GTK_OBJECT (entry_private), "toggled",
                       GTK_SIGNAL_FUNC (ignore_private_toggled), 0);
   gtk_container_add (GTK_CONTAINER (box), entry_private);
   gtk_widget_show (entry_private);

   entry_channel = gtk_check_button_new_with_label ("Channel");
   gtk_signal_connect (GTK_OBJECT (entry_channel), "toggled",
                       GTK_SIGNAL_FUNC (ignore_channel_toggled), 0);
   gtk_container_add (GTK_CONTAINER (box), entry_channel);
   gtk_widget_show (entry_channel);

   entry_notice = gtk_check_button_new_with_label ("Notice");
   gtk_signal_connect (GTK_OBJECT (entry_notice), "toggled",
                       GTK_SIGNAL_FUNC (ignore_notice_toggled), 0);
   gtk_container_add (GTK_CONTAINER (box), entry_notice);
   gtk_widget_show (entry_notice);

   entry_invi = gtk_check_button_new_with_label ("Invite");
   gtk_signal_connect (GTK_OBJECT (entry_invi), "toggled",
                       GTK_SIGNAL_FUNC (ignore_invi_toggled), 0);
   gtk_container_add (GTK_CONTAINER (box), entry_invi);
   gtk_widget_show (entry_invi);

   entry_unignore = gtk_check_button_new_with_label ("Unignore");
   gtk_signal_connect (GTK_OBJECT (entry_unignore), "toggled",
                       GTK_SIGNAL_FUNC (ignore_unignore_toggled), 0);
   gtk_container_add (GTK_CONTAINER (box), entry_unignore);
   gtk_widget_show (entry_unignore);

   wid = gtk_hseparator_new ();
   gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 8);
   gtk_widget_show (wid);

   frame = gtk_frame_new ("Ignore Stats:");
   gtk_widget_show (frame);

   stat_box = gtk_hbox_new (0, 2);
   gtk_container_set_border_width (GTK_CONTAINER (stat_box), 6);
   gtk_container_add (GTK_CONTAINER (frame), stat_box);
   gtk_widget_show (stat_box);

   num_ctcp = ignore_stats_entry (stat_box, "CTCP:", ignored_ctcp);
   num_priv = ignore_stats_entry (stat_box, "Private:", ignored_priv);
   num_chan = ignore_stats_entry (stat_box, "Channel:", ignored_chan);
   num_noti = ignore_stats_entry (stat_box, "Notice:", ignored_noti);
   num_invi = ignore_stats_entry (stat_box, "Invite:", ignored_invi);

   gtk_container_add (GTK_CONTAINER (vbox), frame);

   box = gtk_hbox_new (0, 2);
   gtk_container_set_border_width (GTK_CONTAINER (box), 2);
   gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, TRUE, 2);
   gtk_widget_show (box);

   gtkutil_button (ignorewin, GNOME_STOCK_BUTTON_OK, "OK",
           close_ignore_list, 0, box);
   gtkutil_button (ignorewin, GNOME_STOCK_PIXMAP_NEW, "New",
    ignore_new_entry_clicked, 0, box);
   gtkutil_button (ignorewin, GNOME_STOCK_PIXMAP_CLOSE, "Delete",
                   ignore_delete_entry_clicked, 0, box);
   gtkutil_button (ignorewin, GNOME_STOCK_PIXMAP_SPELLCHECK, "Sort",
         ignore_sort_clicked, 0, box);
   gtkutil_button (ignorewin, GNOME_STOCK_BUTTON_CANCEL, "Cancel",
     gtkutil_destroy, ignorewin, box);

   while (temp)
   {
      ignore_add_clist_entry ((struct ignore *) temp->data);
      temp = temp->next;
   }
   gtk_widget_show (ignorewin);
}

void
ignore_gui_update (int level)
{
   /* some ignores have changed via /ignore, we should update
      the gui now */
   /* level 1 = the list only. */
   /* level 2 = the numbers only. */
   /* for now, ignore level 1, since the ignore GUI isn't realtime,
      only saved when you click OK */
   char buf[16];

   if (level == 2 && ignorewin)
   {
      sprintf (buf, "%d", ignored_ctcp);
      gtk_entry_set_text (GTK_ENTRY (num_ctcp), buf);

      sprintf (buf, "%d", ignored_noti);
      gtk_entry_set_text (GTK_ENTRY (num_noti), buf);

      sprintf (buf, "%d", ignored_chan);
      gtk_entry_set_text (GTK_ENTRY (num_chan), buf);

      sprintf (buf, "%d", ignored_invi);
      gtk_entry_set_text (GTK_ENTRY (num_invi), buf);

      sprintf (buf, "%d", ignored_priv);
      gtk_entry_set_text (GTK_ENTRY (num_priv), buf);
   }
}
