%
% history.sl
% Store/Recall mini-buffer lines
%
% $Id: history.sl,v 1.13 2000/04/07 12:32:01 rocher Exp $
%

static variable Max_Num_Lines = 32; % The same as in 'mini.sl'

%!%+
%\variable{History_File}
%\synopsis{History_File}
%\usage{String_Type History_File = "jed.his";}
%\description
% The variable \var{History_File} is used by the function \var{history_save}
% to know the file name in which to store all non-blank lines of the
% mini-buffer. Its default value is "~/.jed-history" under Unix and
% "~/jed.his" on other platforms.
%\seealso{history_load, history_save}
%!%-
custom_variable ("History_File", 
#ifdef UNIX
		 expand_filename ("~/.jed-history")
#elifdef VMS
		 "sys$login:jed.his"
#else
		 expand_filename ("~/jed.his")
#endif
		 );

%!%+
%\function{history_load}
%\synopsis{history_load}
%\usage{Void_Type history_load ();}
%\description
% This function reads a history file, i.e. each line of the file is
% stored in the mini-buffer, but not evaluated. By default, historical
% records are kept in a file whose name is stored in the \var{History_file}
% variable. This file is written in the current working directory
% associated with jed, i.e. the directory from which you started the jed
% process. For example, to read your history file every time you start
% jed and give an alternative name to the history file, put:
%#v+
%   variable History_File;
%   if (BATCH == 0)
%   {
%      () = evalfile ("history");
%      History_File = ".my-jed-history";
%      history_load ();
%   }
%#v-
% in your .jedrc (or jed.rc) file. The \var{History_File} variable can be
% assigned either a file name or an absolute path+filename. In the first
% case, a file will be saved in the current working directory (the one
% you were in when you started jed), so you will find a history file in
% every directory you work in. In the second one, only one file will be
% created.
%\seealso{history_save, minued}
%\seealso{History_File}
%!%-
define history_load ()
{
   variable fp = fopen (History_File, "r");
   if (fp == NULL)
     return;

   variable lines = fgetslines (fp);
   if (lines == NULL)
     return;
   
   % remove trailing newline
   lines = array_map (String_Type, &strtrim_end, lines, "\n");
   if (length (lines) > 1)
     mini_set_lines (lines[[1:]]);
}


%!%+
%\function{history_save}
%\synopsis{history_save}
%\usage{Int_Type history_save ()}
%\description
% This function saves the contents of the mini-buffer (see \var{history_load}
% for more information) to the file specified by the variable \var{History_File}.
% It returns -1 upon failure, or 0 upon success.
%\notes
% When history.sl is loaded, \var{history_save} will automatically get attached
% to the editors exit hooks.  As a result, it is really not necessary to call 
% this function directly.
%\seealso{history_load}
%\seealso{History_File}
%!%-
define history_save ()
{
   variable lines = mini_get_lines (NULL), not_blank;
   variable fp = fopen (History_File, "w");
   if (fp == NULL)
     {
	verror ("Unable to open %s", History_File);
	return -1;
     }
   () = chmod (History_File, 0600);

   () = fprintf (fp, "%% JED: File generated by 'history_save' on %s\n", time ());

   not_blank = where (array_map (Integer_Type, &strlen, lines) > 0);
   
   foreach (lines [not_blank])
     {
	variable line = ();
	() = fprintf (fp, "%s\n", line);
     }

   return 0;
}

static define save_history_at_exit ()
{
   () = history_save ();
   return 1;
}

add_to_hook ("_jed_exit_hooks", &save_history_at_exit);
