/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>

#include "global.h"
#include "flight.h"
#include "window_state.h"
#include "window.h"

/* This adds a force from a given attractor to a partial sum of forces */
/* The attraction is given as a force magnitude and a vector from one
 * object to the other */
static void
add_force (float magnitude, float delta[3], float sum[3])
{
  /* Magnitude over (distance cubed) */
  float magodist3;
  float dist2;

  /* I need to normalize delta and then divide by dist squared for the
   * inverse square law.  This is equivalent to dividing by dist
   * cubed.  I toss in the force magnitude to make one nice scalar.  */
  dist2 = (delta[0]*delta[0] + delta[1]*delta[1] + delta[2]*delta[2]);
  
  /* If dist == 0 then the flying object is on the attractor and it
   * shouldn't move anyway */
  if (dist2 != 0)
  {
    magodist3 = magnitude / dist2;
    
    //magodist3 *= magnitude / sqrt (magodist3);

    sum[0] += delta[0] * magodist3;
    sum[1] += delta[1] * magodist3;
    sum[2] += delta[2] * magodist3;
  }
}

static void 
attractor (Flight_Model *flight,
	   float magnitude, float p[4], float sum[3])
{
  float delta[3];

  delta[0] = p[0] - flight->X[0];
  delta[1] = p[1] - flight->X[1];
  delta[2] = p[2] - flight->X[2];

  add_force (magnitude, delta, sum);
}


void
flight_update_position (Flight_Model *flight, Demo_State *state)
{
  float F[3] = {0, 0, 0};
  float delta[3];
  float center[3];
  int i;
  float ur[3], ll[3];

  /* Check all of the attractors and repellers and calculate the net
   * force on the bird.  */

  for (i = 0; i < state->n_wstates; i++)
  {
    /* Construct the vector between the flying object and the center
     * of mass of the attractor */
    center[0] = (vtx[state->wstates[i]->index][0].x +
		 vtx[state->wstates[i]->index][1].x) / 2;
    center[1] = (vtx[state->wstates[i]->index][1].y +
		 vtx[state->wstates[i]->index][2].y) / 2;
    center[2] =  vtx[state->wstates[i]->index][1].z;

    delta[0] = center[0] - flight->X[0];
    delta[1] = center[1] - flight->X[1];
    delta[2] = center[2] - flight->X[2];

    /* FIXME: repulsion has to overpower attraction at close range */
    add_force (state->wstates[i]->repulsion, delta, F);
    
    /* Now do the attractors at the four corners */
    attractor (flight, state->wstates[i]->attraction[0], 
	       state->wstates[i]->world_upper_left, F);
    ur[0] = state->wstates[i]->world_lower_right[0]; 
    ur[1] = state->wstates[i]->world_upper_left[1]; 
    ur[2] = state->wstates[i]->world_upper_left[2];
    attractor (flight, state->wstates[i]->attraction[0], ur, F);
    attractor (flight, state->wstates[i]->attraction[0], 
	       state->wstates[i]->world_lower_right, F);
    ll[0] = state->wstates[i]->world_upper_left[0]; 
    ll[1] = state->wstates[i]->world_lower_right[1]; 
    ll[2] = state->wstates[i]->world_upper_left[2];
    attractor (flight, state->wstates[i]->attraction[0], ll, F);
    
  }

  /* Also sum over the 6 offscreen repulsors that keep things on screen */
  ll[0] = -2;
  ll[1] = 0;
  ll[2] = 0;  
  attractor  (flight, -20, ll, F);
  ll[0] = 2;
  attractor  (flight, -20, ll, F);
  ll[0] = 0;
  ll[1] = 2;
  attractor  (flight, -20, ll, F);
  ll[2] = -2;
  attractor  (flight, -20, ll, F);
  ll[2] = 0;
  ll[3] = -75;
  attractor  (flight, -20, ll, F);
  ll[3] = -25;
  attractor  (flight, -20, ll, F);
  
#if 0
  /* FIXME: for now just fly around the center of the screen */
  delta[0] = -flight->X[0];
  delta[1] = -flight->X[1];
  delta[2] = -50 - flight->X[2];
  add_force (1, delta, F);
#endif

  /* Assume some mass for the flying object should have contributed
   * linearly to the force of attraction.  Then to convert the total
   * force to an acceleration I should divide by the mass.  Hence the
   * mass would cancel and the force can be used as the acceleration */
  /* Given the current position, velocity and acceleration I can
   * calculate the new velocity and position.  */
  flight->dX[0] += F[0] * state->dt;
  flight->dX[1] += F[1] * state->dt;
  flight->dX[2] += F[2] * state->dt;  
  //#define TRACE_PATH
#ifdef TRACE_PATH
  printf ("v = (%f %f %f)\n", 
	  flight->dX[0],
	  flight->dX[1],
	  flight->dX[2]);

  printf ("X = (%f %f %f)\n", flight->X[0], 
	  flight->X[1], flight->X[2]);
#endif
  /* And the position */
  flight->X[0] += flight->dX[0] * state->dt;
  flight->X[1] += flight->dX[1] * state->dt;
  flight->X[2] += flight->dX[2] * state->dt;  
}



