/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "tux.h"
#include "window_state.h"
#include "gxf.h"
#include "window.h"
#include "mathutil.h"

/* These arrays are used to automatically update tux's animation
 * frame regardless of which state he is in.
 */
int tux_max_frames [TUX_LAST] = 
{
  TUX_WALK_MAX_FRAME,
  TUX_FLY_MAX_FRAME,
  TUX_JUMP_START_MAX_FRAME,
  TUX_JUMP_END_MAX_FRAME,
  TUX_TURN_MAX_FRAME,
  TUX_REST_MAX_FRAME,
  0, /* TUX_IDLE */
  0  /* TUX_NONE */
};

/* Set various tux velocities */
#define SET_TUX_WALK_V(state) \
  state->tux_velocity[0] = 6 * state->tux_direction; \
  state->tux_velocity[1] = 0; \
  state->tux_velocity[2] = 0; \
  state->tux_acceleration[0] = 0;\
  state->tux_acceleration[1] = 0;\
  state->tux_acceleration[2] = 0

/* This gets used in several places */
/* FIXME: this needs to read the window's locations to determine v[1] */
#define SET_TUX_CORNER_POSITION(v)\
  v[2] = TUX_POSITION_TWEAK_Z;\
  v[1] = -0.01 * v[2];\
  v[0] = 0


/* I feel really bad about this hack.  But all of the tux animations
 * use the same single texture.  I don't have a smart texture cache so
 * I can't really check for duplicate textures.  I am going to remove
 * all of the dumpicates manually.  If this sort of hack makes your
 * squeemish then close your eyes now.  */
#define TEX_HACK(tux_state) \
  state->texloc -= 174768

void
init_tux (Demo_State *state)
{
  Matrix rotate;

  /* Setup the tux state */
  RotateMat (state->tux_orientation, DEG_TO_RAD (-90), 1, 0, 0);
  /* Andrew Fear thinks 1/7.5th scale is 3/4 of what tux's size should be. */
  UniformScaleMat (state->tux_scale, (1.f / 7.5f) * (4.f / 3.f));
  RotateMat (rotate, DEG_TO_RAD (90), 0, 0, 1);
  MatMultMat4x4 (state->tux_orientation, state->tux_orientation, rotate);
  /* Load the tux animations */
  state->tux_anim[TUX_WALK] = open_gxf (state, "data/tux/tux_w.GXF");

  /* The next GXF file uses the same texture.  TEX_HACK backs up the 
   * texture pointer so that the next's GXF's texture will replace this one */
  TEX_HACK (TUX_FLY);

  state->tux_anim[TUX_FLY] = open_gxf (state, "data/tux/TUX_flight.GXF");
  TEX_HACK (TUX_FLY);

  state->tux_anim[TUX_JUMP_START] = open_gxf (state, 
			   "data/tux/TUX_JumpStart.GXF");
  TEX_HACK (TUX_JUMP_START);

  state->tux_anim[TUX_JUMP_END] = open_gxf (state, 
			   "data/tux/TUX_JumpEnd.GXF");
  TEX_HACK (TUX_JUMP_END);

  state->tux_anim[TUX_TURN] = open_gxf (state, 
					"data/tux/TUX_Turn.GXF");
  TEX_HACK (TUX_TURN);

  state->tux_anim[TUX_SIT] = open_gxf (state,
				       "data/tux/TUX_rest.GXF");
  /* No TEX_HACK here.  This is the final Tux GXF so this textures stays. */

  state->tux_anim[TUX_IDLE] = NULL;
  state->tux_anim[TUX_NONE] = NULL;

  state->tux_direction = 1;
  

  state->tux_window = NULL;

  state->tux_state = TUX_WALK;

  state->tux_frame = TUX_WALK_MIN_FRAME;

  /* Put Tux on the corner of his window */
  SET_TUX_CORNER_POSITION (state->tux_position);

  SET_TUX_WALK_V (state);
}

/* Build a parametric equation for a parabola.  This parabola will
 * pass through v1 and v2.  The length of time to traverse the 
 * parabola will be some multiple of base_time.  The vertical
 * speed of the parabola will be based on the motion of an
 * object moving under gravitational attraction near the surface
 * of some object.  (That is due to scaling issues it doesn't 
 * necessarily behave like earth.)  This parbola is a perfect
 * jump path between v1 and v2 given an animated object with
 * cycle time of base_time.
 * The parbola is returned by setting the tux_a, tux_b, and tux_c
 * coefficients in the state.
 */
static int
build_parabola (Demo_State *state, Vector v1, Vector v2, float base_time)
{ 
  float hdist;
  int factor;
  Vector a;

  /* This is the acceleration for my "planet". */
  a[0] = 0;
  a[1] = -3.6;
  a[2] = 0;

/* Defining TRACE_PARABOLA prints out lots of info about How tux's
 * jump arcs get generated. */
  /*#define TRACE_PARABOLA*/
#ifdef  TRACE_PARABOLA
  printf ("parabola:\n\tx1 = %f y1 = %f z1 = %f\n",
	  v1[0], v1[1], v1[2]);	  
  printf ("\tx2 = %f y2 = %f z2 = %f\n",
	  v2[0], v2[1], v2[2]);
  printf ("\tbase_time = %f\n", base_time);
#endif
  /* This is the non-vertical distance to be traveled */
  hdist = sqrt ((v2[0] - v1[0]) * (v2[0] - v1[0]) + 
		(v2[2] - v1[2]) * (v2[2] - v1[2]));
  if (hdist < 0)
    hdist = -hdist;

#ifdef TRACE_PARABOLA
  printf ("\thdist = %f\n", hdist);
#endif

 /* Tux has a maximum horizontal velocity.  If he would have to exceed
   * it then we just jump/fly longer
   */
  factor = (int)ceil (hdist / TUX_MAX_H_VELOCITY);
#ifdef TRACE_PARABOLA
  printf ("\tfactor = %f\n", (double)factor);
#endif 

  /* Just scale base_time */
  base_time *= factor;

  state->tux_b[0] = (v2[0] - v1[0]) / base_time;
  state->tux_b[2] = (v2[2] - v1[2]) / base_time;
  state->tux_b[1] = (v2[1] - v1[1] + 
		     (a[1] * a[1] * base_time * base_time) / 2) / 
    base_time;

  state->tux_a[0] = 0;
  state->tux_a[1] = -a[1] * a[1] / 2;
  state->tux_a[2] = 0;
  
  state->tux_c[0] = v1[0];
  state->tux_c[1] = v1[1];
  state->tux_c[2] = v1[2];

#ifdef TRACE_PARABOLA
  {
    float t = 59.f * factor / 30.f;

    printf ("\tfinal prabola coefficients\n");
    printf ("\t\ta = (%f %f %f)\n", 
	    state->tux_a[0], state->tux_a[1], state->tux_a[2]);
    printf ("\t\tb = (%f %f %f)\n", 
	  state->tux_b[0], state->tux_b[1], state->tux_b[2]);
    printf ("\t\tc = (%f %f %f)\n", 
	    state->tux_c[0], state->tux_c[1], state->tux_c[2]);  
    printf ("best approx of target is (%f %f %f)\n",
	    state->tux_a[0] * t * t +
	    state->tux_b[0] * t + state->tux_c[0],
	    state->tux_a[1] * t * t +
	    state->tux_b[1] * t + state->tux_c[1],
	    state->tux_a[2] * t * t +
	    state->tux_b[2] * t + state->tux_c[2]);
  }
#endif

  return ((int)factor);
}

/* Place tux at the upper left hand corner of his window and reset him
 * to walking forward. */
void
put_tux_on_window_corner (Demo_State *state)
{
  Matrix rotate, xlate;

  if (!state->tux_window)
    return;
  
  SET_TUX_CORNER_POSITION (state->tux_position);  

  /* Check if tux is moving backwards and if so fix his
   * scale matrix. */
  /* Concatenate the scale matrix with a rotation matrix */
  if (state->tux_direction < 0)
  {
    /* Put tux at origin */
    /* FIXME: don't hardcode this.  In fact read it out of the GXF. */
    TranslateMat (xlate, 0, -18.159, 46.312);
    MatMultMat4x4 (state->tux_orientation, state->tux_orientation,
		   xlate);
    RotateMat (rotate, DEG_TO_RAD (180), 0, 0, 1);
    MatMultMat4x4 (state->tux_orientation, state->tux_orientation,
		   rotate);
    /* Back to his spot */
    TranslateMat (xlate, 0, 18.159, -46.312);
    MatMultMat4x4 (state->tux_orientation, state->tux_orientation,
		   xlate);   

    /* Now make him walk forward */
    state->tux_direction = 1;
  }
  
  /* Set the walk velocity */
  SET_TUX_WALK_V(state);

  /* Start the walk state */
  state->tux_state = TUX_WALK;
  state->tux_frame = TUX_WALK_MIN_FRAME;
}

static void
start_tux_jump (Demo_State *state, Vector position)
{
  /* Set tux's state to none as a sentinel */
  state->tux_state = TUX_NONE;
  int i;

  /* This is a list of all windows tux could jump to.  If I don't look
   * at them all tux ends up jumping back and forth between the same
   * two windows and it gets boring. */
  int nice_spots [WINDOW_MAX];
  int num_nice_spots = 0;

  /* If tux is moving to the right then he needs to jump to a left edge */
  /* I guess he doesn't need to, but thats the simplified case I am
   * working with.  It would be cool to fix the jump code up to make
   * tux be able to jump to more windows.  Right now he won't jump to
   * a window that overlaps his current window in the x direction.
   * And he will only jump in the direction he is walking. */
  if (state->tux_velocity[0] > 0)  
  {
    /* Search for a window within jumping range */
    for (i = 0; i < state->n_wstates &&
	   state->tux_state != TUX_JUMP_START; i++)
    {
      /* Only jump forward and to decorated windows */
      if (state->wstates[i]->world_upper_left[0] > 
	  position[0] &&
	  state->wstates[i]->decorated)
      {
	/* Store this window as a possible nice spot */
	nice_spots[num_nice_spots++] = i;
      }
    }

    /* Pick one if the nice spots at random */
    if (num_nice_spots > 0)
    {
      /* Get an index at random */
      /* Use floats to prevent rollover.  This may seem unecessary but
       * it happened.  */
      i = nice_spots[(int)((float)rand () * num_nice_spots / RAND_MAX)];

      /* Next to the edge */
      state->tux_destination[0]=3;
      state->tux_destination[1]=0;
      state->tux_destination[2]=TUX_POSITION_TWEAK_Z;
      
      /* Remember this window */
      state->tux_destination_window = state->wstates[i];
    
      /* Put tux into the start jump state */
      state->tux_state = TUX_JUMP_START;
    }
  }
  else /* If tux is moving to the left then jump to a right edge */
  {
    /* Search for a window within jumping range */
    for (i = 0; i < state->n_wstates &&
	   state->tux_state != TUX_JUMP_START; i++)
    {
      if ( state->wstates[i]->world_lower_right[0] < 
	   position[0] &&
	   state->wstates[i]->decorated)
      {
	/* Store this window as a possible nice spot */
	nice_spots[num_nice_spots++] = i;
      }
    }

    if (num_nice_spots)
    {
      /* Get an index at random */
      /* Use floats to prevent rollover */
      i = nice_spots[(int)((float)rand () * num_nice_spots / RAND_MAX)];

      state->tux_destination[0]=
	state->wstates[i]->world_lower_right[0] -
	state->wstates[i]->world_upper_left[0] - 3;
      state->tux_destination[1]=0;
      state->tux_destination[2]=TUX_POSITION_TWEAK_Z;
      
      /* Remember this window */
      state->tux_destination_window = state->wstates[i];
      
      /* Put tux into the start jump state */
      state->tux_state = TUX_JUMP_START;
    }
  }

  /* Did I find one */
  if (state->tux_state == TUX_JUMP_START)
  {
    /* The starting frame */
    state->tux_frame = TUX_JUMP_START_MIN_FRAME;

    /* Tux now switches to the destination window.  This way if the
     * current window is moved tux doesn't care.  If the destination
     * window is moved tux goes with it.  If the destination window
     * closes then tux goes to his window closed state and ends up
     * picking a new window */

    /* First I need to make tux's position relative to the new window */
    state->tux_position[0] += state->tux_window->world_upper_left[0] -
      state->tux_destination_window->world_upper_left[0];
    state->tux_position[1] += state->tux_window->world_upper_left[1] -
      state->tux_destination_window->world_upper_left[1];
    state->tux_position[2] += state->tux_window->world_upper_left[2] -
      state->tux_destination_window->world_upper_left[2];

    /* Now set the right window */
    state->tux_window = state->tux_destination_window;

    
    /* Jump start happens in place */
    state->tux_velocity[0] = 0;
    state->tux_velocity[1] = 0;
    state->tux_velocity[2] = 0;

    state->tux_acceleration[0] = 0;
    state->tux_acceleration[1] = 0;
    state->tux_acceleration[2] = 0;
  }
  else
  {
    /* Put tux into the turn around animation */
    state->tux_state = TUX_TURN;
    state->tux_frame = TUX_TURN_MIN_FRAME;

    state->tux_velocity[0] = 0;
    state->tux_velocity[1] = 0;
    state->tux_velocity[2] = 0;

    state->tux_acceleration[0] = 0;
    state->tux_acceleration[1] = 0;
    state->tux_acceleration[2] = 0;    
  }	
}

static void
update_tux_animation (Demo_State *state)
{
  float width;
  float walk_distance;
  Vector position;
  Matrix rotate, xlate;

  /* Lots of things need this so I do it here */
  /* Get tux's position in world coordinates */
  position[0] = state->tux_position[0] + 
    state->tux_window->world_upper_left[0];
  position[1] = state->tux_position[1] + 
    state->tux_window->world_upper_left[1];
  position[2] = state->tux_position[2] + 
    state->tux_window->world_upper_left[2];    

  /* We now look at which state just ENDED and decide what to do next */
  switch (state->tux_state)
  {
  case TUX_WALK:
    /* Check if there is enough room for another walk cycle */    
    width = state->tux_window->world_lower_right[0] - 
      state->tux_window->world_upper_left[0];

    /* If the frame rate is locked this is a very good estimate */
    walk_distance = state->tux_velocity[0] * TUX_WALK_FRAMES * state->dt;

    /* FIXME: try to unify the two conditionals below.  They are
     * almost identical except for the contions.  */
    /* Which way is tux walking and does he have enough room */
    if ( (state->tux_velocity[0] > 0 &&
	  state->tux_position[0] + walk_distance < width) )
    {
#ifndef DEBUG 
      /* If im debugging then this takes too much of my time 
	 waiting for tux to wave. */
      /* 25 / 75 chance to keep walking or sit down and take
       * a breather. */
      if (rand () & 0x3 == 0x3)
      {
	/* Rest */
	state->tux_frame = TUX_REST_MIN_FRAME;
	state->tux_state = TUX_SIT;
	state->tux_velocity[0] = 0;
	state->tux_velocity[1] = 0;
	state->tux_velocity[2] = 0;

	state->tux_acceleration[0] = 0;
	state->tux_acceleration[1] = 0;
	state->tux_acceleration[2] = 0;
      }
      else
      {
#endif
	/* Restart the walk cycle */
	state->tux_frame = TUX_WALK_MIN_FRAME;
#ifndef DEBUG
      }
#endif
    }
    /* Use + walk_distance since the calculation above calulcates
     * walk distance to be negative */
    else if (state->tux_velocity[0] < 0 &&
	  ((state->tux_position[0] + walk_distance) > 0))
    {
#ifndef DEBUG
      /* 25 / 75 chance to keep walking or sit down and take
       * a breather.  Tom Griffin, VPG's artist got creative.  Instead
       * tux dances around. */
      if (rand () & 0x3 == 0x3)
      {
	/* Rest */
	state->tux_frame = TUX_REST_MIN_FRAME;
	state->tux_state = TUX_SIT;
	state->tux_velocity[0] = 0;
	state->tux_velocity[1] = 0;
	state->tux_velocity[2] = 0;

	state->tux_acceleration[0] = 0;
	state->tux_acceleration[1] = 0;
	state->tux_acceleration[2] = 0;
      }
      else
      {
#endif
	/* Restart the walk cycle */
	state->tux_frame = TUX_WALK_MIN_FRAME;
#ifndef DEBUG
      }
#endif
    }
    else
    {
      start_tux_jump(state, position);
    }    
    break;
  case TUX_JUMP_START:
    /* Start the flight */
    state->tux_state = TUX_FLY;
    state->tux_frame = TUX_FLY_MIN_FRAME;
    state->tux_fly_frame = 0;

    /* Build a parabola to jump along */
    /* I have to assume the frame rate is constant.  This is to make a
     * very precise calculate of the base time for the flight cycle.
     * The actual frame time can be slowed down by moving windows.
     * This means that if windows move around tux will land his jump
     * wrong.  Since I don't want to reset tux's position when he
     * lands I have to make the landings very acurate.  The result is
     * that Tux's flight animation is controled by frame not time. */
    state->tux_total_fly_frames = build_parabola (state,
						  state->tux_position, 
						  state->tux_destination,
						  TUX_FLY_FRAMES / 
						  (float)FRAMERATE); 
    break;
  case TUX_FLY:
    if (--(state->tux_total_fly_frames) > 0)
    {
      /* If still going then reset the fly cycle */
      state->tux_frame = TUX_FLY_MIN_FRAME;
    }
    else
    {
      /* transition to the end of the jump frame */
      state->tux_state = TUX_JUMP_END;
      state->tux_frame = TUX_JUMP_END_MIN_FRAME;
      state->tux_velocity[0] = 0;
      state->tux_velocity[1] = 0;
      state->tux_velocity[2] = 0;
      
      state->tux_acceleration[0] = 0;
      state->tux_acceleration[1] = 0;
      state->tux_acceleration[2] = 0;      

#ifdef TRACE_PARABOLA
      printf ("Landed at (%f %f %f)\n", 
	      state->tux_position[0],
	      state->tux_position[1],
	      state->tux_position[2]);
#endif
    }
    break;
  case TUX_JUMP_END:
    /* The penguin has landed */
    /* Tux is now on a new window and needs to have his window 
     * relative coordinates adjusted */
    /* Start walking */
    state->tux_state = TUX_WALK;
    state->tux_frame = TUX_WALK_MIN_FRAME;

    SET_TUX_WALK_V(state);
    
    break;
  case TUX_TURN:
    /* If tux just finished turning then he should start walking
     * again but in the other direction */
    /* Start the walk animation in the other direction */
    state->tux_state = TUX_WALK;
    state->tux_frame = TUX_WALK_MIN_FRAME;
    state->tux_direction = -state->tux_direction;

    SET_TUX_WALK_V(state);

    /* Concatenate the scale matrix with a rotation matrix */
    TranslateMat (xlate, 0, -18.159, 46.312);
    RotateMat (rotate, DEG_TO_RAD (180), 0, 0, 1);
    MatMultMat4x4 (state->tux_orientation, state->tux_orientation,
		   xlate);
    MatMultMat4x4 (state->tux_orientation, state->tux_orientation,
		   rotate);
    /* Back to his spot */
    TranslateMat (xlate, 0, 18.159, -46.312);
    MatMultMat4x4 (state->tux_orientation, state->tux_orientation,
		   xlate);
    break;
  case TUX_SIT:
    /* Thats enough of a break, get back up and walk */
    state->tux_state = TUX_WALK;
    state->tux_frame = TUX_WALK_MIN_FRAME;
    SET_TUX_WALK_V (state);
    break;
  default:
    printf ("not handling tux state %d in tux_update_animation\n", 
	    state->tux_state);
    break;
  }
}

void
update_tux (Demo_State *state)
{
  /* Advance to the next animation frame */
  state->tux_frame++;

  if (state->tux_frame > tux_max_frames[state->tux_state])
    update_tux_animation (state);

  /* I must not be sampling frequently enough because if I use the
   * acceleration velocity model for the flight tux gets a bit lost.
   * So I special case the flight.  However the only animation that
   * requires acceleration is flight so maybe I'm just wasting my
   * time. */
  if (state->tux_state == TUX_FLY)
  {   
    /* Compute time manually from frame count */
    /* I do this to combat the effect of an uneven frame rate.  This
     * happens if windows are being moved around a lot or a background
     * process is being slow.  If I don't do this then tux's final
     * position could be a bit off and will never get fixed. */
    state->tux_fly_frame += 1;
    float t = (float)state->tux_fly_frame / (float)FRAMERATE;
      
    state->tux_position[0] = state->tux_b[0] * t +
      state->tux_a[0] * t * t + 
      state->tux_c[0];
    state->tux_position[1] = state->tux_b[1] * t + 
      state->tux_a[1] * t * t + 
      state->tux_c[1];
    state->tux_position[2] = state->tux_b[2] * t +
      state->tux_a[2] * t * t + 
      state->tux_c[2]; 
  }
  else
  {
    /* Update tux's velocity now.  */
    state->tux_velocity[0] += state->tux_acceleration[0] * state->dt;
    state->tux_velocity[1] += state->tux_acceleration[1] * state->dt;
    state->tux_velocity[2] += state->tux_acceleration[2] * state->dt;
    
    state->tux_position[0] += state->tux_velocity[0] * state->dt;
    state->tux_position[1] += state->tux_velocity[1] * state->dt;
    state->tux_position[2] += state->tux_velocity[2] * state->dt;
  }
}


