/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __MATHUTIL_H__
#define __MATHUTIL_H__

enum Axis
{
	X = 0,
	Y,
	Z,
	W
};

enum FPUprecision
{
	PRECISION_24_BIT = 0,
	PRECISION_53_BIT,
	PRECISION_64_BIT
};

typedef float Vector[4];
typedef float Matrix[4][4];
typedef float Vector3[3];

#define PI  (3.141594f)

#define ABS(a)  ((a) >= 0 ? (a) : (-(a)))
#define MIN(a, b)  ((a) < (b) ? (a) : (b))
#define MAX(a, b)  ((a) > (b) ? (a) : (b))
#define CLAMP(a, min, max)  ( ((a) < (min)) ? (min) : ( ((a) > (max)) ? (max) : (a) ) )

#define DEG_TO_RAD(deg) ((deg)*(PI/180.0f))
#define RAD_TO_DEG(rad) ((rad)*(180.0f/PI))

#define SQR(f)  ((f)*(f))

#define SET_VECTOR(v, x, y, z, w)  (v)[X] = (x); (v)[Y] = (y); (v)[Z] = (z); (v)[W] = (w);

#ifndef USE_GLIDE3
// glide2x needs 12.4 x, y coords
#define SNAP(f)  ((f) + (float)(3<<18))
#endif // !USE_GLIDE3

#ifdef USE_ASM
__declspec(naked) inline float fsqrt(float f)
{
	__asm
	{
		fld			dword ptr [esp + 4]
		fsqrt
		ret
	}
}

__declspec(naked) inline void fsincos(float angle, float *sin_angle, float *cos_angle)
{
	__asm
	{
		fld			dword ptr [esp + 4]
		fsincos
		mov			ecx, [esp + 8]
		mov			edx, [esp + 12]
		fstp		dword ptr [edx]
		fstp		dword ptr [ecx]
		ret
	}
}

#define COPY_DWORDS(dst, src, n) \
{ \
	unsigned long *__dst__, *__src__, __n__; \
	__dst__ = (unsigned long *)(dst); \
	__src__ = (unsigned long *)(src); \
	__n__ = (unsigned long)(n); \
	__asm push		edi \
	__asm push		esi \
	__asm push		ecx \
	__asm mov			edi, dword ptr [__dst__] \
	__asm mov			esi, dword ptr [__src__] \
	__asm mov			ecx, dword ptr [__n__] \
	__asm rep			movsd \
	__asm pop			ecx \
	__asm pop			esi \
	__asm pop			edi \
}
#else
#define fsqrt(f) ((float)sqrt(f))

#define fsincos(angle, sin_angle, cos_angle) do { *sin_angle = sin(angle); *cos_angle = cos(angle); } while (0)

#define COPY_DWORDS(dst, src, n)  memcpy (dst, src, n<<2);
#endif
#define UniformScaleMat(m, s)  ScaleMat((m), (s), (s), (s))

// function prototypes
void SetFPUprecision(FPUprecision precision);
void RestoreFPUprecision();
unsigned long PackARGB(float a, float r, float g, float b);
unsigned long PackARGBIntensity(float a, float i);
float fsqrt_inv(float f);
float DotProduct(const float *a, const float *b);
void CrossProduct(float *v, const float *v1, const float *v2);
float NormZ(const float *v0, const float *v1, const float *v2);
float Magnitude(const float *v);
float Distance(const float *v0, const float *v1);
void Normalize(float *v);
void FastApproxNormalize(float *v);
void RotateByXAxis(Vector res, Vector point, float angle);
void RotateByYAxis(Vector res, Vector point, float angle);
void RotateByZAxis(Vector res, Vector point, float angle);
void RotateByAxis(float *res, const float *point, float angle, const float *axis);
void TransposeMatrix(Matrix res, Matrix m);
void MatMultMat4x4(Matrix res, Matrix m1, Matrix m2);
void MatMultMat3x4(Matrix res, Matrix m1, Matrix m2);
void MatMultVec3x3_2(Vector res, Matrix m, Vector v);
void MatMultVec3x4_3(Vector res, Matrix m, Vector v);
void MatMultVec3x4_4(Vector res, Matrix m, Vector v);
void MatMultVec4x4_3(Vector res, Matrix m, Vector v);
void MatMultVec4x4_4(Vector res, Matrix m, Vector v);
void IdentityMat(Matrix m);
void ScaleMat(Matrix m, float sx, float sy, float sz);
void TranslateMat(Matrix m, float dx, float dy, float dz);
void RotateXMat(Matrix m, float angle);
void RotateYMat(Matrix m, float angle);
void RotateZMat(Matrix m, float angle);
void RotateMat(Matrix m, float angle, float x, float y, float z);
void ViewportMat(Matrix m, float min_x, float max_x, float min_y, float max_y, float min_z, float max_z);
void OrthoMat(Matrix m, float left_plane, float right_plane, float bottom_plane, float top_plane, float near_plane, float far_plane);
void FrustumMat(Matrix m, float left_plane, float right_plane, float bottom_plane, float top_plane, float near_plane, float far_plane);
void PerspectiveMat(Matrix m, float fovy, float aspect_ratio, float near_plane, float far_plane);
void ParallelMat(Matrix m, float fovy, float aspect_ratio, float near_plane, float far_plane);
void MatrixCopy(Matrix dst, Matrix src);
float Determinant3x4(Matrix m);
float Determinant4x4(Matrix m);
void AdjointMatrix3x4(Matrix adj_mat, Matrix mat);
void AdjointMatrix4x4(Matrix adj_mat, Matrix mat);
int InvertMatrix3x4Cramer(Matrix inv_mat, Matrix mat);
int InvertMatrix4x4Cramer(Matrix inv_mat, Matrix mat);
int InvertMatrix4x4CramerPrecise(Matrix inv_mat, Matrix mat);
int InvertMatrix3x4Gaussian(Matrix inv_mat, Matrix mat);
int InvertMatrix4x4Gaussian(Matrix inv_mat, Matrix mat);
int InvertMatrix4x4GaussianPrecise(Matrix inv_mat, Matrix mat);

#define InvertMatrix3x4  InvertMatrix3x4Gaussian
#define InvertMatrix4x4  InvertMatrix4x4Gaussian

#endif // __MATHUTIL_H__
