/*
   This file is part of the KDE libraries
   Copyright (c) 1999 Kurt Granroth <granroth@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include <kapp.h>
#include <qstring.h>
#include <qcstring.h>
#include <dcopclient.h>
#include <dcopobject.h>
#include <qmap.h>
#include <qdatetime.h>

/**
g++ -o testxmlrpc testxmlrpc.cpp -I$KDEDIR/include -I$QTDIR/include -L$KDEDIR/lib -L$QTDIR/lib -lkdecore -lqt
**/

class AddressServer : public DCOPObject
{
public:
    AddressServer() : DCOPObject("email")
    {
        m_addressMap.insert("Kurt Granroth", "granroth@kde.org");
        m_addressMap.insert("Preston Brown", "pbrown@kde.org");
        m_addressMap.insert("Matthias Ettrich", "ettrich@kde.org");
    }

    virtual bool process(const QCString &func, const QByteArray &data,
                         QCString &replyType, QByteArray &replyData)
    {
        qDebug("Processing: %s", func.data());
        if (func == "lookup(QString)")
        {
            QDataStream args(data, IO_ReadOnly);
            QString name;
            args >> name;
            qDebug("name = %s", name.data());

            QDataStream reply(replyData, IO_ReadWrite);
            reply << lookup(name);
            replyType = "QString";

            return true;
        }

        if (func == "addAddress(QString,QString)")
        {
            QDataStream args(data, IO_ReadOnly);
            QString name, address;
            args >> name >> address;

            addAddress(name, address);
            replyType = "void";

            return true;
        }

        if (func == "deleteAddress(QString)")
        {
            QDataStream args(data, IO_ReadOnly);
            QString name;
            args >> name;

            deleteAddress(name);
            replyType = "void";

            return true;
        }

        return false;

    }

private:
    QString lookup(const QString& _name)
    {
        qDebug("%s -> %s", _name.data(), m_addressMap[_name].data());
        return m_addressMap[_name];
    }

    void addAddress(const QString& _name, const QString& _address)
    {
        m_addressMap.insert(_name, _address);
    }

    void deleteAddress(const QString& _name)
    {
        m_addressMap.remove(_name);
    }

private:
    QMap<QString, QString> m_addressMap;
};

class TypeServer : public DCOPObject
{
public:
    TypeServer() : DCOPObject("types")
    {
    }

    virtual bool process(const QCString &func, const QByteArray &data,
                         QCString &replyType, QByteArray &replyData)
    {
        qDebug("Processing: %s", func.data());
        if (func == "fromBase64(QByteArray)")
        {
            QDataStream args(data, IO_ReadOnly);
            QByteArray array;
            args >> array;

            QDataStream reply(replyData, IO_ReadWrite);
            reply << QString(array.data());
            replyType = "QString";

            return true;
        }

        if (func == "toBase64(QString)")
        {
            QDataStream args(data, IO_ReadOnly);
            QString param;
            args >> param;

            QDataStream reply(replyData, IO_ReadWrite);
            reply << (QByteArray)QCString(param.data());
            replyType = "QByteArray";

            return true;
        }

        if (func == "currentTime()")
        {
            QDataStream reply(replyData, IO_ReadWrite);
            reply << QDateTime::currentDateTime();
            replyType = "QDateTime";

            return true;
        }

        if (func == "getTime(QDateTime)")
        {
            QDataStream args(data, IO_ReadOnly);
            QDateTime param;
            args >> param;

            QDataStream reply(replyData, IO_ReadWrite);
            reply << param.toString();
            replyType = "QString";

            return true;
        }

        if (func == "returnMyself(QMap<QString,int>)")
        {
            QDataStream args(data, IO_ReadOnly);
            QMap<QString, int> param;
            args >> param;

            QDataStream reply(replyData, IO_ReadWrite);
            reply << param;
            replyType = "QMap<QString,int>";

            return true;
        }


        return false;

    }
};

int main(int argc, char **argv)
{
    KApplication app(argc, argv, "testapp", false);

    // standard DCOP stuff
    DCOPClient *client = app.dcopClient();
    client->registerAs("xmlrpcDCOP");

    // start up my servers
    DCOPObject *address_server = new AddressServer;
    DCOPObject *type_server    = new TypeServer;

    int ret = app.exec();
    client->detach();
    return ret;
}
