/* Yo Emacs, this is -*- C++ -*- */
/*
 *   ksame 0.4 - simple Game
 *   Copyright (C) 1997,1998  Marcus Kreutzberger
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <kapp.h>
#include <qaccel.h>
#include <qwidget.h>
#include <qpushbutton.h>
#include <qpixmap.h>
#include <kiconloader.h>
#include "StoneWidget.h"
#include "version.h"
#include "KSameWidget.moc"
#include <qpopupmenu.h>
#include <kstatusbar.h>
#include <knuminput.h>
#include <klocale.h>

#include <kfiledialog.h>
#include <qmessagebox.h>
#include <kmenubar.h>
#include <kconfig.h>

static int default_colors=3;

KSameWidget::KSameWidget() {

  KMenuBar *menu=new KMenuBar(this);

  QPopupMenu *gamemenu=new QPopupMenu;
  gamemenu->setAccel(CTRL+Key_N,
		     gamemenu->insertItem(i18n("&New"),this, SLOT(m_new())));

  gamemenu->setAccel(CTRL+Key_L,
		     gamemenu->insertItem(i18n("&Open"),this, SLOT(m_load())));
  gamemenu->setAccel(CTRL+Key_S,
		     gamemenu->insertItem(i18n("&Save"),this, SLOT(m_save())));

  gamemenu->insertSeparator();
  gamemenu->setAccel(CTRL+Key_H,
		     gamemenu->insertItem(i18n("S&how Highscore"),this, SLOT(m_showhs())));
  gamemenu->insertSeparator();
  gamemenu->setAccel(CTRL+Key_Q,
		     gamemenu->insertItem(i18n("E&xit"),this, SLOT(m_quit())));

  menu->insertItem(i18n("&Game"),gamemenu);
  menu->insertSeparator();

  editmenu = new QPopupMenu;
  editmenu->setAccel(CTRL+Key_Z,
		     editmenu->insertItem(i18n("&Undo"),this,
					  SLOT(m_undo())));
  menu->insertItem(i18n("&Edit"),editmenu);
  menu->insertSeparator();


  optionsmenu = new QPopupMenu;
  optionsmenu->setCheckable(true);


  random_item = optionsmenu->insertItem(i18n("&Random Board"),this, SLOT(m_tglboard()));
  optionsmenu->setAccel(CTRL+Key_R,random_item);
  menu->insertItem(i18n("&Options"),optionsmenu);
  menu->insertSeparator();

  QString about = i18n("SameGame %1\n\n"
                       "by Marcus Kreutzberger (kreutzbe@informatik.mu-luebeck.de)")
                 .arg(KSAME_VERSION);
  QPopupMenu *help = helpMenu(about);
  menu->insertItem(i18n("&Help"), help);

  setMenu(menu);

  status=new KStatusBar(this);
  status->insertItem(i18n("Colors: XX"),1);
  status->insertItem(i18n("Board: XXXXXX"),2);
  status->insertItem(i18n("Marked: XXXXXX"),3);
  status->insertItem(i18n("Score: XXXXXX"),4);
  // The new KStatusBar doesn't support this;
  // commenting it out for now.  status->setAlignment(4,AlignRight);
  setStatusBar(status);

  stone = new StoneWidget(this,15,10);

  connect( stone, SIGNAL(s_gameover()), this, SLOT(gameover()));

  connect( stone, SIGNAL(s_colors(int)), this, SLOT(setColors(int)));
  connect( stone, SIGNAL(s_board(int)), this, SLOT(setBoard(int)));
  connect( stone, SIGNAL(s_marked(int)), this, SLOT(setMarked(int)));
  connect( stone, SIGNAL(s_score(int)), this, SLOT(setScore(int)));

  connect(stone, SIGNAL(s_sizechanged()), this, SLOT(sizeChanged()));

  setView(stone, false);
  stone->show();

  highscore = new HighScore();


  israndom=1;
  optionsmenu->setItemChecked(random_item, israndom);

  sizeChanged();
  if (!kapp->isRestored()) newGame(kapp->random(),default_colors);
  debug("test");
}

KSameWidget::~KSameWidget() {
  // delete stone;
  delete highscore;
  //  debug("~KSameWidget\n");
}

void KSameWidget::readProperties(KConfig *conf) {
  ASSERT(conf);
  stone->readProperties(conf);
}

void KSameWidget::saveProperties(KConfig *conf) {
  ASSERT(conf);
  stone->saveProperties(conf);
  conf->sync();
}

void KSameWidget::sizeChanged() {
  setFixedSize(stone->sizeHint());
  updateRects();
}

void
KSameWidget::newGame(unsigned int board,int colors) {
  while (board>=1000000) board-=1000000;
  // debug("newgame board %i colors %i",board,colors);
  stone->newGame(board,colors);
  //     setBoard(stone->getBoard());
  //     setColors(stone->getColors());
}

void KSameWidget::m_new() {
  if (israndom) {
    if (stone->isGameover()||
        (QMessageBox::information(this, kapp->caption() + " - " + i18n("New Game"),
				  i18n("Do you want to resign?"),
				  i18n("Yes"), i18n("No"))==0)) {
      newGame(kapp->random(),default_colors);
    }
	
  } else {
    QDialog dlg(0,i18n("Board"),1);
    int w=10,h=10;
	
    KIntNumInput bno(0, this, 10);
    bno.setRange(0, 1000000, 1);
    bno.setLabel(i18n("Select a board"));
    bno.setFocus();
//    QLabel l(&bno,i18n("Please &enter the board number:"),&dlg);
	
    bno.move(10,h);
    bno.resize(bno.sizeHint());
    w=bno.width()+20;
    h+=bno.height()+10;
//    bno.move(10,h);
//    w=QMAX(bno.width()+20,w);
//    h+=bno.height()+10;

    QPushButton ok(i18n("OK"), &dlg);
    ok.setDefault(true);
    QPushButton cancel(i18n("Cancel"), &dlg);
	
    w=QMAX(ok.width()+cancel.width()+40,w);
	
    ok.move(w-ok.width()-20-cancel.width(),h);
    cancel.move(w-cancel.width()-10,h);
    h+=QMAX(ok.height(),cancel.height())+10;
	
    dlg.resize(w,h);
    dlg.setFixedSize(w,h);
    dlg.connect(&ok, SIGNAL(clicked()), SLOT(accept()) );
    dlg.connect(&cancel, SIGNAL(clicked()), SLOT(reject()) );
    dlg.connect(&bno, SIGNAL(returnPressed()), SLOT(accept()) );
    bno.setValue(stone->board());
//    bno.selectAll();
    if (dlg.exec()) newGame(bno.value(),default_colors);
  }
}

void KSameWidget::m_load() {
  debug("menu load not supported");
}

void KSameWidget::m_save() {
  debug("menu save not supported");
}

void KSameWidget::m_undo() {
  ASSERT(stone);
  stone->undo();
}


void KSameWidget::m_showhs() {
  ASSERT(stone);
  stone->unmark();
  highscore->showScore();
}
void KSameWidget::m_quit() {
  ASSERT(stone);
  stone->unmark();
  if (QMessageBox::information(this, i18n("SameGame - Exit"),
			       i18n("Do you really want to exit?"),
			       i18n("Yes"), i18n("No"))==0) {
    kapp->quit();
  }
}

void KSameWidget::m_tglboard() {
  israndom=!israndom;
  optionsmenu->setItemChecked(random_item, israndom);
}


void KSameWidget::setColors(int colors) {
  QString s;
  s.sprintf(i18n("Colors: %01i"),colors);
  status->changeItem(s,1);
}

void KSameWidget::setBoard(int board) {
  QString s;
  s.sprintf(i18n("Board: %06i"),board);
  status->changeItem(s,2);
}

void KSameWidget::setMarked(int m) {
  QString s;
  s.sprintf(i18n("Marked: %06i"),m);
  status->changeItem(s,3);
}

void KSameWidget::setScore(int score) {
  QString s;
  s.sprintf(i18n("Score: %06i"),score);
  status->changeItem(s,4);
}

void KSameWidget::gameover() {
  debug("GameOver");
  stone->unmark();
  highscore->add(stone->board(),stone->score(),stone->colors());
}
