/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <stdlib.h>
#include <stdio.h>

#include <kapp.h>
#include <qtextstream.h>

#include "interrupt.h"
#include "ioaddress.h"
#include "pccarddevice.h"


QString runCardCtl(QString cmd, int socket)
{
  QString result, command;
  
  command.sprintf("/sbin/cardctl %s %d", cmd.data(), socket);

  FILE *pfd = popen(command.data(), "r");
  
  if (pfd >= 0)
    {
      QTextStream is(pfd, IO_ReadOnly);
      
      while (!is.eof())
	result += is.readLine() + "\n";
      
      pclose(pfd);
    }

  return result;
}


PCCardDevice::PCCardDevice(QString ident, int socket)
  : Device(Device::BaseClassPCCard, ident)
{
  _ident = ident;

  _function = -1;
  _socket = socket;
  _vendorname = i18n("Unknown vendor");
  _devicename = i18n("Unknown device");

  lookupData();
}    


QString  PCCardDevice::deviceClassName()
{
  switch (_function) 
    {
    case -1: return QString().setNum(_socket);
      case 0: return QString(i18n("Multifunction card"));
      case 1: return QString(i18n("Memory card"));
      case 2: return QString(i18n("Serial/Modem card"));
      case 3: return QString(i18n("Parallel card"));
      case 4: return QString(i18n("Fixed disk"));
      case 5: return QString(i18n("Video card"));
      case 6: return QString(i18n("Network adapter"));
      case 7: return QString(i18n("AIMS card"));
      case 8: return QString(i18n("SCSI adapter"));
    default:
      return QString(i18n("unknown card type"));
    }
};


QString  PCCardDevice::deviceName()
{
  if (_function == -1)
    return QString(i18n("Empty socket"));
  return _devicename;
}


Device::devstate  PCCardDevice::deviceState()
{
  QString state = runCardCtl("status", _socket);

  if (state.find("  no card") >= 0)
    return DeviceStateEmpty;
  if (state.find("card present"))
    return DeviceStateReady;
  return DeviceStateUnknown;
}


QString  PCCardDevice::vendorName()
{
  return _vendorname;
}


/*
 * Private methods
 ******************/
 
void  PCCardDevice::lookupData()
{
  bool ok;
  int pos, pos2;
  unsigned long value, value2;

  // parse the ident information
  if ((pos = _ident.find("  function:")) >= 0)
    {
      _function = _ident.mid(pos+12,2).stripWhiteSpace().toInt(&ok);
      if (!ok)
      	_function = 999;
    }
  if ((pos = _ident.find("  product info:")) >= 0)
    {
      pos = _ident.find("\"");
      pos2 = _ident.find("\"", pos+1);
      if (pos2 > pos)
	{
	  _vendorname = _ident.mid(pos+1,pos2-pos-1);
	  pos = _ident.find("\"", pos2+1);
	  pos2 = _ident.find("\"", pos+1);
	  if (pos2 > pos)
	    _devicename = _ident.mid(pos+1,pos2-pos-1);
	}
    }

  // get the configuration information
  QString config = runCardCtl("config", _socket);

  // parse config information
  if (config.find("not configured") < 0)
    {
      // interrupt      
      if ((pos = config.find("  IRQ ")) >= 0)
	{
	  value = config.mid(pos+6,2).stripWhiteSpace().toULong(&ok);
	  if (ok)
	    addResource(new Interrupt(value));
	}

      // io ports
      pos = 0;
      while ((pos = config.find("    I/O window", pos)) >= 0)
	{
	  if ((pos2 = config.find(": ", pos)) >= 0)
	    {
	      pos = pos2+2;
	      QString tmp = config.mid(pos, 10);
	      value = strtol(tmp.data(), NULL, 16);

	      if ((pos2 = config.find(" to ", pos)) >= 0)
		{
		  pos = pos2+4;
		  QString tmp = config.mid(pos, 10);
		  value2 = strtol(tmp.data(), NULL, 16);
		  addResource(new IOAddress(value, value2-value+1));
		}
	    }
	}

    }
}


