/* 
 *  Copyright (C) 1999-2000 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the "Artistic License" which comes with this Kit. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Artistic License for more details.
 *
 * You should have received a copy of the Artistic License with 
 * this Kit, in the file named "Artistic".  If not, I'll be glad
 * to provide one.
 */


#include <qlabel.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qlayout.h>

#include <kapp.h>
#include <kbuttonbox.h>
#include <klocale.h>
#include <kstddirs.h>

#include "ftpresults.h"
#include "ftp.h"


FtpTopLevel::FtpTopLevel(QWidget *parent, const char *name)
    : QWidget(parent, name)
{
    setAcceptDrops(true);

    man = new FtpSiteManager();
    
    QBoxLayout *layout = new QVBoxLayout(this, 10, 4);
    
    QGridLayout *grid = new QGridLayout(3, 4, 4);
    layout->addLayout(grid, 1);
    grid->setColStretch(0, 0);
    grid->setColStretch(1, 10);
    grid->setColStretch(2, 0);
    grid->addColSpacing(2, 10);
    grid->addRowSpacing(1, 15);
    grid->setRowStretch(1, 0);
    
    QLabel *wordslabel = new QLabel(i18n("What:"), this);
    grid->addWidget(wordslabel, 0, 0);
    
    wordsedit = new QLineEdit(this);
    wordsedit->setFocus();
    grid->addWidget(wordsedit, 0, 1);
    
    KButtonBox *buttonbox = new KButtonBox(this, Vertical);
    searchbutton = buttonbox->addButton(i18n("&Search"));
    searchbutton->setDefault(true);
    connect( searchbutton, SIGNAL(clicked()), SLOT(slotSearch()) );
    cancelbutton = buttonbox->addButton(i18n("Cancel"));
    connect( cancelbutton, SIGNAL(clicked()), SLOT(slotCancel()) );
    cancelbutton->setEnabled(false);
    buttonbox->layout();
    grid->addMultiCellWidget(buttonbox, 0, 2, 3, 3);
    
    layout->addSpacing(10);
    
    foundlabel = new QLabel(i18n("Found:"), this);
    layout->addWidget(foundlabel);
    
    resultbox = new FtpResultView(this);
    layout->addWidget(resultbox, 10);
    
    connect( resultbox, SIGNAL(urlActivated(const QString&)),
             this, SLOT(urlActivated(const QString&)) );
}


FtpTopLevel::~FtpTopLevel()
{}


void FtpTopLevel::updateButtons()
{
    bool hasjobs = !jobs.isEmpty();
    searchbutton->setEnabled(!hasjobs);
    cancelbutton->setEnabled(hasjobs);
}


void FtpTopLevel::slotSearch()
{
    int n;
    QString searchtext = wordsedit->text();
    while ( (n = searchtext.find('\n')) != -1) 
        searchtext.remove(n, 1);
    
    FtpSite *site = man->find("Lycos");
    if (!site)
        {
            QMessageBox::warning(this, "Columbo", "Could not find site definition file");
            return;
        }
    
    FtpSearch *job = new FtpSearch(site, searchtext);
    jobs.append(job);
    connect( job, SIGNAL(foundItem(const QString&,const QString&,const QString&,const QString&,const QString&)),
             this, SLOT(processItem(const QString&,const QString&,const QString&,const QString&,const QString&)) );
    connect( job, SIGNAL(result(KIO::Job*) ),
             this, SLOT(slotResult(KIO::Job*)) );

    updateButtons();
}


void FtpTopLevel::slotCancel()
{
    QListIterator<FtpSearch> it(jobs);
    for (; it.current(); ++it)
        it.current()->kill();
}


void FtpTopLevel::slotResult(KIO::Job *job)
{
    if (job->error())
        job->showErrorDialog(this);
    jobs.remove(static_cast<FtpSearch*>(job));
    
    updateButtons();
}

    
void FtpTopLevel::processItem(const QString &, const QString &host, const QString &path,
                              const QString &size, const QString &date)
{
    resultbox->insertItem(host, path, size, date);
    foundlabel->setText(i18n("Found %1 matches").arg(resultbox->childCount()));
}


void FtpTopLevel::urlActivated(const QString &url)
{
    qDebug( "Url activated %s", url.ascii() );
    kapp->invokeBrowser(url);  
}
