/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2000  Andrew Clausen, Lennert Buytenhek and Red Hat Inc.

	Andrew Clausen			<clausen@gnu.org>
	Lennert Buytenhek		<buytenh@gnu.org>
	Matt Wilson, Red Hat Inc.	<msw@redhat.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>
#include <string.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) gettext (String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#define	LOOP_SIGNATURE		"GNU Parted Loopback 0"

void ped_disk_loop_init ();
void ped_disk_loop_done ();

static int loop_probe (const PedDevice *dev);
static PedDisk* loop_open (PedDevice* dev);
static PedDisk* loop_create (PedDevice* dev);
static int loop_close (PedDisk* disk);
static int loop_read (PedDisk* disk);
static int loop_write (PedDisk* disk);
static int loop_align_partition (PedDisk* disk, PedPartition* part);
static int loop_enumerate_partition (PedDisk* disk, PedPartition* part);
static int loop_alloc_metadata (PedDisk* disk);
static int loop_get_extended_system ();

static PedDiskOps loop_disk_ops = {
	probe:			loop_probe,
	open:			loop_open,
	create:			loop_create,
	close:			loop_close,
	read:			loop_read,
	write:			loop_write,
	
	align_partition:	loop_align_partition,
	enumerate_partition:	loop_enumerate_partition,
	get_extended_system:	loop_get_extended_system,
	alloc_metadata:		loop_alloc_metadata
};

static PedDiskType loop_disk_type = {
	next:	NULL,
	name:	"loop",
	ops:	&loop_disk_ops
};

void
ped_disk_loop_init ()
{
	ped_register_disk_type (&loop_disk_type);
}

void
ped_disk_loop_done ()
{
	ped_unregister_disk_type (&loop_disk_type);
}

static int
loop_probe (const PedDevice * dev)
{
	PedDisk*	disk;

	disk = loop_open ((PedDevice*) dev);
	if (!disk)
		return 0;

	loop_close (disk);
	return 1;
}

static PedDisk*
loop_open (PedDevice* dev)
{
	char			buf [512];
	PedDisk*		disk;

	PED_ASSERT (dev != NULL, return 0);

	disk = (PedDisk*) ped_malloc (sizeof (PedDisk));
	if (!disk)
		goto fail;

	disk->dev = dev;
	disk->type = &loop_disk_type;
	disk->part_list = ped_partition_new (
		disk, PED_PARTITION_FREESPACE, NULL,
		0, dev->length);
	if (!disk->part_list)
		goto fail;

	if (dev->length < 256)
		goto fail_free_part_list;
	if (!ped_device_open ((PedDevice*) dev))
		goto fail_free_part_list;
	if (!loop_read (disk))
		goto fail_close_dev;

	return disk;

fail_close_dev:
	ped_device_close ((PedDevice*) dev);
fail_free_part_list:
	ped_partition_destroy (disk->part_list);
fail_free_disk:
	ped_free (disk);
fail:
	return NULL;
}

static PedDisk*
loop_create (PedDevice* dev)
{
	char		buf [512];

	PED_ASSERT (dev != NULL, return 0);

	if (dev->length < 256)
		goto error;
	if (!ped_device_open ((PedDevice*) dev))
		goto error;

	memset (buf, 0, 512);
	strcpy (buf, LOOP_SIGNATURE);

	ped_device_write (dev, buf, 0, 1);
	ped_device_close (dev);

	return ped_disk_open (dev);

error:
	return NULL;
}

static int
loop_close (PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);

	ped_disk_delete_all (disk);
	ped_device_close (disk->dev);
	ped_free (disk);

	return 1;
}

static int
loop_read (PedDisk* disk)
{
	char			buf [512];
	PedGeometry*		geom;
	PedFileSystemType*	fs_type;
	PedPartition*		part;

	PED_ASSERT (disk != NULL, return 0);
	ped_disk_delete_all (disk);

	if (!ped_device_read (disk->dev, &buf, 0, 1))
		goto error;
	if (!strncmp (buf, LOOP_SIGNATURE, strlen (LOOP_SIGNATURE)))
		return 1;

	geom = ped_geometry_new (disk, 0, disk->dev->length);
	if (!geom)
		goto error;

	fs_type = ped_file_system_probe (geom);
	if (!fs_type)
		goto error_free_geom;

	part = ped_partition_new (disk, PED_PARTITION_PRIMARY, fs_type,
				  geom->start, geom->end);
	part->fs_type = fs_type;
	ped_geometry_destroy (geom);
	if (!part)
		goto error;

	return ped_disk_add_partition (disk, part);

error_free_geom:
	ped_geometry_destroy (geom);
error:
	return 0;
}

static int
loop_write (PedDisk* disk)
{
	char		buf [512];

	if (disk->part_list)
		return 1;

	memset (buf, 0, 512);
	strcpy (buf, LOOP_SIGNATURE);

	return ped_device_write (disk->dev, buf, 0, 1);
}

static int
loop_align_partition (PedDisk* disk, PedPartition* part)
{
	return 1;
}

static int
loop_enumerate_partition (PedDisk* disk, PedPartition* part)
{
	PedPartition*		first;

	first = ped_disk_get_partition (disk, 1);

	if ((first && part != first) || part->type != PED_PARTITION_PRIMARY) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("Loopback disks can only have one primary partition."));
		return 0;
	}

	part->num = 1;

	return 1;
}

static int
loop_alloc_metadata (PedDisk* disk)
{
	return 1;
}

static int
loop_get_extended_system ()
{
	return 0;
}

