/*
    libparted
    Copyright (C) 1998-2000  Andrew Clausen  <clausen@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef FAT_H_INCLUDED
#define FAT_H_INCLUDED

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) gettext (String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

// #define DEBUG
// #define VERBOSE

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#define BUFFER_SIZE  1024	/* buffer size in sectors (512 bytes) */

typedef __s32			FatCluster;
typedef __s32			FatFragment;

enum _FatType {
	FAT_TYPE_FAT12,
	FAT_TYPE_FAT16,
	FAT_TYPE_FAT32
};
typedef enum _FatType		FatType;

typedef struct _FatSpecific	FatSpecific;
typedef struct _FatDirEntry	FatDirEntry;

/* FIXME: YUCKY */
#include "table.h"
#include "bootsector.h"
#include "context.h"
#include "fatio.h"
#include "traverse.h"
#include "calc.h"
#include "count.h"
#include "clstdup.h"

struct _FatDirEntry {
	__u8		name[8];
	__u8		extension[3];
	__u8		attributes;
	__u8		is_upper_case_name;
	__u8		creation_time_low;      /* milliseconds */
	__u16		creation_time_high;
	__u16		creation_date;
	__u16		access_date;
	__u16		first_cluster_high;     /* for FAT32 */
	__u16		time;
	__u16		date;
	__u16		first_cluster;
	__u32		length;
} __attribute__ ((packed));

struct _FatSpecific {
	FatBootSector	boot_sector;    /* structure of boot sector */
	FatInfoSector	info_sector;    /* fat32-only information sector */

	int		logical_sector_size;	/* illogical sector size :-) */
	PedSector	sector_count;

	int		cluster_size;
	PedSector	cluster_sectors;
	FatCluster	cluster_count;
	int		dir_entries_per_cluster;

	FatType		fat_type;
	int		fat_table_count;
	PedSector	fat_sectors;

	PedSector	info_sector_offset;     /* FAT32 only */
	PedSector	fat_offset;
	PedSector	root_dir_offset;	/* non-FAT32 */
	PedSector	cluster_offset;
	PedSector	boot_sector_backup_offset;

	FatCluster	root_cluster;           /* FAT32 only */
	int		root_dir_entry_count;   /* non-FAT32 */
	PedSector	root_dir_sector_count;  /* non-FAT32 */
	FatCluster	total_dir_clusters;

	FatTable*	fat;
	FatClusterInfo*	cluster_info;

	PedSector	buffer_sectors;
	char*		buffer;

	int		frag_size;
	PedSector	frag_sectors;
	FatFragment	frag_count;
	FatFragment	buffer_frags;
	FatFragment	cluster_frags;
};

#define FAT_SPECIFIC(fs)	((FatSpecific*) fs->type_specific)

#define FAT_ROOT		0

#define DELETED_FLAG		0xe5

#define READ_ONLY_ATTR		0x01
#define HIDDEN_ATTR		0x02
#define SYSTEM_ATTR		0x04
#define VOLUME_LABEL_ATTR	0x08
#define VFAT_ATTR		0x0f
#define DIRECTORY_ATTR		0x10

#define MAX_FAT12_CLUSTERS	4086      
#define MAX_FAT16_CLUSTERS	65526

#define FAT_ROOT_DIR_ENTRY_COUNT	512

extern int fat_probe (const PedGeometry* geom);
extern PedFileSystem* fat_open (const PedGeometry* geom);
extern PedFileSystem* fat_create (const PedGeometry* geom);
extern int fat_close (PedFileSystem* fs);
extern int fat_resize (PedFileSystem* fs, PedGeometry* geom);
extern int fat_copy (PedFileSystem* fs, PedGeometry* geom);
extern int fat_check (PedFileSystem* fs);
extern PedSector fat_get_min_size (const PedFileSystem* fs);
extern char fat_get_system (const PedFileSystem* fs, const PedPartition* part,
			    PedDiskType* disk_type);
extern void fat_print (const PedFileSystem* fs);

extern PedFileSystem* fat_alloc (const PedGeometry* geom);
extern void fat_free (PedFileSystem* fs);
extern int fat_alloc_buffers (PedFileSystem* fs);
extern void fat_free_buffers (PedFileSystem* fs);

extern int fat_set_frag_sectors (PedFileSystem* fs, PedSector frag_sectors);

#define MIN(a,b)	((a<b)? a : b)
#define MAX(a,b)	(a>b) ? a : b;

inline static PedSector
div_round_up (PedSector a, PedSector b)
{
	PedSector	result;

	result = a / b;
	if (a % b) result++;

	return result;
}

#endif /* FAT_H_INCLUDED */
