/* main.c - Main program for gnome-sync.

   Copyright (C) 1998 Tom Tromey

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

#include <config.h>

#include <assert.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>

#include "sync.h"

static void about (GtkWidget *widget, gpointer ignore);
static void sync_callback (GtkWidget *widget, gpointer client_data);
static void quit (GtkWidget *widget, gpointer client_data);
static void new_entry_callback (GtkWidget *widget, gpointer client_data);
static void remove_callback (GtkWidget *widget, gpointer client_data);
static void config_callback (GtkWidget *widget, gpointer client_data);



/* Nonzero if user requested immediate synchronization.  */
static int just_sync = 0;

/* Nonzero if we should quit before calling gtk_main().  */
static int quit_flag = 0;

/* Number of allowable concurrent transfers.  */
int concurrent_transfers;

/* Size of per-file memory cache in kilobytes.  */
int max_file_size;



/*
 * This section deals with the menubar and simple helper functions.
 */

/* The File menu.  */
static GnomeUIInfo file_menu[] =
{
  { GNOME_APP_UI_ITEM, N_("_Synchronize"), NULL, sync_callback, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  GNOMEUIINFO_MENU_PREFERENCES_ITEM(config_callback, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_EXIT_ITEM(quit, NULL),
  GNOMEUIINFO_END
};

/* The Synchronize menu.  */
static GnomeUIInfo sync_menu[] =
{
  { GNOME_APP_UI_ITEM, N_("New entry..."), NULL, new_entry_callback,
    NULL, NULL, GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Remove"), NULL, remove_callback, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Retry"), NULL, /*FIXME*/ NULL, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  GNOMEUIINFO_END
};

/* The Help menu.  */
static GnomeUIInfo help_menu[] =
{
  { GNOME_APP_UI_ITEM, N_("About..."), NULL, about, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT, 0, 0, NULL },
  GNOMEUIINFO_END
};

/* The menubar.  */
static GnomeUIInfo menubar[] =
{
  GNOMEUIINFO_MENU_FILE_TREE(file_menu),
  GNOMEUIINFO_SUBTREE (N_("Synchronize"), &sync_menu),
  GNOMEUIINFO_MENU_HELP_TREE(help_menu),
  GNOMEUIINFO_END
};

static GtkWidget *
find_menu_widget (const GnomeUIInfo *uiinfo, const char *label)
{
  int i;

  for (i = 0; uiinfo[i].type != GNOME_APP_UI_ENDOFINFO; ++i)
    {
      if (uiinfo[i].label && ! strcmp (uiinfo[i].label, label))
	return uiinfo[i].widget;
    }
  return NULL;
}

/* Pop up about box for application.  */
static void
about (GtkWidget *widget, gpointer ignore)
{
  GtkWidget *about;
  gchar *authors[] =
  {
    "Tom Tromey <tromey@cygnus.com>",
    NULL
  };

  about = gnome_about_new (_("Gnome Synchronize"), VERSION,
			   "(C) 1998 the Free Software Foundation",
			   (const gchar **) authors,
			   _("The GNOME file synchronization program."),
			   NULL);
  gtk_widget_show (about);
}

/* Callback function to quit.  */
static void
quit (GtkWidget *widget, gpointer client_data)
{
  force_quit ((struct database *) client_data);
}

/* This is called to start synchronization.  */
static void
sync_callback (GtkWidget *widget, gpointer client_data)
{
  synchronize ((struct database *) client_data);
}

/* Let user create a new entry.  */
static void
new_entry_callback (GtkWidget *widget, gpointer client_data)
{
  new_entry ((struct database *) client_data);
}

/* Configure the properties.  */
static void
config_callback (GtkWidget *widget, gpointer client_data)
{
  show_properties ();
}

/* Remove selected entries.  */
static void
remove_callback (GtkWidget *widget, gpointer client_data)
{
  struct database *database = (struct database *) client_data;

  gtk_clist_freeze (database->list);

  while (database->list->selection != NULL)
    {
      gchar *text;
      gint row = GPOINTER_TO_INT (database->list->selection->data);

      gtk_clist_get_text (database->list, row, 0, &text);
      /* FIXME: error handling.  */
      remove_entry (database->database, text);

      /* This modifies database->list->selection.  */
      gtk_clist_remove (database->list, row);
    }

  gtk_clist_thaw (database->list);
}

/* This is called when a row is selected.  */
static void
row_selected (GtkCList *list, gint row, gint column, GdkEvent *event,
	      gpointer client_data)
{
  struct database *database = (struct database *) client_data;
  gchar *text;

  if (event && event->type == GDK_2BUTTON_PRESS)
    {
      gtk_clist_get_text (list, row, 0, &text);
      edit_entry (database, text);
    }
  gtk_widget_set_sensitive (find_menu_widget (sync_menu, _("Remove")), 1);
}

/* This is called when a row is unselected.  */
static void
row_unselected (GtkCList *list, gint row, gint column, GdkEvent *event,
		gpointer client_data)
{
  if (list->selection == NULL)
    gtk_widget_set_sensitive (find_menu_widget (sync_menu, _("Remove")), 0);
}

/* This is called when a clist column is clicked on.  */
static void
column_clicked (GtkCList *list, gint column, gpointer client_data)
{
  gtk_clist_set_sort_column (list, column);
  gtk_clist_sort (list);
}

/* Create contents that go in main window.  */
static GtkCList *
setup_contents (GnomeApp *app, struct database *db)
{
  GtkCList *list;
  GtkWidget *sw;
  gchar *titles[3];

  titles[0] = _("Local");
  titles[1] = _("Remote");
  titles[2] = _("Status");
  sw = gtk_scrolled_window_new (NULL, NULL);
  list = GTK_CLIST (gtk_clist_new_with_titles (3, titles));
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw),
					 GTK_WIDGET (list));
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  gtk_clist_column_titles_show (list);
  /* FIXME: gtkclist doesn't support GTK_SELECTION_EXTENDED.  */
  gtk_clist_set_selection_mode (list, GTK_SELECTION_MULTIPLE);
  gtk_clist_set_sort_type (list, GTK_SORT_ASCENDING);
  gtk_clist_set_auto_sort (list, TRUE);
  gtk_clist_set_sort_column (list, 0);

  gtk_signal_connect (GTK_OBJECT (list), "select_row",
		      GTK_SIGNAL_FUNC (row_selected), (gpointer) db);
  gtk_signal_connect (GTK_OBJECT (list), "unselect_row",
		      GTK_SIGNAL_FUNC (row_unselected), (gpointer) db);
  gtk_signal_connect (GTK_OBJECT (list), "click_column",
		      GTK_SIGNAL_FUNC (column_clicked), NULL);

  /* FIXME: this is a hack.  */
  gtk_clist_set_column_width (list, 0, 75);
  gtk_clist_set_column_width (list, 1, 75);
  gtk_clist_set_column_width (list, 2, 75);

  database_fill_list (db, list);

  gtk_widget_show_all (GTK_WIDGET (sw));
  gnome_app_set_contents (app, GTK_WIDGET (sw));

  /* FIXME: another hack.  */
  gtk_widget_set_usize (GTK_WIDGET (app), 275, 100);

  return list;
}

/* Add menubar to the application.  */
static void
setup_menu (GnomeApp *app, struct database *database)
{
  gnome_app_create_menus_with_data (app, menubar, (gpointer) database);
  gtk_widget_set_sensitive (find_menu_widget (sync_menu,  _("Remove")), 0);
  gtk_widget_set_sensitive (find_menu_widget (sync_menu, _("Retry")), 0);
}

/* Quit the application.  */
void
force_quit (struct database *db)
{
  db->database->close (db->database);
  gtk_main_quit ();
  /* If this is called before gtk_main(), then we want to
     short-circuit it.  */
  quit_flag = 1;
}

/* Create a new synchronize application.  */
void
new_sync_app (struct database *database)
{
  GnomeApp *app
    = GNOME_APP (gnome_app_new ("Synchronize", _("Gnome Synchronize")));

  setup_menu (app, database);

  assert (! database->top);
  database->top = GTK_WIDGET (app);

  assert (! database->list);
  database->list = setup_contents (app, database);

  gtk_widget_show (GTK_WIDGET (app));
}



/*
 * Session management.
 */

static int
session_save (GnomeClient *client, int phase, GnomeSaveStyle save_style,
	      int is_shutdown, GnomeInteractStyle interact_style,
	      int is_fast, gpointer client_data)
{
  char *args[1];

  args[0] = (char *) client_data;
  gnome_client_set_restart_command (client, 1, args);

  /* FIXME: complete this.  */

  return TRUE;
}

static void
session_die (GnomeClient *client, gpointer client_data)
{
  force_quit ((struct database *) client_data);
}

static const struct poptOption options[] =
{
  { "resync", '\0', POPT_ARG_NONE, &just_sync, 0,
   N_("Immediately synchronize with remote"), NULL },
  { NULL, '\0', 0, NULL, 0 }
};

int
main (int argc, char *argv[])
{
  GnomeClient *client;
  struct database *database;

  bindtextdomain (PACKAGE, GNOMELOCALEDIR);
  textdomain (PACKAGE);

  gnome_init_with_popt_table ("Synchronize", VERSION, argc, argv,
			      options, 0, NULL);

  database = (struct database *) malloc (sizeof (struct database));
  /* FIXME: filename.  */
  database->filename = gnome_util_home_file ("sync.db");
  database->database = database_open (database->filename);
  database->list = NULL;
  database->top = NULL;
  database->flags = 0;
  database->ios = NULL;

  client = gnome_master_client ();
  if (client)
    {
      gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
			  GTK_SIGNAL_FUNC (session_save), argv[0]);
      gtk_signal_connect (GTK_OBJECT (client), "die",
			  GTK_SIGNAL_FUNC (session_die), (gpointer) database);
    }

  max_file_size = gnome_config_get_int (KEY_FILE_SIZE "=64");
  concurrent_transfers = gnome_config_get_int (KEY_TRANSFERS "=1");

  if (just_sync)
    {
      database->flags |= FLAG_DB_SYNC;
      synchronize (database);
    }
  else
    new_sync_app (database);

  if (! quit_flag)
    gtk_main ();

  return 0;
}
