/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-1999 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *                                                                              
 * As a special exception, you have permission to link this program             
 * with the Qt library and distribute executables, as long as you               
 * follow the requirements of the GNU GPL in regard to all of the               
 * software in the executable aside from Qt.                                    
 */


#include "bin.h"
#include "channel.h"
#include "main.h"
#include "toolbox.h"
#include "global.h"
#include "dostime.h"
#include "output.h"
#include "checksums.h"
#include "filetransfer.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

//#include <kwm.h>
#include <kwin.h>

extern TopLevel *toplevel;


/************************************************************************/
/**                      7plus-Autosave                                **/
/************************************************************************/

auto7plus::auto7plus( QWidget *rxchan )
{
   chan = rxchan;
   rxfile = false;              // Noch wird kein File empfangen
   transInfo = NULL;
   textfile = false;
}


auto7plus::~auto7plus()
{
   if (transInfo != NULL)
   {
      ((Channel *)chan)->withoutTransInfo();
      delete transInfo;
      transInfo = NULL;
   }
   toplevel->chanListTable->setMode( chan, MODE_INFO );
}


void auto7plus::proceed( const char *data, int len )
{
   char *str;

   // Lokale Kopie der Daten zum Bearbeiten anlegen
   str = (char *) malloc(len+1);
   memcpy(str,data,len);
   str[len] = '\0';

   // Wenn noch ein CR dahinter ist, wird es geloescht
   if (str[len-1] == '\r')
   {
      str[len-1] = '\0';
      len--;
   }

   // Jede 7plus-Zeile ist laenger als 10 Zeichen
   if (len < 10 && !textfile)
   {
      free(str);
      return;
   }

   if (rxfile)
      saveLine(str, len);       // Diese Zeile speichern und gucken, ob das
                                // File zuende ist.
   else
      lookForLine(str, len);    // Faengt ein 7plus-File an? Wenn ja: Infos
                                // speichern etc

   free(str);
}


bool auto7plus::isRxfile()
{
   return rxfile;
}


void auto7plus::saveLine( char *str, int len )
{
   char tmp[500], *timeptr;
   time_t timediff;


   // String abspeichern
   write(fd,str,len);
   write(fd,"\n",1);
   bytes_got += len+1;

   transInfo->setReceivedBytes( bytes_got );
   if (textfile)
   {
      filelen += len;
      transInfo->setFilesizeBytes( filelen );
   }

   if (textfile)
   {
      if (strncmp(str," stop_text.", 11))
         return;
   }
   else
   {
      COPY(tmp,str,0,10);
      if (strcmp(tmp," stop_7+. "))
         return;
   }

   // Das wars, das 7plus-File ist zu Ende
   ::close(fd);
    
   // Informationen an die Gegenstation verschicken */
   timediff = time(NULL)-starttime;

   if (timediff == 0) timediff = 1;
   timeptr = (char *)spec_time(timediff);
   if (textfile)
      sprintf(tmp,"\r<LinKT>: 7plus-Textsave: '%s' saved.\r"
                  "         %i bytes, %s, %li bit/s.\r\r"
                 ,shortname, bytes_got
                 ,timeptr,(bytes_got*8)/timediff);
   else
      sprintf(tmp,"\r<LinKT>: 7plus-Autosave: '%s' saved.\r"
                  "         %i/%i bytes, %s, %li bit/s.\r\r"
                 ,shortname, bytes_got, filelen
                 ,timeptr,(bytes_got*8)/timediff);
   free(timeptr);

   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }

   if (((Channel *)chan)->userinfo->getType() == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp );
   else
      ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );


   rxfile = false;
   textfile = false;
   toplevel->chanListTable->setMode( chan, MODE_INFO );
}


void auto7plus::lookForLine( char *str, int len )
{
   char tmp[500];
   int binbytes, of, part, i, blocklines;

   // Ist dies ein 7plus-Textfile?
   if (!strncmp(str, " go_text. ", 10))
   {
      textfile = true;

      // Ein 7plus-Textfile. Dateiname?
      memcpy(tmp, str+10, strlen(str)-10);
      tmp[strlen(str)-10] = '\0';
      Klein(tmp);

      strcpy(name, tmp);
      sprintf(tmp,"%s/%s", conf->getDir7plus().latin1(), name);
      strcpy(shortname, name);


      // Dateien nicht ueberschreiben!?!
      if (!conf->getFlag(CFG_OVERWRITEXIST))
      {
         i = 0;
         while (file_exist(tmp))
         {
            sprintf(tmp,"%s/%s--%i", conf->getDir7plus().latin1(), name, i);
            i++;
         }
      }

      if ((fd = open(tmp,O_WRONLY|O_CREAT)) == -1)
         return;

      // Zugriffsrechte einstellen
      fchmod(fd,S_IRUSR|S_IWUSR);

      strcpy(name, tmp);

      // Das ganze scheint ja wohl ein 7plus-Textfile zu sein...
      bytes_got = len+1;
      starttime = time(NULL);

      // Den String schreiben
      write(fd,str,len);
      write(fd,"\n",1);

      // Merken: Das hier ist jetzt ein 7plus-Textfile
      rxfile = true;
      toplevel->chanListTable->setMode( chan, MODE_7AUTOSAVE );

      // Entsprechendes Fenster oeffnen
      transInfo = new TransferInfo( ((Channel *)chan)->centralWidget, TRANSART_SIEBEN, shortname, bytes_got );
      ((Channel *)chan)->withTransInfo( transInfo );
      transInfo->setReceivedBytes( len+1 );
      return;
   }


   if (strncmp(str," go_7+. ",8))
      return;

   // Koennte ein 7plus-File sein

   // Steht " of " an der richtigen Stelle?
   if (strncmp(str+11," of ",4))
      return;   // Nein

   /* Teile-Nummern extrahieren */
   COPY(tmp,str,8,3);
   part = atoi(tmp);
   COPY(tmp,str,15,3);
   of = atoi(tmp);
   // Dateinamen
   COPY(tmp,str,19,12);
   if ((i = POS('.',tmp)) != -1) tmp[i] = '\0';
   KillSpacesLeft(tmp);
   KillSpacesRight(tmp);
   Klein(tmp);
   strcpy(name,tmp);
   // Laenge des dekodierten Files
   COPY(tmp,str,32,7);
   binbytes = atoi(tmp);
   // Anzahl Zeilen in einem Block
   COPY(tmp,str,45,3);
   blocklines = get_hex (tmp);

   // Laenge errechen. Hochgradig abhaengig von der Teil-Nummer
   if (part == 1)
      if (str[65] == '*')
         filelen = (blocklines+3)*70;
      else
         filelen = (blocklines+2)*70;
   else if (part == of)
           filelen = (int) ((((binbytes + 61) / 62) % blocklines)+2)*70;
        else
           filelen = (blocklines+2)*70;

   if ((of == 0) || (part == 0) || (part > of))
      return;

   if (of == 1)
   {
      sprintf(tmp,"%s/%s.7pl", conf->getDir7plus().latin1(), name);
      sprintf(shortname, "%s.7pl", name);
   }
   else
   {
      sprintf(tmp,"%s/%s.p%02x", conf->getDir7plus().latin1(), name, part);
      sprintf(shortname,"%s.p%02x",name,part);
   }

   // Dateien nicht ueberschreiben!?!
   if (!conf->getFlag(CFG_OVERWRITEXIST))
   {
      i = 0;
      while (file_exist(tmp))
      {
         if (of == 1)
         {
            sprintf(tmp,"%s/%s.7pl--%i", conf->getDir7plus().latin1(), name, i);
            sprintf(shortname,"%s.7pl--%i",name, i);
         }
         else
         {
            sprintf(tmp,"%s/%s.p%02x--%i", conf->getDir7plus().latin1(), name, part, i);
            sprintf(shortname,"%s.p%02x--%i",name, part, i);
         }
         i++;
      }
   }

   if ((fd = open(tmp,O_WRONLY|O_CREAT)) == -1)
      return;

   // Zugriffsrechte einstellen
   fchmod(fd,S_IRUSR|S_IWUSR);

   strcpy(name, tmp);

   // Das ganze scheint ja wohl ein 7plus-File zu sein...
   bytes_got = len+1;
   starttime = time(NULL);

   // Den String schreiben
   write(fd,str,len);
   write(fd,"\n",1);

   // Merken: Das hier ist jetzt ein 7plus-File
   rxfile = true;
   toplevel->chanListTable->setMode( chan, MODE_7AUTOSAVE );

   // Entsprechendes Fenster oeffnen
   transInfo = new TransferInfo( ((Channel *)chan)->centralWidget, TRANSART_SIEBEN, shortname, filelen );
   ((Channel *)chan)->withTransInfo( transInfo );
   transInfo->setReceivedBytes( len+1 );
}


/*
 *  void auto7plus::abortRX()
 * Auf die Abort-Taste wurde gedrueckt. Transfer abbrechen.
 */
void auto7plus::abortRX()
{
   char tmp[500];


   // Das wars, das 7plus-File ist zu Ende
   ::close(fd);

   // Informationen an die Gegenstation verschicken */
   strcpy(tmp,"\r<LinKT>: 7plus-Autosave aborted.\r\r");

   if (((Channel *)chan)->userinfo->getType() == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp );
   else
      ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

   rxfile = false;

   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }
   toplevel->chanListTable->setMode( chan, MODE_INFO );
}


void auto7plus::closeTransferwin()
{
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
      toplevel->chanListTable->setMode( chan, MODE_INFO );
   }
}



///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////


BinRX::BinRX( QWidget *rxchan )
{
   chan = rxchan;
   rxfile = false;
   transInfo = NULL;

   filename = NULL;
   shortname = NULL;
}


BinRX::~BinRX()
{
   if (filename != NULL) free(filename);
   if (shortname != NULL) free(shortname);

   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }

   toplevel->chanListTable->setMode( chan, MODE_INFO );
}


void BinRX::deleteDataFile()
{
   unlink(filename);
}


bool BinRX::isRxfile()
{
   return rxfile;
}


/*
 *  void BinRX::proceed( char *data, int len )
 * Verarbeitet Zeile fuer Zeile.
 *
 * Rueckgabe:    0, wenn das Frame komplett abgespeichert wurde,
 *            != 0, naemlich die Anzahl Bytes, die abgespeichert wurden,
 *                  wenn in dem Frame noch etwas anderes drin war.
 */
int BinRX::proceed( const char *data, int len )
{
   char *line, str[100];
   int n;


   line = (char *) malloc(len+1);
   memcpy(line, data, len);

   if (!strncmp(line, "SP\\-", 4))
   {
      memmove(line, line+4, len-4);
      line[len-4] = '\0';
      ((Channel *)chan)->outText( line, strlen(line), OUTCOLOR_RXTEXT );
      free(line);
      return 0;
   }

   // #ABORT#\r oder \r#ABORT#\r empfangen?
   if (!strcmp(line, "#ABORT#\r") || !strcmp(line, "\r#ABORT#\r"))
   {
      ::close(fd);
      fd = -1;

      sprintf(str,"<LinKT>: AutoBIN-RX aborted by peer.\r");

      if (((Channel *)chan)->userinfo->getType() == TYPE_TERMINAL)
          ((Channel *)chan)->sendString( str );
      else
          ((Channel *)chan)->outText( str, strlen(str), OUTCOLOR_TXTEXT );

      if (filename != NULL)
      {
         free(filename);
         filename = NULL;
      }
      if (shortname != NULL)
      {
         free(shortname);
         shortname = NULL;
      }

      rxfile = false;
      if (transInfo != NULL)
      {
         delete transInfo;
         transInfo = NULL;
         ((Channel *)chan)->withoutTransInfo();
      }
      toplevel->chanListTable->setMode( chan, MODE_INFO );

      // Linemode wieder einschalten
      ((Channel *)chan)->flags |= CH_LINEMODE;

      free( line );
      return 0;
   }

   if (rxfile)
   {
      n = saveFrame(line, len);
      free(line);
      return n;
   }
   else
   {
      lookForLine(line, len);
      free(line);
      return 0;
   }
}


void BinRX::lookForLine( char *line, int len )
{
   char tmp[500],str[500];
   int i, tlen;


   if (strncmp(line,"#BIN#",5) != 0) return;

   if (line[len-1] == '\xD')
   {
      line[len-1] = '\0';
      len--;
   }

   memcpy(tmp,line+5,len-5);
   tmp[len-5] = '\0';

   size = -1;
   filename = NULL;
   crc = 0;
   crc_rec = false;
   rxtime = 0;

   while (tmp[0] != '\0')
   {
      if ((i = POS('#',tmp)) == -1)
      {
         strcpy(str,tmp);
         tmp[0] = '\0';
      }
      else
      {
         COPY(str,tmp,0,i);
         COPY(tmp,tmp,i+1,strlen(tmp)-i-1);
      }

      switch (str[0])
      {
         case '|': // CRC
                   crc = atoi(str+1);
                   crc_rec = true;
                   break;
         case '$': // DOS-Time
                   rxtime = get_hex (str+1);
                   break;
        default: if (size == -1)
                    size = atoi(str);
                 else
                    filename = (char *) strdup(str);
      }
   }

   if (size == -1) return;

   if (filename == NULL)
   {
      i = 0;
      do
      {
         sprintf(tmp,"%s/%lX.%.3i", conf->getDirABin().latin1(), time(NULL), i);
         i++;
      }
      while (file_exist(tmp));
      filename = (char *) strdup(tmp);
   }
   else
   {
      // Eventuell vorhandenen Pfad (mit '/' oder '\') enfernen
      if ((i = lPOS('/',filename)) != -1)
      {
	      tlen = strlen(filename)-i-1;
         memmove( filename, filename+i+1, tlen );
         filename[tlen] = '\0';
		}
      if ((i = lPOS('\\',filename)) != -1)
      {
	      tlen = strlen(filename)-i-1;
         memmove( filename, filename+i+1, tlen );
         filename[tlen] = '\0';
		}
printf("filename: %s\n", filename);

      sprintf(tmp,"%s/%s", conf->getDirABin().latin1(), filename);

      // Gucken, ob der Filename schon existiert. Wenn ja wird eine
      // Zahl angehaengt
      i = 0;
      while (file_exist(tmp))
      {
         sprintf(tmp,"%s/%s--%i", conf->getDirABin().latin1(), filename, i);
         i++;
      }
      free(filename);
      filename = (char *) strdup(tmp);
   }

   if ((fd = open(filename,O_WRONLY|O_CREAT|O_TRUNC)) == -1)
   {
      ((Channel *)chan)->sendString( "#NO#\xD" );
         return;
   }

   fchmod(fd,S_IRUSR|S_IWUSR);

   ((Channel *)chan)->sendString( "#OK#\xD" );

   our_crc = 0;
   starttime = time(NULL);
   size_written = 0;

   toplevel->chanListTable->setMode( chan, MODE_AUTOBINRX );

   // Linemode ausschalten
   ((Channel *)chan)->flags &= ~CH_LINEMODE;

   // Shortname
   if ((i = lPOS('/',filename)) == -1)
      shortname = (char *) strdup(filename);
   else
   {
      shortname = (char *) malloc(strlen(filename));
      COPY(shortname, filename,i+1,strlen(filename)-i-1);
   }

   rxfile = true;
}


int BinRX::saveFrame(char *line, int len)
{
   int i,timediff;
   char str[500],*tmp;

   i = size - size_written;

   // i Bytes schreiben.
   if (len < i) i = len;
   write(fd,line,i);
   our_crc = CheckSums::calcCrcthpStr( line, i, our_crc );


   if (transInfo == NULL)
   {
      transInfo = new TransferInfo( ((Channel *)chan)->centralWidget, TRANSART_ABINRX, shortname, size );
      ((Channel *)chan)->withTransInfo( transInfo );
   }


   size_written += i;
   transInfo->setReceivedBytes( size_written );

   // Wurden schon alle Daten empfangen?
   if (size_written == size)
   {
      ::close(fd);
      fd = -1;

      timediff = time(NULL)-starttime;
      if(timediff == 0) timediff = 1;
      tmp = spec_time(timediff);

      if (!crc_rec)
         sprintf(str,"<LinKT>: AutoBIN-RX OK. No CRC-check (time: %s, %li bit/s)\xD",
                      tmp,(size*8)/timediff);
      else
         if (our_crc == crc)
            sprintf(str,"<LinKT>: AutoBIN-RX OK. CRC: %i (time: %s, %li bit/s)\xD",
                        crc,tmp,(size*8)/timediff);
         else
            sprintf(str,"<LinKT>: AutoBIN-RX failed (CRC-ERROR %i != %i)\xD",
                        crc,our_crc);
      free(tmp);

      if (((Channel *)chan)->userinfo->getType() == TYPE_TERMINAL)
          ((Channel *)chan)->sendString( str );
      else
          ((Channel *)chan)->outText( str, strlen(str), OUTCOLOR_TXTEXT );

      if (filename != NULL)
      {
         free(filename);
         filename = NULL;
      }
      if (shortname != NULL)
      {
         free(shortname);
         shortname = NULL;
      }

      rxfile = false;
      if (transInfo != NULL)
      {
         delete transInfo;
         transInfo = NULL;
         ((Channel *)chan)->withoutTransInfo();
      }
      toplevel->chanListTable->setMode( chan, MODE_INFO );

      // Linemode wieder einschalten
      ((Channel *)chan)->flags |= CH_LINEMODE;
   }

   return i;
}


void BinRX::abortRX()
{
   char tmp[500];


   // Das wars, das File ist zu Ende
   ::close(fd);

   // Informationen an die Gegenstation verschicken */
   strcpy(tmp,"\r<LinKT>: AutoBIN-save aborted.\r\r");

   if (((Channel *)chan)->userinfo->getType() == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp );
   else
      ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

   rxfile = false;
   toplevel->chanListTable->setMode( chan, MODE_INFO );

   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }

   if (filename != NULL)
   {
      free(filename);
      filename = NULL;
   }
   if (shortname != NULL)
   {
      free(shortname);
      shortname = NULL;
   }

   // Linemode wieder einschalten
   ((Channel *)chan)->flags |= CH_LINEMODE;
}


void BinRX::closeTransferwin()
{
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
      toplevel->chanListTable->setMode( chan, MODE_INFO );
   }
}




