/*
 * xconvers - GTK+ convers client for amateur radio
 * Copyright (C) 2000-2001 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * net.c - private functions for sending and receiving data, connecting and 
 * disconnecting.
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <resolv.h>
#include <netdb.h>
#include <sys/socket.h>
#include "support.h"
#include "net.h"
#include "types.h"
#include "utils.h"

gint sockethandle;
extern GHashTable *callsigns;
extern GtkWidget *mainwindow;
extern gpointer cvhost, cvport;
extern gchar *preferencesdir, *logfilename;
extern gint rxmonitor, usercolor;
extern gboolean connected, locked;
extern FILE *logfp;
extern preferencestype preferences;

/*
 * Connect routine. Lookup the hostname first, create the socket and recover 
 * from possible errors, which will be displayed in the statusbar. Menus will
 * be updated, so connect becomes insensitive. Logging is actived if enabled.
 * If autologin is activated, this is the place where the '/n <callsign>
 * <channel>' string is sent to the connected host. Gdk_input_add is called,
 * so a received message on the socket will be handled by 'rx'. Finally, the 
 * hash table is created where we will save our callsigns.
 */

void cvconnect(void)
{
  GString *statusmessage = g_string_new(""), *sendstring;
  GtkWidget *menuopen, *menuclose, *mainstatusbar;
  gchar **sendsplit = NULL;
  gint i = 0;
  struct hostent *cvhostent;
  struct sockaddr_in cvaddress;

  mainstatusbar = lookup_widget (mainwindow, "mainstatusbar");
  if ((cvhostent = gethostbyname(cvhost)) == NULL) 
  { 
    g_string_sprintf(statusmessage, _("Unknown host %s..."), (gchar *)cvhost);
    gtk_statusbar_pop (GTK_STATUSBAR(mainstatusbar), 1);
    gtk_statusbar_push (GTK_STATUSBAR(mainstatusbar), 1, statusmessage->str);
    if (!preferences.statusbar)
    {
      g_string_prepend(statusmessage, "--> ");
      g_string_append(statusmessage, "\n");
      maintext_add(statusmessage->str, MESSAGE_TX);
    }
    g_string_free(statusmessage, TRUE); 
    return;
  }  
  if ((sockethandle = socket (AF_INET, SOCK_STREAM, 0)) == -1)
  {
    statusmessage = g_string_new(strerror(errno));
    gtk_statusbar_pop (GTK_STATUSBAR (mainstatusbar), 1);
    gtk_statusbar_push (GTK_STATUSBAR (mainstatusbar), 1, statusmessage->str);
    if (!preferences.statusbar)
    {
      g_string_prepend(statusmessage, "--> ");
      g_string_append(statusmessage, "\n");
      maintext_add(statusmessage->str, MESSAGE_TX);
    }
    g_string_free(statusmessage, TRUE); 
    return;
  }
  cvaddress.sin_family = AF_INET;
  cvaddress.sin_port = htons(atoi(cvport));
  cvaddress.sin_addr = *((struct in_addr *)cvhostent->h_addr);
  bzero (&(cvaddress.sin_zero), 8);
  if (connect(sockethandle, (struct sockaddr *)&cvaddress, 
    sizeof(struct sockaddr)) == -1)
  {
    statusmessage = g_string_new(strerror(errno)); 
    gtk_statusbar_pop (GTK_STATUSBAR (mainstatusbar), 1);
    gtk_statusbar_push (GTK_STATUSBAR (mainstatusbar), 1, statusmessage->str);
    if (!preferences.statusbar)
    {
      g_string_prepend(statusmessage, "--> ");
      g_string_append(statusmessage, "\n");
      maintext_add(statusmessage->str, MESSAGE_TX);
    }
    g_string_free(statusmessage, TRUE); 
    return;
  }

  g_string_sprintf(statusmessage, _("Connected to %s"), (gchar *)cvhost);
  gtk_statusbar_pop (GTK_STATUSBAR (mainstatusbar), 1);
  gtk_statusbar_push (GTK_STATUSBAR (mainstatusbar), 1, statusmessage->str);
  if (!preferences.statusbar)
  {
    g_string_prepend(statusmessage, "--> ");
    g_string_append(statusmessage, "\n");
    maintext_add(statusmessage->str, MESSAGE_TX);
  }
  g_string_free(statusmessage, TRUE);
  
  menuclose = lookup_widget(mainwindow, "close");
  menuopen = lookup_widget(mainwindow, "open");
  gtk_widget_set_sensitive(menuopen, 0);
  gtk_widget_set_sensitive(menuclose, 1);
  if (preferences.logging) 
  {
    logfilename = g_strconcat(preferencesdir, "/", cvhost, ".log", NULL);
    if ((logfp = fopen(logfilename, "w")) == NULL) 
      g_error(_("Creating %s for writing."), logfilename);
  }
  if (preferences.autologin && preferences.name)
  {
    sendstring = g_string_new("/n ");
    g_string_append(sendstring, preferences.name->str);
    if (preferences.channel)
    {
      g_string_append(sendstring, " ");
      g_string_append(sendstring, preferences.channel->str);
    }
    tx(sendstring);
    g_string_free(sendstring, TRUE);
  }
  if (preferences.autologin && preferences.commands)
  {
    sendsplit = g_strsplit(preferences.commands->str, ";", 0);
    while (sendsplit[i])
    {
      sendstring = g_string_new(sendsplit[i]);
      tx(sendstring);
      g_string_free(sendstring, TRUE);
      i++;
    }
    g_strfreev(sendsplit);
  }
  rxmonitor = gdk_input_add(sockethandle, GDK_INPUT_READ, GTK_SIGNAL_FUNC(rx), NULL);
  callsigns = g_hash_table_new(g_str_hash, g_str_equal);
  connected = TRUE;
}

/*
 * Disconnect routine. The socket and logfile are closed, the lockfile for
 * logging is removed, statusbar updated and hash tabled freed. 
 * Also, menu items are adjusted, so only 'connect' will be active. 
 */

void cvdisconnect(void)
{
  GString *statusmessage = g_string_new("");
  GtkWidget *menuopen, *menuclose, *mainstatusbar;

  close(sockethandle);
  if (preferences.logging) fclose(logfp);
  if (rxmonitor) gdk_input_remove(rxmonitor);
  mainstatusbar = lookup_widget (mainwindow, "mainstatusbar");
  g_string_sprintf(statusmessage, _("Connection closed"));
  gtk_statusbar_pop (GTK_STATUSBAR (mainstatusbar), 1);
  gtk_statusbar_push (GTK_STATUSBAR (mainstatusbar), 1, statusmessage->str);
  if (!preferences.statusbar)
  {
    g_string_prepend(statusmessage, "--> ");
    g_string_append(statusmessage, "\n");
    maintext_add(statusmessage->str, MESSAGE_TX);
  }
  g_string_free(statusmessage, TRUE);
  menuclose = lookup_widget(mainwindow, "close");
  menuopen = lookup_widget(mainwindow, "open");
  gtk_widget_set_sensitive(menuopen, 1);
  gtk_widget_set_sensitive(menuclose, 0);
  g_hash_table_foreach_remove(callsigns, (GHRFunc)remove_calls, NULL);
  g_hash_table_destroy(callsigns);
  usercolor = 0;
  connected = FALSE;
}

/*
 * A message is received here. We create a buffer where the incoming message is
 * stored. Then we check for possible problems (displayed in the statusbar). If 
 * all is OK, the message gets displayed and is written to the log if this is
 * opened.
 */

void rx(gpointer data, gint source, GdkInputCondition condition)
{
  gchar cvbuf[8192];
  gint numbytes;
  GString *statusmessage = g_string_new("");
  GtkWidget *mainstatusbar;

  memset(cvbuf, 0, 8192*sizeof(gchar));
  numbytes = read(sockethandle, cvbuf, 4096);
  if (numbytes == - 1 )
  {
    mainstatusbar = lookup_widget (mainwindow, "mainstatusbar");
    statusmessage = g_string_new(strerror(errno)); 
    gtk_statusbar_pop (GTK_STATUSBAR (mainstatusbar), 1);
    gtk_statusbar_push (GTK_STATUSBAR (mainstatusbar), 1, statusmessage->str);
    if (!preferences.statusbar)
    {
      g_string_prepend(statusmessage, "--> ");
      g_string_append(statusmessage, "\n");
      maintext_add(statusmessage->str, MESSAGE_TX);
    }
    g_string_free(statusmessage, TRUE); 
    cvdisconnect();
    return;
  }
  if ( numbytes == 0 )
  {
    mainstatusbar = lookup_widget (mainwindow, "mainstatusbar");
    g_string_sprintf(statusmessage, _("0 byte packet received"));
    gtk_statusbar_pop (GTK_STATUSBAR (mainstatusbar), 1);
    gtk_statusbar_push (GTK_STATUSBAR (mainstatusbar), 1, statusmessage->str);
    if (!preferences.statusbar)
    {
      g_string_prepend(statusmessage, "--> ");
      g_string_append(statusmessage, "\n");
      maintext_add(statusmessage->str, MESSAGE_TX);
    }
    g_string_free(statusmessage, TRUE);
    cvdisconnect();
    return;
  }
  maintext_add(cvbuf, MESSAGE_RX);
  if (preferences.logging)
  {
    fprintf(logfp, "%s", cvbuf);
    fflush(logfp);
  }
}

/*
 * Save messages to history, send out to the socket, echo to the main window,
 * and write to logfile.
 */

void tx(GString *message)
{
  tx_save(message);
  message = g_string_append(message, "\n");
  write(sockethandle, message->str, message->len);
  maintext_add(message->str, MESSAGE_TX);
  if (preferences.logging && connected)
  {
    fprintf(logfp, "%s", message->str);
    fflush(logfp);
  }
}