/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * file_media.c : the file media.
 *
 */

/*********************************************************************
 *                             INCLUDES                              *
 *********************************************************************/

#include <file_media.h>

/*********************************************************************
 *                            VARIABLES                              *
 *********************************************************************/

typedef struct {

  int          f_id;
  char        *filename;
  char        *display_name;
  xmps_data_t *data_stream;
  GList       *output_list; 

} file_media_t;

/*********************************************************************
 *                             FUNCTIONS                             *
 *********************************************************************/

/*
 * get_media_info : MANDATORY
 *
 * Used by the plugin_center to get a pointer to the gui object.
 *
 */

xmps_media_plugin_t *get_media_info(void)
{
  xmps_media_plugin_t *media;

  media = (xmps_media_plugin_t *) malloc(sizeof(xmps_media_plugin_t));

  media->id   = 0;
  media->name = "FILE";
  media->data = (void *) malloc(sizeof(file_media_t));
  media->path = "";

  media->open  = file_open;
  media->get   = file_get;
  media->set   = file_set;
  media->read  = file_read;
  media->seek  = file_seek;
  media->write = file_write;
  media->close = file_close;

  ((file_media_t *) media->data)->f_id         = -1;
  ((file_media_t *) media->data)->filename     = NULL;
  ((file_media_t *) media->data)->display_name = NULL;
  ((file_media_t *) media->data)->output_list  = NULL;

  ((file_media_t *) media->data)->data_stream  = (xmps_data_t *) malloc(sizeof(xmps_data_t));

  return media;
}


/*
 * file_open
 *
 * - open a new file and 
 *   store the f_id.
 *
 */

unsigned int file_open(xmps_media_plugin_t *media, char *full_path)
{
  if(full_path == NULL || media == NULL) {
   
    XMPS_DEBUG("wrong parameters!");
    return 0;
  }
  
  if(strncasecmp(full_path, "FILE://", 6) == 0) {

    ((file_media_t *) media->data)->f_id = open(strstr(full_path, "//") + 2, O_RDONLY);
    
    XMPS_DEBUG("opening %s got f_id %d", 
	       full_path, ((file_media_t *) media->data)->f_id);
    
    if(((file_media_t *) media->data)->f_id == -1) {
      return 0;
    }
    
    media->path = full_path + 6;

    ((file_media_t *) media->data)->display_name = full_path;
    ((file_media_t *) media->data)->filename     = full_path;
    
    return 1;
  }
  else {
    return 0;
  }

}

/*
 * file_get_info
 *
 * - generic info routine
 *
 */

void* file_get(xmps_media_plugin_t *media, unsigned int flag, void *data)
{
  if(media == NULL) {
    return NULL;
  }

  switch(flag)
    {
    case XMPS_FLAG_OUTPUT_LIST:
      {
	if( ((file_media_t *) media->data)->f_id != -1 ) {

	  /*
	   * re-init output list
	   *
	   */

	  XMPS_DEBUG("creating raw output stream");

	  ((file_media_t *) media->data)->output_list = NULL;

	  /*
	   * fill data structure
	   *
	   */

	  ((file_media_t *) media->data)->data_stream->id          = 0;
	  ((file_media_t *) media->data)->data_stream->type        = XMPS_DATA_RAW;

	  ((file_media_t *) media->data)->data_stream->plugin_type = XMPS_PLUGIN_MEDIA;
	  ((file_media_t *) media->data)->data_stream->plugin      = (void *) media;

	  /*
	   * and add the output data
	   *
	   */

	  ((file_media_t *) media->data)->output_list = g_list_prepend(((file_media_t *) media->data)->output_list, 
								       (void *) ((file_media_t *) media->data)->data_stream);
	
	  return (void *) ((file_media_t *) media->data)->output_list;
	}

      }
      break;

    default:
      break;
      
    }
  
  return NULL;
}

/*
 * file_set_info
 *
 * - generic info routine
 *
 */

unsigned int file_set(xmps_media_plugin_t *media, unsigned int flag, void *data)
{
  return 0;
}

/*
 * file_read
 *
 * - read the file.
 *
 */

unsigned int file_read(xmps_media_plugin_t *media, void *buffer, unsigned int size)
{
  return read(((file_media_t *) media->data)->f_id, buffer, size);
}

/*
 * file_seek
 *
 * - supports all the methods
 *
 */

unsigned int file_seek(xmps_media_plugin_t *media, long pos, xmps_seeking_method_t method)
{
  if(media == NULL) {
    return 0;
  }
  
  if(((file_media_t *) media->data)->f_id != -1)
    {
      switch(method)
	{
	case XMPS_SEEK_SET:
	  return lseek(((file_media_t *) media->data)->f_id, pos, SEEK_SET);
	  break;
 	case XMPS_SEEK_CUR:
	  return lseek(((file_media_t *) media->data)->f_id, pos, SEEK_CUR);
	  break;
 	case XMPS_SEEK_END:
	  return lseek(((file_media_t *) media->data)->f_id, pos, SEEK_END);
	  break;
	case XMPS_SEEK_PERCENT:
	  break;
	default:
	  return lseek(((file_media_t *) media->data)->f_id, pos, SEEK_SET);
	  break;
	}
    }  
  else
    return 0;

  return 0;
}

/*
 * file_write
 *
 * - write to a file.
 *
 */

unsigned int file_write(xmps_media_plugin_t *media, const void *buffer, unsigned int size)
{
  if(media == NULL) {
    return 0;
  }

  if(((file_media_t *) media->data)->f_id != -1)
    return write(((file_media_t *) media->data)->f_id, buffer, size);
  else
    return 0;
}

/*
 * file_close
 *
 * - closes the file and
 *   release f_id.
 *
 */

unsigned int file_close(xmps_media_plugin_t *media)
{
  if(media == NULL) {
    return 0;
  }

  if(((file_media_t *) media->data)->f_id != -1)
    {
      XMPS_DEBUG("close");

      close(((file_media_t *) media->data)->f_id);
      
      ((file_media_t *) media->data)->f_id = -1;
      return 1;
    }
  else
    return 0;
}




