/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * http_media.c : the http media.
 *
 */

/*********************************************************************
 *                             INCLUDES                              *
 *********************************************************************/

#include <http_media.h>

/*********************************************************************
 *                            VARIABLES                              *
 *********************************************************************/

typedef struct {

  int          http_id;
  char        *httpname;
  char        *display_name;
  xmps_data_t *data_stream;
  GList       *output_list; 

} http_media_t;

/*********************************************************************
 *                             FUNCTIONS                             *
 *********************************************************************/

/*
 * get_media_info : MANDATORY
 *
 * Used by the plugin_center to get a pointer to the gui object.
 *
 */

xmps_media_plugin_t *get_media_info(void)
{
  xmps_media_plugin_t *media;

  media = (xmps_media_plugin_t *) malloc(sizeof(xmps_media_plugin_t));

  media->id   = 0;
  media->name = "HTTP";
  media->data = (void *) malloc(sizeof(http_media_t));
  media->path = "";

  media->open  = http_open;
  media->get   = http_get;
  media->set   = http_set;
  media->read  = http_read;
  media->seek  = http_seek;
  media->write = http_write;
  media->close = http_close;

  ((http_media_t *) media->data)->http_id         = -1;
  ((http_media_t *) media->data)->httpname     = NULL;
  ((http_media_t *) media->data)->display_name = NULL;
  ((http_media_t *) media->data)->output_list  = NULL;

  ((http_media_t *) media->data)->data_stream  = (xmps_data_t *) malloc(sizeof(xmps_data_t));

  return media;
}

/*
 * internal functions
 *
 */

int tcp_open(char * address, int port)
{
  struct sockaddr_in stAddr;
  struct hostent * host;
  int sock;
  struct linger l;

  memset(&stAddr,0,sizeof(stAddr));
  stAddr.sin_family = AF_INET ;
  stAddr.sin_port = htons(port);

  if((host = gethostbyname(address)) == NULL) return(0);

  stAddr.sin_addr = *((struct in_addr *) host->h_addr_list[0]) ;

  if((sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) return(0);

  l.l_onoff = 1; l.l_linger = 5;
  if(setsockopt(sock, SOL_SOCKET, SO_LINGER, (char*) &l, sizeof(l)) < 0) return(0);

  if(connect(sock, (struct sockaddr *) &stAddr, sizeof(stAddr)) < 0) return(0);

  return(sock);
}

int http_open_fd(char * arg)
{
  char * host;
  int port;
  char * request;
  int tcp_sock;
  char http_request[1024];
  char c;

  /* Check for URL syntax */
  if(strncmp(arg, "HTTP://", strlen("HTTP://"))) {

    XMPS_DEBUG("wrong URL");
    return(0);
  }

  /* Parse URL */
  port = 80;
  host = arg + strlen("HTTP://");
  
  if((request = strchr(host, '/')) == NULL)  {

    XMPS_DEBUG("again wrong URL");
    return(0);
  }
  
  *request++ = 0;
  
  if(strchr(host, ':') != NULL) /* port is specified */
  {
    port = atoi(strchr(host, ':') + 1);
    *strchr(host, ':') = 0;
  } 

  /* Open a TCP socket */
  if(!(tcp_sock = tcp_open(host, port)))
  {
    XMPS_DEBUG("http_open");
    return(0);
  }

  /* Send HTTP GET request */
  sprintf(http_request, 
	  "GET /%s HTTP/1.0\r\n"
	  "User-Agent: Mozilla/2.0 (Win95; I)\r\n"
	  "Pragma: no-cache\r\n"
	  "Host: %s\r\n"
	  "Accept: */*\r\n"
	  "\r\n",
	  request, host);
  send(tcp_sock, http_request, strlen(http_request), 0);
  
  /* Parse server reply */
  do read(tcp_sock, &c, sizeof(char)); while(c != ' ');

  read(tcp_sock, http_request, 4*sizeof(char));

  http_request[4] = 0;

  if(strcmp(http_request, "200 "))
  {
    fprintf(stderr, "http_open: ");
    do { 
      read(tcp_sock, &c, sizeof(char));
      fprintf(stderr, "%c", c); 
    }
    while(c != '\r');
    fprintf(stderr, "\n");
    return(0);
  }
  
  return(tcp_sock);
}


/*
 * http_open
 *
 * - open a new http and 
 *   store the http_id.
 *
 */

unsigned int http_open(xmps_media_plugin_t *media, char *full_path)
{
  if(full_path == NULL || media == NULL) {
   
    XMPS_DEBUG("wrong parameters!");
    return 0;
  }
  
  if(strncasecmp(full_path, "HTTP://", 6) == 0) {

    unsigned char byte;
    
    ((http_media_t *) media->data)->http_id = http_open_fd(full_path);
    
    XMPS_DEBUG("opening %s got http_id %d", 
	       full_path, ((http_media_t *) media->data)->http_id);
    
    if(((http_media_t *) media->data)->http_id == -1) {
      return 0;
    }
    
    media->path = full_path + 6;

    ((http_media_t *) media->data)->display_name = full_path;
    ((http_media_t *) media->data)->httpname     = full_path;

    /*
     * skip info
     */

  loop:

    do {
     
      read(((http_media_t *) media->data)->http_id, &byte, 1);

      XMPS_DEBUG("byte : %c", byte);
    }
    while(byte != '\n');
    
    read(((http_media_t *) media->data)->http_id, &byte, 1);

    XMPS_DEBUG("special ; %c", byte);

    if(byte != '\r')
      goto loop;

    read(((http_media_t *) media->data)->http_id, &byte, 1);

    XMPS_DEBUG("special ; %c", byte);

    if(byte != '\n')
      goto loop;

    return 1;
  }
  else {
    return 0;
  }

}

/*
 * http_get_info
 *
 * - generic info routine
 *
 */

void* http_get(xmps_media_plugin_t *media, unsigned int flag, void *data)
{
  if(media == NULL) {
    return NULL;
  }

  switch(flag)
    {
    case XMPS_FLAG_OUTPUT_LIST:
      {
	if( ((http_media_t *) media->data)->http_id != -1 ) {

	  /*
	   * re-init output list
	   *
	   */

	  XMPS_DEBUG("creating raw output stream");

	  ((http_media_t *) media->data)->output_list = NULL;

	  /*
	   * fill data structure
	   *
	   */

	  ((http_media_t *) media->data)->data_stream->id          = 0;
	  ((http_media_t *) media->data)->data_stream->type        = XMPS_DATA_RAW;

	  ((http_media_t *) media->data)->data_stream->plugin_type = XMPS_PLUGIN_MEDIA;
	  ((http_media_t *) media->data)->data_stream->plugin      = (void *) media;

	  /*
	   * and add the output data
	   *
	   */

	  ((http_media_t *) media->data)->output_list = g_list_prepend(((http_media_t *) media->data)->output_list, 
								       (void *) ((http_media_t *) media->data)->data_stream);
	
	  return (void *) ((http_media_t *) media->data)->output_list;
	}

      }
      break;

    default:
      break;
      
    }
  
  return NULL;
}

/*
 * http_set_info
 *
 * - generic info routine
 *
 */

unsigned int http_set(xmps_media_plugin_t *media, unsigned int flag, void *data)
{
  return 0;
}

/*
 * http_read
 *
 * - read the http.
 *
 */

unsigned int http_read(xmps_media_plugin_t *media, void *buffer, unsigned int size)
{
  return read(((http_media_t *) media->data)->http_id, buffer, size);
}

/*
 * http_seek
 *
 * - supports all the methods
 *
 */

unsigned int http_seek(xmps_media_plugin_t *media, long pos, xmps_seeking_method_t method)
{
  if(media == NULL) {
    return 0;
  }
  
  if(((http_media_t *) media->data)->http_id != -1)
    {
      switch(method)
	{
	case XMPS_SEEK_SET:
	  return lseek(((http_media_t *) media->data)->http_id, pos, SEEK_SET);
	  break;
 	case XMPS_SEEK_CUR:
	  return lseek(((http_media_t *) media->data)->http_id, pos, SEEK_CUR);
	  break;
 	case XMPS_SEEK_END:
	  return lseek(((http_media_t *) media->data)->http_id, pos, SEEK_END);
	  break;
	case XMPS_SEEK_PERCENT:
	  break;
	default:
	  return lseek(((http_media_t *) media->data)->http_id, pos, SEEK_SET);
	  break;
	}
    }  
  else
    return 0;

  return 0;
}

/*
 * http_write
 *
 * - write to a http.
 *
 */

unsigned int http_write(xmps_media_plugin_t *media, const void *buffer, unsigned int size)
{
  if(media == NULL) {
    return 0;
  }

  if(((http_media_t *) media->data)->http_id != -1)
    return write(((http_media_t *) media->data)->http_id, buffer, size);
  else
    return 0;
}

/*
 * http_close
 *
 * - closes the http and
 *   release http_id.
 *
 */

unsigned int http_close(xmps_media_plugin_t *media)
{
  if(media == NULL) {
    return 0;
  }

  if(((http_media_t *) media->data)->http_id != -1)
    {
      XMPS_DEBUG("close");

      close(((http_media_t *) media->data)->http_id);
      
      ((http_media_t *) media->data)->http_id = -1;
      return 1;
    }
  else
    return 0;
}




