/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 *  sdl_renderer : interface to SDL display
 *
 */

/*********************************************************************
 *                             INCLUDES                              *
 *********************************************************************/

/*
 * includes : local
 *
 */

#include "sdl_renderer.h"
/*********************************************************************
 *                            VARIABLES                              *
 *********************************************************************/

/* 
 * variables : SDL
 *
 */

typedef struct _sdl_renderer_t {

  SDL_Surface       *surface;
  SDL_Surface       *display_surface;
  SDL_Surface       *decal_surface;
  SDL_Overlay       *overlay;
  SDL_Rect           r;
  xmps_video_info_t *our_info;
  xmps_video_info_t *codec_info;
  unsigned int       fullscreen;
  unsigned int       fullscreen_width;
  unsigned int       fullscreen_height;
  char               SDL_windowhack[32];
  int                use_yuv;

} sdl_renderer_t;

/*********************************************************************
 *                             FUNCTIONS                             *
 *********************************************************************/

/*
 * ++ get_video_renderer_info ++ : MANDATORY
 *
 * Used by the plugin_center to get a pointer to the object.
 *
 */

xmps_video_renderer_plugin_t *get_video_renderer_info(void *cfgfile) {
  
  xmps_video_renderer_plugin_t *video_renderer;

  video_renderer = (xmps_video_renderer_plugin_t *) malloc(sizeof(xmps_video_renderer_plugin_t));

  video_renderer->name = "SDL Video";
  video_renderer->data = malloc(sizeof(sdl_renderer_t));
  
  video_renderer->open  = sdl_open;
  video_renderer->get   = sdl_get;
  video_renderer->set   = sdl_set;
  video_renderer->draw  = sdl_draw;
  video_renderer->close = sdl_close;

  ((sdl_renderer_t *) video_renderer->data)->surface         = NULL;
  ((sdl_renderer_t *) video_renderer->data)->display_surface = NULL;
  ((sdl_renderer_t *) video_renderer->data)->decal_surface   = NULL;
  ((sdl_renderer_t *) video_renderer->data)->overlay         = NULL;
  ((sdl_renderer_t *) video_renderer->data)->our_info        = NULL;
  ((sdl_renderer_t *) video_renderer->data)->codec_info      = NULL;
  ((sdl_renderer_t *) video_renderer->data)->fullscreen      = 0;
  ((sdl_renderer_t *) video_renderer->data)->use_yuv         = 1;
  
  xmps_config_read_int((xmps_config_file_t *) cfgfile,
		       "sdl_renderer",
		       "sdl_use_yuv",
		       &((sdl_renderer_t *) video_renderer->data)->use_yuv);
  
  return video_renderer;
}

/*
 * ++ sdl_get_formats ++
 *
 *  - return the current format list, will be 
 *    YUV12 or current RBG bit depth
 *
 */

GList* sdl_get_formats(xmps_video_renderer_plugin_t *video_renderer) 
{
  Display             *sdl_display;
  XVisualInfo          template, *sdl_visual_info;
  int                  found;
  GList               *list = NULL;
  xmps_video_format_t *f;
  
  /* 
   * get the options 
   *
   */

  /*
  if(sdl_video_renderer.cfgfile == NULL)
    DEBUG("sdl_renderer", "!!! NULL cfgfile !!!", NULL);
	
  sdl_load_options(sdl_video_renderer.cfgfile);
  */

  /*
   * two posibilities here: overlay or not
   *
   */
  
  if(((sdl_renderer_t *) video_renderer->data)->use_yuv == 1)
    {
      /*
       * return YUV 12 mode (TODO : add more modes) 
       *
       */
  
      f = (xmps_video_format_t *) malloc(sizeof(xmps_video_format_t));
		
      f->type   = XMPS_VIDEO_FORMAT_YUV12;
      f->bpp    = 12;
		
      list = g_list_append(list, f);
    }
  else
    {
      /*
       * return the current video mode 
       *
       */
		
      f = (xmps_video_format_t *) malloc(sizeof(xmps_video_format_t));
		
      sdl_display     = XOpenDisplay(NULL);
      template.screen = XDefaultScreen(sdl_display);
		
      sdl_visual_info = XGetVisualInfo(sdl_display, 
				       VisualScreenMask, 
				       &template,
				       &found);
		
      switch(sdl_visual_info->depth)
	{
	case 15:
	  f->type = XMPS_VIDEO_FORMAT_RGB15;
	  f->bpp    = 15;
	  break;
	case 16:
	  f->type = XMPS_VIDEO_FORMAT_RGB16;
	  f->bpp    = 16;
	  break;
	case 24:
	  f->type = XMPS_VIDEO_FORMAT_RGB24;
	  f->bpp    = 24;
	  break;
	case 32:
	  f->type = XMPS_VIDEO_FORMAT_RGB32;
	  f->bpp    = 32;
	  break;
	}
		
      list = g_list_append(list, f);
    }
	
  return list;
}

unsigned int sdl_open(xmps_video_renderer_plugin_t *video_renderer)
{

  return 1;
}

/*
 * sdl_init
 *
 * - initialize the video renderer 
 *   accoring to the format requested in info
 */

unsigned int sdl_init(xmps_video_renderer_plugin_t *video_renderer, xmps_video_info_t *info)
{
  Uint32          flags;
  sdl_renderer_t *data;

  if(video_renderer == NULL) {
    return 0;
  }
	
  data = (sdl_renderer_t *) video_renderer->data;

  /*
   * always start in non-fullscreen mode.
   *
   */

  data->fullscreen = 0;
  
  /*
   * load any available options.
   *
   */

  //sdl_load_options(sdl_video_renderer.cfgfile);


  if(info == NULL)
    {
      XMPS_DEBUG("init : wrong parameter!");
      return 0;
    }

  /*
   * here we get setup of the WINDOW_ID
   *
   */

  if(info->windowid != 0)
    {
      XMPS_DEBUG("Detected windowid!");

      sprintf(data->SDL_windowhack,"SDL_WINDOWID=%d", info->windowid);
      putenv( data->SDL_windowhack);
    }
  
  /* 
   * and intialize SDL video subsystem 
   *
   */
  
  if(!(SDL_WasInit(0) & SDL_INIT_VIDEO))
    {
      XMPS_DEBUG("Initializing video...");

      SDL_InitSubSystem(SDL_INIT_VIDEO);
    }


  
  flags = SDL_HWSURFACE | SDL_RESIZABLE;
  
  XMPS_DEBUG("init : %dx%d+%d+%d, format = %d", 
	     info->width, 
	     info->height, 
	     info->x, 
	     info->y,
	     info->format.type);
  
  /*
   * have we been requested 
   * an overlay or not?
   *
   */

  switch(info->format.type)
    {
    case XMPS_VIDEO_FORMAT_YUV12:
    
      XMPS_DEBUG("creating YUV overlay");

      /*
       * setup of the surface
       *
       */
  
      data->surface = SDL_SetVideoMode(info->width + info->x, 
				       info->height + info->y, 
				       0, flags);
      
      if(data->surface == NULL) {

	XMPS_DEBUG("COULD NOT CREATE MAIN SURFACE!");
      }
      

      /*
       * and create the overlay over it
       *
       */
      
      data->overlay = SDL_CreateYUVOverlay(info->width, 
					   info->height, 
					   SDL_YV12_OVERLAY, 
					   data->surface);
      
      /*
       * if we had a problem then quit cleanly
       *
       */
      
      if(data->overlay == NULL) {
	
	XMPS_DEBUG("could not create overlay!!");
	
	SDL_QuitSubSystem(SDL_INIT_VIDEO);
	return 0;
      }
      
      break;
    default:

      /*
       * here in are in standard RGB 
       * request so we test the mode
       *
       */

      XMPS_DEBUG("creating RGB surface");


      if(SDL_VideoModeOK(info->width, 
			 info->height, 
			 info->format.bpp, 
			 flags))
	{
	  /*
	   * create the main surface
	   *
	   */

	  data->surface = SDL_SetVideoMode(info->width + info->x, 
					   info->height + info->y, 
					   info->format.bpp, 
					   flags);
	
	  /*
	   * and the surface to do the 
	   * -> decal if we need it.
	   *
	   */

	  if(info->x != 0)
	    {
	      XMPS_DEBUG("init : detected x decal!");
	      
	      switch(info->format.bpp)
		{
		case 16:
		  data->decal_surface = SDL_CreateRGBSurface(SDL_SWSURFACE, 
							     info->width, info->height, 
							     info->format.bpp, 
							     0xF800, 0x07E0, 
							     0x001F, 0x0000);
		  break;
		case 24:
		  data->decal_surface = SDL_CreateRGBSurface(SDL_SWSURFACE, 
							     info->width, info->height, 
							     info->format.bpp,  
							     0xFF0000, 0x00FF00, 
							     0x0000FF, 0x000000);
		  break;
		case 32:
		  data->decal_surface = SDL_CreateRGBSurface(SDL_SWSURFACE, 
							     info->width, info->height, 
							     info->format.bpp,  
							     0xFF000000, 0x00FF0000, 
							     0x0000FF00, 0x00000000);
		  break;
		default:
		  break;
		}	
	    }
	}
      else
	{
	  /*
	   * we had a problem
	   *
	   */

	  XMPS_DEBUG("can't set video mode");
	  SDL_QuitSubSystem(SDL_INIT_VIDEO);

	  return 0;
	}
      break;
    }
  
  /* 
   * now copy info information 
   * so we may modify our copy 
   * afterwards/ 
   *
   */

  data->our_info         = (xmps_video_info_t *) malloc(sizeof(xmps_video_info_t));
  
  data->our_info->width          = info->width;
  data->our_info->height         = info->height;
  data->our_info->format.bpp     = info->format.bpp;
  data->our_info->format.type    = info->format.type;
  data->our_info->x              = info->x;
  data->our_info->y              = info->y;
  data->our_info->windowid       = info->windowid;

  /* 
   * and we keep a pointer
   * to the original one.
   *
   */

  data->codec_info = info;

  /*
   * evrything went ok so we return 1
   *
   */

  return 1;
}

/*
 * ++ sdl_get_surface ++
 *
 *  - return a pointer to where the data must be written
 *
 */

void* sdl_get_surface(xmps_video_renderer_plugin_t *video_renderer)
{
  void           *return_surface;
  sdl_renderer_t *data;

  if(video_renderer == NULL) {
    return NULL;
  }

  data = (sdl_renderer_t *) video_renderer->data;

  /*
   * first we check for security
   *
   */

  if(data->surface == NULL) {
    
    XMPS_DEBUG("null surface!");
    
    return NULL;
  }
  
  /*
   * check if we are in YUV or RGB
   *
   */

  if(data->overlay != NULL)
    {
      /*
       * YUV
       *
       */

      XMPS_DEBUG("get_surface : giving overlay surface");

      XMPS_DEBUG("pitches : %d, %d, %d", data->overlay->pitches[0], 
		 data->overlay->pitches[1], 
		 data->overlay->pitches[2]);

      /*
       * check for consistency
       *
       */
      
      return (void *) data->overlay->pixels;
    }  
  else
    {
      /*
       * RGB
       *
       */

      if(data->fullscreen) {
	
	SDL_LockSurface(data->surface);

	return_surface = (void *) data->surface->pixels;
	
	SDL_UnlockSurface(data->surface);

	return return_surface;
      }
		/*
		 * we are not in fullscren
		 * 
		 */

      if(data->our_info->x != 0)
	{
	  XMPS_DEBUG("get_surface : giving decal surface");
	  
	  SDL_LockSurface(data->surface);

	  return_surface = (void *) data->decal_surface->pixels;
	  
	  SDL_UnlockSurface(data->surface);
	  
	  return return_surface;
	}

      /*
       * we are not -> the image
       *
       */
      
      if(data->display_surface != NULL)
	{
	  /*
	   * we get here if woftware 
	   * stretching is enabled
	   *
	   */
	  
	  XMPS_DEBUG("get_surface : giving normal resize surface");

	  SDL_LockSurface(data->surface);

	  if(data->surface->pixels == NULL)
	    {
	      XMPS_DEBUG("surface has null pixels!!");
	      
	      return NULL;
	    }

	  return_surface = (void *) data->surface->pixels;

	  SDL_UnlockSurface(data->surface);

	  return return_surface;
	}
      else
	{
	  /*
	   * and here is the most normal situation
	   *
	   */

	  XMPS_DEBUG("get_surface : giving normal surface");
	  
	  SDL_LockSurface(data->surface);

	  return_surface = (void *)(data->surface->pixels + 
				    (data->our_info->y*data->our_info->width)*data->our_info->format.bpp/8);
	  SDL_UnlockSurface(data->surface);

	  return return_surface;

	}
    }
}

/*
 * sdl_get_video_info
 *
 * - return a xmps_video_info about current playback
 *
 */

xmps_video_info_t *sdl_get_video_info(xmps_video_renderer_plugin_t *video_renderer)
{
  if(video_renderer == NULL) {
    return NULL;
  }

  return ((sdl_renderer_t *) video_renderer->data)->our_info;
}

/*
 * sdl_get_info
 *
 * - gives the desired info
 *
 */

GList *sdl_get_xml_config(xmps_video_renderer_plugin_t *video_renderer)
{
  GList *list = NULL;
  sdl_renderer_t *data;

  /*
   * security check
   *
   */
  
  if(video_renderer == NULL)
    return 0;
  
  data = (sdl_renderer_t *) video_renderer->data;
  
  list = g_list_prepend(list, (void *) xmps_xml_config_entry_int_new("use_yuv", data->use_yuv, "checkbox", "Use YUV Overlays"));

  return list;
}

void* sdl_get(xmps_video_renderer_plugin_t *video_renderer, unsigned int flag, void* data)
{

  if(video_renderer == NULL) {
    return NULL;
  }
  
  switch(flag)
    {

    case XMPS_FLAG_XML_CONFIG:
      
      return (void *) sdl_get_xml_config(video_renderer);
      break;
      
    case XMPS_FLAG_VIDEO_INFO:
      return (void *) sdl_get_video_info(video_renderer);
      break;
      
    case XMPS_FLAG_VIDEO_FORMAT_LIST:
      return (void *) sdl_get_formats(video_renderer);
      break;

    case XMPS_FLAG_VIDEO_BUFFER:
      return sdl_get_surface(video_renderer);
      break;

    default:
      break;
    }
	
  return NULL;
}


unsigned int sdl_resize(xmps_video_renderer_plugin_t *video_renderer, unsigned int width, unsigned int height)
{
  sdl_renderer_t *data;

  /*
   * security check
   *
   */
  
  if(video_renderer == NULL)
    return 0;
  
  data = (sdl_renderer_t *) video_renderer->data;

  if(data->surface == NULL) {
    XMPS_DEBUG("null surface!");
    return 0;
  }

  XMPS_DEBUG("resizing to %dx%d", width, height);

  /*
   * if we really need to resize
   *
   */ 

  if(width != data->our_info->width || height != data->our_info->height)
    {
      if(data->overlay != NULL)
	{
	  /*
	   * YUV
	   * 
	   * we just resize the surface, 
	   * the overlay will do with it 
	   *
	   */

	  XMPS_DEBUG("resizing overlay!");

	  data->surface = SDL_SetVideoMode(width + data->our_info->x, 
					   height + data->our_info->y, 
					   data->our_info->format.bpp, 
					   data->surface->flags);
	}
      else
	{
	  /*
	   * RGB
	   *
	   * we create a new surface so 
	   * the codec continues writing 
	   * on the old one 
	   *
	   */

	  data->display_surface = SDL_SetVideoMode(width + data->our_info->x, 
						   height + data->our_info->y, 
						   data->our_info->format.bpp, 
						   data->surface->flags);
	  
	  switch(data->codec_info->format.bpp)
	    {
	    case 16:
	      XMPS_DEBUG("creating 16 bpp surface");
	      
	      data->surface = SDL_CreateRGBSurface(SDL_SWSURFACE, 
						   data->codec_info->width, 
						   data->codec_info->height, 
						   data->codec_info->format.bpp, 
						   0xF800, 0x07E0, 0x001F, 0x0000);
	      break;
	    case 24:
	      
	      XMPS_DEBUG("creating 24 bpp surface");
	      
	      data->surface = SDL_CreateRGBSurface(SDL_HWSURFACE, 
						   data->codec_info->width, 
						   data->codec_info->height, 
						   data->codec_info->format.bpp,  
						   0xFF0000, 0x00FF00, 0x0000FF, 0x000000);
	      break;
	    case 32:
	      
	      XMPS_DEBUG("creating 32 bpp surface");
	      
	      data->surface = SDL_CreateRGBSurface(SDL_HWSURFACE, 
						   data->codec_info->width, 
						   data->codec_info->height, 
						   data->codec_info->format.bpp,  
						   0xFF000000, 0x00FF0000, 0x0000FF00, 0x00000000);
	      break;
	    default:

	      XMPS_DEBUG("problem!");
	      break;
	    }
	  
	  /*
	   * check if everything is ok
	   *
	   */

	  if(data->display_surface == NULL)
	    {
	      XMPS_DEBUG("cannot resize video mode...");
	      
	      return 0;
	    }

	  SDL_LockSurface(data->surface);
	  
	  if(data->surface == NULL || data->surface->pixels == NULL) {

	    XMPS_DEBUG("resize, null surface");
	    
	    return 0;
	  }

	  SDL_UnlockSurface(data->surface);

	  XMPS_DEBUG("software stretch from %dx%d to %dx%d in %dbpp", 
		     data->codec_info->width, data->codec_info->height, 
		     width, height, data->our_info->format.bpp);
	}
  
      /* 
       * if everything went well, 
       * update our info 
       *
       */

      XMPS_DEBUG("updating info!");

      data->our_info->width  = width;
      data->our_info->height = height;
    }

  return 1;
}

void sdl_parse_xml(xmps_video_renderer_plugin_t *video_renderer, GList * xml)
{
  sdl_renderer_t *data;
  GList          *node;
  char           *xml_data;

  if(video_renderer == NULL) {

    XMPS_DEBUG("wrong params!");
    return;
  }

  data = (sdl_renderer_t *) video_renderer->data;

  if(data == NULL) {

    XMPS_DEBUG("error!");
    return;
  }
  
  for(node = xml; node != NULL; node = g_list_next(node)) {

    xml_data = (char *) node->data;

    if(xml_data == NULL) {
      break;
    }
    
    if(strstr(xml_data, "use_yuv") != NULL) {
      sscanf(xmps_xml_get_value(xml_data, "value"), "%d", &data->use_yuv);
    }
  }
  
  XMPS_DEBUG("use_yuv = %d", data->use_yuv);
}

/*
 * sdl_set_info
 *
 *  - sets up particular flags (FULLSCREEN)
 * 
 */

unsigned int sdl_set(xmps_video_renderer_plugin_t *video_renderer, unsigned int flag, void *arg)
{
  sdl_renderer_t *data;
  unsigned int    display = 0;

  if(video_renderer == NULL) {

    XMPS_DEBUG("wrong params!");
    return 0;
  }

  data = (sdl_renderer_t *) video_renderer->data;

  if(data == NULL) {

    XMPS_DEBUG("error!");
    return 0;
  }

  switch(flag) {

  case XMPS_FLAG_SAVE_CONFIG:
    {
      xmps_config_file_t *cfgfile;

      cfgfile = (xmps_config_file_t *) arg;

      xmps_config_write_int(cfgfile, 
			    "sdl_renderer", 
			    "sdl_use_yuv", 
			    data->use_yuv);

      xmps_config_write_default_file(cfgfile);

      return 1;
    }
    break;

  case XMPS_FLAG_XML_CONFIG:

    XMPS_DEBUG("updating config");

    sdl_parse_xml(video_renderer, (GList *) arg);

    return 1;
    break;

  case XMPS_FLAG_VIDEO_RESIZE:
    {
      xmps_video_info_t *info;

      info = (xmps_video_info_t *) arg;

      if(info == NULL) {
	return 0;
      }

      XMPS_DEBUG("resizing!");

      sdl_resize(video_renderer, info->width, info->height);
      return 1;
    }
    break;

  case XMPS_FLAG_VIDEO_INFO: 
    {
      return sdl_init(video_renderer, (xmps_video_info_t *) arg);
    }
    break;

  case XMPS_FLAG_FULLSCREEN_SET:
    {
      if( !data->fullscreen )
        {
	  int          full_width, full_height;
	  Display     *sdl_display;

	  XMPS_DEBUG("setting fullscreen");

          /*
           * we can go to fullscreen
           *
           */
	  
          if(data->overlay != NULL) {
	    
	    XMPS_DEBUG("fullscreen overlay");

	    /*
	     * we need to unset the WINDOW_ID
	     *
	     */
	    
            SDL_QuitSubSystem(SDL_INIT_VIDEO);
            unsetenv("SDL_WINDOWID");
            SDL_InitSubSystem(SDL_INIT_VIDEO);
	    
	    /*
	     * and recreate surface and overlay in fullscreen mode
	     *
	     */

	    /*
	     * FIXME : real screen size please !!!
	     * + respect the pitch!!
	     *
	     */

	    sdl_display = XOpenDisplay(NULL);
	    full_width  = DisplayWidth(sdl_display, 0); 
	    full_height = DisplayHeight(sdl_display, 0);

	    data->surface = SDL_SetVideoMode(full_width,
					     full_height,
					     data->our_info->format.bpp,
					     SDL_HWSURFACE | SDL_FULLSCREEN);
	    if(data->surface != NULL)
	      {
		/*
		 * create the overlay
		 *
		 */
		
		data->overlay = SDL_CreateYUVOverlay(data->our_info->width,
						     data->our_info->height,
						     SDL_YV12_OVERLAY,
						     data->surface);
	      }
	    else
	      {
		XMPS_DEBUG("fullscreen : NULL surface!");

		SDL_QuitSubSystem(SDL_INIT_VIDEO);
		return 0;
	      }

	    /* 
	     * and update the info 
	     *
	     */
	    
	    data->fullscreen_width  = data->our_info->width;
	    data->fullscreen_height = data->our_info->height;


	    data->our_info->width  = full_width;
	    data->our_info->height = data->codec_info->height*full_width/data->codec_info->width;
	    data->our_info->x      = 0;
	    data->our_info->y      = (full_height - data->our_info->height)/2;

	    /*
	     * finally remember that we are in fullscreen
	     *
	     */
	    
	    data->fullscreen = 1;
	    SDL_ShowCursor(0);
	    
	  }
          else {
	    if(data->surface != NULL) {
	      
	      /*
	       * here we are in standard RGB
	       *
	       */

	      XMPS_DEBUG("fullscreen RGB");
	      
	      /* 
	       * try to see the better way not to strech the image 
	       *
	       */

              if(data->display_surface != NULL)
                {
                  display = 1;
                  SDL_FreeSurface(data->display_surface);
                }
	      
              SDL_FreeSurface(data->surface);
	      
	      /*
	       * unset the WINDOW_ID
	       *
	       */

              SDL_QuitSubSystem(SDL_INIT_VIDEO);
              unsetenv("SDL_WINDOWID");
              SDL_InitSubSystem(SDL_INIT_VIDEO);

	      /*
	       * and re-create all the needed surfaces
	       *
	       */
	      
              if(display)
                {
                  data->display_surface = SDL_SetVideoMode(data->our_info->width,
							   data->our_info->height,
							   data->our_info->format.bpp,
							   SDL_HWSURFACE | SDL_FULLSCREEN);
		  
                  switch(data->codec_info->format.bpp)
                    {
                    case 16:
                      data->surface = SDL_CreateRGBSurface(SDL_SWSURFACE,
							   data->codec_info->width,
							   data->codec_info->height,
							   data->codec_info->format.bpp,
							   0xF800, 0x07E0, 0x001F, 0x0000);
                      break;
                    case 24:
                      data->surface = SDL_CreateRGBSurface(SDL_HWSURFACE,
							   data->codec_info->width,
							   data->codec_info->height,
							   data->codec_info->format.bpp,
							   0xFF0000, 0x00FF00, 0x0000FF, 0x000000);
                      break;
                    case 32:
                      data->surface = SDL_CreateRGBSurface(SDL_HWSURFACE,
							   data->codec_info->width,
							   data->codec_info->height,
							   data->codec_info->format.bpp,
							   0xFF000000, 0x00FF0000, 0x0000FF00, 0x00000000);
                      break;
                    default:
                      break;
                    }
                }
              else
                {
		  /* 
                   * we do not need software stretch 
                   *
                   */

                  data->surface = SDL_SetVideoMode(data->codec_info->width,
						   data->codec_info->height,
						   data->our_info->format.bpp,
						   SDL_HWSURFACE | SDL_FULLSCREEN);
                }
	      
	      /*
	       * if womething went wrong...
	       *
	       */
	      

              if(data->surface == NULL) {
		XMPS_DEBUG("fullscreen : null surface");
		return 0;
	      }

	      /*
	       * otherwise hide cursor
	       *
	       */
	      
              SDL_ShowCursor(0);
	      
	      /*
	       * and we're done
	       *
	       */
	      
              data->fullscreen = 1;
            }
	  }
	}
      else
        {
          /*
           * we already are in fullscreen
           *
           */

          XMPS_DEBUG("already in fullscreen mode");
        }
    }
    break;

  case XMPS_FLAG_FULLSCREEN_UNSET:
    {
      if( data->fullscreen )
	{
                                
	  /*
	   * we can go
	   *
	   */

	  if(data->overlay != NULL)
	    {
	      /*
	       * YUV
	       *
	       */

	      /*
	       * we eventually reset the WINDOW_ID
	       *
	       */

	      SDL_QuitSubSystem(SDL_INIT_VIDEO);

	      if(data->our_info->windowid != 0)
		{
		  sprintf(data->SDL_windowhack,
			  "SDL_WINDOWID=%d",
			  data->our_info->windowid);

		  putenv(data->SDL_windowhack);
		}

              SDL_InitSubSystem(SDL_INIT_VIDEO);

	      /*
	       * and re-create the surface
	       *
	       */

	      data->surface = SDL_SetVideoMode(data->codec_info->width,
					       data->codec_info->height,
					       data->our_info->format.bpp,
					       SDL_HWSURFACE | SDL_RESIZABLE);
	      if(data->surface == NULL)
		{
		  XMPS_DEBUG("fulscreen : NULL surface re-created!!");
		  
		  SDL_QuitSubSystem(SDL_INIT_VIDEO);
		  return 0;
		}
	      else
		{
		  data->overlay = SDL_CreateYUVOverlay(data->codec_info->width,
						       data->codec_info->height,
						       SDL_YV12_OVERLAY,
						       data->surface);
		}

                /* 
                 * re-update our info 
                 *
                 */

                data->our_info->width  = data->fullscreen_width;
                data->our_info->height = data->fullscreen_height;
                data->our_info->x = data->codec_info->x;
                data->our_info->y = data->codec_info->y;

                /*
                 * and we're done
                 *
                 */

                data->fullscreen = 0;
		SDL_ShowCursor(1);

              }
            else
              {
                /*
                 * RGB
                 *
                 */


                if(data->display_surface != NULL)
                  {
                    display = 1;
                    SDL_FreeSurface(data->display_surface);
                  }

                /*
                 * eventually re-set the WINDOW_ID
                 *
                 */

                SDL_FreeSurface(data->surface);

                SDL_QuitSubSystem(SDL_INIT_VIDEO);

                if(data->our_info->windowid != 0)
                  {
                    sprintf(data->SDL_windowhack,"SDL_WINDOWID=%d", data->our_info->windowid);
                    putenv(data->SDL_windowhack);
                  }
                SDL_InitSubSystem(SDL_INIT_VIDEO);

                /*
                 * and re-cretae the surfaces
                 *
                 */

                if(display)
                  {
                    data->display_surface = SDL_SetVideoMode(data->our_info->width + data->our_info->x,
							     data->our_info->height + data->our_info->y,
							     data->our_info->format.bpp,
							     SDL_HWSURFACE | SDL_RESIZABLE);

                    switch(data->codec_info->format.bpp)
                      {
                      case 16:
                        data->surface = SDL_CreateRGBSurface(SDL_SWSURFACE,
							     data->codec_info->width,
							     data->codec_info->height,
							     data->codec_info->format.bpp,
							     0xF800, 0x07E0, 0x001F, 0x0000);
                        break;
                      case 24:
                        data->surface = SDL_CreateRGBSurface(SDL_HWSURFACE,
							     data->codec_info->width,
							     data->codec_info->height,
							     data->codec_info->format.bpp,  
							     0xFF0000, 0x00FF00, 0x0000FF, 0x000000);
                      case 32:
                        data->surface = SDL_CreateRGBSurface(SDL_HWSURFACE,
							     data->codec_info->width,
							     data->codec_info->height,
							     data->codec_info->format.bpp,  
							     0xFF000000, 0x00FF0000, 0x0000FF00, 0x00000000);
                        break;
                      default:
                        break;
                      }

                  }
                else
                  {
                    data->surface = SDL_SetVideoMode(data->codec_info->width + data->our_info->x,
						     data->codec_info->height + data->our_info->y,
						     data->our_info->format.bpp,
						     SDL_HWSURFACE | SDL_RESIZABLE);
                  }

                /*
                 * we show the cursor again
                 *
                 */

                SDL_ShowCursor(1);
		
		/*
		 * and that's all
                 *
                 */

                data->fullscreen = 0;
              }
          }
        else
          {
                                
	    /*
	     * we are not infullscreen
	     *
	     */

            XMPS_DEBUG("not in fullscreen");
          }


      
    }
    break;

  default:
    break;

  }  

  return 0;
}

/*
 * sdl_draw_frame
 *
 * - performs video update.
 *
 */

unsigned int sdl_draw(xmps_video_renderer_plugin_t *video_renderer, void *image)
{
  
  sdl_renderer_t *data;

  data = (sdl_renderer_t *) video_renderer->data;

  /*
   * if in fullscreen we check for events
   */ 

  if(data->fullscreen) {

    SDL_Event event;

    /*
     * do all wainting events
     *
     */

    while(SDL_PollEvent(&event))
      {
	switch(event.type)
	  {
	  case SDL_MOUSEMOTION:
	    
	    SDL_ShowCursor(1);
	    break;

	  case SDL_KEYDOWN:
	    switch(event.key.keysym.sym)
	      {
	      case SDLK_ESCAPE:
	      case SDLK_f:
		
		push_xmps_event(XMPS_EVENT_BACK_FROM_FULLSCREEN);
		break;
	      default:
		break;
	      }
	  }
      }
  }

  if(data->overlay != NULL) {

    /*
     * YUV
     *
     * overlay display setup
     *
     */
    
    data->r.x = data->our_info->x;
    data->r.y = data->our_info->y;
    data->r.w = data->our_info->width;
    data->r.h = data->our_info->height;
    
    SDL_DisplayYUVOverlay(data->overlay, &data->r);
    
    return 1;

  }
  else {
    
    /*
     * RGB
     *
     */

      if(data->display_surface != NULL)
	{

	  /*
	   * we are in software stretch
	   */

	  SDL_LockSurface(data->surface);
	  SDL_LockSurface(data->display_surface);

	  if(data->fullscreen)
	    {
	      AS_stretchBlit(data->surface->pixels, 
			     data->display_surface->pixels,
			     data->codec_info->width, 
			     data->codec_info->height, 
			     data->our_info->width, 
			     data->our_info->height, 
			     data->our_info->format.bpp);
	    }
	  else
	    {
	      AS_stretchBlit(data->surface->pixels,
			     data->display_surface->pixels + 
			     (data->our_info->y*data->our_info->width)*data->our_info->format.bpp/8,
			     data->codec_info->width, 
			     data->codec_info->height, 
			     data->our_info->width, 
			     data->our_info->height, 
			     data->our_info->format.bpp);
	    }

	  SDL_UnlockSurface(data->display_surface);
	  SDL_UnlockSurface(data->surface);

	  /*
	   * now update the screen
	   *
	   */

	  if ( data->display_surface->flags & SDL_DOUBLEBUF ) 
	    {
	      SDL_Flip(data->display_surface);
	    } 
	  else 
	    {
	      if(data->fullscreen)
		SDL_UpdateRect(data->display_surface, 
			       0, 
			       0, 
			       data->our_info->width, 
			       data->our_info->height);
	      else
		SDL_UpdateRect(data->display_surface, 
			       data->our_info->x, 
			       data->our_info->y, 
			       data->our_info->width, 
			       data->our_info->height);
	    }
	}
      else
	{
	  /*
	   * we are in normal mode
	   *
	   */

	  if ( data->surface->flags & SDL_DOUBLEBUF ) 
	    {
	      SDL_Flip(data->surface);
	    } 
	  else 
	    {
	      if(data->fullscreen)
		{
		  SDL_UpdateRect(data->surface, 0, 0, data->our_info->width, data->our_info->height);
		}
	      else
		{
		  /*
		   * check for -> software decal
		   *
		   */

		  if(data->our_info->x != 0)
		    {
		      AS_copy_decal(data->decal_surface->pixels, 
				    data->surface->pixels, 
				    data->our_info->x, 
				    data->our_info->y,
				    data->our_info->width, 
				    data->our_info->height, 
				    data->our_info->format.bpp);
		    }

		  /*
		   * and update
		   *
		   */

		  SDL_UpdateRect(data->surface, 
				 data->our_info->x, 
				 data->our_info->y, 
				 data->our_info->width, 
				 data->our_info->height);
		}
	    }
	}
    }

  return 1;
}

/*
 * sdl_close
 *
 * - performs all needed cleanup.
 *
 */

unsigned int sdl_close(xmps_video_renderer_plugin_t *video_renderer)
{
  sdl_renderer_t *data;

  if(video_renderer == NULL)
    return 0;
  
  data = (sdl_renderer_t *) video_renderer->data;
  
  if(data == NULL)
    return 0;  
    
  /*
   * free all the surfaces
   *
   */

  if(data->surface != NULL)
    {
      SDL_FreeSurface(data->surface);
      data->surface = NULL;
    }
  
  if(data->display_surface != NULL)
    {
      SDL_FreeSurface(data->display_surface);
      data->display_surface = NULL;
    }
  
  if(data->overlay != NULL)
    {
      SDL_FreeYUVOverlay(data->overlay);
      data->overlay = NULL;
    }
  
  /*
   * and quit
   *
   */

  SDL_QuitSubSystem(SDL_INIT_VIDEO);
  
  return 1;
}


