/*******************************************************************
*                                                                  *
*             This software is part of the ast package             *
*                Copyright (c) 1984-2001 AT&T Corp.                *
*        and it may only be used by you under license from         *
*                       AT&T Corp. ("AT&T")                        *
*         A copy of the Source Code Agreement is available         *
*                at the AT&T Internet web site URL                 *
*                                                                  *
*       http://www.research.att.com/sw/license/ast-open.html       *
*                                                                  *
*        If you have copied this software without agreeing         *
*        to the terms of the license you are infringing on         *
*           the license and copyright and are violating            *
*               AT&T's intellectual property rights.               *
*                                                                  *
*                 This software was created by the                 *
*                 Network Services Research Center                 *
*                        AT&T Labs Research                        *
*                         Florham Park NJ                          *
*                                                                  *
*               Glenn Fowler <gsf@research.att.com>                *
*******************************************************************/
#pragma prototyped
/*
 * Glenn Fowler
 * AT&T Research
 *
 * make - maintain and update programs
 *
 * history
 *
 *   1970's                   pre-make           (research)
 *                                .
 *                                .
 *   1976                       make             S. I. Feldman
 *                              .  .
 *                            .     .
 *                          .        .
 *                        .           .
 *   1978        augmented-make        .         E. G. Bradford
 *                    . . .             .
 *                  .   .   .           .
 *                .     .     .         .
 *              .       .       .       .
 *   1979     build     .         .     .        V. B. Erickson, J. F. Pellegrin
 *              .       .           .   .
 *              .       .             . .
 *   1984       .       .            V8-make     S. I. Feldman
 *              .       .             . .
 *              .       .           .   .
 *              .       .         .     .
 *              .       .       .       .
 *              .       .     .         .
 *              .       .   .           .
 *              .       . .             .
 *   1985       .     nmake             .        G. S. Fowler
 *              .     . .               .
 *              .   .   .               .
 *              . .     .               .
 *   1986    LC-nmake   .               .        J. E. Shopiro, M. S. Freeland
 *              . .     .               . 
 *              .   .   .               mk       A. G. Hume
 *              .     . .             . .
 *   1987       .     nmake         .   .        G. S. Fowler
 *              .       .         .     .
 *            (1.4)     .       .       .
 *                .     .     .         .
 *                  .   .   .           .
 *                    . . .             .
 *   1988             nmake             .        G. S. Fowler
 *                      .               .
 *                    (2.0)             .
 *                      .               .
 *   1990             (2.1)             .
 *                      .               .
 *   1991             (2.2)             .
 *                      .               .
 *   1992             (2.3)             .
 *                      .               .
 *   1993             (3.0)             .
 *                      .               .
 *   1994             (3.1)             .
 *                      .               .
 *   1995             (3.2)             .	AT&T Bell Labs
 *                      .               .
 *   1996             (3.3)             .	AT&T Research
 *                      .               .
 *   1997             (3.4)             .	AT&T Research
 *                      .               .
 *   1998             (3.5)             .	AT&T Research
 *                      .               .
 *   1999             (3.6)             .	AT&T Research
 *                      .               .
 *   2000             (4.0)             .	AT&T Research
 *                      .               .
 *   2001             (4.1)             .	AT&T Research
 *
 * command line arguments are of three types
 *
 *	make [ [ option ] [ script ] [ target ] ... ]
 *
 * options are described in options.h
 *
 * debug trace levels are controlled by error_info.trace
 * debug level n enables tracing for all levels less than or equal to n
 * levels 4 and higher are conditionally compiled with DEBUG!=0
 * if debug level < 20 then all levels are disabled during early bootstrap
 * if debug level = 1 then all levels are disabled until after .INIT is made
 *
 *	level	trace
 *	  0	no trace
 *	  1	basic make trace
 *	  2	major actions
 *	  3	option settings, make object files
 *	  4	state variables
 *	  5	directory and archive scan, some OPT_explain'ations
 *	  6	coshell commands and messages
 *	  7	makefile input lines
 *	  8	lexical analyzer states
 *	  9	metarules applied
 *	 10	variable expansions
 *	 11	bindfile() trace
 *	 12     files added to hash
 *	 13	new atoms
 *	 14	hash table usage
 *		...
 *	 20	bootstrap trace
 *
 * state.questionable registry (looks like it should go but not sure)
 *
 *	0x00000001 *temporary*
 *	0x00000002 *temporary*
 *	0x00000004 *temporary*
 *	0x00000008 1996-10-11 meta rhs prefix dir check even if rhs has prefix
 *	0x00000010 1994-01-01 old out of date if prereq in higher view
 *	0x00000020 1994-01-01 old scan advance logic
 *	0x00000040 1994-08-11 foiled alias means not found
 *	0x00000080 1994-08-11 don't catrule() ../.. dir prefixes in bindfile()
 *	0x00000100 1994-10-01 metaclose() recursion even if rule.action!=0
 *	0x00000200 1994-11-11 disable generate() % transformation check
 *	0x00000400 1995-01-19 forceread even if global
 *	0x00000800 1995-07-17 D_global if not P_target in bindfile()
 *	0x00001000 1995-07-17 less agressive aliasing in bindfile()
 *	0x00002000 1995-07-17 less agressive putbound() in bindalias()
 *	0x00004000 1995-11-11 P_target && D_dynamic does not imply generated
 *	0x00008000 1996-02-29 :P=D: returns at most one dir per item
 *	0x00010000 1996-10-11 don't alias check path suffixes in bindfile()
 *	0x00020000 1998-11-11 don't force require_p make() recheck
 *	0x00040000 1999-09-07 generate() intermediates only if (sep & LT)
 *	0x00080000 1999-11-19 disable touch steady state loop
 *	0x00100000 2000-02-14 apply metarule even if dir on unbound lhs
 *	0x00200000 2000-09-08 0x00100000 implied if !P_implicit
 *
 * state.test registry (conditionally compiled with DEBUG!=0)
 *
 *	0x00000001 *temporary*
 *	0x00000002 *temporary*
 *	0x00000004 *temporary*
 *	0x00000008 bindfile() directory prune trace
 *	0x00000010 expand() trace
 *	0x00000020 force external archive table of contents generation
 *	0x00000040 bindalias() trace
 *	0x00000080 scan state trace
 *	0x00000100 statetime() trace
 *	0x00000200 staterule() view trace
 *	0x00000400 more staterule() view trace
 *	0x00000800 mergestate() trace
 *	0x00001000 job status trace
 *	0x00002000 dump Vmheap stats on exit
 *	0x00004000 dump atom address with name
 *	0x00008000 force state file garbage collection
 *	0x00010000 alarm status trace
 *	0x00020000 close internal.openfd before job exec
 */

#include "make.h"
#include "options.h"

#include <sig.h>
#include <stak.h>
#include <vecargs.h>
#include <vmalloc.h>

#define settypes(c,t)	for(s=c;*s;settype(*s++,t))

#ifndef PATHCHECK
#define PATHCHECK	IDNAME
#endif

/*
 * global initialization -- external engine names are defined in initrule()
 */

struct internal		internal;	/* internal rules and vars	*/
struct state		state;		/* program state		*/
struct tables		table;		/* hash table info		*/
char			null[] = "";	/* null string			*/
char			tmpname[MAXNAME];/* temporary name buffer	*/
short			ctypes[UCHAR_MAX+1];/* istype() character types	*/

#if __OBSOLETE__ < 20020101
#include "../../lib/libast/string/fmtident.c"
#endif

/*
 * user error message intercept
 */

static int
intercept(Sfio_t* sp, int level, int flags)
{
	register struct rule*	r;
	char*			m;
	char*			s;
	char*			t;
	int			n;
	int			i;
	Sfio_t*			tmp;

	NoP(sp);
	NoP(flags);
	if ((state.mam.level = level) > 0 && !state.hold && (r = internal.error) && (r->property & (P_functional|P_target)) == (P_functional|P_target) && !state.compileonly && !state.interrupt && (m = stakptr(0)) && (n = staktell()) > 0)
	{
		/*
		 * make the error trap
		 */

		state.hold = m;
		while (*m && *m != ':')
			m++;
		while (isspace(*++m));
		n -= m - state.hold;
		tmp = sfstropen();
		sfprintf(tmp, "%d %-.*s", level, n, m);
		s = sfstruse(tmp);

		/*
		 * return [ level | - [ message ] ]
		 * level is new level or - to retain old
		 * omitted message means it has been printed
		 */

		if (t = call(r, s))
		{
			if (*t == '-' && isspace(*(t + 1)))
				t += 2;
			else if (i = strtol(t, &t, 0))
				level = i;
			while (isspace(*t))
				t++;
		}
		if (!t || !*t)
		{
			level |= ERROR_OUTPUT;
			message((-1, "suppressed %s message: `%-.*s'", level == 1 ? "warning" : "error", n, m));
		}
		sfstrclose(tmp);
		state.hold = 0;
	}
	return level;
}

/*
 * make entry point
 */

int
main(int argc, char** argv)
{
	register char*		s;
	register struct rule*	r;
	register struct list*	p;
	int			i;
	int			args;
	int			debug;
	char*			t;
	char*			buf;
	char*			tok;
	struct var*		v;
	struct stat		st;
	struct stat		ds;
	Sfio_t*			tmp;

	/*
	 * initialize dynamic globals
	 */

	version = strdup(fmtident(version));
	setlocale(LC_ALL, "");
	error_info.id = idname;
	error_info.version = version;
	error_info.exit = finish;
	error_info.auxilliary = intercept;
	if (pathcheck(PATHCHECK, error_info.id, NiL)) exit(1);
	settypes("*?[]", C_MATCH);
	settypes("+-|=", C_OPTVAL);
	settypes(" \t\n", C_SEP);
	settype(0, C_SEP);
	settypes(" \t\v\n:+&=;\"\\", C_TERMINAL);
	settype(0, C_TERMINAL);
	settypes("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_", C_ID1|C_ID2|C_VARIABLE1|C_VARIABLE2);
	settypes(".", C_VARIABLE1|C_VARIABLE2);
	settypes("0123456789", C_ID2|C_VARIABLE2);

	/*
	 * close garbage fd's -- we'll be tidy from this point on
	 * 3 may be /dev/tty on some systems
	 * 0..9 for user redirection in shell
	 * 10..19 left open by bugs in some shells
	 * any close-on-exec fd's must have been done for us
	 */

	i = 3;
	if (isatty(i))
		i++;
	for (; i < 20; i++)
		if (!fcntl(i, F_GETFD, 0))
			close(i);

	/*
	 * initialize the hash tables
	 */

	inithash();

	/*
	 * allocate the very temporary buffer streams
	 */

	internal.met = sfstropen();
	internal.nam = sfstropen();
	internal.tmp = sfstropen();
	internal.val = sfstropen();
	tmp = sfstropen();
	sfstrrsrv(tmp, 2 * MAXNAME);

	/*
	 * set the default state
	 */

	state.alias = 1;
	state.exec = 1;
	state.global = 1;
	state.init = 1;
#if DEBUG
	state.intermediate = 1;
#endif
	state.io[0] = sfstdin;
	state.io[1] = sfstdout;
	state.io[2] = sfstderr;
	state.jobs = 1;
	state.pid = getpid();
	state.readstate = MAXVIEW;
	state.scan = 1;
	state.start = CURTIME;
	state.stateview = -1;
	state.tabstops = 8;
	state.targetview = -1;
#if BINDINDEX
	state.view[0].path = makerule(".");
#else
	state.view[0].path = ".";
#endif
	state.view[0].pathlen = 1;
	state.writeobject = 1;
	state.writestate = 1;

	/*
	 * pwd initialization
	 *
	 * for project management, if . is group|other writeable
	 * then change umask() for similar write protections
	 */

	buf = sfstrbase(tmp);
	internal.pwd = (s = getcwd(buf, MAXNAME)) ? strdup(s) : ".";
	internal.pwdlen = strlen(internal.pwd);
	if (stat(".", &st))
		error(3, "cannot stat .");
	if (S_ISDIR(st.st_mode) && (st.st_mode & (S_IWGRP|S_IWOTH)))
		umask(umask(0) & ~(st.st_mode & (S_IWGRP|S_IWOTH)));

	/*
	 * set some variable default values
	 */

	hashclear(table.var, HASH_ALLOCATE);
	setvar(external.make, argv[0], V_import);
	t = "lib/make";
	setvar(external.lib, strdup((s = pathpath(buf, t, argv[0], PATH_EXECUTE)) ? s : t), V_import);
	setvar(external.pwd, internal.pwd, V_import);
	setvar(external.version, version, V_import);
	hashset(table.var, HASH_ALLOCATE);

	/*
	 * read the environment
	 */

	readenv();
	if (v = getvar(external.nproc))
		state.jobs = (int)strtol(v->value, NiL, 0);
	if ((v = getvar(external.pwd)) && !streq(v->value, internal.pwd))
	{
		if (!stat(v->value, &st) && !stat(internal.pwd, &ds) && st.st_ino == ds.st_ino && st.st_dev == ds.st_dev)
		{
			internal.pwd = strdup(v->value);
			internal.pwdlen = strlen(v->value);
		}
		else
		{
			v->property &= ~V_import;
			v->property |= V_free;
			v->value = strdup(internal.pwd);
		}
	}

	/*
	 * initialize the internal rule pointers
	 */

	initrule();

	/*
	 * read the static initialization script
	 */

	sfputr(tmp, initstatic, -1);
	parse(NiL, sfstruse(tmp), "initstatic", 0);

	/*
	 * check and read the args file
	 */

	if (s = colonlist(tmp, external.args, ' '))
	{
		i = fs3d(0);
		tok = tokopen(s, 1);
		while (s = tokread(tok))
			if (vecargs(vecfile(s), &argc, &argv) >= 0)
				break;
			else if (errno != ENOENT)
				error(1, "cannot read args file %s", s);
		tokclose(tok);
		fs3d(i);
	}
	state.argf = newof(0, int, argc, 0);

	/*
	 * set the command line options
	 * read the command line assignments
	 * mark the command line scripts and targets
	 */

	state.init = 0;
	state.readonly = 1;
	state.argv = argv;
	state.argc = argc;
	args = scanargs(state.argc, state.argv, state.argf);
	state.readonly = 0;
	state.init = 1;
	if (state.base)
		state.readstate = 0;
	if (state.compileonly)
	{
		state.forceread = 1;
		state.virtualdot = 0;
	}
#if _UWIN
	state.test ^= 0x00020000;
#endif

	/*
	 * tone down the bootstrap noise
	 */

	if ((debug = error_info.trace) == -1)
		error_info.trace = 0;

	/*
	 * check explicit environment overrides
	 */

	if (s = colonlist(tmp, external.import, ' '))
	{
		tok = tokopen(s, 1);
		while (s = tokread(tok))
		{
			if (i = *s == '!')
				s++;
			if (v = getvar(s))
			{
				if (i)
					v->property &= ~V_import;
				else
					v->property |= V_readonly;
			}
		}
		tokclose(tok);
	}

	/*
	 * set up the traps
	 */

	inittrap();

	/*
	 * announce the version
	 */

	if (error_info.trace < 0)
	{
		errno = 0;
		error(error_info.trace, "%s [%d %s]", version, getpid(), strtime(state.start));
	}

	/*
	 * initialize the views
	 */

	state.global = 0;
	state.user = 1;
	initview();

	/*
	 * check for mam
	 */

	if (state.mam.out)
	{
		if (!state.mam.statix || *state.mam.label)
			error_info.write = mamerror;
		sfprintf(state.mam.out, "%sinfo mam %s %05d 1994-07-17 %s\n", state.mam.label, state.mam.type, state.mam.parent, version);
		if (state.mam.regress)
			sfprintf(state.mam.out, "%sinfo start regression\n", state.mam.label);
		else if (!state.mam.statix || *state.mam.label)
		{
			sfprintf(state.mam.out, "%sinfo start %lu\n", state.mam.label, CURTIME);
			if (!state.mam.root || streq(state.mam.root, internal.pwd))
				sfprintf(state.mam.out, "%sinfo pwd %s\n", state.mam.label, internal.pwd);
			else
				sfprintf(state.mam.out, "%sinfo pwd %s %s\n", state.mam.label, state.mam.root, mamname(makerule(internal.pwd)));
			buf = sfstrbase(tmp);
			if (state.fsview && !mount(NiL, buf, FS3D_GET|FS3D_ALL|FS3D_SIZE(sfstrsize(tmp)), NiL))
				sfprintf(state.mam.out, "%sinfo view %s\n", state.mam.label, buf);
		}
	}

	/*
	 * read the dynamic initialization script
	 */

	if ((i = error_info.trace) > -20)
		error_info.trace = 0;
	sfputr(tmp, initdynamic, -1);
	parse(NiL, sfstruse(tmp), "initdynamic", 0);
	error_info.trace = i;
	state.user = 0;
	state.init = 0;

	/*
	 * read the explicit makefiles
	 * readfile() handles the base and global rules
	 *
	 * NOTE: internal.tmplist is used to handle the effects of
	 *	 load() on internal list pointers
	 */

	compref(0, NiL, 0);
	if (p = internal.makefiles->prereqs)
	{
		p = internal.tmplist->prereqs = listcopy(p);
		for (; p; p = p->next)
			readfile(p->rule->name, COMP_FILE, NiL);
		freelist(internal.tmplist->prereqs);
		internal.tmplist->prereqs = 0;
	}

	/*
	 * if no explicit makefiles then try external.{convert,files}
	 */

	if (!state.makefile)
	{
		int	sep;
		Sfio_t*	imp;

		imp = sfstropen();
		sep = 0;
		s = 0;
		if (*(t = getval(external.convert, 1)))
		{
			sfputr(tmp, t, 0);
			sfstrrsrv(tmp, MAXNAME);
			i = sfstrtell(tmp);
			tok = tokopen(sfstrbase(tmp), 0);
			while (s = tokread(tok))
			{
				if (*s == '*' && *(s + 1) == '.')
				{
					edit(tmp, internal.pwd, DELETE, KEEP, s + 1);
					sfputc(tmp, 0);
					s = sfstrset(tmp, i);
				}
				if (readfile(s, COMP_INCLUDE|COMP_DONTCARE, NiL))
					break;
				if (sep)
					sfputc(imp, ',');
				else
					sep = 1;
				sfputr(imp, s, -1);
				if (!(s = tokread(tok)))
					break;
			}
			tokclose(tok);
			sfstrset(tmp, 0);
		}
		if (!s && (s = colonlist(tmp, external.files, ' ')))
		{
			tok = tokopen(s, 1);
			while (s = tokread(tok))
			{
				if (readfile(s, COMP_INCLUDE|COMP_DONTCARE, NiL))
					break;
				if (sep)
					sfputc(imp, ',');
				else
					sep = 1;
				sfputr(imp, s, -1);
			}
			tokclose(tok);
		}
		if (!s)
		{
			if (*(s = sfstruse(imp)))
				error(3, "a makefile must be specified when %s omitted", s);
			else
				error(3, "a makefile must be specified");
		}
		sfstrclose(imp);
	}

	/*
	 * validate external command line options
	 */

	optcheck();

	/*
	 * check for listing of variable and rule definitions
	 */

	if (state.list)
	{
		dump(sfstdout, 0);
		exit(0);
	}

	/*
	 * check if makefiles to be compiled
	 */

	if (state.compile && !state.virtualdot && state.writeobject && state.writestate)
	{
		/*
		 * make the compinit trap
		 */

		if (r = getrule(external.compinit))
		{
			state.reading = 1;
			maketop(r, P_dontcare|P_foreground, NiL);
			state.reading = 0;
		}
		if (state.exec && state.objectfile)
		{
			message((-2, "compiling makefile input into %s", state.objectfile));
			compile(state.objectfile, NiL);
		}

		/*
		 * make the compdone trap
		 */

		if (r = getrule(external.compdone))
		{
			state.reading = 1;
			maketop(r, P_dontcare|P_foreground, NiL);
			state.reading = 0;
		}
	}

	/*
	 * makefile read cleanup
	 */

	if (state.compileonly)
		exit(0);
	compref(0, NiL, 0);
	sfstrclose(tmp);
	state.compile = COMPILED;
	if (state.believe)
	{
		if (!state.maxview)
			state.believe = 0;
		else if (state.fsview)
			error(3, "%s: option currently works in 2d only", opentry(OPT_believe, 0)->name);
	}

	/*
	 * read the state file
	 */

	readstate(state.statefile);

	/*
	 * place the command line targets in internal.args
	 */

	if (internal.main->dynamic & D_dynamic)
		dynamic(internal.main);
	internal.args->prereqs = p = 0;
	for (i = args; i < state.argc; i++)
		if (state.argf[i] & ARG_TARGET)
		{
			struct list*	q;

			q = cons(makerule(state.argv[i]), NiL);
			if (p)
				p = p->next = q;
			else
				internal.args->prereqs = p = q;
		}

	/*
	 * the engine bootstrap is complete -- start user activities
	 */

	state.user = 1;

	/*
	 * make the makeinit trap
	 */

	if (r = getrule(external.makeinit))
		maketop(r, P_dontcare|P_foreground, NiL);

	/*
	 * read the command line scripts
	 */

	for (i = args; i < state.argc; i++)
		if (state.argf[i] & ARG_SCRIPT)
		{
			state.reading = 1;
			parse(NiL, state.argv[i], "command line script", 0);
			state.reading = 0;
		}

	/*
	 * make the init trap
	 */

	if (r = getrule(external.init))
		maketop(r, P_dontcare|P_foreground, NiL);

	/*
	 * freeze the parameter files and candidate state variables
	 */

	state.user = 2;
	candidates();

	/*
	 * internal.args default to internal.main
	 */

	if (!internal.args->prereqs && internal.main->prereqs)
		internal.args->prereqs = listcopy(internal.main->prereqs);

	/*
	 * turn up the volume again
	 */

	if (!error_info.trace)
		error_info.trace = debug;

	/*
	 * make the prerequisites of internal.args
	 */

	if (internal.args->prereqs)
		while (internal.args->prereqs)
		{
			/*
			 * we explicitly allow internal.args modifications
			 */

			r = internal.args->prereqs->rule;
			internal.making->prereqs = internal.args->prereqs;
			internal.args->prereqs = internal.args->prereqs->next;
			internal.making->prereqs->next = 0;
			maketop(r, 0, NiL);
		}
	else if (state.makefile)
		error(3, "%s: a main target must be specified", state.makefile);

	/*
	 * finish up
	 */

	finish(0);
	exit(0);
}

/*
 * clean up and exit
 */

void
finish(int n)
{
	int		i;
	struct rule*	r;

	/*
	 * old error intercept
	 */

	if (!state.hold && (r = internal.error) && (r->property & (P_target|P_functional)) == P_target)
	{
		state.hold = null;
		if (n && error_info.errors && !state.compileonly && !state.interrupt)
		{
			if (r->status == NOTYET)
				maketop(r, P_dontcare|P_foreground, NiL);
			state.hold = 0;
			if (r->status == EXISTS)
			{
				r->status = NOTYET;
				return;
			}
		}
	}

	/*
	 * children exit without cleanup
	 */

	if (getpid() != state.pid)
		_exit(n);
	unparse(0);
	switch (state.finish)
	{

	case 0:
		/*
		 * disable listing and make the done trap
		 */

		state.finish++;
		alarm(0);
		state.list = 0;
		message((-1, "%s cleanup", state.interrupt ? "interrupt" : n > 0 ? "error" : "normal"));
		if (!state.compileonly && (r = getrule(external.done)))
			maketop(r, P_dontcare, NiL);
		/*FALLTHROUGH*/

	case 1:
		/*
		 * make the makedone trap
		 */

		state.finish++;
		if (!state.compileonly && (r = getrule(external.makedone)))
			maketop(r, P_dontcare, NiL);
		/*FALLTHROUGH*/

	case 2:
		/*
		 * wait for any job(s) to finish
		 */

		state.finish++;
		complete(NiL, NiL, NiL, 0);
		/*FALLTHROUGH*/

	case 3:
		/*
		 * put all jobs in foreground and save state
		 */

		state.finish++;
		state.jobs = 0;
		savestate();
		/*FALLTHROUGH*/

	case 4:
		/*
		 * clean up temporaries
		 */

		state.finish++;
		remtmp(1);
		/*FALLTHROUGH*/

	case 5:
		/*
		 * drop the coshell
		 */
		
		state.finish++;
		drop();
		/*FALLTHROUGH*/

	case 6:
		/*
		 * dump
		 */

		state.finish++;
		if (state.test & 0x00002000)
		{
			Vmstat_t	vs;

			vmstat(Vmheap, &vs);
			error(0, "vm region %lu segments %lu busy %lu:%lu:%lu free %lu:%lu:%lu", vs.extent, vs.n_seg, vs.n_busy, vs.s_busy, vs.m_busy, vs.n_free, vs.s_free, vs.m_free);
		}
		dump(sfstdout, error_info.trace <= -14);
		/*FALLTHROUGH*/

	case 7:
		/*
		 * final output
		 */

		state.finish++;
		if (state.errors && state.keepgoing)
		{
			error(2, "*** %d action%s failed", state.errors, state.errors == 1 ? null : "s");
			if (!n) n = 1;
		}
		if (state.mam.out)
		{
			if (state.mam.regress)
				sfprintf(state.mam.out, "%sinfo finish regression\n", state.mam.label);
			else if (state.mam.dynamic || *state.mam.label)
				sfprintf(state.mam.out, "%sinfo finish %lu %d\n", state.mam.label, CURTIME, n);
		}
		for (i = 0; i < elementsof(state.io); i++)
			if (state.io[i] != sfstdin && state.io[i] != sfstdout && state.io[i] != sfstderr)
				sfclose(state.io[i]);
		message((-1, "%s exit", state.interrupt ? "interrupt" : n ? "error" : "normal"));
		break;

	}
	if (state.interrupt)
	{
		n = 3;
		signal(state.interrupt, SIG_DFL);
		kill(getpid(), state.interrupt);
		pause();
	}
	exit(n);
}
