/* Copyright (C) 2000 MySQL AB & MySQL Finland AB & TCX DataKonsult AB
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* Testing of the basic functions of a MyISAM table */

#include "myisam.h"
#include <getopt.h>
#include <m_ctype.h>


#define MAX_REC_LENGTH 1024

static int rec_pointer_size=0,verbose=0,flags[50];
static int key_field=FIELD_SKIPP_PRESPACE,extra_field=FIELD_SKIPP_ENDSPACE;
static int key_type=HA_KEYTYPE_NUM;
static int create_flag=0;

static uint insert_count= 1000,update_count=1000,remove_count=1000;
static uint pack_keys=0,pack_seg=0,null_fields=0,key_length=6,skip_update=0;
static uint unique_key=HA_NOSAME,key_cacheing=0,opt_unique=0;
static uint silent;
static MI_COLUMNDEF recinfo[4];
static MI_KEYDEF keyinfo[10];
static MI_KEYSEG keyseg[10];
static MI_KEYSEG uniqueseg[10];

static int run_test(const char *filename);
static void get_options(int argc, char *argv[]);
static void create_key(char *key,uint rownr);
static void create_record(char *record,uint rownr);
static void update_record(char *record);

int main(int argc,char *argv[])
{
  MY_INIT(argv[0]);
  my_init();
  if (key_cacheing)
    init_key_cache(IO_SIZE*16,(uint) IO_SIZE*4*10);
  get_options(argc,argv);

  exit(run_test("test1"));
}


int run_test(const char *filename)
{
  MI_INFO *file;
  int i,j,error,deleted,rec_length,uniques=0;
  ha_rows found,row_count;
  my_off_t pos;
  char record[MAX_REC_LENGTH],key[MAX_REC_LENGTH],read_record[MAX_REC_LENGTH];
  MI_UNIQUEDEF uniquedef;
  MI_CREATE_INFO create_info;

  bzero((char*) recinfo,sizeof(recinfo));

  /* First define 2 columns */
  recinfo[0].type=FIELD_NORMAL; recinfo[0].length=1; /* For NULL bits */
  recinfo[1].type=key_field;
  recinfo[1].length= (key_field == FIELD_BLOB ? 4+mi_portable_sizeof_char_ptr :
		      key_length);
  if (key_field == FIELD_VARCHAR)
    recinfo[1].length+=2;
  recinfo[2].type=extra_field;
  recinfo[2].length= (extra_field == FIELD_BLOB ? 4 + mi_portable_sizeof_char_ptr : 24);
  if (extra_field == FIELD_VARCHAR)
    recinfo[2].length+=2;
  if (opt_unique)
  {
    recinfo[3].type=FIELD_CHECK;
    recinfo[3].length=MI_UNIQUE_HASH_LENGTH;
  }
  rec_length=recinfo[0].length+recinfo[1].length+recinfo[2].length+
    recinfo[3].length;


  /* Define a key over the first column */
  keyinfo[0].seg=keyseg;
  keyinfo[0].keysegs=1;
  keyinfo[0].seg[0].type= key_type;
  keyinfo[0].seg[0].flag= pack_seg;
  keyinfo[0].seg[0].start=1;
  keyinfo[0].seg[0].length=key_length;
  keyinfo[0].seg[0].null_bit= null_fields ? 2 : 0;
  keyinfo[0].seg[0].null_pos=0;
  keyinfo[0].seg[0].language=MY_CHARSET_CURRENT;
  if (pack_seg & HA_BLOB_PART)
  {
    keyinfo[0].seg[0].bit_start=4;		/* Length of blob length */
  }
  keyinfo[0].flag = (uint8) (pack_keys | unique_key);

  bzero((byte*) flags,sizeof(flags));
  if (opt_unique)
  {
    uint start;
    uniques=1;
    bzero((char*) &uniquedef,sizeof(uniquedef));
    bzero((char*) uniqueseg,sizeof(uniqueseg));
    uniquedef.seg=uniqueseg;
    uniquedef.keysegs=2;

    /* Make a unique over all columns (except first NULL fields) */
    for (i=0, start=1 ; i < 2 ; i++)
    {
      uniqueseg[i].start=start;
      start+=recinfo[i+1].length;
      uniqueseg[i].length=recinfo[i+1].length;
      uniqueseg[i].language=MY_CHARSET_CURRENT;
    }
    uniqueseg[0].type= key_type;
    uniqueseg[0].null_bit= null_fields ? 2 : 0;
    uniqueseg[1].type= HA_KEYTYPE_TEXT;
    if (extra_field == FIELD_BLOB)
    {
      uniqueseg[1].length=0;			/* The whole blob */
      uniqueseg[1].bit_start=4;			/* long blob */
      uniqueseg[1].flag|= HA_BLOB_PART;
    }
    else if (extra_field == FIELD_VARCHAR)
      uniqueseg[1].flag|= HA_VAR_LENGTH;
  }
  else
    uniques=0;

  if (!silent)
    printf("- Creating isam-file\n");
  bzero((char*) &create_info,sizeof(create_info));
  create_info.max_rows=(ulong) (rec_pointer_size ?
				(1L << (rec_pointer_size*8))/40 :
				0);
  if (mi_create(filename,1,keyinfo,3+opt_unique,recinfo,
		uniques, &uniquedef, &create_info,
		create_flag))
    goto err;
  if (!(file=mi_open(filename,2,HA_OPEN_ABORT_IF_LOCKED)))
    goto err;
  if (!silent)
    printf("- Writing key:s\n");

  my_errno=0;
  row_count=deleted=0;
  for (i=49 ; i>=1 ; i-=2 )
  {
    if (insert_count-- == 0) { VOID(mi_close(file)) ; exit(0) ; }
    j=i%25 +1;
    create_record(record,j);
    error=mi_write(file,record);
    if (!error)
      row_count++;
    flags[j]=1;
    if (verbose || error)
      printf("J= %2d  mi_write: %d  errno: %d\n", j,error,my_errno);
  }

  /* Insert 2 rows with null values */
  if (null_fields)
  {
    create_record(record,0);
    error=mi_write(file,record);
    if (!error)
      row_count++;
    if (verbose || error)
      printf("J= NULL  mi_write: %d  errno: %d\n", error,my_errno);
    error=mi_write(file,record);
    if (!error)
      row_count++;
    if (verbose || error)
      printf("J= NULL  mi_write: %d  errno: %d\n", error,my_errno);
    flags[0]=2;
  }

  if (!skip_update)
  {
    if (opt_unique)
    {
      if (!silent)
	printf("- Checking unique constraint\n");
      create_record(record,j);
      if (!mi_write(file,record) || my_errno != HA_ERR_FOUND_DUPP_UNIQUE)
      {
	printf("unique check failed\n");
      }
    }
    if (!silent)
      printf("- Updating rows\n");

    /* Update first last row to force extend of file */
    if (mi_rsame(file,read_record,-1))
    {
      printf("Can't find last row with mi_rsame\n");
    }
    else
    {
      memcpy(record,read_record,rec_length);
      update_record(record);
      if (mi_update(file,read_record,record))
      {
	printf("Can't update last row: %.*s\n",
	       keyinfo[0].seg[0].length,read_record+1);
      }
    }

    /* Read through all rows and update them */
    pos=(my_off_t) 0;
    found=0;
    while ((error=mi_rrnd(file,read_record,pos)) == 0)
    {
      if (update_count-- == 0) { VOID(mi_close(file)) ; exit(0) ; }
      memcpy(record,read_record,rec_length);
      update_record(record);
      if (mi_update(file,read_record,record))
      {
	printf("Can't update row: %.*s, error: %d\n",
	       keyinfo[0].seg[0].length,record+1,my_errno);
      }
      found++;
      pos=HA_OFFSET_ERROR;
    }
    if (found != row_count)
      printf("Found %ld of %ld rows\n", found,row_count);
  }

  if (!silent)
    printf("- Reopening file\n");
  if (mi_close(file)) goto err;
  if (!(file=mi_open(filename,2,HA_OPEN_ABORT_IF_LOCKED))) goto err;
  if (!skip_update)
  {
    if (!silent)
      printf("- Removing keys\n");

    for (i=0 ; i <= 10 ; i++)
    {
      /* testing */
      if (remove_count-- == 0) { VOID(mi_close(file)) ; exit(0) ; }
      j=i*2;
      if (!flags[j])
	continue;
      create_key(key,j);
      my_errno=0;
      if ((error = mi_rkey(file,read_record,0,key,0,HA_READ_KEY_EXACT)))
      {
	if (verbose || (flags[j] >= 1 ||
			(error && my_errno != HA_ERR_KEY_NOT_FOUND)))
	  printf("key: '%.*s'  mi_rkey:  %3d  errno: %3d\n",
		 (int) key_length,key+test(null_fields),error,my_errno);
      }
      else
      {
	error=mi_delete(file,read_record);
	if (verbose || error)
	  printf("key: '%.*s'  mi_delete: %3d  errno: %3d\n",
		 (int) key_length, key+test(null_fields), error, my_errno);
	if (! error)
	{
	  deleted++;
	  flags[j]--;
	}
      }
    }
  }
  if (!silent)
    printf("- Reading rows with key\n");
  for (i=0 ; i <= 25 ; i++)
  {
    create_key(key,i);
    my_errno=0;
    error=mi_rkey(file,read_record,0,key,0,HA_READ_KEY_EXACT);
    if (verbose ||
	(error == 0 && flags[i] == 0 && unique_key) ||
	(error && (flags[i] != 0 || my_errno != HA_ERR_KEY_NOT_FOUND)))
    {
      printf("key: '%.*s'  mi_rkey: %3d  errno: %3d  record: %s\n",
	     (int) key_length,key+test(null_fields),error,my_errno,record+1);
    }
  }

  if (!silent)
    printf("- Reading rows with position\n");
  for (i=1,found=0 ; i <= 30 ; i++)
  {
    my_errno=0;
    if ((error=mi_rrnd(file,read_record,i == 1 ? 0L : HA_OFFSET_ERROR)) == -1)
    {
      if (found != row_count-deleted)
	printf("Found only %ld of %ld rows\n",found,row_count-deleted);
      break;
    }
    if (!error)
      found++;
    if (verbose || (error != 0 && error != HA_ERR_RECORD_DELETED &&
		    error != HA_ERR_END_OF_FILE))
    {
      printf("pos: %2d  mi_rrnd: %3d  errno: %3d  record: %s\n",
	     i-1,error,my_errno,read_record+1);
    }
  }
  if (mi_close(file)) goto err;
  my_end(MY_CHECK_ERROR);

  return (0);
err:
  printf("got error: %3d when using myisam-database\n",my_errno);
  return 1;			/* skipp warning */
}


static void create_key_part(char *key,uint rownr)
{
  if (!unique_key)
    rownr&=7;					/* Some identical keys */
  if (keyinfo[0].seg[0].type == HA_KEYTYPE_NUM)
  {
    sprintf(key,"%*d",keyinfo[0].seg[0].length,rownr);
  }
  else if (keyinfo[0].seg[0].type == HA_KEYTYPE_VARTEXT)
  {						/* Alpha record */
    /* Create a key that may be easily packed */
    bfill(key,keyinfo[0].seg[0].length,rownr < 10 ? 'A' : 'B');
    sprintf(key+keyinfo[0].seg[0].length-2,"%-2d",rownr);
    if ((rownr & 7) == 0)
    {
      /* Change the key to force a unpack of the next key */
      bfill(key+3,keyinfo[0].seg[0].length-4,rownr < 10 ? 'a' : 'b');
    }
  }
  else
  {						/* Alpha record */
    if (keyinfo[0].seg[0].flag & HA_SPACE_PACK)
      sprintf(key,"%-*d",keyinfo[0].seg[0].length,rownr);
    else
    {
      /* Create a key that may be easily packed */
      bfill(key,keyinfo[0].seg[0].length,rownr < 10 ? 'A' : 'B');
      sprintf(key+keyinfo[0].seg[0].length-2,"%-2d",rownr);
      if ((rownr & 7) == 0)
      {
	/* Change the key to force a unpack of the next key */
	key[1]= (rownr < 10 ? 'a' : 'b');
      }
    }
  }
}


static void create_key(char *key,uint rownr)
{
  if (keyinfo[0].seg[0].null_bit)
  {
    if (rownr == 0)
    {
      key[0]=1;					/* null key */
      key[1]=0;					/* Fore easy print of key */
      return;
    }
    *key++=0;
  }
  if (keyinfo[0].seg[0].flag & (HA_BLOB_PART | HA_VAR_LENGTH))
  {
    uint tmp;
    create_key_part(key+2,rownr);
    tmp=strlen(key+2);
    int2store(key,tmp);
  }
  else
    create_key_part(key,rownr);
}


static char blob_key[MAX_REC_LENGTH];
static char blob_record[MAX_REC_LENGTH+20*20];


static void create_record(char *record,uint rownr)
{
  char *pos;
  bzero((char*) record,MAX_REC_LENGTH);
  record[0]=1;					/* delete marker */
  if (rownr == 0 && keyinfo[0].seg[0].null_bit)
    record[0]|=keyinfo[0].seg[0].null_bit;	/* Null key */

  pos=record+1;
  if (recinfo[1].type == FIELD_BLOB)
  {
    uint tmp;
    char *ptr;
    create_key_part(blob_key,rownr);
    tmp=strlen(blob_key);
    int4store(pos,tmp);
    ptr=blob_key;
    memcpy_fixed(pos+4,&ptr,sizeof(char*));
    pos+=recinfo[1].length;
  }
  else if (recinfo[1].type == FIELD_VARCHAR)
  {
    uint tmp;
    create_key_part(pos+2,rownr);
    tmp=strlen(pos+2);
    int2store(pos,tmp);
    pos+=recinfo[1].length;
  }
  else
  {
    create_key_part(pos,rownr);
    pos+=recinfo[1].length;
  }
  if (recinfo[2].type == FIELD_BLOB)
  {
    uint tmp;
    char *ptr;;
    sprintf(blob_record,"... row: %d", rownr);
    strappend(blob_record,max(MAX_REC_LENGTH-rownr,10),' ');
    tmp=strlen(blob_record);
    int4store(pos,tmp);
    ptr=blob_record;
    memcpy_fixed(pos+4,&ptr,sizeof(char*));
  }
  else if (recinfo[2].type == FIELD_VARCHAR)
  {
    uint tmp;
    sprintf(pos+2,"... row: %d", rownr);
    tmp=strlen(pos+2);
    int2store(pos,tmp);
  }
  else
  {
    sprintf(pos,"... row: %d", rownr);
    strappend(pos,recinfo[2].length,' ');
  }
}

/* change row to test re-packing of rows and reallocation of keys */

static void update_record(char *record)
{
  char *pos=record+1;
  if (recinfo[1].type == FIELD_BLOB)
  {
    char *column,*ptr;
    int length;
    length=uint4korr(pos);			/* Long blob */
    memcpy_fixed(&column,pos+4,sizeof(char*));
    memcpy(blob_key,column,length);		/* Move old key */
    ptr=blob_key;
    memcpy_fixed(pos+4,&ptr,sizeof(char*));	/* Store pointer to new key */
    if (keyinfo[0].seg[0].type != HA_KEYTYPE_NUM)
      casedn(blob_key,length);
    pos+=recinfo[1].length;
  }
  else if (recinfo[1].type == FIELD_VARCHAR)
  {
    uint length=uint2korr(pos);
    casedn(pos+2,length);
    pos+=recinfo[1].length;
  }
  else
  {
    if (keyinfo[0].seg[0].type != HA_KEYTYPE_NUM)
      casedn(pos,keyinfo[0].seg[0].length);
    pos+=recinfo[1].length;
  }

  if (recinfo[2].type == FIELD_BLOB)
  {
    char *column;
    int length;
    length=uint4korr(pos);
    memcpy_fixed(&column,pos+4,sizeof(char*));
    memcpy(blob_record,column,length);
    bfill(blob_record+length,20,'.');	/* Make it larger */
    length+=20;
    int4store(pos,length);
    column=blob_record;
    memcpy_fixed(pos+4,&column,sizeof(char*));
  }
  else if (recinfo[2].type == FIELD_VARCHAR)
  {
    /* Second field is longer than 10 characters */
    uint length=uint2korr(pos);
    bfill(pos+2+length,recinfo[2].length-length-2,'.');
    length=recinfo[2].length-2;
    int2store(pos,length);
  }
  else
  {
    bfill(pos+recinfo[2].length-10,10,'.');
  }
}


static struct option long_options[] =
{
  {"checksum",		no_argument,		0, 'c'},
#ifndef DBUG_OFF
  {"debug",		required_argument,	0, '#'},
#endif
  {"delete_rows",	required_argument,	0, 'd'},
  {"help",		no_argument,		0, '?'},
  {"insert_rows",	required_argument,	0, 'i'},
  {"key_alpha",		no_argument,		0, 'a'},
  {"key_binary_pack",	no_argument,		0, 'B'},
  {"key_blob",		required_argument,	0, 'b'},
  {"key_cache",		no_argument,		0, 'K'},
  {"key_length",	required_argument,	0, 'k'},
  {"key_multiple",	no_argument,		0, 'm'},
  {"key_prefix_pack",	no_argument,		0, 'P'},
  {"key_space_pack",	no_argument,		0, 'p'},
  {"key_varchar",	no_argument,		0, 'w'},
  {"null_fields",	no_argument,		0, 'N'},
  {"row_fixed_size",	no_argument,		0, 'S'},
  {"row_pointer_size",	required_argument,	0, 'R'},
  {"silent",		no_argument,		0, 's'},
  {"skip_update",	no_argument,		0, 'U'},
  {"unique",		no_argument,		0, 'C'},
  {"update_rows",	required_argument,	0, 'u'},
  {"verbose",		no_argument,		0, 'v'},
  {"version",		no_argument,		0, 'V'},
  {0, 0, 0, 0}
};


/* Read options */

static void get_options(int argc,char *argv[])
{
  int c,option_index=0;

  while ((c=getopt_long(argc,argv,"abBcCd:i:k:KmPR:SspNu:UvVw#:",
			long_options, &option_index)) != EOF)
  {
    switch(c) {
    case 'a':
      key_type= HA_KEYTYPE_TEXT;
      break;
    case 'c':
      create_flag|= HA_CREATE_CHECKSUM;
      break;
    case 'C':
      opt_unique=1;
      break;
    case 'R':				/* Length of record pointer */
      rec_pointer_size=atoi(optarg);
      if (rec_pointer_size > 3)
	rec_pointer_size=0;
      break;
    case 'P':
      pack_keys= HA_PACK_KEY;		/* Use prefix compression */
      break;
    case 'B':
      pack_keys= HA_BINARY_PACK_KEY;	/* Use binary compression */
      break;
    case 'S':
      if (key_field == FIELD_VARCHAR)
      {
	create_flag=0;			/* Static sized varchar */
      }
      else if (key_field != FIELD_BLOB)
      {
	key_field=FIELD_NORMAL;		/* static-size record */
	extra_field=FIELD_NORMAL;
      }
      break;
    case 'p':
      pack_keys=HA_PACK_KEY;		/* Use prefix + space packing */
      pack_seg=HA_SPACE_PACK;
      key_type=HA_KEYTYPE_TEXT;
      break;
    case 'N':
      null_fields=1;			/* First key part may be null */
      break;
    case 'v':				/* verbose */
      verbose=1;
      break;
    case 'd':
      remove_count=atoi(optarg);
      break;
    case 'i':
      insert_count=atoi(optarg);
      break;
    case 'u':
      update_count=atoi(optarg);
      break;
    case 'U':
      skip_update=1;
      break;
    case 'm':
      unique_key=0;
      break;
    case 'b':
      key_field=FIELD_BLOB;			/* blob key */
      extra_field= FIELD_BLOB;
      pack_seg|= HA_BLOB_PART;
      key_type= HA_KEYTYPE_VARTEXT;
      break;
    case 'k':
      key_length=atoi(optarg);
      if (key_length < 4 || key_length > MI_MAX_KEY_LENGTH)
      {
	fprintf(stderr,"Wrong key length\n");
	exit(1);
      }
      break;
    case 's':
      silent=1;
      break;
    case 'w':
      key_field=FIELD_VARCHAR;			/* varchar keys */
      extra_field= FIELD_VARCHAR;
      key_type= HA_KEYTYPE_VARTEXT;
      pack_seg|= HA_VAR_LENGTH;
      create_flag|= HA_PACK_RECORD;
      break;
    case 'K':				/* Use key cacheing */
      key_cacheing=1;
      break;
    case 'V':
      printf("test1 Ver 1.0 \n");
      exit(0);
    case '#':
      DEBUGGER_ON;
      DBUG_PUSH (optarg);
      break;
    }
  }
  return;
} /* get options */
