/*	SC	A Spreadsheet Calculator
 *		Curses based Screen driver
 *
 *		original by James Gosling, September 1982
 *		modifications by Mark Weiser and Bruce Israel,
 *			University of Maryland
 *
 *              More mods Robert Bond, 12/86
 *		More mods by Alan Silverstein, 3-4/88, see list of changes.
 *		$Revision: 7.7 $
 *
 */


#include <curses.h>
#include <time.h>
#include "sc.h"

#ifndef MSDOS
#include <unistd.h>
#endif

#ifdef VMS
extern int VMS_read_raw;   /*sigh*/
    VMS_read_raw = 1;
#endif

#ifdef BROKENCURSES
		/* nl/nonl bug fix */
#undef nl
#undef nonl
#define nl()	 (_tty.sg_flags |= CRMOD,_pfast = _rawmode,stty(_tty_ch, &_tty))
#define nonl()	 (_tty.sg_flags &= ~CRMOD, _pfast = TRUE, stty(_tty_ch, &_tty))
#endif

void	repaint(int x, int y, int len);

char	under_cursor = ' '; /* Data under the < cursor */
char	mode_ind = 'i';
extern	char    revmsg[];

int	rows, lcols;
int	lastmx, lastmy;	/* Screen address of the cursor */
int	lastcol;	/* Spreadsheet Column the cursor was in last */
int	rescol = 4;	/* columns reserved for row numbers */
extern	int *fwidth;
extern	int showrange;	/* Causes ranges to be highlighted	*/
extern	int showneed;	/* Causes cells needing values to be highlighted */
extern	int showexpr;	/* Causes cell exprs to be displayed, highlighted */
extern	int shownote;	/* Causes cells with attached notes to be highlighted */
#ifdef RIGHT_CBUG
extern	int wasforw;	/* Causes screen to be redisplay if on lastcol */
#endif
extern struct go_save gs;

/*
 * update() does general screen update
 *
 * standout last time in update()?
 *	At this point we will let curses do work
 */
int	standlast	= FALSE;

void
update(int anychanged)		/* did any cell really change in value? */
{
    int				row, col;
    struct ent			**pp;
    int				mxrow, mxcol;
    int				minsr = 0, minsc = 0, maxsr = 0, maxsc = 0;
    int				r;
    int				i;
    static int			lastcurcol = -1, lastcurrow = -1;
    struct frange		*fr;
    static struct frange	*lastfr = 0;
    int				ftoprows, fbottomrows, fleftcols, frightcols;
    int				ftrows, fbrows, flcols, frcols;
    static int			lastftoprows = 0;
    static int			lastfleftcols = 0;
    static bool			frTooLarge = 0;

#ifndef MSDOS
    /*
     * If receiving input from a pipeline, don't display spreadsheet data
     * on screen.
     */
    if (!isatty(STDIN_FILENO)) return;
#endif

    fr = lastfr;
    if (!(fr && fr->or_left->row <= currow &&	/* If we've left the	    */
	    fr->or_left->col <= curcol &&	/* previous framed range... */
	    fr->or_right->row >= currow &&
	    fr->or_right->col >= curcol)) {
	fr = find_frange(currow, curcol);
	FullUpdate++;
    }
    if (fr) {
	ftoprows = fr->ir_left->row - fr->or_left->row;
	fbottomrows = fr->or_right->row - fr->ir_right->row;
	fleftcols = fr->ir_left->col - fr->or_left->col;
	frightcols = fr->or_right->col - fr->ir_right->col;
	ftrows = RESROW + ftoprows + fbottomrows;
	flcols = rescol;
	for (r = fr->or_left->row - 1, i = ftoprows; i; i--)
	    if (row_hidden[r+i])
		ftrows--;
	for (r = fr->or_right->row - 1, i = fbottomrows; i; i--)
	    if (row_hidden[r-i])
		ftrows--;
	for (r = fr->or_left->col - 1, i = fleftcols; i; i--) {
	    if (col_hidden[r+i])
		continue;
	    flcols += fwidth[r+i];
	}
	for (r = fr->or_right->col - 1, i = frightcols; i; i--) {
	    if (col_hidden[r-i])
		continue;
	    flcols += fwidth[r-i];
	}
	if (ftrows >= LINES || flcols >= COLS) {
	    frTooLarge = TRUE;
	    if (FullUpdate) {
		error("Frame too large for screen size - ignoring");
	    }
	    ftoprows = fbottomrows = fleftcols = frightcols = 0;
	    strow -= lastftoprows;
	    stcol -= lastfleftcols;
	} else {
	    frTooLarge = FALSE;
	    if (strow >= fr->or_left->row) {
		if (currow > fr->ir_left->row)
		    ftoprows = fr->ir_left->row - fr->or_left->row;
		else
		    ftoprows = currow - fr->or_left->row;
		if (strow < fr->or_left->row + ftoprows)
		    strow = fr->or_left->row + ftoprows;
		else if (strow > fr->ir_right->row) {
		    strow = fr->ir_right->row + 1;
		    FullUpdate++;
		}
	    } else
		ftoprows = 0;
	    if (stcol >= fr->or_left->col) {
		if (curcol > fr->ir_left->col)
		    fleftcols = fr->ir_left->col - fr->or_left->col;
		else
		    fleftcols = curcol - fr->or_left->col;
		if (stcol < fr->or_left->col + fleftcols)
		    stcol = fr->or_left->col + fleftcols;
		else if (stcol > fr->ir_right->col) {
		    stcol = fr->ir_right->col + 1;
		    FullUpdate++;
		}
	    } else
		fleftcols = 0;
	    if (fr) {
		if (currow < fr->ir_right->row)
		    fbottomrows = fr->or_right->row - fr->ir_right->row;
		else
		    fbottomrows = fr->or_right->row - currow;
		if (curcol < fr->ir_right->col)
		    frightcols = fr->or_right->col - fr->ir_right->col;
		else
		    frightcols = fr->or_right->col - curcol;
	    }
	}
    } else
	ftoprows = fbottomrows = fleftcols = frightcols = 0;
    if (fr != lastfr && !gs.stflag) {
	strow -= lastftoprows;
	stcol -= lastfleftcols;
    }

    ftrows = ftoprows;
    fbrows = fbottomrows;
    flcols = frcols = 0;
    if (fr) {
	for (r = fr->or_left->row - 1, i = ftrows; i; i--)
	    if (row_hidden[r+i])
		ftrows--;
	for (r = fr->or_right->row + 1, i = fbrows; i; i--)
	    if (row_hidden[r-i])
		fbrows--;
	for (r = fr->or_left->col - 1, i = fleftcols; i; i--) {
	    if (col_hidden[r+i])
		continue;
	    flcols += fwidth[r+i];
	}
	for (r = fr->or_right->col + 1, i = frightcols; i; i--) {
	    if (col_hidden[r-i])
		continue;
	    frcols += fwidth[r-i];
	}
    }

    /*
     * Place the cursor on the screen.  Set col, curcol, stcol, lastcol as
     * needed.  If strow and stcol are negative, centering is forced.
     */
    if ((curcol != lastcurcol) || FullUpdate) {
	while (col_hidden[curcol])   /* You can't hide the last row or col */
	    curcol++;
	if (fwidth[curcol] > COLS - rescol - 2) {
	    error("column %s too wide - resizing", coltoa(curcol));
	    doformat(curcol, curcol, COLS - rescol - 2,
		    precision[curcol], realfmt[curcol]);
	}

	/* First see if the last display still covers curcol */
	if (stcol >= 0 && stcol <= curcol) { 
	    if (fr && fr != lastfr) {
		if (stcol >= fr->or_left->col &&
			stcol < fr->or_left->col + fleftcols)
		    stcol = fr->or_left->col + fleftcols;
		else if (stcol < fr->or_left->col) {
		    fleftcols = 0;
		    flcols = 0;
		} else if (!gs.stflag) {
		    stcol += fleftcols;
		    if (stcol > fr->ir_right->col)
			stcol = fr->ir_right->col + 1;
		}
	    }
	    for (i = stcol, lcols = 0, col = rescol + flcols + frcols;
		    (col + fwidth[i]) < COLS-1 && i < maxcols; i++) {
		lcols++;
		if (fr && i == fr->ir_right->col + 1) {
		    col -= frcols;
		    frcols = frightcols = 0;
		}
		if (col_hidden[i])
		    continue;
		col += fwidth[i];
	    }
	}

	while (stcol < 0 || curcol < stcol || stcol + lcols - 1 < curcol ||
		(colsinrange != fwidth[curcol] && stcol != curcol &&
		stcol + lcols - 1 < gs.g_lastcol)) {

	    FullUpdate++;

		/* How about back one? */
	    if (stcol - 1 == curcol) {
		stcol--;
		/* Forward one? */
	    } else if (stcol >= 0 && stcol + lcols == curcol) {
		stcol++;
	    } else if (stcol >= 0 && fr && curcol >= fr->or_left->col &&
		    curcol <= fr->ir_left->col && stcol < curcol &&
		    curcol <= stcol + lcols + fr->ir_left->col -
			    fr->or_left->col) {
		while (stcol + lcols < fr->ir_left->col && !frTooLarge)
		    if (col_hidden[++stcol]) lcols--;
	    } else {
		/* Try to put the cursor in the center of the screen.
		 * If we've just jumped to a range using the goto command,
		 * center the range instead.
		 */
		colsinrange = (colsinrange > COLS - rescol -
			flcols - frcols - 2 ?
			COLS - rescol - flcols - frcols - 2 : colsinrange);
		col = (COLS - rescol - flcols - frcols - colsinrange)/2; 
		stcol = curcol;
		for (i = curcol - 1;
			i >= (fr ? fr->or_left->col + fleftcols : 0) &&
			(col - fwidth[i] > 0 || col_hidden[i]);
			i--) {
		    stcol--;
		    if (col_hidden[i])
			continue;
		    col -= fwidth[i];
		}
		if (fr && stcol < fr->or_left->col + fleftcols)
		    stcol = fr->or_left->col + fleftcols;
	    }
	    /* Now pick up the counts again */
	    for (i = stcol, lcols = 0, col = rescol + flcols + frcols;
		    (col + fwidth[i]) < COLS-1 && i < maxcols; i++) {
		lcols++;
		if (fr && i == fr->ir_right->col + 1) {
		    col -= frcols;
		    frcols = frightcols = 0;
		}
		if (col_hidden[i])
		    continue;
		col += fwidth[i];
	    }
	}
	lastcurcol = curcol;
    }

    /* Now - same process on the rows as the columns */
    if ((currow != lastcurrow) || FullUpdate) {
	while (row_hidden[currow])   /* You can't hide the last row or col */
	    currow++;
	if (strow >= 0 && strow <= currow) { 
	    if (fr && fr != lastfr) {
		if (strow >= fr->or_left->row &&
			strow < fr->or_left->row + ftoprows)
		    strow = fr->or_left->row + ftoprows;
		else if (strow < fr->or_left->row) {
		    ftoprows = 0;
		    ftrows = 0;
		} else if (!gs.stflag) {
		    strow += ftoprows;
		    if (strow > fr->ir_right->row)
			strow = fr->ir_right->row + 1;
		}
	    }
	    for (i = strow, rows = 0, row = RESROW + ftrows + fbrows;
		    row < LINES && i < maxrows; i++) {
		rows++;
		if (fr && i == fr->ir_right->row + 1) {
		    row -= fbrows;
		    fbrows = fbottomrows = 0;
		}
		if (row_hidden[i])
		    continue;
		row++;
	    }
	    if (fr && currow <= strow + rows &&
		    fr->ir_left->row >= strow + rows) {
		while (strow + rows < fr->ir_left->row && !frTooLarge)
		    if (row_hidden[++strow]) rows--;
	    }
	}

	while (strow < 0 || currow < strow || strow + rows - 1 < currow ||
		strow + rows < currow + rowsinrange) {

	    FullUpdate++;

		/* How about up one? */
	    if (strow - 1 == currow) {
		strow--;
		/* Down one? */
	    } else if (strow >= 0 && strow + rows == currow) {
		strow++;
	    } else if (strow >= 0 && fr && currow >= fr->or_left->row &&
		    currow <= fr->ir_left->row && strow < currow &&
		    currow <= strow + rows + fr->ir_left->row -
			    fr->or_left->row) {
		while (strow + rows < fr->ir_left->row && !frTooLarge)
		    if (row_hidden[++strow]) rows--;
	    } else {
		/* Try to put the cursor in the center of the screen.
		 * If we've just jumped to a range using the goto command,
		 * center the range instead.
		 */
		rowsinrange = (rowsinrange > LINES - RESROW - ftrows - fbrows ?
			LINES - RESROW : rowsinrange);
		row = (LINES - RESROW - ftrows - fbrows - rowsinrange)/2; 
		strow = currow;
		for (i = currow - 1;
			i >= (fr ? fr->or_left->row + ftoprows : 0) &&
			row > 0; i--) {
		    strow--;
		    if (row_hidden[i])
			continue;
		    row--;
		}
		if (fr && strow < fr->or_left->row + ftoprows)
		    strow = fr->or_left->row + ftoprows;
	    }
	    /* Now pick up the counts again */
	    for (i = strow, rows = 0, row = RESROW + ftrows + fbrows;
		    row < LINES && i < maxrows; i++) {
		rows++;
		if (fr && i == fr->ir_right->row + 1) {
		    row -= fbrows;
		    fbrows = fbottomrows = 0;
		}
		if (row_hidden[i])
		    continue;
		row++;
	    }
	}
	lastcurrow = currow;
    }
    mxcol = frightcols ? fr->or_right->col : stcol + lcols - 1;
    mxrow = fbottomrows ? fr->or_right->row : strow + rows - 1;
    gs.stflag = 0;
    lastfr = fr;
    lastftoprows = ftoprows;
    lastfleftcols = fleftcols;

    /* Get rid of cursor standout on the cell at previous cursor position */
    if (!FullUpdate) {
	if (showcell)
	    repaint(lastmx, lastmy, fwidth[lastcol]);

	(void) move(lastmy, lastmx+fwidth[lastcol]);

	if ((inch() & A_CHARTEXT ) == '<')
	    (void) addch(under_cursor);

	standout();
	repaint(lastmx, RESROW - 1, fwidth[lastcol]);
	repaint(0, lastmy, rescol - 1);
	standend();
    }

    /* where is the the cursor now? */
    lastmy =  RESROW + ftrows;
    for (row = strow; row < currow; row++)
	if (!row_hidden[row])
	    lastmy++;

    lastmx = rescol + flcols;
    for (col = stcol; col < curcol; col++)
	if (!col_hidden[col])
	    lastmx += fwidth[col];
    lastcol = curcol;

    if (FullUpdate || standlast) {
	(void) move(2, 0);
	(void) clrtobot();
	(void) standout();

	for (row = RESROW, i = (ftoprows ? fr->or_left->row : strow);
		i <= (fbottomrows ? fr->or_right->row : mxrow); i++) {
	    if (ftoprows && row == RESROW + ftrows)
		i = strow;
	    if (fbottomrows && row == LINES - fbrows)
		i = fr->or_right->row - fbottomrows + 1;
	    if (row_hidden[i]) 
		continue;
	    (void) move(row, 0);
	    (void) printw("%*d", rescol - 1, i);
	    row++;
	}
#ifdef RIGHT_CBUG
	if (wasforw) {
	    clearok(stdscr, TRUE);
	    wasforw = 0;
	}
#endif
	(void) move(2, 0);
	(void) printw("%*s", rescol, " ");

	for (col = rescol, i = (fleftcols ? fr->or_left->col : stcol);
		i <= (frightcols ? fr->or_right->col : mxcol); i++) {
	    register int k;
	    if (fleftcols && col == rescol + flcols)
		i = stcol;
	    if (frightcols && col + fwidth[i] >= COLS - 1 - frcols &&
		    i < fr->or_right->col - frightcols + 1)
		i = fr->or_right->col - frightcols + 1;
	    if (col_hidden[i])
		continue;
	    (void) move(2, col);
	    k = fwidth[i]/2;
	    if (k == 0)
		(void) printw("%1s", coltoa(i));
	    else
	        (void) printw("%*s%-*s", k, " ", fwidth[i]-k, coltoa(i));
	    col += fwidth[i];
	}
	(void) standend();
    }

    if (showrange) {
	minsr = showsr < currow ? showsr : currow;
	minsc = showsc < curcol ? showsc : curcol;
	maxsr = showsr > currow ? showsr : currow;
	maxsc = showsc > curcol ? showsc : curcol;

	if (showtop) {
	    (void) move(1,0);
	    (void) clrtoeol();
	    (void) printw("Default range:  %s",
			    r_name(minsr, minsc, maxsr, maxsc));
	}
    }


    /* Repaint the visible screen */
    if (showrange || anychanged || FullUpdate || standlast) {
	/* may be reset in loop, if not next time we will do a FullUpdate */
      if (standlast) {
      	FullUpdate = TRUE;
	standlast = FALSE;
      }

      for (row = (ftoprows ? fr->or_left->row : strow), r = RESROW;
	    row <= (fbottomrows ? fr->or_right->row : mxrow); row++) {
	int c = rescol;
	int do_stand = 0;
	int fieldlen;
	int nextcol;

	if (row_hidden[row])
	    continue;
	if (ftoprows && r == RESROW + ftrows)
	    row = strow;
	if (fbottomrows && r == LINES - fbrows)
	    row = fr->or_right->row - fbottomrows + 1;
	for (pp = ATBL(tbl, row, col = (fleftcols ? fr->or_left->col : stcol));
		col <= mxcol;
		pp += nextcol - col,  col = nextcol, c += fieldlen) {

	    if (col_hidden[col]) {
		fieldlen = 0;
		continue;
	    }
	    if (fleftcols && c == rescol + flcols) {
		col = stcol;
		pp = ATBL(tbl, row, col);
	    }
	    if (frightcols && c + fwidth[col] >= COLS - 1 - frcols &&
		    col < fr->or_right->col - frightcols + 1) {
		col = fr->or_right->col - frightcols + 1;
		pp = ATBL(tbl, row, col);
	    }
	    nextcol = col + 1;

	    fieldlen = fwidth[col];

	    /*
	     * Set standout if:
	     *
	     * - showing ranges, and not showing cells which need to be filled
	     *	 in, and not showing cell expressions, and in a range, OR
	     *
	     * - showing cells which need to be filled in and this one is
	     *	 of that type (has a value and doesn't have an expression,
	     *	 or it is a string expression), OR
	     *
	     * - showing cells which have expressions and this one does.
	     */
	    if ((showrange && (!showneed) && (!showexpr)
			&& (row >= minsr) && (row <= maxsr)
			&& (col >= minsc) && (col <= maxsc))
		    || (showneed && (*pp) && ((*pp)->flags & is_valid) &&
			(((*pp)->flags & is_strexpr) || !((*pp)->expr)))
		    || (showexpr && (*pp) && ((*pp)->expr))
		    || (shownote && (*pp) && ((*pp)->nrow >= 0))) {

		(void) move(r, c);
		(void) standout();
		standlast++;
		if (!*pp) {	/* no cell, but standing out */
		    (void) printw("%*s", fwidth[col], " ");
		    (void) standend();
		    continue;
		} else
		    do_stand = 1;
	    } else
		do_stand = 0;

	    if ((*pp) && (((*pp)->flags & is_changed || FullUpdate) ||
		    do_stand)) {
		if (do_stand) {
		    (*pp)->flags |= is_changed; 
		} else {
		    (void) move(r, c);
		    (*pp)->flags &= ~is_changed;
		}

		/*
		 * Show expression; takes priority over other displays:
		 */

		if ((*pp)->cellerror)
		    (void) printw("%*.*s", fwidth[col], fwidth[col],
			(*pp)->cellerror == CELLERROR ? "ERROR" : "INVALID");
		else
		if (showexpr && ((*pp)->expr)) {
		    linelim = 0;
		    editexp(row, col);		/* set line to expr */
		    linelim = -1;
		    showstring(line, /* leftflush = */ 1, /* hasvalue = */ 0,
			    row, col, &nextcol, mxcol, &fieldlen, r, c,
			    fr, frightcols, flcols, frcols);
		} else {
		    /*
		     * Show cell's numeric value:
                     */

		    if ((*pp)->flags & is_valid) {
			char field[FBUFLEN];

			if ((*pp)->format) {
			    if (*((*pp)->format) == ctl('d')) {
				time_t v = (time_t) ((*pp)->v);
				strftime(field, sizeof(field),
					((*pp)->format)+1, localtime(&v));
			    } else
			    (void) format((*pp)->format, (*pp)->v,
				    field, sizeof(field));
			} else {
			    (void) engformat(realfmt[col], fwidth[col], 
				precision[col], (*pp)->v, 
				field, sizeof(field));
			}
			if (strlen(field) > fwidth[col]) {
			    for(i = 0; i<fwidth[col]; i++)
				(void)addch('*');
			} else {
			    if ((*pp)->format && *((*pp)->format) != ctl('d'))
			    for(i = 0; i < fwidth[col] - strlen(field);i++)
				(void)addch(' ');
			    (void)addstr(field);
			}
		    }

		    /*
		     * Show cell's label string:
		     */

		    if ((*pp)->label) {
			showstring((*pp)->label,
				    (*pp)->flags & (is_leftflush|is_label),
				    (*pp)->flags & is_valid,
				    row, col, &nextcol, mxcol, &fieldlen,
				    r, c, fr, frightcols, flcols, frcols);
		    }
		    else	/* repaint a blank cell: */
		    if ((do_stand || !FullUpdate) &&
				((*pp)->flags & is_changed) &&
				!((*pp)->flags & is_valid) && !(*pp)->label) {
			(void) printw("%*s", fwidth[col], " ");
		    }
		} /* else */

		if (do_stand) {
		    (void) standend();
		    do_stand = 0;
		}
	    }
	}
	r++;
      }
    }

    /* place 'cursor marker' */
    if (showcell && (!showneed) && (!showexpr) && (!shownote)) {
	(void) move(lastmy, lastmx);
        (void) standout();
        repaint(lastmx, lastmy, fwidth[lastcol]);
        (void) standend();
    }

    repaint(lastmx, RESROW - 1, fwidth[lastcol]);
    repaint(0, lastmy, rescol - 1);

    (void) move(lastmy, lastmx+fwidth[lastcol]);
    under_cursor = (inch() & A_CHARTEXT);
    if (! showcell)
	(void) addch('<');

    (void) move(0, 0);
    (void) clrtoeol();
    if (linelim >= 0) {
	int ctlchars;

	for (i = ctlchars = 0; i < linelim; i++)
	    if ((unsigned char) line[i] < ' ')
		ctlchars++;
	(void) addch(mode_ind);
	(void) addstr("> ");
	(void) addstr(line);
	(void) move((linelim+3+ctlchars)/COLS, (linelim+3+ctlchars)%COLS);
    } else {
	if (showtop) {			/* show top line */
	    register struct ent *p1;
	    int printed = 0;		/* printed something? */

	    (void) printw("%s%d ", coltoa(curcol), currow);

	    if ((p1 = *ATBL(tbl, currow, curcol)) && p1->nrow > -1)
		printw("{*%s} ", r_name(p1->nrow, p1->ncol,
			p1->nlastrow, p1->nlastcol));

	    /* show the current cell's format */
	    if ((p1) && p1->format)
		printw("(%s) ", p1->format);
	    else
		printw("(%d %d %d) ", fwidth[curcol], precision[curcol],
				realfmt[curcol]);

	    if (p1) {
		if (p1->expr) {
		    /* has expr of some type */
		    linelim = 0;
		    editexp(currow, curcol);	/* set line to expr */
		    linelim = -1;
		}

		/*
		 * Display string part of cell:
		 */

		if ((p1->expr) && (p1->flags & is_strexpr)) {
 		    if (p1->flags & is_label)
			(void) addstr("|{");
		    else
			(void) addstr((p1->flags & is_leftflush) ? "<{" : ">{");
		    (void) addstr(line);
		    (void) addstr("} ");	/* and this '}' is for vi % */
		    printed = 1;

		} else if (p1->label) {
		    /* has constant label only */
		    if (p1->flags & is_label)
			(void) addstr("|\"");
		    else
			(void) addstr((p1->flags & is_leftflush) ? "<\"" : ">\"");
		    (void) addstr(p1->label);
		    (void) addstr("\" ");
		    printed = 1;
		}

		/*
		 * Display value part of cell:
		 */

		if (p1->flags & is_valid) {
		    /* has value or num expr */
		    if ((!(p1->expr)) || (p1->flags & is_strexpr))
			(void) sprintf (line, "%.15g", p1->v);

		    (void) addch('[');
		    (void) addstr(line);
		    (void) addch(']');
		    *line = '\0'; /* this is the input buffer ! */
		    printed = 1;
		}
	    }
	    if (!printed)
		(void) addstr("[]");
	    /* Display if cell is locked */
	    if (p1 && p1->flags&is_locked)
		(void) addstr(" locked");
	}
	if (showcell)
	    move(LINES - 1, COLS - 1);
	else
	    (void) move(lastmy, lastmx+fwidth[lastcol]);
    }

    if (revmsg[0]) {
	(void) move(0, 0);
	(void) clrtoeol();	/* get rid of topline display */
	(void) printw(revmsg);
	*revmsg = '\0';		/* don't show it again */
	if (showcell)
	    move(LINES - 1, COLS - 1);
	else
	    (void) move(lastmy, lastmx + fwidth[lastcol]);
    }

    FullUpdate = FALSE;
}

/* redraw what is under the cursor from curses' idea of the screen */
void
repaint(int x, int y, int len)
{
    int c;

    while(len-- > 0) {
	(void) move(y, x);
	c = inch() & A_CHARTEXT;
	(void) addch(c);
	x++;
    }
}

int seenerr;

/* error routine for yacc (gram.y) */
void
yyerror(char *err)
{
    if (seenerr) return;
    seenerr++;
    (void) move(1,0);
    (void) clrtoeol();
    (void) printw("%s: %.*s<=%s",err,linelim,line,line+linelim);
}

#ifdef XENIX2_3
struct termio tmio;
#endif

void
startdisp()
{
#if sun
    int	 fd;
    fd = dup(0);
#endif
#ifndef MSDOS
    if (isatty(STDIN_FILENO)) {
#endif
#ifdef TIOCGSIZE
	{   struct ttysize size;
	    if (ioctl(0, TIOCGSIZE, &size) == 0) { 
		LINES = size.ts_lines;
		COLS = size.ts_cols;
	    }
	}
#endif

#ifdef XENIX2_3
	(void) ioctl(fileno(stdin), TCGETA, & tmio);
#endif
	(void) initscr();
#if sun
	close(0);
	dup(fd);
	close(fd);
#endif
	(void) clear();
#ifdef VMS
	VMS_read_raw = 1;
#else
	nonl();
	noecho();
	cbreak();
#endif
	initkbd();
	scrollok(stdscr, 1);

#if defined(SYSV3) && !defined(NOIDLOK)
# ifndef IDLOKBAD
	/*
	 * turn hardware insert/delete on, if possible.
	 * turn on scrolling for systems with SYSVr3.{1,2} (SYSVr3.0 has
	 * this set as the default)
	 */
	idlok(stdscr,TRUE);
# else	/*
	 * This seems to fix (with an empty spreadsheet):
	 *	a) Redrawing the bottom half of the screen when you
	 *		move between row 9 <-> 10
	 *	b) the highlighted row labels being trash when you
	 *		move between row 9 <-> 10
	 *	c) On an xterm on Esix Rev. D+ from eating lines
	 *	 -goto (or move) a few lines (or more) past the bottom
	 *	 of the screen, goto (or move) to the top line on the
	 *	 screen, move upward and the current line is deleted, the
	 *	 others move up even when they should not, check by
	 *	 noticing the rows become 2, 3, 40, 41, 42... (etc).
	 */
	idlok(stdscr,FALSE);
# endif
#endif

	FullUpdate++;
#ifndef MSDOS
    }
#endif
}

void
stopdisp()
{
#ifndef MSDOS
    if (isatty(STDIN_FILENO)) {
#endif
	deraw();
	resetkbd();
	endwin();
#ifdef XENIX2_3
	(void) ioctl(fileno(stdin), TCSETAW, & tmio);
#endif
#ifndef MSDOS
    }
#endif
}

/* init curses */
#ifdef VMS

goraw()
{
    VMS_read_raw = 1;
    FullUpdate++;
}

deraw()
{
    (void) move (LINES - 1, 0);
    (void) clrtoeol();
    (void) refresh();
    VMS_read_raw = 0;
}

#else /* VMS */
void
goraw()
{
#if SYSV2 || SYSV3
    fixterm();
#else /* SYSV2 || SYSV3 */
    cbreak();
    nonl();
    noecho ();
#endif /* SYSV2 || SYSV3 */
    kbd_again();
    (void) clear();
    FullUpdate++;
}

/* clean up curses */
void
deraw()
{
    (void) move(LINES - 1, 0);
    (void) clrtoeol();
    (void) refresh();
#if SYSV2 || SYSV3
    resetterm();
#else
    nocbreak();
    nl();
    echo();
#endif
    resetkbd();
}

#endif /* VMS */
