/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <libgnome/libgnome.h>

#include <gmime/gmime.h>

#include <pan/debug.h>
#include <pan/pan-glib-extensions.h>
#include <pan/util-mime.h>
#include <pan/util-file.h>

/***
****
***/

/**
 * parse_uu_begin_line
 * @param line the line to check for "begin filename mode"
 * @param filename if parse is successful, is set with the
 *        starting character of the filename.
 * @param mode if parse is successful, mode is set with the
 *        mode requested by the line.
 * @return 0 on success, -1 on failure
 */
static int
parse_uu_begin_line (const gchar  * b,
                     const gchar ** file,
                            gint  * mode)
{
	gint m = 0;

	/* skip to the permissions */
	while (*b==' ') ++b;
	if (!isdigit((int)*b)) return -1;

	/* parse the permissions */
	while (isdigit((int)*b)) {
		m = m*8 + (*b-'0');
		++b;
	};

	/* skip to the filename */
	if (*b!=' ') return -1;
	while (*b==' ') ++b;
	if (*b < ' ') return -1;

	/* return the results */
	if (mode!=NULL) *mode = m;
	if (file!=NULL) *file = b;
	return 0;
}

/**
 * uu_is_beginning
 * @param line line to test & see if it's the beginning of a uu-encoded block
 * @return true if it is, false otherwise
 */
static gboolean
uu_is_beginning (const gchar * line)
{
	return line!=NULL
	 	&& (line[0]=='b' || line[0]=='B')
	 	&& (line[1]=='e' || line[1]=='E')
	 	&& (line[2]=='g' || line[2]=='G')
	 	&& (line[3]=='i' || line[3]=='I')
	 	&& (line[4]=='n' || line[4]=='N')
		&& line[5]==' '
		&& !parse_uu_begin_line (line+6, NULL, NULL);
}


/**
 * uu_is_ending
 * @param line line to test & see if it's the end of a uu-encoded block
 * @return true if it is, false otherwise
 */
static gboolean
uu_is_ending (const gchar * line)
{
	return line!=NULL
	 	&& (line[0]=='e' || line[0]=='E')
	 	&& (line[1]=='n' || line[1]=='N')
	 	&& (line[2]=='d' || line[2]=='D')
	 	&& (line[3]=='\0' || line[3]=='\n' || line[3]==' ');
}

static void
uu_get_file_info (const gchar       * begin,
                  gchar            ** setme_filename,
                  gulong            * setme_mode)
{
	gchar * filename;
	const gchar * end;

	g_return_if_fail (uu_is_beginning(begin));
	g_return_if_fail (setme_filename != NULL);
	g_return_if_fail (setme_mode != NULL);

	*setme_mode = strtol (begin+6, NULL, 8);
	
	begin += 10;
	end = strchr (begin, '\n');
	filename = g_strndup (begin, end-begin);
	g_strstrip (filename);
	*setme_filename = filename;
}

static int
get_char_len (gint octet_len)
{
	gint char_len = (octet_len / 3) * 4;
	switch (octet_len % 3) {
		case 0: break;
		case 1: char_len += 2;
		case 2: char_len += 3;
	}
	return octet_len;
}

static gboolean
is_uu_line (const gchar * line, gint len)
{
	gint i;
	gint octet_len;
	gint char_len;

	g_return_val_if_fail (is_nonempty_string(line), FALSE);

	if (*line=='\0' || len<1)
		return FALSE;

	if (len==1 && *line=='`')
		return TRUE;

	/* get octet length */
	octet_len = *line - 0x20;
	if (octet_len > 45)
		return FALSE;

	/* get character length */
	char_len = get_char_len (octet_len);
	if (char_len+1 > len)
		return FALSE;

	/* make sure each character is in the uuencoded range */
	for (i=0; i<char_len; ++i)
		if (line[i+1]<0x20 || line[i+1]>0x60)
			return FALSE;

	/* looks okay */
	return TRUE;
}

typedef struct
{
	gchar * tmp_filename;
	gchar * uu_decoded_filename;
	gboolean is_uu;
}
UUTempFilePart;

static void
add_temp_part (GPtrArray * parts, UUTempFilePart ** part, FILE ** fp)
{
	g_return_if_fail (parts!=NULL);

	if (*part != NULL) {
		g_ptr_array_add (parts, *part);
		*part = NULL;
	}
	if (*fp != NULL) {
		fclose (*fp);
		*fp = NULL;
	}
}

static void
separate_uu_from_non_uu (FILE * fp_in,
                         GPtrArray * appendme_uu_file_parts)
{
	UUTempFilePart * cur = NULL;
	FILE * fp = NULL;
	gboolean uu = FALSE;
	gchar line[2048];

	/* sanity clause */
	g_return_if_fail (fp_in!=NULL);
	g_return_if_fail (appendme_uu_file_parts!=NULL);

	while (fgets(line, sizeof(line), fp_in))
	{
		/* beginning of a UU part */
		if (!uu && uu_is_beginning(line))
		{
			gulong mode = 0;
			gchar * decoded_filename = NULL;

			add_temp_part (appendme_uu_file_parts, &cur, &fp);

			/* new entry */
			uu = TRUE;
			uu_get_file_info (line, &decoded_filename, &mode);
			cur = g_new0 (UUTempFilePart, 1);
			cur->tmp_filename = pan_make_temp (&fp);
			cur->is_uu = TRUE;
			cur->uu_decoded_filename = decoded_filename;
		}

		/* ending of a UU part */
		else if (uu && uu_is_ending(line))
		{
			add_temp_part (appendme_uu_file_parts, &cur, &fp);
			uu = FALSE;
		}

		/* all other lines */
		else
		{
			/* new non-uu part */
			if (cur == NULL) {
				cur = g_new0 (UUTempFilePart, 1);
				cur->tmp_filename = pan_make_temp (&fp);
				cur->is_uu = FALSE;
				cur->uu_decoded_filename = NULL;
			}

			/* write the current line */
			if (fp!=NULL && (!cur->is_uu || is_uu_line(line, strlen(line)-1)))
				fwrite (line, sizeof(char), strlen(line), fp);
		}
	}

	/* close old entry */
	add_temp_part (appendme_uu_file_parts, &cur, &fp);
}

static void
guess_part_type_from_filename (const gchar   * filename,
                               gchar        ** setme_type,
                               gchar        ** setme_subtype)
{
	const gchar * type = gnome_mime_type (filename);
	const gchar * delimiter = strchr (type, '/');
	*setme_type = g_strndup (type, delimiter-type);
	*setme_subtype = g_strdup (delimiter+1);
}

static void
look_for_uuencoded_data (GMimePart * parent, gpointer data)
{
	gint i;
	guint content_len;
	const gchar * body;
	GPtrArray * parts;

	if (!g_mime_content_type_is_type (parent->mime_type, "text", "plain"))
		return;

	content_len = 0;
	body = g_mime_part_get_content (parent, &content_len);
	parts = g_ptr_array_new ();

	/* convert into N temp files -- uu and text parts */
	if (1)
	{
		FILE * full_temp_fp = NULL;
		gchar * full_temp_filename = pan_make_temp (&full_temp_fp);
		fwrite (body, sizeof(char), content_len, full_temp_fp);
		fclose (full_temp_fp);
		full_temp_fp = fopen (full_temp_filename, "r");
		separate_uu_from_non_uu (full_temp_fp, parts);
		fclose (full_temp_fp);
		unlink (full_temp_filename);
		g_free (full_temp_filename);
	}

	/* split? */
	if (parts->len>1 || (parts->len==1 && ((UUTempFilePart*)g_ptr_array_index(parts,0))->is_uu))
	{
		/* recast this part as a multipart/mixed instead of a text/plain */
		if (1) {
			GMimeContentType * mime_type;
			g_mime_part_set_content (parent, NULL, 0);
			mime_type = g_mime_content_type_new ("multipart", "mixed");
			g_mime_part_set_content_type (parent, mime_type);
			body = NULL;
		}

		/* add the children */
		for (i=0; i<parts->len; ++i)
		{
			FILE * fp = NULL;
			const UUTempFilePart * part = g_ptr_array_index(parts,i);
			GMimePart * child = NULL;
			char line[2048];

			if (part->is_uu)
			{
				gchar * pch = NULL;
				gchar * type = NULL;
				gchar * subtype = NULL;
				gint state = 0;
				guint32 save = 0;
				gchar uulen = 0;
				guess_part_type_from_filename (
					part->uu_decoded_filename,
					&type, &subtype);
				child = g_mime_part_new_with_type (type, subtype);
				g_mime_part_set_filename (
					child, part->uu_decoded_filename);
				pch = g_strdup_printf ("<%s>", part->uu_decoded_filename);
				g_mime_part_set_content_id (child, pch);
				g_free (pch);
				g_mime_part_set_content_disposition (child, "inline");

				fp = fopen (part->tmp_filename, "r");
				while (fgets(line, sizeof(line), fp))
				{
					gint len;
					gchar buf[2048];
					len = g_mime_utils_uudecode_step (
						line, strlen(line)-1, buf,
						&state, &save, &uulen);
					g_mime_part_append_pre_encoded_content (
						child, buf, len,
						GMIME_PART_ENCODING_8BIT);
				}
			}
			else
			{
				child = g_mime_part_new_with_type ("text", "plain");
				fp = fopen (part->tmp_filename, "r");
				while (fgets(line, sizeof(line), fp))
					g_mime_part_append_pre_encoded_content (
						child, line, strlen(line),
						GMIME_PART_ENCODING_8BIT);
			}

			fclose (fp);

			if (child->content!=NULL && child->content->len>0)
				g_mime_part_add_subpart (parent, child);
			else
				g_mime_part_destroy (child);
		}
	}

	/* remove parts */
	for (i=0; i<parts->len; ++i) {
		UUTempFilePart * part = g_ptr_array_index(parts,i);
		unlink (part->tmp_filename);
		g_free (part->tmp_filename);
		g_free (part->uu_decoded_filename);
		g_free (part);
	}
	g_ptr_array_free (parts, TRUE);
}

/***
****
***/

GMimeMessage*
pan_g_mime_parser_construct_message (const gchar * data)
{
	gint force_uu_check;
	GMimeMessage * message;

	message  = g_mime_parser_construct_message (data, TRUE);

	/* HogWasher does this */
	force_uu_check = strstr (data, "x-uuencode") != NULL;

	g_mime_message_foreach_part (message,
	                             look_for_uuencoded_data,
	                             GINT_TO_POINTER(force_uu_check));

	return message;
}

GMimeMessage*
pan_g_mime_parser_construct_message_from_file (FILE * fp)
{
	gint force_uu_check;
	GMimeMessage * message;

	message = g_mime_parser_construct_message_from_file (fp, TRUE);

	/* HogWasher does this */
	force_uu_check = TRUE;

	g_mime_message_foreach_part (message,
	                             look_for_uuencoded_data,
	                             GINT_TO_POINTER(force_uu_check));

	return message;
}
