/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc.h"
#include "embed.h"
#include "bookmarks.h"

#include <string.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gtkpixmapmenuitem.h>


static void bookmarks_menu_destroy (GaleonWindow *window);
static GtkWidget *bookmarks_menu_create_item (BookmarkItem *bm, GtkMenu *menu);
static void bookmarks_create_file_menu (GtkMenuItem *file_menuitem,
					BookmarkItem *bookmark);
static void bookmarks_create_file_submenu (GtkMenuItem *file_menuitem, 
					   BookmarkItem *bookmark);
static void bookmarks_menu_create_extra_items (GtkMenu *menu,
					       BookmarkItem *bookmark,
					       gint opts_location);
static void bookmarks_menu_create_recursively_cb (GtkMenuItem *menuitem,
						  gpointer data);
static void bookmarks_create_file_menu_cb (GtkMenuItem *mi, BookmarkItem *b);
GtkMenu *bookmarks_create_copy_menu_recursively (BookmarkItem *source,
						 BookmarkItem *b);
static gint bookmarks_smart_bm_dialog_single (GtkWidget *dialog, 
					      GaleonWindow *window, 
					      BookmarkItem *bi,
					      gboolean new_window);
static gint bookmarks_smart_bm_dialog_multi (GtkWidget *dialog, 
					     GaleonWindow *window, 
					     BookmarkItem *bi,
					     gboolean new_window);
static gboolean bookmark_has_subcategories (BookmarkItem *b);

/**
 * bookmarks_menu_create: create the bookmarks menu for a single window
 */
void
bookmarks_menu_create (GaleonWindow *window) 
{
	GtkMenuItem *file_menuitem;
	GtkWidget *app;
	GtkMenu *menu;
	
	return_if_not_window (window);

	/* skip if no menu on this window */
	if (!(window->menuBarOn))
	{
		return;
	}

	app = window->WMain;
	menu = GTK_MENU (gtk_object_get_data (GTK_OBJECT(app), 
					      "bookmarks_menu"));
	file_menuitem = GTK_MENU_ITEM (gtk_object_get_data (GTK_OBJECT(app), 
							    "file_bookmark"));

	if (window->bookmarks_tooltips != NULL)
	{
		gtk_object_destroy (GTK_OBJECT (window->bookmarks_tooltips));
	}
	
	window->bookmarks_tooltips = gtk_tooltips_new ();
	bookmarks_create_file_submenu (file_menuitem, bookmarks_root);
	bookmarks_menu_create_recursively (bookmarks_root, menu, file_menuitem,
					   GTK_TOOLTIPS(window->bookmarks_tooltips), 
					   TRUE, 
					   FALSE);
}

/**
 * bookmarks_menu_recreate: recreate the bookmarks menu for a single window
 */
void
bookmarks_menu_recreate (GaleonWindow *window)
{
	bookmarks_menu_destroy (window);
	bookmarks_menu_create (window);
}

static void
bookmarks_menu_destroy (GaleonWindow *window)
{
	GtkWidget *WMain = GTK_WIDGET (window->WMain);
	GtkWidget *bookmarks_menu;
	GtkWidget *separator;
	GList *li, *l;
	
	/* skip if no menu on this window */
	if (!(window->menuBarOn))
	{
		return;
	}
	
	separator = gtk_object_get_data (GTK_OBJECT (WMain),
					 "bookmarks_separator");
	bookmarks_menu = gtk_object_get_data (GTK_OBJECT (WMain),
					      "bookmarks_menu");
	l = gtk_container_children (GTK_CONTAINER (bookmarks_menu));
	
	/* first look for the separator */
	for (li = l; li != NULL; li = li->next)
	{
		if (li->data == separator)
		{
			break;
		}
	}
	
	/* then, destroy every widget after it */
	for (li = li->next; li != NULL; li = li->next)
	{
		if (GTK_IS_WIDGET (li->data))
		{
			gtk_widget_destroy (GTK_WIDGET (li->data));
		}
	}
	//FIXME: right???
	g_list_free (l);
}

/**
 * bookmarks_menu_create_recursively: Create menus for bookmarks
 * @bookmark: the root bookmark of the menu
 * @menu: the menu to fill with @bookmark children
 * @file_menuitem: the menu to fill with categories (to add bookmarks)
 * @tooltips: the tooltips object to use (or NULL if not needed)
 * @options: set to TRUE if you want the special options (add bookmark here...)
 * 
 * Creates the bookmarks menu and the file submenu if necessary. If you don't
 * need to create a file submenu pass NULL as file_menuitem. 
 * (usefull for the toolbar menus).
 **/
void
bookmarks_menu_create_recursively (BookmarkItem *bookmark, GtkMenu *menu, 
				   GtkMenuItem *file_menuitem,
				   GtkTooltips *tooltips, gboolean options, 
				   gboolean disable_tearoffs)
{
	GList *l; 
	GtkWidget *mi;
	gint opts_location = gnome_config_get_int
				("/galeon/General/bm_extra_items_location=1");

	g_return_if_fail (bookmark->type == BM_CATEGORY || 
			  bookmark->type == BM_AUTOBOOKMARKS);
	
	if (file_menuitem != NULL) 
		bookmarks_create_file_menu (file_menuitem, bookmark);
			
	if (bookmark == bookmarks_root)
	{
		if (opts_location == 0)
		{
			bookmarks_menu_create_extra_items (menu, bookmark, 0);
		}
		else if (opts_location == 2)
		{
			mi = bookmarks_menu_create_item (bookmark, menu);
			gtk_signal_connect (GTK_OBJECT(mi),
					    "button-release-event",
					    GTK_SIGNAL_FUNC (
					     bookmarks_button_release_event_cb),
					    bookmark);
			gtk_widget_show (mi);
			gtk_menu_append (menu, mi);
			mi = gtk_menu_item_new ();
			gtk_widget_show (mi);
			gtk_menu_append (menu, mi);
		}
	}

	for (l = bookmark->list; l != NULL; l = l->next) 
	{
		BookmarkItem *b = l->data;
		GtkWidget *submenu;

		switch (b->type)
		{
		case BM_CATEGORY:
		case BM_AUTOBOOKMARKS:
			submenu = gtk_menu_new ();
			gtk_menu_set_title (GTK_MENU (submenu), b->name);
			mi = bookmarks_menu_create_item (b, menu);
			gtk_menu_item_set_submenu (GTK_MENU_ITEM(mi), submenu);
			gtk_signal_connect 
				(GTK_OBJECT (submenu), "button-release-event", 
				 GTK_SIGNAL_FUNC (
					bookmarks_button_release_event_cb), b);

			gtk_object_set_data (GTK_OBJECT (mi), "bookmark", b);
			gtk_object_set_data (GTK_OBJECT (mi), "submenu", submenu);
			gtk_object_set_data (GTK_OBJECT (mi), "tooltips", tooltips);
			gtk_object_set_data (GTK_OBJECT (mi), "options", 
					     GINT_TO_POINTER (options));
			gtk_object_set_data (GTK_OBJECT (mi), "disable_tearoffs", 
					     GINT_TO_POINTER (disable_tearoffs));
			
			gtk_signal_connect (GTK_OBJECT (mi), "map", 
					    bookmarks_menu_create_recursively_cb,
					    NULL);
			break;
		case BM_SITE:
			mi = bookmarks_menu_create_item (b, menu);
			gtk_signal_connect_after
				(GTK_OBJECT (mi), "button_release_event", 
				 GTK_SIGNAL_FUNC (
					 bookmarks_button_release_event_cb), b);
			gtk_signal_connect_after
				(GTK_OBJECT (mi), "activate", 
				 GTK_SIGNAL_FUNC (bookmarks_activate_cb), b);
			if (tooltips)
				gtk_tooltips_set_tip (tooltips, GTK_WIDGET (mi),
						      b->url, NULL);
			break;
		case BM_SEPARATOR:
		default:
			mi = gtk_menu_item_new ();
			break;
		}
		gtk_widget_show (mi);
		gtk_menu_append (menu, mi);
	}

	if ( !((bookmark == bookmarks_root) && (opts_location == 0)) 
	     && options)
		bookmarks_menu_create_extra_items (menu, bookmark,
						   opts_location);

	if (gnome_preferences_get_menus_have_tearoff ()
	    && (bookmark != bookmarks_root)
	    && !disable_tearoffs)
	{
		GtkWidget *tearoff = gtk_tearoff_menu_item_new ();
		gtk_menu_prepend (GTK_MENU (menu), tearoff);
		gtk_widget_show (tearoff);
		gtk_signal_connect (GTK_OBJECT (tearoff),
				    "button-release-event",
				    GTK_SIGNAL_FUNC ( 
					bookmarks_button_release_event_cb),
				    NULL);
	}
}

static void
bookmarks_create_file_menu (GtkMenuItem *file_menuitem, BookmarkItem *bookmark)
{
	GList *l;
	GtkWidget *file_menu;

	if (!GTK_MENU_ITEM (file_menuitem)->submenu) {
		gtk_signal_disconnect_by_data
			(GTK_OBJECT (file_menuitem),
			 bookmark);
		bookmarks_create_file_submenu
			(file_menuitem, bookmark);
	}
	
	file_menu = 
		GTK_MENU_ITEM (file_menuitem)->submenu;
	
	for (l = bookmark->list; l ; l = l->next) {
		BookmarkItem *b = l->data;
		if ((b->type == BM_CATEGORY) || (b->type == BM_AUTOBOOKMARKS)) {
			GtkWidget *file = bookmarks_menu_create_item (b,
							GTK_MENU (file_menu));
			gtk_widget_show (file);
			gtk_menu_append (GTK_MENU (file_menu), file);
			if (bookmark_has_subcategories (b)) {
				bookmarks_create_file_submenu (GTK_MENU_ITEM (file), b);
				gtk_signal_connect (GTK_OBJECT(file),
						    "map",
						    bookmarks_create_file_menu_cb,
						    b);
			} else {
				gtk_signal_connect (GTK_OBJECT(file),
						    "activate",
						    bookmarks_file_bookmark_cb,
						    b);
			}			
		} else if (b->type == BM_SEPARATOR) {
			GtkWidget *mi = gtk_menu_item_new ();
			gtk_widget_show (GTK_WIDGET (mi));
			gtk_menu_append (GTK_MENU (file_menu), 
					 GTK_WIDGET (mi));
		}
	}
}

static void
bookmarks_menu_create_recursively_cb (GtkMenuItem *mi, gpointer data)
{
	BookmarkItem *b = gtk_object_get_data (GTK_OBJECT (mi), "bookmark");
	GtkMenu *submenu = gtk_object_get_data (GTK_OBJECT (mi), "submenu");
	GtkTooltips *tooltips = gtk_object_get_data (GTK_OBJECT (mi), "tooltips");
	gboolean options = 
		GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (mi), "options"));
	gboolean disable_tearoffs = 
		GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (mi), 
						      "disable_tearoffs"));

	gtk_signal_disconnect_by_func (GTK_OBJECT (mi),  
			    bookmarks_menu_create_recursively_cb,
			    NULL);

	gtk_object_remove_data (GTK_OBJECT (mi), "bookmark");
	gtk_object_remove_data (GTK_OBJECT (mi), "submenu");
	gtk_object_remove_data (GTK_OBJECT (mi), "tooltips");
	gtk_object_remove_data (GTK_OBJECT (mi), "options");
	gtk_object_remove_data (GTK_OBJECT (mi), "disable_tearoffs");

	bookmarks_menu_create_recursively (b, GTK_MENU (submenu), NULL, 
					   tooltips, options, disable_tearoffs);

}

static void
bookmarks_create_file_menu_cb (GtkMenuItem *mi, BookmarkItem *b)
{
	gtk_signal_disconnect_by_func (GTK_OBJECT (mi),  
			    bookmarks_create_file_menu_cb,
			    b);
	bookmarks_create_file_menu (mi, b);
}

static gboolean 
bookmark_has_subcategories (BookmarkItem *b) 
{
	GList *l;
	for (l = b->list; l; l = l->next) {
		BookmarkItem *bi = l->data;
		if ((bi->type == BM_CATEGORY) || (bi->type == BM_AUTOBOOKMARKS))
			return TRUE;
	}
	return FALSE;
}

/**
 * Creates the extra "Add bookmark to this folder" and "Open all bookmarks
 * in this folder" menuitems if necessary and places them appropriately in
 * the bookmarks menu.
 **/
static void
bookmarks_menu_create_extra_items (GtkMenu *menu, BookmarkItem *bookmark,
				   gint opts_location)
{
	GtkWidget *add_item, *open_all_item, *create_toolbar_item, *separator;

	g_return_if_fail (GTK_IS_MENU (menu));

	if (opts_location == 2)
		return;

	add_item = gtk_menu_item_new_with_label ("");
	label_set_accel_text (_("Add _bookmark to this folder"), 
			GTK_BIN (add_item)->child, GTK_WIDGET (menu),
			add_item);
	gtk_signal_connect (GTK_OBJECT (add_item), "button-release-event",
			    GTK_SIGNAL_FUNC(bookmarks_button_release_event_cb),
			    bookmark);
	gtk_signal_connect (GTK_OBJECT (add_item), "activate",
			    GTK_SIGNAL_FUNC (bookmarks_activate_cb), bookmark);
	gtk_object_set_data (GTK_OBJECT (add_item), "add_item",
			     GINT_TO_POINTER (1));

	open_all_item = gtk_menu_item_new_with_label ("");
	label_set_accel_text (_("_Open all bookmarks in this folder"),
			GTK_BIN (open_all_item)->child, GTK_WIDGET (menu),
			open_all_item);
	gtk_signal_connect (GTK_OBJECT(open_all_item), "button-release-event",
			    GTK_SIGNAL_FUNC(bookmarks_button_release_event_cb),
			    bookmark);
	gtk_signal_connect (GTK_OBJECT (open_all_item), "activate",
			    GTK_SIGNAL_FUNC (bookmarks_activate_cb), bookmark);
	gtk_object_set_data (GTK_OBJECT (open_all_item), "open_all_item",
			     GINT_TO_POINTER (1));

	create_toolbar_item = gtk_check_menu_item_new_with_label ("");
	label_set_accel_text (_("_Create toolbar from this folder"),
			GTK_BIN (create_toolbar_item)->child,
			GTK_WIDGET (menu), create_toolbar_item);
	gtk_check_menu_item_set_active 
				(GTK_CHECK_MENU_ITEM(create_toolbar_item), 
				 bookmark->create_toolbar);
	gtk_signal_connect (GTK_OBJECT(create_toolbar_item), 
			    "button-release-event",
			    GTK_SIGNAL_FUNC(bookmarks_button_release_event_cb),
			    bookmark);
	gtk_signal_connect (GTK_OBJECT (create_toolbar_item), "activate",
			    GTK_SIGNAL_FUNC (bookmarks_activate_cb), bookmark);
	gtk_object_set_data (GTK_OBJECT (create_toolbar_item), 
			     "create_toolbar_item",
			     GINT_TO_POINTER (1));

	separator = gtk_menu_item_new();

	if (opts_location == 0)
	{
		if ( (bookmark == bookmarks_root) &&
		     gtk_menu_get_attach_widget (menu) )
		{
			gtk_menu_append (GTK_MENU (menu), add_item);
			gtk_menu_append (GTK_MENU (menu), open_all_item);
			gtk_menu_append (GTK_MENU (menu), create_toolbar_item);
			gtk_menu_append (GTK_MENU (menu), separator);
		}
		else
		{	
			gtk_menu_prepend (GTK_MENU (menu), separator);
			gtk_menu_prepend (GTK_MENU (menu), create_toolbar_item);
			gtk_menu_prepend (GTK_MENU (menu), open_all_item);
			gtk_menu_prepend (GTK_MENU (menu), add_item);
		}
	}
	else
	{
		gtk_menu_append (GTK_MENU (menu), separator);
		gtk_menu_append (GTK_MENU (menu), add_item);
		gtk_menu_append (GTK_MENU (menu), open_all_item);
		gtk_menu_append (GTK_MENU (menu), create_toolbar_item);
	}
	gtk_widget_show (add_item);
	gtk_widget_show (open_all_item);
	gtk_widget_show (create_toolbar_item);
	gtk_widget_show (separator);
}

/**
 * Creates the file submenu for a bookmark catergory that has subcategories.
 * This functions does not fill the submenu.
 */
static void
bookmarks_create_file_submenu (GtkMenuItem *file_menuitem,
			       BookmarkItem *bookmark)
{
	GtkWidget *w;
	GtkWidget *file_menu = gtk_menu_new();

	gchar *stripped = strip_uline_accel (bookmark->name);
	gtk_menu_set_title (GTK_MENU (file_menu), stripped);
	g_free(stripped);

	gtk_menu_item_set_submenu
		(GTK_MENU_ITEM (file_menuitem),
		 file_menu);

	w = bookmarks_menu_create_item (bookmark, GTK_MENU (file_menu));
	gtk_signal_connect (GTK_OBJECT (w), "activate", 
			    bookmarks_file_bookmark_cb, bookmark);
	gtk_widget_show (w);
	gtk_menu_append (GTK_MENU (file_menu), w);
	
	w = gtk_menu_item_new ();
	gtk_widget_show (w);
	gtk_menu_append (GTK_MENU (file_menu), w);

	if (gnome_preferences_get_menus_have_tearoff ()){
		GtkWidget *tearoff = gtk_tearoff_menu_item_new ();
		gtk_menu_prepend (GTK_MENU (file_menu), tearoff);
		gtk_widget_show (tearoff);
	}
}

/* FIXME: obsoleted
 * These two functions need rewriting or just dumping them...
 * They are useful now just for dropping utls on the bookmark menu
 */
GtkMenu*
bookmarks_create_copy_menu (BookmarkItem *source)
{
	GtkMenu *res;
	res = bookmarks_create_copy_menu_recursively (source,
						      bookmarks_root);
	if (! res) {
		BookmarkItem *item = bookmarks_root;
		GtkWidget *menu_item = bookmarks_menu_create_item (item, res);
		res = GTK_MENU (gtk_menu_new ());
		gtk_widget_show (menu_item);
		gtk_menu_prepend (res, menu_item);
		gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
				    bookmarks_editor_bookmark_copy_cb, item);
		gtk_object_set_user_data (GTK_OBJECT (menu_item), 
					  source);
	} else {
		BookmarkItem *item = bookmarks_root;
		GtkWidget *menu_item = bookmarks_menu_create_item (item, res);
		GtkWidget *sep = gtk_menu_item_new ();
		gtk_widget_show (menu_item);
		gtk_widget_show (sep);
		gtk_menu_prepend (res, sep);
		gtk_menu_prepend (res, menu_item);
		gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
				    bookmarks_editor_bookmark_copy_cb, item);
		gtk_object_set_user_data (GTK_OBJECT (menu_item), 
					  source);
	}
	return res;
}

GtkMenu*
bookmarks_create_copy_menu_recursively (BookmarkItem *source, BookmarkItem *b)
{
	GtkMenu *res = GTK_MENU (gtk_menu_new ());
	GList *l;
	gboolean has_sub_categories = FALSE;
	g_assert (b->type == BM_CATEGORY);
	for (l = b->list; l != NULL; l = g_list_next (l)) {
		BookmarkItem *item = l->data;
		GtkWidget *menu_item, *sub_menu_item, *sep;
		if (item->type == BM_CATEGORY && !(item->alias_of)) {
			GtkMenu *submenu = bookmarks_create_copy_menu_recursively (source, item);
			has_sub_categories = TRUE;
			sub_menu_item = bookmarks_menu_create_item (item, res);
			if (submenu) {
				gtk_menu_item_set_submenu 
					(GTK_MENU_ITEM (sub_menu_item), 
					 GTK_WIDGET (submenu));
				menu_item = bookmarks_menu_create_item (item, submenu);
				gtk_widget_show (menu_item);
				sep = gtk_menu_item_new ();
				gtk_widget_show (sep);
				gtk_menu_prepend (submenu, sep);
				gtk_menu_prepend (submenu, menu_item);
				gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
						    bookmarks_editor_bookmark_copy_cb, item);
				gtk_object_set_user_data (GTK_OBJECT (menu_item), 
							  source);
			} else {
				gtk_signal_connect (GTK_OBJECT (sub_menu_item), 
						    "activate", bookmarks_editor_bookmark_copy_cb, item);
				gtk_object_set_user_data (GTK_OBJECT (sub_menu_item), 
							  source);
			}
			gtk_widget_show (sub_menu_item);
			gtk_menu_append (res, sub_menu_item);
		}
		
	}
	if (has_sub_categories) {
		return res;
	} else {
		gtk_widget_destroy (GTK_WIDGET (res));
		return NULL;
	}
}

/**
 * Returns a menuitem with the appropiate icon
 * menu must be the menu in which the item will be placed, this is required
 * for uline accel creation.
 */
/* This may be improved */
static GtkWidget *
bookmarks_menu_create_item (BookmarkItem *bm, GtkMenu *menu)
{
	GtkWidget *pixmap;
	const PixmapData *bm_icon;
	
	GtkWidget *m = gtk_pixmap_menu_item_new ();
	GtkWidget *l;
	GtkWidget *hb = gtk_hbox_new(FALSE, 0);

	g_return_val_if_fail (bm->name != NULL, NULL);

	if (bm->pixmap_data == NULL || bm->pixmap_data->pixmap == NULL)
	{
		l = gtk_label_new ("");
		if (strlen (bm->name) > BOOKMARKS_MENU_MAX_LENGTH)
		{
			gchar *s, *t;

			s = g_strndup(bm->name, BOOKMARKS_MENU_MAX_LENGTH);
			t = g_strconcat (s, "...", NULL);
			label_set_accel_text (t, l, GTK_WIDGET (menu), m);
			g_free (t);
			g_free (s);
		} 
		else
		{
			label_set_accel_text (bm->name, l, GTK_WIDGET (menu), m);
		}
		gtk_box_pack_start (GTK_BOX (hb), l, FALSE, FALSE, 2);
	}
	else
	{
		GtkWidget *image = gtk_pixmap_new (bm->pixmap_data->pixmap,
						   bm->pixmap_data->mask);
		gtk_box_pack_start (GTK_BOX (hb), GTK_WIDGET (image),
				    FALSE, FALSE, 2);
	}
	
	if (bm->type == BM_CATEGORY || bm->type == BM_AUTOBOOKMARKS)
	{
		pixmap = gtk_pixmap_new (folder_pixmap_data->pixmap,
					 folder_pixmap_data->mask);
	}
	else
	{
		bm_icon = bookmarks_get_siteicon (bm->url);
		pixmap = gtk_pixmap_new (bm_icon->pixmap, bm_icon->mask);
	}

	gtk_container_add (GTK_CONTAINER (m), hb);
	gtk_pixmap_menu_item_set_pixmap (GTK_PIXMAP_MENU_ITEM (m), pixmap);
	gtk_widget_show_all (hb);
	if (gnome_preferences_get_menus_have_icons ())
		gtk_widget_show (pixmap);
	return m;
}

/**
 * bookmarks_smart_bm_dialog: prompt user to complete the URL
 */
void
bookmarks_smart_bm_dialog (GaleonWindow *window, BookmarkItem *bi,
			   gboolean new_window)
{	
	GtkWidget *dialog;
	gchar *s = bi->url, *strippedname;
	int num_parameters = 0;
	int dlg_status = 0;

	return_if_not_window (window);	

	strippedname = strip_uline_accel (bi->name);
	dialog = gnome_dialog_new (strippedname,
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gnome_dialog_set_default (GNOME_DIALOG(dialog), 0);
	
	if (bi->pixmap_data == NULL)
	{
		GtkWidget *label;
		label = gtk_label_new (strippedname);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
				    label, TRUE, TRUE, 0);
	}
	else
	{
		GtkWidget *image = gtk_pixmap_new (bi->pixmap_data->pixmap,
						   bi->pixmap_data->mask);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
				    GTK_WIDGET (image), TRUE, FALSE, 0);
	}
	g_free (strippedname);

	/* count %s in the URL */
	s = bi->url;
	while ((s = strstr(s, "%s"))) {
		++num_parameters;
		++s;
	}

	if (num_parameters == 1)
	{
		dlg_status = bookmarks_smart_bm_dialog_single (dialog, window,
							       bi, new_window);
	}
	else if (num_parameters > 1)
	{
		dlg_status = bookmarks_smart_bm_dialog_multi (dialog, window,
							      bi, new_window);
	}
	if (dlg_status != -1)
		gnome_dialog_close (GNOME_DIALOG (dialog));
}

/**
 * bookmarks_smart_bm_dialog_single: prompt user to resolve smart bookmark with
 * exactly one parameter
 * Returns: return value of the dialog 
 */
static gint
bookmarks_smart_bm_dialog_single (GtkWidget *dialog, GaleonWindow *window, 
				  BookmarkItem *bi, gboolean new_window)
{
	GtkWidget *entry;
	gchar *s, *tmp, *result, *text, *translated_text;
	gint dlg_status;
	gboolean tabbed_mode;

	entry = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
			    entry, TRUE, TRUE, 0);
	
	gtk_widget_grab_focus (GTK_WIDGET (entry));
			    
	gnome_dialog_editable_enters (GNOME_DIALOG(dialog),
				      GTK_EDITABLE(entry));
	gtk_widget_show_all (dialog);

	if ((dlg_status = gnome_dialog_run (GNOME_DIALOG (dialog))) == 0) {
		s = strstr (bi->url, "%s");
		g_return_val_if_fail (s, 0);

		tmp = g_strndup (bi->url, s - bi->url);
		text = gtk_entry_get_text (GTK_ENTRY (entry));
		translated_text = bookmarks_translate_string (text);
		result = g_strconcat (tmp, translated_text, s+2, NULL);

		tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
		if (new_window)
			embed_create_from_url(window->active_embed, result,
					      TRUE, !tabbed_mode);
		else
			embed_load_url (window->active_embed, result);

		g_free (tmp);
		g_free (translated_text);
		g_free (result);
	}
	return dlg_status;
}

/**
 * bookmarks_smart_bm_dialog_multi: prompt user to resolve smart bookmark with
 * 2 or more parameters
 * Returns: return value of the dialog
 */
static gint
bookmarks_smart_bm_dialog_multi (GtkWidget *dialog, GaleonWindow *window, 
				 BookmarkItem *bi, gboolean new_window)
{
	gchar *start, *point, *tmp, *result;
	int i, dlg_status;
	GtkWidget *seperator, *entry, *label;
	GPtrArray *entries;
	gboolean tabbed_mode;

	seperator = gtk_hseparator_new();
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
			    seperator, TRUE, TRUE, 0);

	entries = g_ptr_array_new();

	start = bi->url;
	while(1)
	{
		point = strstr(start, "%s");
		if (point == start)
		{
			/* user-input part of URL */
			entry = gtk_entry_new ();
			gtk_box_pack_start
					(GTK_BOX (GNOME_DIALOG (dialog)->vbox),
					 entry, TRUE, TRUE, 0);
			g_ptr_array_add(entries, (gpointer) entry);
			start = point + 2;
			continue;
		}
		if (point == NULL)
			point = start + strlen(start);
		if (point == start)
			break;
		
		/* static part of URL */
		tmp = g_strndup(start, point-start);
		label = gtk_label_new (tmp);
		g_free(tmp);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
				    label, TRUE, TRUE, 0);
		g_ptr_array_add(entries, (gpointer) label);
		start = point;
	}

	/* set focus on the first entry */
	for (i = 0; i < entries->len; ++i) {
		if (GTK_IS_ENTRY (entries->pdata[i]))
		{
			gtk_widget_grab_focus (GTK_WIDGET (entries->pdata[i]));
			break;
		}
	}

	/* activating (hitting enter) on the last entry should select "OK" */
	for (i = entries->len - 1; i >= 0; --i) {
		if (GTK_IS_ENTRY (entries->pdata[i]))
		{
			gnome_dialog_editable_enters
				(GNOME_DIALOG(dialog), 
				 GTK_EDITABLE(entries->pdata[i]));
			break;
		}
	}

	gtk_widget_show_all (dialog);

	if ((dlg_status = gnome_dialog_run (GNOME_DIALOG (dialog))) == 0)
	{
		/* user selected OK, construct URL and go there */
		result = g_strnfill(0,0);
		for (i = 0; i < entries->len; ++i)
		{
			gchar *text = NULL, *translated_text = NULL;

			if (GTK_IS_ENTRY (entries->pdata[i]))
			{
				text = gtk_entry_get_text 
					(GTK_ENTRY (entries->pdata[i]));
				translated_text = bookmarks_translate_string
						   (text);
				tmp = g_strconcat (result, translated_text,
						   NULL);
				g_free (translated_text);
			}
			else
			{
				gtk_label_get 
					(GTK_LABEL (entries->pdata[i]), &text);
				tmp = g_strconcat (result, text, NULL);
			}
			g_free (result);
			result = tmp;
		}

		tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
		if (new_window)
			embed_create_from_url(window->active_embed, result,
					      TRUE, !tabbed_mode);
		else
			embed_load_url (window->active_embed, result);

		g_free(result);
	}
	g_ptr_array_free (entries, FALSE);
	return dlg_status;
}
