/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc.h"
#include "window.h"
#include "embed.h"
#include "mime.h"
#include "bookmarks.h"
#include "mozilla.h"

#include <gtk/gtkfilesel.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-dialog-util.h>

/* saved open URL dialog string */
gchar *open_url_str = NULL;

/* local function prototypes */
void file_open_ok_button_clicked_cb (const gchar *filename, gpointer data);
void file_open_cancel_button_clicked_cb (GtkButton *button, 
					 GaleonWindow *window);
void file_save_cancel_button_clicked_cb(GtkButton *button, 
					GaleonWindow *window);
void file_save_ok_button_clicked_cb(GtkButton *button, GaleonWindow *window);
void download_dir_cancel_button_clicked_cb (GtkButton *button, gpointer data);
void download_dir_ok_button_clicked_cb (GtkButton *button, gpointer data);

/**
 * closes a popup menu with a popup menus child item is selected
 */
gint popup_selection_done_cb (GtkMenuShell *menushell, GtkWidget *popup)
{
	gtk_menu_shell_deactivate (GTK_MENU_SHELL (popup));
	return FALSE;
}

/** 
 * open_url_ok_button_clicked_cb: open url dialog ok button pressed
 */
void
open_url_ok_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	GtkWidget *entry;
	gchar *string;

	return_if_not_window (window);

	entry = gtk_object_get_data (GTK_OBJECT (window->WMain),
				     "open_url_entry");
	g_return_if_fail (entry != NULL);

	string = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	if (string)
	{
		gchar *string2;

		if (open_url_str)
		{
			g_free (open_url_str);
		}

		string2 = bookmarks_parse_nick (string, NULL);
		if (string2) 
		{
			open_url_str = string2;
			g_free(string);
		} 
		else 
		{
			open_url_str = string;
		}
		window_load_url (window, open_url_str);
	}

	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/** 
 * open_url_clear_button_clicked_cb: open url dialog clear button clicked
 */
void
open_url_clear_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	GtkWidget *entry = gtk_object_get_data (GTK_OBJECT (window->WMain),
						"open_url_entry");

	g_return_if_fail (entry != NULL);

	gtk_editable_delete_text (GTK_EDITABLE (entry), 0, -1);
}

/** 
 * open_url_cancel_button_clicked_cb : open url dialog cancel button clicked
 */
void
open_url_cancel_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/**
 * file_open_ok_button_clicked_cb: file open dialog ok button clicked
 */
void
file_open_ok_button_clicked_cb (const gchar *file, gpointer data)
{
	GaleonWindow *window = (GaleonWindow *) data;
	char *url;
	
	url = g_strconcat ("file://", file, NULL);
	window_load_url(window, url);
	g_free (url);
	
	gnome_config_set_string (CONF_DIR_OPEN, file);
	gnome_config_sync();
}

/**
 * history_menu_menuitem_activate_cb: go to a selected item in the 
 * back/forward history menu 
 */
void
history_menu_menuitem_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
	GaleonEmbed *embed;
	gint index;

	/* fetch data */
	index = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_if_not_embed (embed);

	mozilla_session_history_go (embed, index);
}

/**
 * up_menu_menuitem_activate_cb: go to a selected item in the 
 * up directory level menu 
 */
void
up_menu_menuitem_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
	GaleonEmbed *embed;
	gint levels;

	/* fetch data */
	levels = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_if_not_embed (embed);

	embed_go_up (embed, levels);
}
/**
 * file_save_cancel_button_clicked_cb: file save dialog cancel button clicked
 */
void 
file_save_cancel_button_clicked_cb(GtkButton *button, GaleonWindow *window)
{
	GtkWidget *fs = glade_lookup_widget(GTK_WIDGET(button), "fssave");

	return_if_not_window (window);
	gtk_widget_destroy(fs);
}

/**
 * file_save_ok_button_clicked_cb: file save dialog ok button clicked
 */
void 
file_save_ok_button_clicked_cb(GtkButton *button, GaleonWindow *window)
{
	char *fname;
	GtkWidget *fs = glade_lookup_widget (GTK_WIDGET(button), "fssave");
	gboolean result = FALSE;
	gchar *url;

	return_if_not_window (window);
	fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs));

	url = (char*)gtk_object_get_data (GTK_OBJECT(fs), "url");

	if (url != NULL)
	{
		if (save_dir != NULL) {
			g_free (save_dir);
		}
		save_dir = g_dirname (fname);

		result = mozilla_save_url (window->active_embed, url, fname);
		
		g_free (url);
	}

	if (!result) gnome_error_dialog(_("Cannot save the file"));

	gtk_widget_destroy(fs);
}

/**
 * client_save_yourself_cb: called during window manager session shutdown,
 * we use our internal exit-with-session functionality to restore next
 * time the user starts Galeon.
 */
gboolean
client_save_yourself_cb (GnomeClient *client, gint phase, 
			 GnomeSaveStyle save_style, gint shutdown,
			 GnomeInteractStyle interact_style, gint fast)
{
	/* save session and exit */
	galeon_quit_with_session ();

	/* we're done */
	return TRUE;
}
