/* 
 *  Copyright (C) 2001 Ross Burton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc.h"

#include <string.h>
#include <gnome-xml/tree.h>

/* Export Prototypes */
void xbel_export_xbel(xmlDocPtr doc, BookmarkItem *b);
void xbel_export_notes(xmlNodePtr xmlNode, BookmarkItem *b);
void xbel_export_galeon_data(xmlNodePtr xmlNode, BookmarkItem *b);
void xbel_export_site(xmlNodePtr xmlNode, BookmarkItem *b);
void xbel_export_category(xmlNodePtr xmlNode, BookmarkItem *b);
void xbel_export_separator(xmlNodePtr xmlNode, BookmarkItem *b);
void xbel_export_node(xmlNodePtr xmlNode, BookmarkItem *b);
gboolean xbel_export_bookmarks (const gchar *filename, BookmarkItem *root);

/**
 * Write the bookmark tree as a XBEL file.  Takes a gchar* as a
 * filename, and a BookmarkItem as the root of the bookmark tree.
 */
gboolean
xbel_export_bookmarks (const gchar *filename, BookmarkItem *root)
{
        xmlDocPtr doc;

        /* Sanity checks */
        g_return_val_if_fail (root->type == BM_CATEGORY, -1);
        g_return_val_if_fail (filename != NULL, -1);

        /* Create a new XML document TODO: add the DTD*/
        doc = xmlNewDoc("1.0");
        /* Add the root XBEL node */
        xbel_export_xbel(doc, root);
        /* Try and save this document */
        if (xmlSaveFile(filename, doc) == -1)
        {
                xmlFreeDoc(doc);
                return FALSE;
        }
        xmlFreeDoc(doc);
        return 0;
}

/**
 * Treat a BM_CATEGORY item specially and use it as the root of the XBEL
 * tree
 */
void
xbel_export_xbel(xmlDocPtr doc, BookmarkItem *b)
{
        GList *list;
        gchar *strippedname;
        xmlNodePtr newNode;
        g_return_if_fail(doc != NULL);

        newNode = xmlNewDocNode(doc, NULL, "xbel", NULL);
        doc->root = newNode;
        xmlSetProp(newNode, "version", "1.0");
        /*
          In Galeon the root of the tree is a BM_CATEGORY. However, XBEL
          complicates the situation by have a special root element,
          XBEL. This is identical to FOLDER apart from the name of the
          tag... Thus this hack.
        */
        xbel_export_galeon_data(newNode, b);
        xbel_export_notes(newNode, b);
        strippedname = strip_uline_accel (b->name);
        xmlNewChild(newNode, NULL, "title", strippedname);
        g_free(strippedname);
        for (list = b->list; list != NULL; list = list->next)
                xbel_export_node(newNode, list->data);
}

/**
 * Add an item of unknown type to the tree. This delegates the
 * responsibility of adding nodes to the tree to other functions.
 */
void
xbel_export_node(xmlNodePtr xmlNode, BookmarkItem *b)
{
        g_return_if_fail(xmlNode != NULL);
        switch (b->type)
        {
        case BM_SITE:
                xbel_export_site(xmlNode, b);
                break;
        case BM_CATEGORY:
                xbel_export_category(xmlNode, b);
                break;
        case BM_SEPARATOR:
                xmlNewChild(xmlNode, NULL, "seperator", NULL);
                break;
        case BM_AUTOBOOKMARKS:
                g_warning ("Found autobookmarks - what do I do here?");
                break;
        default:
                g_warning ("Detected a bookmark item type that I can't export!");
                break;
        }
}

/**
 * Export the data which Galeon supports but XBEL doesn't by putting
 * it in metadata elements. This is done by create a metadata element
 * with an owner of "http://galeon.sourceforge.net" and adding child
 * nodes which describe features which XBEL doesn't support.
 */
void
xbel_export_galeon_data(xmlNodePtr xmlNode, BookmarkItem *b) {
        xmlNodePtr infoNode, newNode;
        g_return_if_fail(xmlNode != NULL);
        if (strlen (b->nick) > 0 || strlen(b->pixmap_file) > 0 || b->create_toolbar 
	    || b->create_context_menu) {
                infoNode = xmlNewChild(xmlNode, NULL, "info", NULL);
                infoNode = xmlNewChild(infoNode, NULL, "metadata", NULL);
                xmlSetProp(infoNode, "owner", "http://galeon.sourceforge.net/");
                /* export the nick name */
                if (strlen (b->nick) > 0)
                {
                        newNode = xmlNewChild (infoNode, NULL, "nick", b->nick);
                }
                /* pixmap path */
                if (strlen (b->pixmap_file) > 0)
                {
                        newNode = xmlNewChild (infoNode, NULL, "pixmap",
                                               b->pixmap_file);
                }
                /* create a toolbar */
                if (b->create_toolbar)
                {
                        newNode = xmlNewChild (infoNode, NULL, "create_toolbar", 
                                               b->create_toolbar ? "yes" : "no");
                }
                /* create a context menu */
                if (b->create_context_menu)
                {
                        newNode = xmlNewChild (infoNode, NULL, "create_context", 
                                               b->create_context_menu ? "yes" : "no");
                }
        }
}

void
xbel_export_notes(xmlNodePtr xmlNode, BookmarkItem *b) {
        if (strlen(b->notes) > 0)
        {
                xmlNewChild(xmlNode, NULL, "desc", b->notes);
        }
}


/**
 * Add a bookmark to the tree
 */
void
xbel_export_site(xmlNodePtr xmlNode, BookmarkItem *b)
{
        xmlNodePtr newNode;
        gchar *strippedname;
        g_return_if_fail(xmlNode != NULL);
        newNode = xmlNewChild(xmlNode, NULL, "bookmark", NULL);
        xmlSetProp(newNode, "href", b->url);
        strippedname = strip_uline_accel (b->name);
        xmlNewChild(newNode, NULL, "title", strippedname);
        g_free(strippedname);
        xbel_export_notes(newNode, b);
        xbel_export_galeon_data(newNode, b);
        /* TODO */
	/* TODO: aliases */
}

/**
 * Add a folder to the tree
 */
void
xbel_export_category(xmlNodePtr xmlNode, BookmarkItem *b)
{
        GList *list;
        gchar *strippedname;
        xmlNodePtr newNode;
        g_return_if_fail(xmlNode != NULL);

        newNode = xmlNewChild(xmlNode, NULL, "folder", NULL);
        xmlSetProp(newNode, "folded", b->expanded ? "no" : "yes");
        strippedname = strip_uline_accel (b->name);
        xmlNewChild(newNode, NULL, "title", strippedname);
        g_free(strippedname);
        xbel_export_notes(newNode, b);
        xbel_export_galeon_data(newNode, b);
	/* TODO: aliases. At the moment, this just looks if this node is an alias 
	   and does not ecport any children (to avoid infinite recursion) */
        /* now iterate over the children of this category */
	if (!b->alias_of)
		for (list = b->list; list != NULL; list = list->next)
			xbel_export_node(newNode, list->data);
}

