/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999,2000,2001 Jens Hoefkens
  jens@hoefkens.com

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/*

  $Log: kbgfibs.cpp,v $
  Revision 1.11  2001/02/10 15:29:34  hoefkens
  Proper use of i18n() allows for better translations.

  Revision 1.10  2001/01/25 20:38:26  leitner
  const in function implementations need to math const in function declarations
  in the *.h file.

  Revision 1.9  2001/01/08 06:52:03  hoefkens
  Yet another rewrite of the FIBS message handling. The next step is to use a
  separate class to parse the messages.  Move the TODO entries in a new file.

  Revision 1.8  2001/01/06 23:31:17  hoefkens
  Lots of small changes and fixed. Big thing is a rewrite of the join menu
  in kbgfibs.

  Revision 1.7  2001/01/03 22:29:43  hoefkens
  Added the total user count to the player list caption and made the FIBS
  menu tearable & removed the aboutToShow handler in the engine.

  Revision 1.6  2001/01/02 20:57:53  waba
  __FUNCTION__ macro is not supported on all compilers.

  Revision 1.5  2000/12/29 22:43:33  hoefkens
  New version number and landing of the first FIBS options. More cleanup
  of the FIBS engine. Almost done with cleanup.

  Revision 1.4  2000/12/28 20:44:00  hoefkens
  Separated the connection and the invitation dialog from the kbgfibs files.
  The latter was just too long (and in the future, other network-based
  engines might use the same classes).

  Revision 1.3  2000/12/28 18:59:43  hoefkens
  Reworked the configuration and setup handling for the FIBS engine. More
  cleaning of the FIBS files and a bug fix: opponent can't move now enables
  cube and dice, and plays a sound.

  Revision 1.2  2000/12/26 20:03:28  hoefkens
  Many additions to the FIBS engine. Small addition to the Playerlist and a
  rather long message to lukas.

  Revision 1.1  2000/12/21 19:58:57  hoefkens
  Initial checkin of application's source files

*/


#include "kbgfibs.h"
#include "kbgfibs.moc"

#include <kapp.h>
#include <kconfig.h>
#include <qtimer.h>
#include <qspinbox.h>
#include <qlayout.h>
#include <qstring.h>
#include <qgroupbox.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <kmainwindow.h>
#include <klineeditdlg.h>
#include <kmessagebox.h>
#include <qdatetime.h>
#include <qwhatsthis.h>
#include <kaudioplayer.h>
#include <kstddirs.h>

#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>

#include "kbgboard.h"
#include "clip.h"
#include "version.h"


// == constructor, destructor and setup ========================================

/*
 * Constructor
 */
KBgEngineFIBS::KBgEngineFIBS(QWidget *parent, QString *name)
	: KBgEngine(parent, name)
{
	/*
	 * No connection and not playing
	 */
	connection = 0;
	playing = false;

	/*
	 * No invitation dialog
	 */
	invitationDlg = 0;

	/*
	 * Creating, initializing and connecting the player list
	 */
	playerlist = new KFibsPlayerList(0, "fibs player list");

	connect(this, SIGNAL(fibsWhoInfo(const QString &)), this, SLOT(changeJoin(const QString &)));
	connect(this, SIGNAL(fibsLogout (const QString &)), this, SLOT(cancelJoin(const QString &)));
	connect(this, SIGNAL(gameOver()), this, SLOT(endGame()));

	connect(this, SIGNAL(fibsWhoInfo(const QString &)), playerlist, SLOT(changePlayer(const QString &)));
	connect(this, SIGNAL(fibsLogout (const QString &)), playerlist, SLOT(deletePlayer(const QString &)));
	connect(this, SIGNAL(fibsWhoEnd()),           playerlist, SLOT(stopUpdate()));
	connect(this, SIGNAL(fibsConnectionClosed()), playerlist, SLOT(stopUpdate()));
	connect(this, SIGNAL(changePlayerStatus(const QString &, int, bool)), 
		playerlist, SLOT(changePlayerStatus(const QString &, int, bool)));
	connect(playerlist, SIGNAL(fibsCommand(const QString &)), this, SLOT(handleCommand(const QString &)));
	connect(playerlist, SIGNAL(fibsInvite(const QString &)), this, SLOT(fibsRequestInvitation(const QString &)));

	/*
	 * Create, initialize and connect the chat window
	 */
	chatWindow = new KFibsChat(0, "chat window");

	connect(this, SIGNAL(chatMessage(const QString &)), chatWindow, SLOT(handleData(const QString &)));
	connect(this, SIGNAL(fibsStartNewGame(const QString &)), chatWindow, SLOT(startGame(const QString &)));
	connect(this, SIGNAL(gameOver()), chatWindow, SLOT(endGame()));
	connect(chatWindow, SIGNAL(fibsCommand(const QString &)), this, SLOT(handleCommand(const QString &)));
	connect(chatWindow, SIGNAL(fibsRequestInvitation(const QString &)), this, SLOT(fibsRequestInvitation(const QString &)));
	connect(chatWindow, SIGNAL(personalMessage(const QString &)), this, SLOT(personalMessage(const QString &)));
	connect(playerlist, SIGNAL(fibsTalk(const QString &)), chatWindow, SLOT(fibsTalk(const QString &)));

	/*
	 * Creating, initializing and connecting the menu
	 * ----------------------------------------------
	 */
	menu     = new QPopupMenu();
	fibsMenu = new QPopupMenu();
	respMenu = new QPopupMenu();
	joinMenu = new QPopupMenu();
	cmdMenu  = new QPopupMenu();
	optsMenu = new QPopupMenu();

	/*
	 * put the menus in the proper places
	 */
	createMenu(((KMainWindow *)parent)->actionCollection());

	menu->insertItem(i18n("&FIBS"), fibsMenu);

	fibsMenu->setCaption(i18n("FIBS Menu"));
	fibsMenu->insertTearOffHandle();
	

	/*
	 * Initialize the FIBS submenu - this is also put in the play menu
	 */
	conAction  = new KAction(i18n("&Connect"),    0, this, SLOT(connectFIBSWrapper()), this);
	newAction  = new KAction(i18n("New Account"), 0, this, SLOT(        newAccount()), this);
	disAction  = new KAction(i18n("&Disconnect"), 0, this, SLOT(    disconnectFIBS()), this);

	conAction->setEnabled(!connection); conAction->plug(fibsMenu);
	disAction->setEnabled( connection); disAction->plug(fibsMenu);
	newAction->setEnabled(!connection); newAction->plug(fibsMenu);

	fibsMenu->insertSeparator();

	(invAction  = new KAction(i18n("&Invite..."), 0, this, SLOT(inviteDialog()), this))->plug(fibsMenu);

	/*
	 * Create and fill the response menu. This is for all these: type this or 
	 * that messages from FIBS.
	 */
	cmdMenuID = fibsMenu->insertItem(i18n("&Commands"), cmdMenu); {
		
		(actAway  = new KAction(i18n("Away"), 0, this, SLOT(away()), this))->plug(cmdMenu);
		(actBack  = new KAction(i18n("Back"), 0, this, SLOT(back()), this))->plug(cmdMenu);

		actAway->setEnabled(true);
		actBack->setEnabled(false);
	}
	
	/*
	 * Create the server side options. This is preliminary and needs more work.
	 * The available options are skewed, since they refelect the needs of the
	 * author. Contact jens@hoefkens.com if your favorite option is missing.
	 */
	optsMenuID = fibsMenu->insertItem(i18n("&Options"), optsMenu); {
	
		for (int i = 0; i < NumFIBSOpt; i++) 
			fibsOpt[i] = 0;

		fibsOpt[OptReady]  = new KToggleAction(i18n("Ready to Play"),   0, this, SLOT(toggle_ready()),  this);
		fibsOpt[OptGreedy] = new KToggleAction(i18n("Greedy Bearoffs"), 0, this, SLOT(toggle_greedy()), this);
		fibsOpt[OptDouble] = new KToggleAction(i18n("Ask for Doubles"), 0, this, SLOT(toggle_double()), this);

		for (int i = 0; i < NumFIBSOpt; i++) 
			if (fibsOpt[i])
				fibsOpt[i]->plug(optsMenu);

	}
	
	/*
	 * Create and fill the response menu. This is for all these: type this or 
	 * that messages from FIBS.
	 */
	respMenuID = fibsMenu->insertItem(i18n("&Response"), respMenu); {
	
		(actAccept  = new KAction(i18n("Accept"), 0, this, SLOT(accept()), this))->plug(respMenu);
		(actReject  = new KAction(i18n("Reject"), 0, this, SLOT(reject()), this))->plug(respMenu);
		
		actAccept->setEnabled(false);
		actReject->setEnabled(false);

		respMenu->insertSeparator();

		(actConti  = new KAction(i18n("Join"),  0, this, SLOT(match_conti()), this))->plug(respMenu);
		(actLeave  = new KAction(i18n("Leave"), 0, this, SLOT(match_leave()), this))->plug(respMenu);

		actConti->setEnabled(false);
		actLeave->setEnabled(false);
	}
	
	/*
	 * Create the join menu and do not fill it (this happens at first 
	 * action setup.
	 */
	joinMenuID = fibsMenu->insertItem(i18n("&Join"), joinMenu); {
		numJoin = -1;
		
		actJoin[0] = new KAction("", 0, this, SLOT(join_0()), this);
		actJoin[1] = new KAction("", 0, this, SLOT(join_1()), this);
		actJoin[2] = new KAction("", 0, this, SLOT(join_2()), this);
		actJoin[3] = new KAction("", 0, this, SLOT(join_3()), this);
		actJoin[4] = new KAction("", 0, this, SLOT(join_4()), this);
		actJoin[5] = new KAction("", 0, this, SLOT(join_5()), this);
		actJoin[6] = new KAction("", 0, this, SLOT(join_6()), this);
		actJoin[7] = new KAction("", 0, this, SLOT(join_7()), this);
	}

	fibsMenu->setItemEnabled(joinMenuID, false);
	fibsMenu->setItemEnabled( cmdMenuID, false);
	fibsMenu->setItemEnabled(respMenuID, false);
	fibsMenu->setItemEnabled(optsMenuID, false);

	/*
	 * Continue with the FIBS menu
	 */
	fibsMenu->insertSeparator();
	
	(listAct = new KToggleAction(i18n("&Player List"), 0, this, SLOT(showList()), this))->plug(fibsMenu);
	(chatAct = new KToggleAction(i18n("&Chat"),        0, this, SLOT(showChat()), this))->plug(fibsMenu);

	connect(playerlist, SIGNAL(windowVisible(bool)), listAct, SLOT(setChecked(bool)));
	connect(chatWindow, SIGNAL(windowVisible(bool)), chatAct, SLOT(setChecked(bool)));

	/*
	 * Create message IDs. This sets up a lot of regular expressions.
	 */
	initPattern();

	/*
	 * Commit timer setup
	 */
	commitTimeout = new QTimer(this);
	connect(commitTimeout, SIGNAL(timeout()), this, SLOT(done()));

	/*
	 * Restore old settings
	 */
	readConfig();

	/*
	 * Update the menu actions
	 */
	listAct->setChecked(playerlist->isVisible());
	chatAct->setChecked(chatWindow->isVisible());
}

/*
 * Destructor deletes child objects if necessary
 */
KBgEngineFIBS::~KBgEngineFIBS()
{
	delete fibsMenu;
	delete joinMenu;
	delete respMenu;
	delete cmdMenu;
	delete optsMenu;

	delete menu;

	if (connection) delete connection;
	if (invitationDlg) delete invitationDlg;

	delete playerlist;
	delete chatWindow;
}


// == configuration handling ===================================================

/*
 * Restore settings and ask children to do the same
 */
void KBgEngineFIBS::readConfig()
{
	KConfig *config = kapp->config();
	config->setGroup("fibs engine");

	// history variables
	lastAway = config->readEntry("away_hist", "");

	// various options
	showMsg  = config->readBoolEntry("pers_msg", false);
	whoisInvite = config->readBoolEntry("whois_invite", false);

	// sound settings
	ringBell = config->readBoolEntry("ring_bell", false);
	useSound = config->readBoolEntry("use_sound", false);

	// commit timeout
	commitTimeoutLength = config->readNumEntry("timer", 2500);

	// connection information
	infoFIBS[FIBSHost] = config->readEntry("server", "fibs.com");
	infoFIBS[FIBSPort] = config->readEntry("port", "4321");
	infoFIBS[FIBSUser] = config->readEntry("user", "");
	infoFIBS[FIBSPswd] = config->readEntry("password", "");

	// automatic messages
	useAutoMsg[MsgBeg] = config->readBoolEntry("auto-beg", false);
	useAutoMsg[MsgLos] = config->readBoolEntry("auto-los", false);
	useAutoMsg[MsgWin] = config->readBoolEntry("auto-win", false);

	autoMsg[MsgBeg] = config->readEntry("msg-beg", "");
	autoMsg[MsgLos] = config->readEntry("msg-los", "");
	autoMsg[MsgWin] = config->readEntry("msg-win", "");

	// ask the children to read their config options
	playerlist->readConfig();
	chatWindow->readConfig();
}

/*
 * Save the engine specific settings and tell all clients
 */
void KBgEngineFIBS::saveConfig()
{
	KConfig *config = kapp->config();
	config->setGroup("fibs engine");

	// history variables
	config->writeEntry("away_hist", lastAway);

	// various options
	config->writeEntry("pers_msg", showMsg);
	config->writeEntry("whois_invite", whoisInvite);

	// sound settings
	config->writeEntry("ring_bell", ringBell);
	config->writeEntry("use_sound", useSound);

	// commit timeout
	config->writeEntry("timer", commitTimeoutLength);

	// connection information
	config->writeEntry("server", infoFIBS[FIBSHost]);
	config->readEntry("port", infoFIBS[FIBSPort]);
	config->readEntry("user", infoFIBS[FIBSUser]);
	config->readEntry("password", infoFIBS[FIBSPswd]);

	// automatic messages
	config->writeEntry("auto-beg", useAutoMsg[MsgBeg]);
	config->writeEntry("auto-los", useAutoMsg[MsgLos]);
	config->writeEntry("auto-win", useAutoMsg[MsgWin]);

	config->writeEntry("msg-beg", autoMsg[MsgBeg]);
	config->writeEntry("msg-los", autoMsg[MsgLos]);
	config->writeEntry("msg-win", autoMsg[MsgWin]);

	// ask the children to read their config options
	playerlist->saveConfig();
	chatWindow->saveConfig();
}
 
/*
 * Called when the setup dialog is positively closed
 */
void KBgEngineFIBS::setupOk()
{
	// various options
	showMsg = cbp->isChecked();
	whoisInvite = cbi->isChecked();

	// sound settings
	useSound = rbb->isChecked() || rbf->isChecked();
	ringBell = rbb->isChecked();

	// commit timeout
	commitTimeoutLength = 100*sbt->value();

	// connection information
	for (int i = 0; i < NumFIBS; i++)
		infoFIBS[i] = lec[i]->text();
	
	// automatic messages
	for (int i = 0; i < NumMsg; i++) {
		useAutoMsg[i] = cbm[i]->isChecked();
		autoMsg[i] = lem[i]->text();
	}

	// save settings
	saveConfig();
}

/*
 * Puts the FIBS specific setup into the dialog nb
 */
void KBgEngineFIBS::getSetupPages(QTabDialog *nb)
{
	/*
	 * FIBS, local options
	 * ===================
	 */
	QWidget *w = new QWidget(nb);
	QGridLayout *gl = new QGridLayout(w, 3, 1, 15);
  
	/*
	 * Group boxes
	 * -----------
	 */
	QButtonGroup *gbs = new QButtonGroup(i18n("Sounds:"), w);
	QGroupBox *gbt = new QGroupBox(i18n("Timer:"), w);
	QGroupBox *gbo = new QGroupBox(i18n("Options:"), w);

	gl->addWidget(gbt, 2, 0);
	gl->addWidget(gbo, 0, 0);
	gl->addWidget(gbs, 1, 0);

	/*
	 * Timer box
	 * ---------
	 */
	sbt = new QSpinBox(0, 600, 1, gbt);
	sbt->setValue(commitTimeoutLength/100);
	QLabel *lbt = new QLabel(i18n("Move Timout (1/10 seconds), 0 to disable"), gbt);

	QWhatsThis::add(sbt, i18n("After you finish your move, the command has to be sent to FIBS. "
				  "You can either do that manually (in which case you should set this "
				  "to 0) or you can specify an amount of time that has to pass before "
				  "the move is commited. If you undo a move during the timeout, the "
				  "timeout will be reset and restarted once you finish the move."));

	gl = new QGridLayout(gbt, 1, 2, 20);
	gl->addWidget(sbt, 0, 0, AlignLeft);
	gl->addWidget(lbt, 0, 1);

	/*
	 * Options
	 * -------
	 */
	cbp = new QCheckBox(i18n("Show copy of personal messages in main window"), gbo);
	cbi = new QCheckBox(i18n("Automatically request player info on invitation"), gbo);

	QWhatsThis::add(cbp, i18n("Usually, all messages sent directly to you by other players "
				  "are displayed only in the chat window. Check this box if you "
				  "would like to get a copy of these messages in the main window."));
	QWhatsThis::add(cbi, i18n("Check this box if you would like to receive information on "
				  "players that invite you to games."));

	cbp->setChecked(showMsg);
	cbi->setChecked(whoisInvite);

	gl = new QGridLayout(gbo, 2, 1, 20);
	gl->addWidget(cbp, 0, 0);
	gl->addWidget(cbi, 1, 0);

	/*
	 * Sound box
	 * ---------
	 */
	rbs = new QRadioButton(i18n("No sounds to signal game events"), gbs);
	rbb = new QRadioButton(i18n("Use system beep to signal game events"),  gbs);
	rbf = new QRadioButton(i18n("Play audio files to signal game events"), gbs);
		
	QWhatsThis::add(rbs, i18n("Check this if you don't want any sound events."));
	QWhatsThis::add(rbb, i18n("Check this if you would like to be reminded of certain game "
				  "events (e.g. your turn to roll or move) by the system bell."));
	QWhatsThis::add(rbf, i18n("Check this if you would like small audio clips to be played "
				  "as reminders of certain game events (e.g. when you have to roll "
				  "or move)."));

	rbs->setChecked(!useSound);
	rbb->setChecked( useSound &&  ringBell);
	rbf->setChecked( useSound && !ringBell);

	gl = new QGridLayout(gbs, 3, 1, 20);
	gl->addWidget(rbs, 0, 0);
	gl->addWidget(rbb, 1, 0);
	gl->addWidget(rbf, 2, 0);
	
	/*
	 * Put the page into the notebook
	 * ==============================
	 */
	gl->activate();
	nb->addTab(w, i18n("FIBS, &Local"));


	/*
	 * FIBS, other options
	 * ===================
	 */
	w = new QWidget(nb);
	gl = new QGridLayout(w, 3, 1, 20);

	QGroupBox *gbc = new QGroupBox(i18n("Connection:"), w);
	QGroupBox *gbm = new QGroupBox(i18n("Automatic Messages:"), w);

	gl->addWidget(gbc, 0, 0);
	gl->addWidget(gbm, 1, 0);

	/*
	 * Connection box
	 * --------------
	 */
	gl = new QGridLayout(gbc, 4, 2, 20);
	
	QLabel *lbc[NumFIBS];

	lbc[FIBSHost] = new QLabel(i18n("Server Name:"), gbc);
	lbc[FIBSPort] = new QLabel(i18n("Server Port:"), gbc);
	lbc[FIBSUser] = new QLabel(i18n("User Name:"), gbc);
	lbc[FIBSPswd] = new QLabel(i18n("Password:"), gbc);

	for (int i = 0; i < NumFIBS; i++) {
		lec[i] = new QLineEdit(infoFIBS[i], gbc);
		gl->addWidget(lbc[i], i, 0); 
		gl->addWidget(lec[i], i, 1);
	}
	lec[FIBSPswd]->setEchoMode(QLineEdit::Password);

	QWhatsThis::add(lec[FIBSHost], i18n("Enter here the host name of FIBS. With almost "
					    "absolute certainty this should be \"fibs.com\". "
					    "If you leave this blank, you will be asked again "
					    "at connection time."));
	QWhatsThis::add(lec[FIBSPort], i18n("Enter here the port number of FIBS. With almost "
					    "absolute certainty this should be \"4321\". "
					    "If you leave this blank, you will be asked again "
					    "at connection time."));
	QWhatsThis::add(lec[FIBSUser], i18n("Enter your login on FIBS here. If you do not have a "
					    "login yet, you should first create an account using "
					    "the corresponding menu entry. If you leave this blank, "
					    "you will be asked again at connection time."));
	QWhatsThis::add(lec[FIBSPswd], i18n("Enter your password on FIBS here. If you do not have a "
					    "login yet, you should first create an account using "
					    "the corresponding menu entry. If you leave this blank, "
					    "you will be asked again at connection time. The password "
					    "will not be visible."));

	/*
	 * Automatic messages
	 * ------------------
	 */
	gl = new QGridLayout(gbm, NumMsg, 2, 20);
		
	cbm[MsgBeg] = new QCheckBox(i18n("Start match :"), gbm);
	cbm[MsgWin] = new QCheckBox(i18n("Win match :"), gbm);
	cbm[MsgLos] = new QCheckBox(i18n("Lose match :"), gbm);	

	QWhatsThis::add(cbm[MsgBeg], i18n("If you want to send a standard greeting to your "
					  "opponent whenever you start a new match, check "
					  "this box and write the message into the entry "
					  "field."));
	QWhatsThis::add(cbm[MsgWin], i18n("If you want to send a standard message to your "
					  "opponent whenever you won a match, check this box "
					  "and write the message into the entry field."));
	QWhatsThis::add(cbm[MsgLos], i18n("If you want to send a standard message to your "
					  "opponent whenever you lost a match, check this box "
					  "and write the message into the entry field."));

	for (int i = 0; i < NumMsg; i++) {
		lem[i] = new QLineEdit(autoMsg[i], gbm);
		gl->addWidget(cbm[i], i, 0); 
		gl->addWidget(lem[i], i, 1);
		connect(cbm[i], SIGNAL(toggled(bool)), lem[i], SLOT(setEnabled(bool)));
		cbm[i]->setChecked(useAutoMsg[i]);
		lem[i]->setEnabled(useAutoMsg[i]);
		QWhatsThis::add(lem[i], QWhatsThis::textFor(cbm[i]));
	}

	/*
	 * put in the page
	 * ===============
	 */
	gl->activate();
	nb->addTab(w, "FIBS, &Other");


	/*
	 * Make sure we get the OK signal
	 * ==============================
	 */
	connect(nb, SIGNAL(applyButtonPressed()), this, SLOT(setupOk()));

	/*
	 * Fill setup of the children
	 * ==========================
	 */
	playerlist->getSetupPages(nb);
}


// == engine specific handling of messages =====================================

/*
 * Parse a rawboard description from FIBS and initialize all relevant 
 * parties - the results are all emitted
 */
void KBgEngineFIBS::newBoard(const QString &line)
{
	KBgBoardStatus game;
	const char *format = ("%*[^:]%*[:]%[^:]%*[:]%[^:]%*[:]%[^:]%*[:]%[^:]%*[:]%[^:]%*[:]"
			      "%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]"
			      "%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]"
			      "%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]"
			      "%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]"
			      "%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]%i%*[:]"
			      "%i%*[:]%i%*[:]");

	int board[26], dice[2][2], maydouble[2], scratch[4], onhome[2], onbar[2];
	char matchlength[4], pointsus[4], pointsthem[4];
	int turn, cube, wasdoubled, color, direction, redoubles;

	// split the incoming line at colons - latin1() is fine, since the string comes from FIBS.
	sscanf (line.latin1(), format,
		player, opponent, matchlength, pointsus, pointsthem,
		&board[ 0], &board[ 1], &board[ 2], &board[ 3],	&board[ 4], &board[ 5], 
		&board[ 6], &board[ 7],	&board[ 8], &board[ 9], &board[10], &board[11],
		&board[12], &board[13], &board[14], &board[15],	&board[16], &board[17], 
		&board[18], &board[19],	&board[20], &board[21], &board[22], &board[23],
		&board[24], &board[25], 
		&turn,
		&dice[US  ][0], &dice[US  ][1],	&dice[THEM][0], &dice[THEM][1],
		&cube,
		&maydouble[US], &maydouble[THEM],
		&wasdoubled,
		&color,
		&direction,
		&scratch[0], &scratch[1],    // home & bar
		&onhome[US], &onhome[THEM],  // on home
		&onbar[US],  &onbar[THEM],   // on bar
		&toMove,
		&scratch[2], &scratch[3],    // forced move & did crawford
		&redoubles);

	playing = (strcmp(player, "You") == 0);
	undoCounter = 0;

	game.setCube(cube, maydouble[US], maydouble[THEM]);
	game.setDirection(direction);
	game.setColor(color);
	for (int i = 1; i < 25; i++) {
		if (board[i] == 0 || color == board[i]/abs(board[i]))
			game.setBoard(i, US, abs(board[i]));
		else
			game.setBoard(i, THEM, abs(board[i]));
	}
	game.setDice(US  , 0, dice[US  ][0]); 
	game.setDice(US  , 1, dice[US  ][1]);
	game.setDice(THEM, 0, dice[THEM][0]); 
	game.setDice(THEM, 1, dice[THEM][1]);	

	game.setHome(US,   onhome[US  ]); 
	game.setHome(THEM, onhome[THEM]);

	game.setBar(US,   onbar[US  ]); 
	game.setBar(THEM, onbar[THEM]);

	/*
	 * Send the new board
	 */
	emit newState(&game);

	/*
	 * Create new caption string
	 */
	if (turn == 0) 
		caption = i18n("%1 (%2) vs. %3 (%4) - game over").
			arg(player).arg(pointsus).arg(opponent).arg(pointsthem);
	else if (strcmp(matchlength, "9999") == 0)
		caption = i18n("%1 (%2) vs. %3 (%4) - unlimited match").
			arg(player).arg(pointsus).arg(opponent).arg(pointsthem);
	else
		caption = i18n("%1 (%2) vs. %3 (%4) - %5 point match").
			arg(player).arg(pointsus).arg(opponent).arg(pointsthem).arg(matchlength);

	/*
	 * Emit everything
	 */
	emit statText(caption);
	emit allowMoving(playing && (color*turn > 0));
	emit allowCommand(Load, true );
	emit allowCommand(Undo, false);
	emit allowCommand(Redo, false);
	emit allowCommand(Done, false);
}


// == functions related to the invitation menu =================================

/*
 * Remove a player from the invitation list in the join menu
 */
void KBgEngineFIBS::cancelJoin(const QString &info)
{
	QRegExp patt = QRegExp("^" + info + " ");
	
	for (int i = 0; i <= numJoin; i++) {
		if (actJoin[i]->text().contains(patt)) {	
			// move all entries starting at i+1 up by one...
			for (int j = i; j < numJoin; j++)
				actJoin[j]->setText(actJoin[j+1]->text());
			actJoin[numJoin--]->unplug(joinMenu); 
			break;
		}	
	}
}

/*
 * Parse the information in info for the purposes of the invitation
 * submenu
 */
void KBgEngineFIBS::changeJoin(const QString &info)
{
	char name_p[100], name_o[100];
	float rate;
	int expi;
	
	/*
	 * Extract the name of the player, her opponent, rating and experience.
	 * It is okay to use latin1(), since the string is coming from FIBS.
	 */
	sscanf(info.latin1(), "%s %s %*s %*s %*s %f %i %*s %*s %*s %*s %*s", 
	       name_p, name_o, &rate, &expi);

	QString name = name_p;
	QString oppo = name_o;

	QString rate_s; rate_s.setNum(rate);
	QString expi_s; expi_s.setNum(expi);

	QRegExp patt = QRegExp("^" + name + " ");

	/*
	 * We have essentially two lists of names to check against: the ones
	 * that have invited us and are not yet in the menu and the ones that
	 * are already in the menu.
	 */

	if (numJoin > -1 && oppo != "-")
		cancelJoin(name);

	for (QStringList::Iterator it = invitations.begin(); it != invitations.end(); ++it) {

		if ((*it).contains(patt)) {

			QString text, menu;
			
			if ((*it).contains(QRegExp(" r$"))) {
				menu = i18n("R means resume", "%1 (R)").arg(name);
				text = i18n("%1 (experience %2, rating %3) wants to resume a saved match with you. "
					    "If you want to play, use the corresponding menu entry to join (or type "
					    "'join %4').").arg(name).arg(expi_s).arg(rate_s).arg(name);
			} else if ((*it).contains(QRegExp(" u$"))) {
				menu = i18n("U means unlimited", "%1 (U)").arg(name);
				text = i18n("%1 (experience %2, rating %3) wants to play an unlimited match with you. "
					    "If you want to play, use the corresponding menu entry to join (or type "
					    "'join %4').").arg(name).arg(expi_s).arg(rate_s).arg(name);
			} else {
				QString len = (*it).right((*it).length() - name.length() - 1);
				menu = i18n("If the format of the (U) and (R) strings is changed, it should also be changed here",
					    "%1 (%2)").arg(name).arg(len);
				text = i18n("%1 (experience %2, rating %3) wants to play a %4 point match with you. "
					    "If you want to play, use the corresponding menu entry to join (or type "
					    "'join %5').").arg(name).arg(expi_s).arg(rate_s).arg(len).arg(name);
			}
			emit serverString("rawwho " + name); // this avoids a race
			if (whoisInvite) {
				emit serverString("whois " + name);
				emit infoText("<font color=\"red\">" + text + "</font>");
			} else 
				emit infoText("<font color=\"red\">" + text + "</font><br/>");
			audioPlay(SoundJoin);
		

			for (int i = 0; i <=numJoin; i++)
				actJoin[i]->unplug(joinMenu);

			if (++numJoin > 7) numJoin = 7;

			for (int i = numJoin; i > 0; i--)
				actJoin[i]->setText(actJoin[i-1]->text());

			actJoin[0]->setText(menu);

			for (int i = 0; i <= numJoin; i++)
				actJoin[i]->plug(joinMenu); 
		       
			invitations.remove(it);
			break;
		}
	}
       
	/*
	 * If there are entries in the menu, enable it
	 */
	fibsMenu->setItemEnabled(joinMenuID, numJoin > -1);
}


// == various slots and functions ==============================================

/*
 * Several bookkeeping operations that have to be done at the
 * end of every game. Some of these may or may not be necessary
 * at a particular time, but they don't hurt either.
 */
void KBgEngineFIBS::endGame()
{
	playing = false;
	
	emit serverString("rawwho " + infoFIBS[FIBSUser]);
	
	actConti->setEnabled(false);
	actLeave->setEnabled(false);
	
	actAccept->setEnabled(false);
	actReject->setEnabled(false);
	
	emit allowCommand(Load, false);
	emit allowCommand(Undo, false);
	emit allowCommand(Done, false);
	emit allowCommand(Cube, false);
	emit allowCommand(Roll, false);
}

/*
 * Add some engine specific entried to the menu m. For consistency,
 * make sure that this uses the same menu shortcut as the 'other'
 * occurence of the FIBS menu in the constructor.
 */
void KBgEngineFIBS::addMenuItems(QPopupMenu *m)
{
	fibsMenuID = m->insertItem(i18n("&FIBS"), fibsMenu);
}

/*
 * Remove our entries from the menu m
 */
void KBgEngineFIBS::removeMenuItems(QPopupMenu *m)
{
 	m->removeItem(fibsMenuID);
}

/*
 * Toggle visibility of the player list
 */
void KBgEngineFIBS::showList()
{
	playerlist->isVisible() ? playerlist->hide() : playerlist->show();
}

/*
 * Toggle visibility of the chat window
 */
void KBgEngineFIBS::showChat()
{
	chatWindow->isVisible() ? chatWindow->hide() : chatWindow->show();	
}

/*
 * Plays a sound, depending on the event. This funtion should be called
 * for all possible events and it decides whether something has to be
 * played and whether the user wants a system beep or a sound clip.
 */
void KBgEngineFIBS::audioPlay(int event)
{
	/*
	 * Do nothing if the user doesn't like sound
	 */
	if (!useSound)
		return;

	/*
	 * Just beep if the user doesn't like fancy audio clips
	 */
	if (ringBell) {
		kapp->beep();
		return;
	}

	/*
	 * Individual sounds: play the correct one
	 */
	switch (event) {
	case SoundRoll:
		KAudioPlayer::play(locate("appdata", "sounds/" PROG_NAME "-roll.wav"));
		break;
	case SoundMove:
	case SoundJoin:
		KAudioPlayer::play(locate("appdata", "sounds/" PROG_NAME "-move.wav"));
		break;	
	case SoundWon:
		KAudioPlayer::play(locate("appdata", "sounds/" PROG_NAME "-won.wav"));
		break;
	case SoundLost:
		KAudioPlayer::play(locate("appdata", "sounds/" PROG_NAME "-lost.wav"));
		break;
	default:
		kapp->beep();
		break;
	}
}

/*
 * Process the last move coming from the board
 */
void KBgEngineFIBS::handleMove(QString *s)
{
	lastMove = *s;
	QString t = lastMove.left(1);
	int moves = t.toInt();

	emit allowCommand(Done, moves == toMove);
	emit allowCommand(Undo, moves > 0);

	/*
	 * Allow undo and possibly start the commit timer
	 */
	redoPossible &= ((moves < toMove) && (undoCounter > 0));
	emit allowCommand(Redo, redoPossible);
	if (moves == toMove && commitTimeoutLength) {
		emit allowMoving(false);
		commitTimeout->start(commitTimeoutLength, true);
	}	
}

/*
 * Done with the move
 */
void KBgEngineFIBS::done()
{
	// prevent the timer from expiring
	commitTimeout->stop();
	
	// no more moves
	emit allowMoving(false);

	// no more commands until it's our turn
	emit allowCommand(Load, false);
	emit allowCommand(Undo, false);
	emit allowCommand(Done, false);
	emit allowCommand(Cube, false);
	emit allowCommand(Roll, false);

	// Transform the string to FIBS cormat
	lastMove.replace(0, 2, "move ");
	lastMove.replace(pat[PlsChar], "-");

	// sent it to the server
	emit serverString(lastMove);
}

/*
 * Undo the last move
 */
void KBgEngineFIBS::undo()
{
	commitTimeout->stop();

	redoPossible = true;
	++undoCounter;

	emit allowMoving(true);
	emit allowCommand(Done, false);
	emit allowCommand(Redo, true);
	emit undoMove();
}

/*
 * Redo the last undone move
 */
void KBgEngineFIBS::redo()
{
	--undoCounter;
	emit redoMove();
}

/*
 * Double the cube - coming from the board
 */
void KBgEngineFIBS::doubleCube(const int w) 
{
	if (playing && w == US) cube();
}

/*
 * Roll the dice - coming from the board
 */
void KBgEngineFIBS::rollDice(const int w) 
{
	if (playing && w == US) roll();
}

/*
 * This engine passes all commands unmodified to the server
 */
void KBgEngineFIBS::handleCommand(QString const &cmd) 
{
	emit serverString(cmd);
}

/*
 * If we have a connection, we don't quit right away
 */
bool KBgEngineFIBS::queryExit()
{
	if (connection)
		disconnectFIBS();
	return true;
}

/*
 * Return the context menu
 */
QPopupMenu* KBgEngineFIBS::getMenu()
{
	return menu;
}

/*
 * This displays a copy of personal messages in the main window. 
 * Normally, these only get displayed in the chat window.  
 */
void KBgEngineFIBS::personalMessage(const QString &msg)
{
	if (showMsg)
		emit infoText(msg);
}


// == slots and functions for FIBS commands ====================================

/*
 * Accept the offer
 */
void KBgEngineFIBS::accept()
{
	actAccept->setEnabled(false);
	actReject->setEnabled(false);

	emit serverString("accept");
}

/*
 * Reject the offer
 */
void KBgEngineFIBS::reject()
{
	actAccept->setEnabled(false);
	actReject->setEnabled(false);
	
	emit serverString("reject");
}

/*
 * Continue a multi game match
 */
void KBgEngineFIBS::match_conti()
{
	actConti->setEnabled(false);
	actLeave->setEnabled(false);

	emit serverString("join");
}

/*
 * Leave a multi game match
 */
void KBgEngineFIBS::match_leave()
{
	actConti->setEnabled(false);
	actLeave->setEnabled(false);

	emit serverString("leave");
}

/*
 * Go away from the server for a little while. Offer the last know away
 * message as a default to the user.
 */
void KBgEngineFIBS::away()
{
	bool ret;
	QString msg = KLineEditDlg::getText(i18n("Please type the message that should be displayed to other\n"
						 "users while you are away."), 
					    lastAway, &ret, (QWidget *)parent());
	if (ret) {
		lastAway = msg;
		emit serverString("away " + msg);	
		actAway->setEnabled(false);
	}
}

/*
 * Toggle being ready for games
 */
void KBgEngineFIBS::toggle_ready()
{
	emit serverString("toggle ready");
}

/*
 * Toggle the use of greedy bearoffs
 */
void KBgEngineFIBS::toggle_greedy()
{
	emit serverString("toggle greedy");
}

/*
 * Toggle whether we will be asked to double/roll or not
 */
void KBgEngineFIBS::toggle_double()
{
	emit serverString("toggle double");
}

/*
 * Come back after being away.
 */
void KBgEngineFIBS::back()
{
	emit serverString("back");
}

/*
 * Double the cube
 */
void KBgEngineFIBS::cube()
{
	emit serverString("double");
}

/*
 * Roll the dice
 */
void KBgEngineFIBS::roll()
{
	emit serverString("roll");
}

/*
 * Reload the board
 */
void KBgEngineFIBS::load()
{
	emit serverString("board");
}

/*
 * Handle the menu short cuts for joining. This is not as pretty as it
 * could or should be, but it works and is easy to understand.
 */
void KBgEngineFIBS::join(const QString &msg)
{ 
	emit serverString("join " + msg.left(msg.find('(')));
}
void KBgEngineFIBS::join_0() { join(actJoin[0]->text()); }
void KBgEngineFIBS::join_1() { join(actJoin[1]->text()); }
void KBgEngineFIBS::join_2() { join(actJoin[2]->text()); }
void KBgEngineFIBS::join_3() { join(actJoin[3]->text()); }
void KBgEngineFIBS::join_4() { join(actJoin[4]->text()); }
void KBgEngineFIBS::join_5() { join(actJoin[5]->text()); }
void KBgEngineFIBS::join_6() { join(actJoin[6]->text()); }
void KBgEngineFIBS::join_7() { join(actJoin[7]->text()); }


// == invitation handling ======================================================

/*
 * Show the invitation dialog and set the name to player
 */
void KBgEngineFIBS::inviteDialog()
{
	fibsRequestInvitation("");
}

/*
 * Show the invitation dialog and set the name to player
 */
void KBgEngineFIBS::fibsRequestInvitation(const QString &player)
{
	if (!invitationDlg) {
		QString p = player;
		invitationDlg = new KBgInvite("invite");
		connect(invitationDlg, SIGNAL(inviteCommand(const QString &)), this, SLOT(handleCommand(const QString &)));
		connect(invitationDlg, SIGNAL(dialogDone()), this, SLOT(invitationDone()));
	}
	invitationDlg->setPlayer(player);
	invitationDlg->show();
}

/*
 * Finish off the invitation dialog
 */
void KBgEngineFIBS::invitationDone()
{
	delete invitationDlg;
	invitationDlg = 0;
}


// == connection handling ======================================================

/*
 * Establish a connection to the server and log in if the parameter login 
 * is true. 
 */
void KBgEngineFIBS::connectFIBS(const bool login)
{
	/*
	 * Make sure the connection parameter are properly set.
	 */
	if (!queryConnection(false))
		return;

	/*
	 * Visual feedback
	 */
	emit infoText(i18n("Connecting..."));

	/*
	 * Connect
	 */
	QString name   = "connection";

	if (!connection) {

		connection = new KBgConnection(parent(), &name, &infoFIBS[FIBSHost], infoFIBS[FIBSPort].toUShort());

		if (!connection || (connection->status() < 0)) {    
			emit infoText(i18n("Connection to the server could not be established!"));
			if (connection) delete connection;
			connection = 0;
			return;
		}
		
		conAction->setEnabled(false);
		newAction->setEnabled(false);
		disAction->setEnabled(true);
		
		fibsMenu->setItemEnabled( cmdMenuID, true);
		fibsMenu->setItemEnabled(respMenuID, true);
		fibsMenu->setItemEnabled(optsMenuID, true);

		/* 
		 * connect the various slots
		 */
		connect(connection, SIGNAL(connectionDown()), this, SLOT(serverDisconnected()));
		connect(connection, SIGNAL(newData(QString *)), this,  SLOT(handleServerData(QString *)));
		connect(this, SIGNAL(serverString(const QString&)), connection,  SLOT(sendData(const QString&)));
	}

	/*
	 * Initialize the rx state machine
	 */
	rxStatus = RxConnect;
	rxCollect = "";		
		
	/*
	 * Depending on whether somebody else wants to handle the login or not
	 */
	if (login) {

		/*
		 * Make sure the player list is empty when the whole list comes
		 * right after login
		 */
		playerlist->clear();
		
		/*
		 * Login, using the autologin feature of FIBS, before we even receive anything.
		 */
		QString entry;
		entry.setNum(CLIP_VERSION);
		emit serverString(QString("login ") + PROG_NAME + "-" + PROG_VERSION + " " + entry + " " 
				  + infoFIBS[FIBSUser] + " " + infoFIBS[FIBSPswd]);
	
	}

	/* 
	 * Some visual feedback and done
	 */
	emit infoText(i18n("Connected") + "<br/>"); 
}

/*
 * This is a simple interface to the connectFIBS function (we needed a 
 * void slot for the connect action).
 */
void KBgEngineFIBS::connectFIBSWrapper()
{
	connectFIBS(true);
}

/*
 * Create a new account on FIBS. Obviously, this will also create
 * a connection. The actual login is handled in the message parsing
 * state machine.
 */
void KBgEngineFIBS::newAccount()
{
	if (!queryConnection(true))
		return;
	rxStatus = RxNewLogin;
	rxCollect = "";
	connectFIBS(false);
	emit serverString("guest");
}

/*
 * Send a disconnection request to the server. As soon as the server has 
 * closed the connection, the serverDisconnected() slot is called
 */
void KBgEngineFIBS::disconnectFIBS()
{
	// send two lines in case we are stuck in the login phase
	emit serverString("quit");
	emit serverString("quit");
	emit infoText("<br/>" + i18n("Closing connection..."));
	emit fibsConnectionClosed();
}

/*
 * Connection to the server is closed for some (unknown) reason. Delete 
 * the connection object and get the actions into a proper state.
 */
void KBgEngineFIBS::serverDisconnected()
{
	delete connection;
	connection = 0;
	
	/*
	 * Flush whatever is left in the rxBuffer and send a note
	 */
	emit infoText(rxCollect + "<br/><hr/>");
	emit infoText(i18n("Connection closed.") + "<br/>");

	conAction->setEnabled(true);	
	newAction->setEnabled(true);	
	disAction->setEnabled(false);

	fibsMenu->setItemEnabled(joinMenuID, false);
	fibsMenu->setItemEnabled( cmdMenuID, false);
	fibsMenu->setItemEnabled(respMenuID, false);
	fibsMenu->setItemEnabled(optsMenuID, false);
}

/*
 * To establish a connection, we need to query the server name, the port 
 * number, the login and the password.
 */
bool KBgEngineFIBS::queryConnection(const bool newlogin)
{
	QString text, msg;
	bool first, ret = true;

	/* 
	 * query the connection parameter
	 */
	if (newlogin || infoFIBS[FIBSHost].isEmpty()) {

		msg = KLineEditDlg::getText(i18n("Enter the name of the server you want to connect to.\n"
						 "This should almost always be \"fibs.com\"."),
					    infoFIBS[FIBSHost], &ret, (QWidget *)parent());

		if (ret)
			infoFIBS[FIBSHost] = msg;
		else
			return false;
	      
	}
	if (newlogin || infoFIBS[FIBSPort].isEmpty()) {

		msg = KLineEditDlg::getText(i18n("Enter the port number on the server. "
						 "It should almost always be \"4321\"."),
					    infoFIBS[FIBSPort], &ret, (QWidget *)parent());
		
		if (ret)
			infoFIBS[FIBSPort] = msg;
		else
			return false;
	}
	if (newlogin || infoFIBS[FIBSUser].isEmpty()) {

		if (newlogin)	

			text = i18n("Enter the login you would like to use on the server %1. The login may not\n"
				    "contain spaces or colons. If the login you choose is not available, you'll later be\n"
				    "given the opportunity to pick another one.\n\n").arg(infoFIBS[FIBSHost]);
		
		else
			
			text = i18n("Enter your login on the server %1. If you don't have a login, you\n"
				    "should create one using the corresponding menu option.\n\n").arg(infoFIBS[FIBSHost]);


		first = true;
		do {
			msg = (KLineEditDlg::getText(text, infoFIBS[FIBSUser], &ret, 
						     (QWidget *)parent())).stripWhiteSpace();
			if (first) {
				text += i18n("The login may not contain spaces or colons!");
				first = false;
			}

		} while (msg.contains(' ') || msg.contains(':'));

		if (ret)
			infoFIBS[FIBSUser] = msg;
		else
			return false;
	}
	if (newlogin || infoFIBS[FIBSPswd].isEmpty()) {
		
		if (newlogin)		
	
			text = i18n("Enter the password you would like to use with the login %1\n"
				    "on the server %2. It may not contain colons.\n\n").
				arg(infoFIBS[FIBSUser]).arg(infoFIBS[FIBSHost]);

		else

			text = i18n("Enter the password for the login %1 on the server %2.\n\n").
				arg(infoFIBS[FIBSUser]).arg(infoFIBS[FIBSHost]);

		first = true;
		do {
			msg = (KLineEditDlg::getText(text, infoFIBS[FIBSPswd], &ret, 
						     (QWidget *)parent())).stripWhiteSpace();
			if (first) {
				text += i18n("The password may not contain colons!");
				first = false;
			}
			
		} while (msg.contains(' ') || msg.contains(':'));
		
		if (ret)
			infoFIBS[FIBSPswd] = msg;
		else
			return false;
	}
	
	/*
	 * Made it here, all paramters aquired
	 */
	return true;
}


// == message parsing ==========================================================

/*
 * Pattern setup - rather long and boring
 */
void KBgEngineFIBS::initPattern()
{
	QString pattern;
	
	/*
	 * Initialize the search pattern array
	 */
	pat[Welcome] = QRegExp(pattern.sprintf("^%d ", CLIP_WELCOME));
	pat[OwnInfo] = QRegExp(pattern.sprintf("^%d ", CLIP_OWN_INFO));
	pat[WhoInfo] = QRegExp(pattern.sprintf("^%d ", CLIP_WHO_INFO));
	pat[WhoEnde] = QRegExp(pattern.sprintf("^%d$", CLIP_WHO_END));
	pat[MotdBeg] = QRegExp(pattern.sprintf("^%d" , CLIP_MOTD_BEGIN));
	pat[MotdEnd] = QRegExp(pattern.sprintf("^%d" , CLIP_MOTD_END));
	pat[MsgPers] = QRegExp(pattern.sprintf("^%d ", CLIP_MESSAGE));
	pat[MsgDeli] = QRegExp(pattern.sprintf("^%d ", CLIP_MESSAGE_DELIVERED));
	pat[MsgSave] = QRegExp(pattern.sprintf("^%d ", CLIP_MESSAGE_SAVED));
	pat[ChatSay] = QRegExp(pattern.sprintf("^%d ", CLIP_SAYS));
	pat[ChatSht] = QRegExp(pattern.sprintf("^%d ", CLIP_SHOUTS));
	pat[ChatWis] = QRegExp(pattern.sprintf("^%d ", CLIP_WHISPERS));
	pat[ChatKib] = QRegExp(pattern.sprintf("^%d ", CLIP_KIBITZES));
	pat[SelfSay] = QRegExp(pattern.sprintf("^%d ", CLIP_YOU_SAY));
	pat[SelfSht] = QRegExp(pattern.sprintf("^%d ", CLIP_YOU_SHOUT));
	pat[SelfWis] = QRegExp(pattern.sprintf("^%d ", CLIP_YOU_WHISPER));
	pat[SelfKib] = QRegExp(pattern.sprintf("^%d ", CLIP_YOU_KIBITZ));
	pat[UserLin] = QRegExp(pattern.sprintf("^%d ", CLIP_LOGIN));
	pat[UserLot] = QRegExp(pattern.sprintf("^%d ", CLIP_LOGOUT));

	pat[NoLogin] = QRegExp("\\*\\* Unknown command: 'login'");
	pat[BegRate] = QRegExp("^rating calculation:$");
	pat[EndRate] = QRegExp("^change for ");
	pat[HTML_lt] = QRegExp("<");
	pat[HTML_gt] = QRegExp(">");
	pat[BoardSY] = QRegExp("^Value of 'boardstyle' set to 3");
	pat[BoardSN] = QRegExp("^Value of 'boardstyle' set to [^3]");
	pat[WhoisBG] = QRegExp("^Information about ");
	pat[WhoisE1] = QRegExp("^  No email address\\.$");
	pat[WhoisE2] = QRegExp("^  Email address: ");
	pat[SelfSlf] = QRegExp("^You say to yourself:");
	pat[Goodbye] = QRegExp("^                             Goodbye\\.");
	pat[GameSav] = QRegExp("The game was saved\\.$");
	pat[RawBord] = QRegExp("^board:");
	pat[YouTurn] = QRegExp("^It's your turn\\. Please roll or double");
	pat[PlsMove] = QRegExp("^Please move [1-6]+ pie");
	pat[EndWtch] = QRegExp("^You stop watching ");
	pat[BegWtch] = QRegExp("^You're now watching ");
	pat[BegGame] = QRegExp("^Starting a new game with ");
	pat[Reload1] = QRegExp("^You are now playing with ");
	pat[Reload2] = QRegExp(" has joined you. Your running match was loaded\\.$");
	pat[OneWave] = QRegExp(" waves goodbye.$");
	pat[TwoWave] = QRegExp(" waves goodbye again.$");
	pat[YouWave] = QRegExp("^You wave goodbye.$");
	pat[GameBG1] = QRegExp("start a [0-9]+ point match");
	pat[GameBG2] = QRegExp("start an unlimited match");
	pat[GameRE1] = QRegExp("are resuming their [0-9]+-point match");
	pat[GameRE2] = QRegExp("are resuming their unlimited match");
	pat[GameEnd] = QRegExp("point match against");
	pat[TabChar] = QRegExp("\\t");
	pat[SpcChar] = QRegExp(" ");
	pat[PlsChar] = QRegExp("\\+");
	pat[Invite0] = QRegExp(" wants to play a [0-9]+ point match with you\\.$");
	pat[Invite1] = QRegExp("^.+ wants to play a ");
	pat[Invite2] = QRegExp(" wants to resume a saved match with you\\.$");
	pat[Invite3] = QRegExp(" wants to play an unlimited match with you\\.$");
	pat[TypJoin] = QRegExp("^Type 'join ");
	pat[OneName] = QRegExp("^ONE USERNAME PER PERSON ONLY!!!");
	pat[YouAway] = QRegExp("^You're away. Please type 'back'");
	pat[YouBack] = QRegExp("^Welcome back\\.$");
	pat[YouMove] = QRegExp("^It's your turn to move\\.");
	pat[YouRoll] = QRegExp("^It's your turn to roll or double\\.");
	pat[TwoStar] = QRegExp("^\\*\\* ");
	pat[OthrNam] = QRegExp("^\\*\\* Please use another name\\. ");
	pat[BoxHori] = QRegExp("^ *\\+-*\\+ *$");
	pat[BoxVer1] = QRegExp("^ *|");
	pat[BoxVer2] = QRegExp("| *$");
	pat[YourNam] = QRegExp("Your name will be ");
	pat[GivePwd] = QRegExp("Please give your password:");	
	pat[RetypeP] = QRegExp("Please retype your password:");
	pat[HelpTxt] = QRegExp("^NAME$");
	pat[MatchB1] = QRegExp(" has joined you for a [0-9]+ point match\\.$");
	pat[MatchB2] = QRegExp(" has joined you for an unlimited match\\.$");
	pat[EndLose] = QRegExp(" wins the [0-9]+ point match [0-9]+-[0-9]+");
	pat[EndVict] = QRegExp(" win the [0-9]+ point match [0-9]+-[0-9]+");
	pat[RejAcpt] = QRegExp("Type 'accept' or 'reject'\\.$");
	pat[YouAcpt] = QRegExp("^You accept the double\\. The cube shows [0-9]+\\.");
	pat[ConLeav] = QRegExp("^Type 'join' if you want to play the next game, type 'leave' if you don't\\.$");
	pat[GreedyY] = QRegExp("^\\*\\* Will use automatic greedy bearoffs\\.");
	pat[GreedyN] = QRegExp("^\\*\\* Won't use automatic greedy bearoffs\\.");
	pat[BegBlnd] = QRegExp("^\\*\\* You blind ");
	pat[EndBlnd] = QRegExp("^\\*\\* You unblind ");
        pat[MatchB3] = QRegExp("^\\*\\* You are now playing a [0-9]+ point match with ");
        pat[MatchB4] = QRegExp("^\\*\\* You are now playing an unlimited match with ");
	pat[RejCont] = QRegExp("^You reject\\. The game continues\\.");
	pat[AcptWin] = QRegExp("^You accept and win ");
	pat[YouGive] = QRegExp("^You give up\\.");
	pat[DoubleY] = QRegExp("^\\*\\* You will be asked if you want to double\\.");
	pat[DoubleN] = QRegExp("^\\*\\* You won't be asked if you want to double\\.");
}

/*
 * Parse an incoming line and notify all interested parties - first match 
 * decides.
 */
void KBgEngineFIBS::handleServerData(QString *line)
{
	QString rawline = *line; // contains the line before it is HTML'fied

	/*
	 * Fix-up any HTML-like tags in the line
	 */
	line->replace(pat[HTML_lt], "&lt;");
	line->replace(pat[HTML_gt], "&gt;");

	/*
	 * FIBS sometimes sends tabs, where it should send 8 spaces...
	 */
	line->replace(pat[TabChar], "        ");

	switch (rxStatus) {

	case RxConnect: 
		handleMessageConnect(*line, rawline);
		break;

	case RxMotd:
		handleMessageMotd(*line);
		return;

	case RxWhois:
		handleMessageWhois(*line);
		break;

	case RxRating: 
		handleMessageRating(*line);
		break;
		
	case RxNewLogin:
		handleMessageNewLogin(*line);
		break;

	case RxIgnore:
		/*
		 * Ignore _ALL_ incoming strings - this is needed during the
		 * login phase, when the message box is open.
		 */
		break;

	case RxGoodbye:
		/*
		 * Receive the logout sequence. The string will be flushed by the 
		 * disconnectFIBS() callback
		 */
		line->replace(pat[SpcChar], "&nbsp;");
		rxCollect += QString("<font color=\"blue\"><tt>") + *line + "</tt></font><br/>";
		break;

	case RxNormal:
		handleMessageNormal(*line, rawline);
		break;
				
	default:
		/*
		 * This is a serious problem - latin1() is fine since the line comes from FIBS.
		 */
		cerr << "PROBLEM in KBgEngineFIBS::handleServerData: " << line->latin1() << endl;
	}
}

/*
 * Handle messages during the RxWhois state
 */
void KBgEngineFIBS::handleMessageWhois(const QString &line)
{
	rxCollect += "<br/>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;" + line;
	if (line.contains(pat[WhoisE1]) || line.contains(pat[WhoisE2])) {
		rxStatus = RxNormal;
		emit infoText("<font color=\"darkgreen\">" + rxCollect + "<br/></font>");
	}
}

/*
 * Handle messages during the RxRating state
 */
void KBgEngineFIBS::handleMessageRating(const QString &line)
{
	rxCollect += "<br/>" + line;
	if (line.contains(pat[EndRate]) && ++rxCount == 2) {
		emit infoText("<font color=\"blue\">" + rxCollect + "<br/></font>");
		rxStatus = RxNormal;
	}
}

/*
 * Handle messages during the RxMotd state
 */
void KBgEngineFIBS::handleMessageMotd(const QString &line)
{
	if (line.contains(pat[MotdEnd])) {
		rxStatus = RxNormal;
		emit infoText("<font color=\"blue\">" + rxCollect + "</font>");
		/*
		 * just to be on the safe side, we set the value of boardstyle.
		 * we do it here, since this is reasonably late in the login 
		 * procedure
		 */
		emit serverString("set boardstyle 3");
	} else {
		QString tline = line;
		tline.replace(pat[BoxHori], "<br/><hr/>");
		tline.replace(pat[BoxVer1], "");
		tline.replace(pat[BoxVer2], "");
		rxCollect += "<br/>" + tline;
	}
}

/*
 * Handle messages during the RxConnect state
 */
void KBgEngineFIBS::handleMessageConnect(const QString &line, const QString &rawline)
{
	/*
	 * Two possibilities: either we are logged in or we sent bad password/login
	 */
	if (line.contains("login:")) {
		/*
		 * This can only happen if the password/login is wrong.
		 */
		if (rxCollect.isEmpty()) {
			rxStatus = RxIgnore;
			int ret = KMessageBox::warningContinueCancel
				((QWidget *)parent(), i18n("There was a problem with "
							   "your login and password. "
							   "You can reenter\n"
							   "your login and password and " 
							   "try to reconnect."),
				 i18n("Wrong Login/Password"),
				 i18n("Reconnect"));
			if (ret == KMessageBox::Continue) {
				infoFIBS[FIBSUser] = "";
				infoFIBS[FIBSPswd] = "";
				connectFIBS(true); // will reset the rxStatus
			} else {
				rxStatus = RxConnect;
				emit serverString("");
				emit serverString("");
			}
			return;
		}			
		rxCollect.replace(pat[SpcChar], "&nbsp;");
		emit infoText("<hr/><tt>" + rxCollect + "<br/></tt>");
		rxCollect = "";
		return;
	}
	
	/*
	 * Ok, we are logged in! Now receive personal information. These
	 * are completely useless but what the heck.
	 */
	if (line.contains(pat[Welcome])) {
		char p[3][256];
		time_t tmp;
		// Using latin1() is okay, since the string comes from FIBS.
		int words = sscanf (line.latin1(), "%s%s%li%s", p[0], p[1], &tmp, p[2]);
		if (words >= 4) {
			QDateTime d; d.setTime_t(tmp);
			QString text = i18n("%1, last logged in from %2 at %3.").arg(p[1]).arg(p[2]).arg(d.toString());
			emit infoText("<hr><br/>" + text);
		}
		return;
	}
	
	/*
	 * Initial parsing of user options and making sure that settings needed
	 * by us are at the correct value. We use and ignore values according
	 * to the following list:
	 *
	 *     p[ 0]   - CLIP_OWN_INFO
	 *     p[ 1]   - name        -- IGNORE
	 *   OptAllowPip
	 *     n[ 0]   - autoboard   -- IGNORE
	 *   OptAutoDouble
	 *   OptAutoMove 
	 *     n[ 1]   - away        -- IGNORE 
	 *     n[ 2]   - bell        -- IGNORE
	 *   OptCrawford
	 *     n[ 3]   - double      -- IGNORE
	 *     n[ 4]   - expierience -- IGNORE
	 *   OptGreedy
	 *     n[ 6]   - moreboards  -- IGNORE and set to YES
	 *   OptMoves
	 *     n[ 8]   - notify      -- IGNORE and set to YES
	 *     rating  - rating      -- IGNORE
	 *   OptRatings
	 *   OptReady 
	 *     n[10]   - redoubles   -- IGNORE
	 *     n[11]   - report      -- IGNORE and set to YES
	 *   OptSilent
	 *     p[3]    - timezone
	 *
	 */
	if (line.contains(pat[OwnInfo])) {

		rxStatus = RxNormal;
		
		int fibsOptions[NumFIBSOpt];
		
		char p[3][256];
		int n[12];
		double rating;    
		
		// Using latin1() is okay, since the string comes from FIBS.
		int words = sscanf (line.latin1(), "%s%s%i%i%i%i%i%i%i%i%i%i%i%i%i%lf%i%i%i%i%i%s", 
				    p[0], p[1], 
				    &fibsOptions[OptAllowPip], 
				    &n[0], 
				    &fibsOptions[OptDouble], 
				    &fibsOptions[OptAutoMove], // equivalent to OptDouble, can be ignored
				    &n[1], &n[2], 
				    &fibsOptions[OptCrawford], 
				    &n[3], &n[4], 
				    &fibsOptions[OptGreedy], 
				    &n[6],
				    &fibsOptions[OptMoves],
				    &n[8], 
				    &rating,
				    &fibsOptions[OptRatings], 
				    &fibsOptions[OptReady], 
				    &n[10], &n[11], 
				    &fibsOptions[OptSilent], 
				    p[2]);
		
		if (words >= 22 && n[6] != 1) {
			/*
			 * need to get boards after new dice have arrived
			 */
			emit infoText("<font color=\"red\">" + i18n("The moreboards toggle has been set.") + "</font>");
			emit serverString("toggle moreboards");
		}
		if (words >= 22 && n[8] != 1) {
			/*
			 * need to know who logs out
			 */
			emit infoText("<font color=\"red\">" + i18n("The notify toggle has been set.") + "</font>");
			emit serverString("toggle notify");
		}
		if (words >= 22 && n[11] != 1) {
			/*
			 * want to know who starts playing games
			 */
			emit infoText("<font color=\"red\">" + i18n("The report toggle has been set.") + "</font>");
			emit serverString("toggle report");			
		}
		
		/*
		 * Set the correct toggles in the options menu
		 */
		fibsOpt[OptReady]->setChecked(fibsOptions[OptReady]);
		fibsOpt[OptDouble]->setChecked(!fibsOptions[OptDouble]);

		return;
	}
	
	/*
	 * The beginning of a new login procedure starts starts here
	 */
	if (line.contains(pat[OneName])) {
		rxStatus = RxNewLogin;
		emit infoText(QString("<font color=\"red\">") + rxCollect + "</font>");
		rxCollect = "";
		QString tmp = rawline;
		handleServerData(&tmp);
		return;
	}
	
	/*
	 * Still in the middle of the login sequence, still collecting information
	 */
	rxCollect += "<br/>" + line;
}

/*
 * Handle messages during the RxNewLogin state
 */
void KBgEngineFIBS::handleMessageNewLogin(const QString &line)
{
	/*
	 * Request the new login
	 */
	if (line.contains(pat[OneName])) {
		emit serverString(QString("name ") + infoFIBS[FIBSUser]);
		return;
	}
	/*
	 * Ooops, user name already exists
	 */
	if (line.contains(pat[OthrNam])) {
		QString text = i18n("The selected login is alreay in use! Please select another one.");
		bool ret, first = true;
		QString msg;
		
		do {
			msg = (KLineEditDlg::getText(text, infoFIBS[FIBSUser], &ret, 
						     (QWidget *)parent())).stripWhiteSpace();
			if (first) {
				text += i18n("\n\nThe login may not contain spaces or colons!");
				first = false;
			}
		} while (msg.contains(' ') || msg.contains(':'));
		
		if (ret) {
			infoFIBS[FIBSUser] =  msg;
			emit serverString("name " + msg);
		} else 
			emit serverString("bye");
		
		return;
	}
	/*
	 * first time we send the password
	 */
	if (line.contains(pat[YourNam])) {
		emit serverString(infoFIBS[FIBSPswd]);
		return;
	}
	/*
	 * second time we send the password
	 */
	if (line.contains(pat[GivePwd])) {
		emit serverString(infoFIBS[FIBSPswd]);
		return;
	}
	/*
	 * at this point we are done creating the account
	 */
	if (line.contains(pat[RetypeP])) {

		QString text = i18n("Your account has been created. Your new login is <u>%1</u>. To fully activate "
				    "this account, I will now close the connection. Once you reconnect, you can start "
				    "playing backgammon on FIBS.").arg(infoFIBS[FIBSUser]);
		emit infoText("<br/><hr/><font color=\"blue\">" + text + "</font><br/><hr/>");
		emit serverString("bye");
		rxStatus = RxNormal;
		rxCollect = "";
		return;
	}
	return;
}

/*
 * Handle all normal messages - during the RxNormal state
 */
void KBgEngineFIBS::handleMessageNormal(QString &line, QString &rawline)
{
	// - ignored ----------------------------------------------------------------------

	/*
	 * For now, the waves are ignored. They should probably go into
	 * the chat window -- but only optional
	 */
	if (line.contains(pat[OneWave]) || line.contains(pat[TwoWave]) || line.contains(pat[YouWave])) {
		
		return; 
	}

	/*
	 * These messages used to go into the games window. If KBackgammon
	 * ever gets a games window, they should be in there. For now, they
	 * are ignored.
	 */
	else if (line.contains(pat[GameBG1]) || line.contains(pat[GameBG2]) || line.contains(pat[GameRE1]) || 
		 line.contains(pat[GameRE2]) || line.contains(pat[GameEnd])) {
		
		return;
	}

	/*
	 * Artefact caused by the login test procedure utilized.
	 */
	else if (line.contains(pat[NoLogin])) {
			
		return;
	}
	
	// --------------------------------------------------------------------------------

	/*
	 * Chat and personal messages - note that the chat window sends these messages 
	 * back to us so we can display them if the user wants that.
	 */
	else if (line.contains(pat[ChatSay]) || line.contains(pat[ChatSht]) || line.contains(pat[ChatWis]) ||
		 line.contains(pat[ChatKib]) || line.contains(pat[SelfSay]) || line.contains(pat[SelfSht]) || 
		 line.contains(pat[SelfWis]) || line.contains(pat[SelfKib]) || line.contains(pat[SelfSlf]) || 
		 line.contains(pat[MsgPers]) || line.contains(pat[MsgDeli]) || line.contains(pat[MsgSave])) {
		
		emit chatMessage(line);
		return;
	}
	
	// --------------------------------------------------------------------------------

	/*
	 * Beginning of games. In all these cases we are playing and not watching.
	 */
	else if (line.contains(pat[MatchB1]) || line.contains(pat[MatchB2])) {
		
		if (useAutoMsg[MsgBeg] && !autoMsg[MsgBeg].stripWhiteSpace().isEmpty())
			emit serverString("kibitz " + autoMsg[MsgBeg]);
	} 
	else if (line.contains(pat[MatchB3]) || line.contains(pat[MatchB4])) {
		
		if (useAutoMsg[MsgBeg] && !autoMsg[MsgBeg].stripWhiteSpace().isEmpty())
			emit serverString("kibitz " + autoMsg[MsgBeg]);
		line = "<font color=\"red\">" + line + "</font>";
	}

	// --------------------------------------------------------------------------------

	/*
	 * The help should be handled separately. A fairly complete implementation of a 
	 * help parsing can be found in KFibs.
	 */
	else if (line.contains(pat[HelpTxt])) {
		
		// do nothing
	}
	
	// --------------------------------------------------------------------------------

	/*
	 * Simple cases without the need for many comments...
	 */
	else if (line.contains(pat[RawBord])) {
		
		newBoard(rawline);
		return;
	}
	else if (line.contains(pat[PlsMove]) || line.contains(pat[YouMove])) {
		
		audioPlay(SoundMove);
	}
	
	// --------------------------------------------------------------------------------
	
	/*
	 * Being away and coming back
	 */
	else if (line.contains(pat[YouAway])) {
		
		emit changePlayerStatus(infoFIBS[FIBSUser], Away, true);
		actBack->setEnabled(true);
		line += "<br/>&nbsp;&nbsp;" + i18n("(or use the corresponding menu entry to join the match)");
	}
	else if (line.contains(pat[YouBack])) {	      

		emit changePlayerStatus(infoFIBS[FIBSUser], Away, false);
		actBack->setEnabled(false);
		actAway->setEnabled(true);
	}

	// --------------------------------------------------------------------------------

	/*
	 * Catch the response of the user responding to double or resign
	 */
	else if (line.contains(pat[YouGive]) || line.contains(pat[RejCont]) || line.contains(pat[AcptWin])) {

		actAccept->setEnabled(false);
		actReject->setEnabled(false);
	}

	// --------------------------------------------------------------------------------

	/*
	 * Catch the responses to newly set toggles
	 */
	else if (line.contains(pat[GreedyY]) || line.contains(pat[GreedyN])) {

		fibsOpt[OptGreedy]->setChecked(line.contains(pat[GreedyY]));
		line = "<font color=\"red\">" + line + "</font>";
	}
	else if (line.contains(pat[DoubleY]) || line.contains(pat[DoubleN])) {

		fibsOpt[OptDouble]->setChecked(line.contains(pat[DoubleY]));
		line = "<font color=\"red\">" + line + "</font>";
	}

	// --------------------------------------------------------------------------------

	/*
	 * It's our turn to roll or double
	 */
	else if (line.contains(pat[YouTurn]) || line.contains(pat[YouRoll])) {

		emit allowCommand(Cube, playing);
		emit allowCommand(Roll, playing);
		emit statText(caption);
		audioPlay(SoundRoll);
	}
	
	// --------------------------------------------------------------------------------

	/*
	 * Got an invitation for a match
	 */
	else if (line.contains(pat[Invite0]) || line.contains(pat[Invite2]) || line.contains(pat[Invite3])) {
		
		rxCollect = rawline.left(rawline.find(' '));
		emit serverString("rawwho " + rxCollect);

		if (line.contains(pat[Invite0])) {
			rawline.replace(pat[Invite1], "");
			rawline = rxCollect + " "+ rawline.left(rawline.find(' '));
		} else if (line.contains(pat[Invite2])) {
			rawline = rxCollect + " r";
		} else if (line.contains(pat[Invite3])) {
			invitations += rxCollect + " u";
		}			
		invitations += rawline;
		return; // will be printed once the rawwho is received
	}

	// - rx status changes ------------------------------------------------------------

	else if (line.contains(pat[WhoisBG])) {
		rxStatus = RxWhois;
		rxCollect = QString("<br/><u>") + line + "</u>";
		return;
	}
	else if (line.contains(pat[MotdBeg])) {
		rxStatus = RxMotd;
		rxCollect = "";
		return;
	}
	else if (line.contains(pat[BegRate])) {
		rxStatus = RxRating;
		rxCount = 0;
		rxCollect = "<br/>" + line;
		return;
	}
	else if (line.contains(pat[Goodbye])) {
		rxStatus = RxGoodbye;
		rxCollect = "<br/><hr/><br/>";
		handleServerData(&rawline); // danger: recursion!
		return;
	}

	// --------------------------------------------------------------------------------
	// --------------------------------------------------------------------------------
	// --------------------------------------------------------------------------------
	// --------------------------------------------------------------------------------

	/*
	 * Continue a mutli game match? We have to either leave or continue
	 */
	else if (line.contains(pat[ConLeav])) {
		actConti->setEnabled(true);
		actLeave->setEnabled(true);
		line.append("<br/>&nbsp;&nbsp;" + i18n("(or use the corresponding menu "
						       "entry to leave or continue the match)"));
	}
	/*
	 * Beginning and end of user updates
	 */
	else if (line.contains(pat[WhoInfo])) {
		rawline.replace(pat[WhoInfo], "");
		if (rawline.contains(QRegExp("^" + infoFIBS[FIBSUser] + " "))) {
			int ready;
			// Using latin1() is fine, since the string is coming from FIBS.
			sscanf(rawline.latin1(), "%*s %*s %*s %i %*s %*s %*s %*s %*s %*s %*s %*s", &ready);
			fibsOpt[OptReady]->setChecked(ready);
		}
		emit fibsWhoInfo(rawline);
		return;
	}
	else if (line.contains(pat[WhoEnde])) {
		emit fibsWhoEnd();
		return;
	}
	/*
	 * This message is ignored. The instruction is given elsewhere (and slightly 
	 * delayed in the flow of time).
	 */
	if (line.contains(pat[TypJoin])) {
		return;
	}
	/*
	 * Watching other players
	 */
	else if (line.contains(pat[BegWtch])) {
		emit allowCommand(Load, true);
		rawline.replace(pat[BegWtch], "");
		rawline.truncate(rawline.length()-1);
		emit fibsStartNewGame(rawline);
		load();
	}
	else if (line.contains(pat[EndWtch])) {
		emit gameOver();
	}
	/*
	 * Blinding of players, the actual blind is handled by 
	 * the player list
	 */
	else if (line.contains(pat[BegBlnd])) {
		rawline.replace(pat[BegBlnd], "");
		rawline.truncate(rawline.length()-1);
		emit changePlayerStatus(rawline, Blind, true);
		line = "<font color=\"red\">" + line + "</font>";
	}
	else if (line.contains(pat[EndBlnd])) {
		rawline.replace(pat[EndBlnd], "");
		rawline.truncate(rawline.length()-1);
		emit changePlayerStatus(rawline, Blind, false);
		line = "<font color=\"red\">" + line + "</font>";
	}
	/*
	 * Starting or reloading games or matches 
	 */
	else if (line.contains(pat[BegGame])) {
		rawline.replace(pat[BegGame], "");
		rawline.truncate(rawline.length()-1);
		emit fibsStartNewGame(rawline);
		fibsOpt[OptDouble]->setChecked(true);
		fibsOpt[OptGreedy]->setChecked(false);
		actConti->setEnabled(false);
		actLeave->setEnabled(false);
	}
	else if (line.contains(pat[Reload1])) {
		rawline.replace(pat[Reload1], "");      
		rawline = rawline.left(rawline.find(' '));
		rawline.truncate(rawline.length()-1);
		emit fibsStartNewGame(rawline);
		fibsOpt[OptDouble]->setChecked(true);
		fibsOpt[OptGreedy]->setChecked(false);
		actConti->setEnabled(false);
		actLeave->setEnabled(false);
		load();
	}
	else if (line.contains(pat[Reload2])) {
		rawline.replace(pat[Reload2], "");      
		emit fibsStartNewGame(rawline);
		fibsOpt[OptDouble]->setChecked(true);
		fibsOpt[OptGreedy]->setChecked(false);
		actConti->setEnabled(false);
		actLeave->setEnabled(false);
		load();
	}
	/*
	 * Opponent offered resignation or the cube. We have to accept 
	 * or reject the offer.
	 */
	else if (line.contains(pat[RejAcpt])) {
		actAccept->setEnabled(true);
		actReject->setEnabled(true);
		line += "<br/>&nbsp;&nbsp;" + i18n("(or use the corresponding menu "
						   "entry to accept or reject the offer)");
	}
	/*
	 * This is strange: FIBS seems to not send a newline at the 
	 * end of this pattern. So we work around that.
	 */
	else if (line.contains(pat[YouAcpt])) {
		actAccept->setEnabled(false);
		actReject->setEnabled(false);
		rawline.replace(pat[YouAcpt], "");
		line.truncate(line.length()-rawline.length());
		if (!rawline.stripWhiteSpace().isEmpty()) { 
			handleServerData(&rawline);
		}
	}
	/*
	 * Ending of games
	 */
	else if (line.contains(pat[EndLose])) {
		if (playing) {
			audioPlay(SoundLost);
			if (useAutoMsg[MsgLos] && !autoMsg[MsgLos].stripWhiteSpace().isEmpty())
				emit serverString(QString("tell ") + opponent + " " + autoMsg[MsgLos]);
		}
		emit gameOver();
	}
	else if (line.contains(pat[EndVict])) {
		if (playing) {
			audioPlay(SoundWon);
			if (useAutoMsg[MsgWin] && !autoMsg[MsgWin].stripWhiteSpace().isEmpty())
				emit serverString(QString("tell ") + opponent + " " + autoMsg[MsgWin]);
		}
		emit gameOver();
	}
	else if (line.contains(pat[GameSav])) {
		emit gameOver();
	}
	/*
	 * User logs out. This has to be signalled to the player 
	 * list. Get the true user names by working on the rawline.
	 */
	else if (line.contains(pat[UserLot])) {
		rawline.replace(pat[UserLot], "");
		emit fibsLogout(rawline.left(rawline.find(' ')));
		return;
	}
	/*
	 * Emit the name of the newly logged in user.
	 */
	else if (line.contains(pat[UserLin])) {
		rawline.replace(pat[UserLin], "");
		emit fibsLogin(rawline.left(rawline.find(' ')));
		return;
	}
	/*
	 * Special attention has to be paid to the proper setting of 
	 * the 'boardstyle' variable, since we will not be able to display
	 * the board properly without it.
	 */
	else if (line.contains(pat[BoardSY])) {
		// ignored
		return;
	}
	else if (line.contains(pat[BoardSN])) {
		emit serverString("set boardstyle 3");
		emit infoText(QString("<font color=\"red\"><br/>")
			      + i18n("You should never set the 'boardstyle' variable "
				     "by hand! It is vital for proper functioning of "
				     "this program that it remains set to 3. It has "
				     "been reset for you.")
			      + "<br/></font>");
		return;
	}
	/*
	 * This is the final fall through: if the line started with ** and 
	 * hasn't been processed, make it red, since it is a server resp.
	 * to something we just did.
	 */
	else if (line.contains(pat[TwoStar])) {
		line = "<font color=\"red\">" + line + "</font>"; 
	}

	// --------------------------------------------------------------------------------

	/*
	 * Print whatever part of the line made it here
	 */
	emit infoText(line);
}

// EOF
