/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2001 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#include "catalog.h"
#include "catalogsettings.h"
#include "editcmd.h"
#include "dictchooser.h"
#include "kbabeldictbox.h"
#include "roughtransdlg.h"
#include "tagextractor.h"

#include <qarray.h>
#include <qcheckbox.h>
#include <qhbuttongroup.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qvgroupbox.h>
#include <qvbox.h>
#include <qwhatsthis.h>

#include <kapp.h>
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kprogress.h>


#include <kdebug.h>

RoughTransDlg::RoughTransDlg(KBabelDictBox *dict, Catalog *cat
                , QWidget *parent,const char *name)
        : KDialogBase(parent,name,true
                ,i18n("Caption of dialog","Rough Translation") 
                , User1|User2|User3|Close)
        ,dictBox(dict)
        ,catalog(cat)
        ,active(false)
        ,stop(false)
        ,cancel(false)
        ,exactTransCounter(0)
        ,partTransCounter(0)
        ,totalTried(0)
{
    setButtonBoxOrientation(Vertical);
    setButtonText(User1,i18n("&Start"));
    setButtonText(User2,i18n("S&top"));
    setButtonText(User3,i18n("C&ancel"));

    enableButton(User2,false);
    enableButton(User3,false);
    
    QWidget *mw = new QWidget(this);
    setMainWidget(mw);

    QVBoxLayout *mainLayout = new QVBoxLayout(mw);

    configWidget = new QVBox(mw);
    mainLayout->addWidget(configWidget);

    QVGroupBox *box = new QVGroupBox(i18n("What to translate"),configWidget);

    QHButtonGroup *bBox = new QHButtonGroup(box);
    bBox->setMargin(0);
    bBox->setFrameStyle(QFrame::NoFrame);
    allButton = new QRadioButton(i18n("All &entries"),bBox);
    untransButton = new QRadioButton(i18n("U&ntranslated and fuzzy entries"),bBox);

    connect(bBox,SIGNAL(clicked(int)),this,SLOT(msgButtonClicked()));

    QWhatsThis::add(bBox,i18n("<qt><p><b>What entries to translate</b></p>"
                "<p>Choose here, for which entries of the file KBabel "
                "tries to find a translation. Changed entries are always "
                "marked as fuzzy, no matter which option you choose.</p></qt>"));
    
    bBox = new QHButtonGroup(box);
    bBox->setFrameStyle(QFrame::NoFrame);
    bBox->setMargin(0);
    wholeMsgButton = new QRadioButton(i18n("&Only whole message"),bBox);
    singleWordButton = new QRadioButton(i18n("A&llow single word translation")
            ,bBox);

    QWhatsThis::add(bBox,i18n("<qt><p><b>How messages get translated</b></p>"
                "<p>Here you can define if a message can only get translated "
                "completely or if KBabel is supposed to try translating "
                "the single words of a message if no translation of the "
                "complete message was found.</p></qt>"));

    box = new QVGroupBox(i18n("Options"),configWidget);
    
    markFuzzyButton = new QCheckBox(i18n("&Mark changed entries as fuzzy"),box);
    markFuzzyButton->setChecked(true);
    QWhatsThis::add(markFuzzyButton,
            i18n("<qt><p><b>Mark changed entries as fuzzy</b></p>"
          "<p>When an translation for a message was found, the entry "
          "will marked <b>fuzzy</b> by default. This is, because the "
          "translation is just guessed by KBabel and you should always "
          "check the results carefully. Deactivate this option only, if "
          "you know what you are doing.</p></qt>"));

    connect(markFuzzyButton, SIGNAL(toggled(bool))
            , this, SLOT(fuzzyButtonToggled(bool)));

    QVGroupBox *dBox = new QVGroupBox(i18n("Dictionaries"),configWidget);
    configWidget->setStretchFactor(dBox,1);

    QList<ModuleInfo> moduleList = dict->moduleInfos();

    KConfig *config = KGlobal::config();
    KConfigGroupSaver gs(config,"RoughTranslation");
    QStringList selectedList=config->readListEntry("Selected");
    if(selectedList.isEmpty())
    {
        int a = dict->activeModule();
        ModuleInfo *mi = moduleList.at(a);
        if(mi)
        {
            selectedList.append(mi->id);
        }
    }
    dictChooser = new DictChooser(moduleList,selectedList,dBox,"dictChooser");

    QWhatsThis::add(dictChooser,i18n("<qt><p><b>Dictionaries</b></p>"
                "<p>Choose here, what dictionaries has to be used for "
                "finding a translation. If you select more than one "
                "dictionary, they are used in the same order as they "
                "are displayed in the list.</p></qt>"));
 
    progressbar = new KProgress(mw,"progressbar");
    progressbar->setTextEnabled(true);
    progressbar->setFormat("%v/%m (%p%)");
    mainLayout->addWidget(progressbar);

    bool flag = config->readBoolEntry("AllMessages",true);
    if(flag)
    {
        allButton->setChecked(true);
    }
    else
    {
        untransButton->setChecked(true);
    }

    flag = config->readBoolEntry("WholeMessage",true);
    if(flag)
    {
        wholeMsgButton->setChecked(true);
    }
    else
    {
        singleWordButton->setChecked(true);
    }

    msgButtonClicked();
}

RoughTransDlg::~RoughTransDlg()
{
}

void RoughTransDlg::slotUser1()
{
    configWidget->setEnabled(false);
    enableButton(User1,false);
    enableButton(Close,false);
    enableButton(User2,true);
    enableButton(User3,true);

    active=true;
    stop=false;
    cancel=false;

    bool markFuzzy = markFuzzyButton->isChecked();

    int total;
    QArray<int> indizes;
    int index=0;
    if(allButton->isChecked())
    {
        total = catalog->numberOfEntries();
        indizes.resize(total);
        for(index = 0; index < total; index++)
        {
            indizes[index]=index;
        }
    }
    else
    {
        total = catalog->numberOfUntranslated()+catalog->numberOfFuzzies();
        indizes.resize(total);
        int counter=0;
        if(catalog->isFuzzy(0) || catalog->isUntranslated(0))
        {
            indizes[0]=index;
            counter++;
        }
        while(index >= 0 && counter < total)
        {
            int fuzzyIndex=catalog->nextFuzzy(index);
            int untransIndex=catalog->nextUntranslated(index);

            if(fuzzyIndex<0)
                fuzzyIndex=untransIndex;
            if(untransIndex<0)
                untransIndex=fuzzyIndex;

            index=(fuzzyIndex < untransIndex)? fuzzyIndex : untransIndex;

            if(index>=0)
            {
                indizes[counter]=index;
                counter++;
            }
        }
    }

    QStringList dictList = dictChooser->selectedDicts();
    
    EditCommand* tmp = new BeginCommand();
    tmp->setPart(EditCommand::Msgstr);
    tmp->setIndex(0);
    catalog->applyEditCommand(tmp,0);
    
    exactTransCounter=0;
    partTransCounter=0;
    bool singleWords=singleWordButton->isChecked();
    QRegExp contextReg=catalog->miscSettings().contextInfo;
    QChar accelMarker=catalog->miscSettings().accelMarker;
    QRegExp endPunctReg("[\\.?!: ]+$");
    
    for(int i = 0; i < total; i++)
    {
        if(stop || cancel)
        {
            totalTried=i;
            break;
        }
        
        progressbar->setValue(i+1);
        kapp->processEvents();
     
        QString msg=catalog->msgid(indizes[i]);
        QString translation;

        // this is KDE specific:
        if(msg.find("_: NAME OF TRANSLATORS\\n")==0)
        {
            translation=catalog->identitySettings().authorName;
        }
        else if(msg.find("_: EMAIL OF TRANSLATORS\\n")==0)
        {
            translation=catalog->identitySettings().authorEmail;
        }
        // end of KDE specific part
        else if(msg.contains(contextReg))
        {
            msg.replace(contextReg,"");
        }


        QStringList::Iterator dit = dictList.begin();
        while(translation.isEmpty() && dit != dictList.end())
        {
            dictBox->setActiveModule(*dit);
            translation = dictBox->translate(msg);
            
            ++dit;
        }

        if(!translation.isEmpty())
        {
            exactTransCounter++;
        }
        else if(singleWords)
        {
            QStringList wordList;
            QChar accel;
            QString endingPunctuation;
            int pos = msg.findRev(endPunctReg);
            if(pos >= 0)
            {
                endingPunctuation = msg.right(msg.length()-pos);
            }
             
            msg=msg.simplifyWhiteSpace();
            msg=msg.stripWhiteSpace();


            TagExtractor te(msg);
            msg=te.tagsReplaced(" KBABELTAG ");
            QString word;
            int length = msg.length();
            QRegExp digitReg("^[0-9]*$");
            for(int index=0; index < length; index++)
            {                
                QChar c=msg[index];
                
                if(c==accelMarker)
                {
                    index++;
                    if(index < length)
                    {
                        if(msg[index].isLetterOrNumber())
                        {
                            word+=msg[index];
                            accel=msg[index];
                        }
                        else if(!word.isEmpty() )
                        {
                            if(!word.contains(digitReg))
                                wordList.append(word);
                                
                            word=QString::null;
                        }
                    }
                    else if(!word.isEmpty())
                    {
                        if(!word.contains(digitReg))
                            wordList.append(word);

                        word=QString::null;
                    }
                        
                }
                else if(c.isLetterOrNumber())
                {
                    word+=c;
                }
                else if(c == '\\')
                {
                    if(index < length-2)
                    {
                        if(msg[index+1]=='n' && msg[index+2].isSpace())
                        {
                            if(!word.isEmpty() && !word.contains(digitReg))
                                wordList.append(word);

                            word=QString::null;
                            
                            wordList.append("\\n\n");
                            index+=2;
                        }
                        else if(!word.isEmpty() )
                        {
                            if(!word.contains(digitReg))
                                wordList.append(word);
                                
                            word=QString::null;
                        }
                    }
                    else if(!word.isEmpty())
                    {
                        if(!word.contains(digitReg))
                            wordList.append(word);

                        word=QString::null;
                    }
                }
                else if(!word.isEmpty())
                {
                    if(!word.contains(digitReg))
                        wordList.append(word);

                    word=QString::null;
                }
            }
            
            dit = dictList.begin();
            int wordCounter=0;
            while(wordCounter==0 && dit != dictList.end())
            {
                dictBox->setActiveModule(*dit);

                for(QStringList::Iterator it=wordList.begin();
                        it!=wordList.end(); ++it)
                {
                    if( (*it)=="\\n\n" )
                    {
                        translation+="\\n\n";
                    }
                    else if( (*it)=="KBABELTAG" )
                    {
                        translation+=te.nextTag();
                    }
                    else
                    {
                        QString trans = dictBox->translate(*it);
                    
                        if(!trans.isEmpty())
                        {
                            wordCounter++;
                            if(!translation.isEmpty())
                            {
                                translation += ' ';
                            }
                            translation += trans;
                        }
                    }
                }

                if(wordCounter==0)
                    translation=QString::null;

                ++dit;
            }

            if(!translation.isEmpty())
            {
                partTransCounter++;
                // try to set the correct keyboard accelerator
                if(!accel.isNull())
                {
                    int index = translation.find(accel,0,false);
                    if(index >= 0)
                    {
                        translation.insert(index,accelMarker);
                    }
                }

                translation+=endingPunctuation;
            }
        }

        if(!translation.isEmpty())
        {
            if(!catalog->isUntranslated(indizes[i]))
            {
                DelTextCmd* delCmd = new DelTextCmd(0
                        ,catalog->msgstr(indizes[i]));
                delCmd->setPart(EditCommand::Msgstr);
                delCmd->setIndex(indizes[i]);
                catalog->applyEditCommand(delCmd,0);
            }
            
            InsTextCmd* insCmd = new InsTextCmd(0,translation);
            insCmd->setPart(EditCommand::Msgstr);
            insCmd->setIndex(indizes[i]);
            catalog->applyEditCommand(insCmd,0);

            if(markFuzzy)
            {
                catalog->setFuzzy(indizes[i],true);
            }
        }
    }

    tmp = new EndCommand();
    tmp->setPart(EditCommand::Msgstr);
    tmp->setIndex(0);
    catalog->applyEditCommand(tmp,0);

    if(!stop && !cancel)
        totalTried=total;

    int nothing=totalTried-partTransCounter-exactTransCounter;
    KLocale *locale = KGlobal::locale();
    QString statMsg = i18n("Result of the translation:\n\n"
            "Edited entries: %1\n"
            "Exact translations: %2 (%3%)\n"
            "Non exact translations: %4 (%5%)\n"
            "Nothing found: %6 (%7%)")
            .arg( locale->formatNumber(totalTried,0) )
            .arg( locale->formatNumber(exactTransCounter,0) )
            .arg( locale->formatNumber( ((double)(10000*exactTransCounter/totalTried))/100) )
            .arg( locale->formatNumber(partTransCounter,0) )
            .arg( locale->formatNumber(((double)(10000*partTransCounter/totalTried))/100) )
            .arg( locale->formatNumber(nothing,0) )
            .arg( locale->formatNumber(((double)(10000*nothing/totalTried))/100) );
    
    if(stop || cancel)
    {
        if(cancel)
        {
            catalog->undo();
        }
        else
        {
            KMessageBox::information(this, statMsg
                    , i18n("Rough translation statistics"));

            msgButtonClicked();
        }
        progressbar->setValue(0);
        configWidget->setEnabled(true);
        active = false;
        stop=false;
        cancel=false;
        
        enableButton(User1,true);
        enableButton(Close,true);
        enableButton(User2,false);
        enableButton(User3,false);
        
        return;
    }
    
    KMessageBox::information(this, statMsg
            , i18n("Rough translation statistics"));
    
    KConfig *config=KGlobal::config();
    KConfigGroupSaver gs(config,"RoughTranslation");
    config->writeEntry("Selected",dictChooser->selectedDicts());

    bool flag=allButton->isChecked();
    config->writeEntry("AllMessages",flag);
    flag=wholeMsgButton->isChecked();
    config->writeEntry("WholeMessage",flag);
    
    accept();
}

void RoughTransDlg::slotClose()
{
    if(active)
    {
        cancel = true;
        return;
    }
    else
    {
       accept(); 
    }
}

void RoughTransDlg::slotUser2()
{
    stop=true;
}

void RoughTransDlg::slotUser3()
{
    cancel=true;
}

void RoughTransDlg::msgButtonClicked()
{
    int total;
    if(allButton->isChecked())
    {
        total = catalog->numberOfEntries();
    }
    else
    {
        total = catalog->numberOfUntranslated()+catalog->numberOfFuzzies();
    }
    
    progressbar->setRange(0,total);

    enableButton(User1,total>0);
}

void RoughTransDlg::fuzzyButtonToggled(bool on)
{
    if(!on)
    {
        QString msg=i18n("<qt><p>"
          "When an translation for a message was found, the entry "
          "will marked <b>fuzzy</b> by default. This is, because the "
          "translation is just guessed by KBabel and you should always "
          "check the results carefully. Deactivate this option only, if "
          "you know what you are doing.</p></qt>");
          
        KMessageBox::information(this, msg, QString::null,"MarkFuzzyWarningInRoughTransDlg");
    }
}

void RoughTransDlg::statistics(int &total, int& exact, int& part) const
{
    total = totalTried;
    exact = exactTransCounter;
    part = partTransCounter;
}

#include "roughtransdlg.moc"
