/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "window.h"
#include "prefs.h"
#include "history.h"
#include "embed.h"
#include "menubar.h"
#include "misc_general.h"
#include "bookmarks.h"
#include "bookmarks_editor.h"
#include "print.h"
#include "find.h"
#include "session.h"
#include "xlink.h"
#include "dialog.h"
#include "gfilepicker.h"
#include "mozilla.h"
#include "link_interfaces.h"
#include "persistent_data_manager.h"
#include "autoreload.h"
#include "page_info.h"
#include "js_console.h"

#include <string.h>
#include <gtkmozembed.h>
#include <gtk/gtkspinbutton.h>
#include <libgnomeui/gnome-about.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

void 
window_menu_recent_activate_cb (GtkMenuItem *menuitem, gpointer user_data);
void 
window_menu_recent_release_cb (GtkMenuItem *menuitem, GdkEventButton *event,
			       gpointer user_data);

static GtkWidget *window_menu_edit_find_focus(GaleonWindow *window);

/**
 * window_menu_prefs_activate_cb: show preferences dialog
 */
void
window_menu_prefs_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	prefs_open (window);
}

/**
 * window_menu_bookmarks_edit_activate_cb: show edit bookmarks dialog
 */
void
window_menu_bookmarks_edit_activate_cb (GtkMenuItem *menuitem,
				        GaleonWindow *window) 
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	if (eel_gconf_get_boolean (CONF_STATE_BOOKMARKS_DOCK_ENABLE))
	{
		if (window->dock_type != DOCK_BOOKMARKS)
		{
			bookmarks_editor_show_dock (window);
		}
		else
		{
			window_undock (window);
		}
	}
	else
	{
		bookmarks_editor_show_dialog (window);
	}
}

/**
 * window_menu_find_activate_cb: show find dialog
 */
void
window_menu_find_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GaleonEmbed *embed;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);
	
	find_show_dialog (embed);
}

/**
 * window_menu_find_next_activate_cb: find next occurence
 */
void
window_menu_find_next_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GaleonEmbed *embed;
	gchar *val, *utf8;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	val = eel_gconf_get_string (CONF_FIND_WORD);
	if (val)
	{
		utf8 = mozilla_utf8_to_locale (val);
		find_next (embed, utf8);
		g_free (utf8);
		g_free (val);
	}
}
/**
 * window_menu_find_previous_activate_cb: find previous occurence
 */
void
window_menu_find_previous_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GaleonEmbed *embed;
	gchar *val, *utf8;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	val = eel_gconf_get_string (CONF_FIND_WORD);
	if (val)
	{
		utf8 = mozilla_utf8_to_locale (val);
		find_previous (embed, utf8);
		g_free (utf8);
		g_free (val);
	}
}

/**
 * window_menu_select_all_activate_cb: select all text on the current page
 */
void
window_menu_select_all_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GtkEditable *editable;
	GtkWidget *widget;

	if ((widget=window_menu_edit_find_focus(window)) == NULL)
	{
		mozilla_select_all (window->active_embed);
		return;
	}

	editable = GTK_EDITABLE (widget);

	gtk_editable_select_region (editable, 0, -1);
}

/*
 * Find which widget is currently in focus
 */
static GtkWidget *
window_menu_edit_find_focus(GaleonWindow *window)
{
	GtkWidget *widget;
	GList *tb_entry;

	if (window->location_entry != NULL)
	{
		if (GTK_WIDGET_HAS_FOCUS (window->location_entry))
		{
			return (window->location_entry);
		}
	}

	if (window->find_entry != NULL)
	{
		if (GTK_WIDGET_HAS_FOCUS (window->find_entry))
		{
			return (window->find_entry);
		}
	}

	/* Search through the toolbars, then down each toolbar
	** looking for a focused widget
	*/

	for (widget=NULL, tb_entry = g_list_first(window->bookmarks_toolbars);
		widget==NULL && tb_entry!=NULL;
		tb_entry = g_list_next(tb_entry))
	{

		GtkToolbar *toolbar=(GtkToolbar *) tb_entry->data;
		GList *tbc_entry;

		/* Step through the toolbar children */
		for (tbc_entry = g_list_first(toolbar->children);
			widget==NULL && tbc_entry!=NULL;
			tbc_entry = g_list_next(tbc_entry))
		{

			GtkToolbarChild *toolbarchild =
				(GtkToolbarChild *) tbc_entry->data;
			GtkWidget *entry;

			if (toolbarchild->type != GTK_TOOLBAR_CHILD_WIDGET)
				continue;
			
			entry = gtk_object_get_data
				(GTK_OBJECT (toolbarchild->widget), "entry");

			if (entry && GTK_WIDGET_HAS_FOCUS (entry))
			{
				widget = entry;
			}
		}
	}

	if (widget)
		return(widget);

	/* Add getting the embed Mozilla widget here */

	return(NULL);
}

gboolean window_menu_edit_expose_event_cb (GtkWidget *editmenu, 
					   GdkEvent *event,
					   GaleonWindow *window)
{
	gboolean can_cut, can_copy, can_paste;
	GtkWidget *focus_widget;
	
	can_cut = FALSE;
	can_copy = FALSE;
	can_paste = FALSE;

	focus_widget = window_menu_edit_find_focus (window);
	if (focus_widget != NULL && GTK_IS_EDITABLE (focus_widget))
	{
		GtkEditable *editable = GTK_EDITABLE(focus_widget);

		if (editable->has_selection && editable->editable)
		{
			can_cut = TRUE;
		}
		if (editable->has_selection)
		{
			can_copy = TRUE;
		}
		if (editable->editable && galeon_x_clipboard_present()) 
		{
			can_paste = TRUE;
		}
	}
	else
	{	
		if (window->active_embed != NULL) 
		{
			can_cut = mozilla_can_cut_selection (window->active_embed);
			can_copy = mozilla_can_copy_selection (window->active_embed);
			can_paste = mozilla_can_paste (window->active_embed);
		}
	}

	gtk_widget_set_sensitive (window->edit_cut, can_cut);
	gtk_widget_set_sensitive (window->edit_copy, can_copy);
	gtk_widget_set_sensitive (window->edit_paste, can_paste);

	return FALSE;
}

/**
 * window_menu_edit_cut_cb: Cut selected text into clipboard
 */
void
window_menu_edit_cut_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GtkEditable *editable;
	GtkWidget *widget;

	if ((widget=window_menu_edit_find_focus(window)) == NULL)
	{
		mozilla_cut_selection (window->active_embed);
		return;
	}

	editable = GTK_EDITABLE (widget);

	gtk_editable_cut_clipboard (editable);
}

/**
 * window_menu_edit_copy_cb: Copy selected text into clipboard
 */
void
window_menu_edit_copy_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GtkEditable *editable;
	GtkWidget *widget;

	if ((widget=window_menu_edit_find_focus(window)) == NULL)
	{
		mozilla_copy_selection (window->active_embed);
		return;
	}

	editable = GTK_EDITABLE (widget);

	gtk_editable_copy_clipboard (editable);
}

/**
 * window_menu_edit_paste_cb: Paste selected text into clipboard
 */
void
window_menu_edit_paste_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GtkEditable *editable;
	GtkWidget *widget;

	if ((widget=window_menu_edit_find_focus(window)) == NULL)
	{
		mozilla_paste (window->active_embed);
		return;
	}

	editable = GTK_EDITABLE (widget);

	gtk_editable_paste_clipboard (editable);
}

/**
 * Goes to the url of the item
 */
void 
window_menu_recent_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
        GaleonWindow *window = window_from_widget (GTK_WIDGET (menuitem));
        RecentMenuEntry *entry = gtk_object_get_user_data (GTK_OBJECT (menuitem));

	return_if_not_window (window);
	window_load_url(window, entry->url);
}

void 
window_menu_recent_release_cb (GtkMenuItem *menuitem, GdkEventButton *event,
			       gpointer user_data)
{
	GaleonWindow *window = window_from_widget (GTK_WIDGET (menuitem));
        RecentMenuEntry *entry = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	GtkMenu *menu = GTK_MENU (GTK_WIDGET (menuitem)->parent);
	
	return_if_not_window (window);

	gtk_menu_shell_deactivate (GTK_MENU_SHELL (menu));
	
	embed_activate_link (window->active_embed, NULL, entry->url,
			     misc_general_mouse_state_to_link_state (
				     		event->button, event->state));
}

/**
 * window_menu_view_source_activate_cb: view source
 */
void
window_menu_view_source_activate_cb (GtkMenuItem *menuitem,
				     GaleonWindow *window)
{
	gboolean tabbed_mode;

	return_if_not_window (window);

	tabbed_mode = eel_gconf_get_boolean (CONF_TABS_TABBED);
	embed_view_source (window->active_embed, TRUE, !tabbed_mode);
}

/**
 * window_menu_history_activate_cb: show history dialog
 */
void window_menu_history_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);

	if (eel_gconf_get_boolean (CONF_STATE_HISTORY_DOCK_ENABLE))
	{
		if (window->dock_type != DOCK_HISTORY)
		{
			history_show_dock (window);
		}
		else
		{
			window_undock (window);
		}
	}
	else
	{
		history_show_dialog (window);
	}
}

/**
 * window_run_first_time_druid_activate_cb: reruns the fisrt time druid
 */
void window_run_first_time_druid_activate_cb (GtkMenuItem *menuitem,
					      GaleonWindow *window)
{
	newuser_show_druid ();
}

/**
 * window_menu_help_about_activate_cb: show about dialog 
 */
void
window_menu_help_about_activate_cb (GtkMenuItem *menuitem, 
				    GaleonWindow *window)
{
	static gchar *authors[] = {
		"Marco Pesenti Gritti <mpeseng@tin.it>",
		"Remco de Jong <rdj@rdj.cg.nu>",
		/* if your encoding allows it, use aacute (U00E1)
		 * for the 'a' of 'Fernandez' */
		N_("Ricardo Fernandez Pascual <ric@users.sourceforge.net>"),
		"Nate Case <nd@kracked.com>",
		"Bob Phan <ex-freek@yahoo.com>",
		"Matt Aubury <matt@ookypooky.com>",
		"Joshua Horvath <jmhorvat@home.com>",
		/* this one is supposed to be in cyrillic */
		N_("Yanko Kaneti <yaneti@declera.com>"),
		"Philip Langdale <philipl@mail.utexas.edu>",
		"Matthew Mueller <donut@azstarnet.com>",
		"Daniel Erat <dan_erat@pobox.com>",
		"Jorn Baayen <jorn@nl.linux.org>",
		" ", /* dont delete the space */
		N_("Graphic arts: "),
		/* if your encoding allows it, use eacute (U00E9)
		 * for the two 'e's of Frederic */
		N_("Toussaint Frederic <fred@linuxgraphic.org>"),
		NULL
	};
	
	static GtkWidget * about;
	
	return_if_not_window (window);

	if (!about)
	{
		gint i, num_authors = sizeof(authors) / sizeof(gchar *);
		gchar *translated_names[num_authors];
		gchar *bottom_line;
		gchar *file;
		
		if (strcmp (_("Translation:"), "Translation:") == 0) {
			bottom_line = g_strdup
				("A GNOME browser based on Mozilla");
		} else {
			bottom_line = g_strconcat (
				_("A GNOME browser based on Mozilla"),
				"\n \n",
			/* Translators should localize the following string
			 * which will be displayed at the bottom of the about
			 * box to give credit to the translator(s).
			 * Translate the "Translation:", add your name and an
			 * email address. You can span several lines with a
			 * newline character if needed, but it shouldn't be
			 * too long; vertical space is limited in the about
			 * dialog. 
			 * If you dont translate it nothing will be added */
				_("Translation:"), NULL);
		}
		
		for (i = 0; i < num_authors ; i++)
			translated_names[i] = _(authors[i]);
		
		file = misc_general_user_file ("about_logo.png", FALSE);
		about = gnome_about_new 
			(_(PACKAGE), VERSION,
			/* Translators: Please change the (C) to a real
			 * copyright character if your character set allows it
			 * (Hint: iso-8859-1 is one of the character sets that
			 * has this symbol). */
			 _("Copyright (C) 2000-2002 Marco Pesenti Gritti"),
				 (const gchar **)translated_names, 
				 bottom_line, file);
		g_free (bottom_line);
		g_free (file);

		gtk_signal_connect (GTK_OBJECT (about), "destroy",
			    GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about);
		gtk_window_set_wmclass (GTK_WINDOW (about), "dialog_about",
					"galeon_browser");
		gtk_widget_show (about);
	}

	dialog_set_parent (about, window->wmain);
}

/**
 * window_menu_help_manual_activate: show the main Galeon manual page
 */
void
window_menu_help_manual_activate_cb (GtkMenuItem *menuitem, 
				     GaleonWindow *window)
{
	gnome_help_goto (NULL, "ghelp:galeon-manual");
}

/**
 * window_menu_help_homepage_activate_cb: go to the Galeon homepage
 */
void
window_menu_help_homepage_activate_cb (GtkWidget *menuitem, 
				       GaleonWindow *window)
{
	return_if_not_window (window);
	window_load_url (window, GALEON_HOMEPAGE_URL);
}

/**
 * window_menu_help_faq_activate_cb: go to the Galeon FAQ page
 */
void
window_menu_help_faq_activate_cb (GtkWidget *menuitem, 
				  GaleonWindow *window)
{
	return_if_not_window (window);
	window_load_url (window, GALEON_FAQ_URL);
}

/**
 * window_menu_help_bugreport_activate_cb: go to the Galeon bug report page
 */
void
window_menu_help_bugreport_activate_cb (GtkWidget *menuitem, 
					GaleonWindow *window)
{
	return_if_not_window (window);
	window_load_url (window, GALEON_BUGREPORT_URL);
}

/**
 * window_menu_help_about_plugins_activate_cb: go to about:plugins
 */
void
window_menu_help_about_plugins_activate_cb (GtkWidget *menuitem, 	
					    GaleonWindow *window)
{
	return_if_not_window (window);
	window_load_url (window, "about:plugins");
}

/**
 * window_menu_help_generic_release_cb: 
 */
void
window_menu_help_generic_release_cb (GtkWidget *menuitem, 
				 GdkEventButton *event,
				 GaleonWindow *window)
{
	gchar *url =
		(gchar*) gtk_object_get_data (GTK_OBJECT (menuitem), "url");
	GaleonEmbed *new = NULL;
	GtkMenu *menu = GTK_MENU (menuitem->parent);
	
	return_if_not_window (window);

	gtk_menu_shell_deactivate (GTK_MENU_SHELL (menu));
	
	embed_activate_link (window->active_embed, &new, url,
		misc_general_mouse_state_to_link_state (event->button,
							event->state));
}

/** 
 * window_menu_new_window_activate_cb: open a new window
 */
void
window_menu_new_window_activate_cb (GtkMenuItem *menuitem,
				    GaleonWindow *window)
{
	GaleonEmbed *embed;

	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	embed = embed_create_after_embed (window->active_embed, TRUE, NULL,
					  EMBED_CREATE_LOAD_DEFAULT_URL |
					  EMBED_CREATE_FORCE_JUMP);

	return_if_not_window (embed->parent_window);
	window_grab_location_focus (embed->parent_window);
}

/** 
 * window_menu_new_tab_activate_cb: open a new tab
 */
void
window_menu_new_tab_activate_cb (GtkMenuItem *menuitem,
				 GaleonWindow *window)
{
	GaleonEmbed *embed;
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	window_grab_location_focus (window);
	embed = embed_create_after_embed (window->active_embed, FALSE, NULL,
					  EMBED_CREATE_LOAD_DEFAULT_URL |
					  EMBED_CREATE_FORCE_JUMP);
}

/** 
 * window_menu_open_activate_cb: show open file dialog
 */
void
window_menu_open_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	window_show_open_dialog (window);
}

/** 
 * on_new_window_activate: show open url dialog
*/
void
window_menu_open_url_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	window_show_openurl_dialog (window);
}

/** 
 * on_save_as1_actuvate: show save dialog
 */
void
window_menu_save_as_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	embed_save_document (window->active_embed, TRUE);
}

/** 
 * window_menu_exit_activate_cb: exit galeon
 */
void
window_menu_exit_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	session_quit (FALSE);
}

/** 
 * window_menu_exit_with_session_activate_cb: exit galeon
 */
void
window_menu_exit_with_session_activate_cb (GtkMenuItem *menuitem, 
					   GaleonWindow *window)
{
	return_if_not_window (window);

	session_quit (TRUE);
}

/** 
 * window_menu_print_activate_cb: print document
 */
void
window_menu_print_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GaleonEmbed *embed;
	
	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	print_show_dialog (embed);
}

/**
 * window_menu_send_to_activate_cb: send link to
 */
void
window_menu_send_to_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	GaleonEmbed *embed;
	gchar *url, *site;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	if (embed->location == NULL) return;
	
	/* FIXME: i don't like this... it means that the URL could be
	 * encoded incorrectly (e.g. ?s that are valid in the context of
	 * the URL will be %HH-escaped).  i don't know what we can do
	 * instead, though. - dan */
	site = bookmarks_encode_smb_string (embed->location);
	url = g_strconcat ("mailto:",
			   /* xgettext:no-c-format */
			   _("?Subject=Check%20this%20out%21"),
			   "&Body=", site, NULL); 
	gtk_moz_embed_load_url (GTK_MOZ_EMBED (embed->mozembed), url);

	g_free (url);
	g_free (site);
}

/** 
 * window_menu_close_activate_cb: close current active embed
 */
void
window_menu_close_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window) 
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_close (window->active_embed);
}

/** 
 * window_menu_back_activate_cb: back menu activated
 */
void
window_menu_back_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_moz_embed_go_back (GTK_MOZ_EMBED(window->active_embed->mozembed));
}

/** 
 * window_menu_up_activate_cb: up menu activated
 */
void
window_menu_up_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_go_up (window->active_embed, 0, 0);
}

/** 
 * window_menu_forward_activate_cb: forward menu activated
 */
void
window_menu_forward_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_moz_embed_go_forward 
		(GTK_MOZ_EMBED (window->active_embed->mozembed));
}

/** 
 * window_menu_detach_tab_activate_cb: detach current tab into a new window
 */
void
window_menu_detach_tab_activate_cb (GtkMenuItem *menuitem, 
				    GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_move_to_new_window (window->active_embed);
}

/** 
 * window_menu_prev_tab_activate_cb: previous tab activated
 */
void
window_menu_prev_tab_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	gint current, last;  

	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	
	current =
		gtk_notebook_get_current_page (GTK_NOTEBOOK (window->notebook));
	last = g_list_length (GTK_NOTEBOOK (window->notebook)->children) - 1;
	
	if (current == 0)
		gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook), last);
	else
		gtk_notebook_prev_page (GTK_NOTEBOOK (window->notebook));
}

/** 
 * window_menu_next_tab_activate_cb: next tab activated
 */
void
window_menu_next_tab_activate_cb (GtkMenuItem *menuitem, GaleonWindow *window)
{
	gint current, last;
	
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	
	current =
		gtk_notebook_get_current_page (GTK_NOTEBOOK (window->notebook));
	last = g_list_length (GTK_NOTEBOOK (window->notebook)->children) - 1;
	
	if (current == last)
		gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook), 0);
	else
		gtk_notebook_next_page (GTK_NOTEBOOK (window->notebook));
}

/** 
 * window_menu_move_tab_left_cb: 
 */
void
window_menu_shift_tab_left_activate_cb (GtkMenuItem *menuitem,
					GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_shift_tab (window->active_embed, -1);
}

/** 
 * window_menu_move_tab_right_cb:
 */
void
window_menu_shift_tab_right_activate_cb (GtkMenuItem *menuitem,
					 GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_shift_tab (window->active_embed, 1);
}

/** 
 * window_menu_stop_activate_cb: stop menu activated
 */
void
window_menu_stop_activate_cb (GtkMenuItem *menuitem,
			      GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	gtk_moz_embed_stop_load(GTK_MOZ_EMBED(window->active_embed->mozembed));
}

/** 
 * window_menu_reload_activate_cb: reload menu activated
 */
void
window_menu_reload_activate_cb (GtkMenuItem *menuitem,
				GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);
	embed_reload (window->active_embed, GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
}

/** 
 * window_menu_home_page_activate_cb: home page menu item activated
 */
void
window_menu_home_page_activate_cb (GtkMenuItem *menuitem,
				   GaleonWindow *window)
{
	return_if_not_window (window);
	window_go_home (window, 0);
}

void
window_menu_add_bookmark_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);

	/* use bookmarks file cb */
	bookmarks_file_bookmark_cb (window->wmain, NULL);
}

/** 
 * window_menu_toolbar_activate_cb: hide/show the toolbar
 */
void window_menu_toolbar_activate_cb (GtkCheckMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);

	g_return_if_fail (menuitem);

	if (menuitem->active) 
	{		
		window_toolbar_show (window);
	}
	else
	{
		window_toolbar_hide (window);
	}
}

/** 
 * window_menu_bookmarks_toolbars_activate_cb: hide/show the bookmarks toolbars
 */
void window_menu_bookmarks_toolbars_activate_cb (GtkCheckMenuItem *menuitem,
						 GaleonWindow *window)
{
	return_if_not_window (window);

	bookmarks_toolbars_set_visibility (window, menuitem->active);
	if (window->hbox)
	{
		gtk_widget_queue_resize (GTK_WIDGET (window->hbox));
	}
}

/** 
 * window_menu_menubar_activate_cb: hide/show the menubar
 */
void window_menu_menubar_activate_cb (GtkCheckMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);

	g_return_if_fail (menuitem);

	if (menuitem->active) 
	{
		window_menubar_show (window);
	}
	else
	{
		window_menubar_hide (window);
	}
}

/** 
 * window_menu_statusbar_activate_cb: hide/show the menubar
 */
void window_menu_statusbar_activate_cb (GtkCheckMenuItem *menuitem,
					GaleonWindow *window)
{
	return_if_not_window (window);

	g_return_if_fail (menuitem);

	if (menuitem->active) 
	{
		window_statusbar_show (window);
	}	
	else
	{
		window_statusbar_hide (window);
	}
}

/** 
 * window_menu_fullscreen_activate_cb: set fullscreen mode
 */
void window_menu_fullscreen_activate_cb (GtkCheckMenuItem *menuitem,
					 GaleonWindow *window)
{
	return_if_not_window (window);

	if (menuitem)
		window_set_fullscreen_mode (window, menuitem->active);
	else
		window_toggle_fullscreen_mode (window);
}

void
window_menu_own_colors_activate_cb (GtkCheckMenuItem *menuitem,
				    GaleonWindow *window)
{
	return_if_not_window (window);
	eel_gconf_set_boolean(CONF_RENDERING_USE_OWN_COLORS, 
			      menuitem->active);
}

void
window_menu_allow_javascript_activate_cb (GtkCheckMenuItem *menuitem,
					  GaleonWindow *window)
{
	return_if_not_window (window);
	eel_gconf_set_boolean(CONF_FILTERING_JAVASCRIPT_ENABLED, 
			      menuitem->active);
}

void
window_menu_allow_popups_activate_cb (GtkCheckMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	eel_gconf_set_boolean (CONF_FILTERING_ALLOW_POPUPS, 
			       menuitem->active);
}

void
window_menu_own_fonts_activate_cb (GtkCheckMenuItem *menuitem,
				   GaleonWindow *window)
{
	return_if_not_window (window);
	eel_gconf_set_boolean(CONF_RENDERING_USE_OWN_FONTS, menuitem->active);
}

void
window_menu_allow_java_activate_cb (GtkCheckMenuItem *menuitem,
				     GaleonWindow *window)
{
	return_if_not_window (window);
	eel_gconf_set_boolean(CONF_FILTERING_JAVA_ENABLED, menuitem->active);
}

void
window_menu_images_never_activate_cb (GtkCheckMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		eel_gconf_set_integer(CONF_FILTERING_IMAGE_LOADING_TYPE,
				      IMAGE_LOADING_NEVER);
	}
}

void
window_menu_images_current_server_activate_cb (GtkCheckMenuItem *menuitem,
					       GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		eel_gconf_set_integer(CONF_FILTERING_IMAGE_LOADING_TYPE,
				      IMAGE_LOADING_FROM_CURRENT_SERVER_ONLY);
	}
}

void
window_menu_images_always_activate_cb (GtkCheckMenuItem *menuitem,
				       GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		eel_gconf_set_integer (CONF_FILTERING_IMAGE_LOADING_TYPE, 
				       IMAGE_LOADING_ALWAYS);
	}
}

void
window_block_images_from_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_permission (window->active_embed, FALSE, IMAGEPERMISSION);
}

void
window_allow_images_from_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_permission (window->active_embed, TRUE, IMAGEPERMISSION);
}

void
window_view_image_sites_activate_cb (GtkMenuItem *menuItem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	pdm_show_page (window, PDM_IMAGESITES);
}

void
window_menu_animate_always_activate_cb (GtkCheckMenuItem *menuitem,
					GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		eel_gconf_set_integer(CONF_FILTERING_ANIMATE_TYPE, 
				      ANIMATE_ALWAYS);
	}
}

void
window_menu_animate_once_through_activate_cb (GtkCheckMenuItem *menuitem,
					      GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		eel_gconf_set_integer(CONF_FILTERING_ANIMATE_TYPE,
				      ANIMATE_ONCE_THROUGH);
	}
}

void
window_menu_animate_never_activate_cb (GtkCheckMenuItem *menuitem,
				       GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active) 
	{
		eel_gconf_set_integer(CONF_FILTERING_ANIMATE_TYPE,
				      ANIMATE_NEVER);
	}
}

void
window_view_cookies_activate_cb (GtkMenuItem *menuitem,
				 GaleonWindow *window)
{
	return_if_not_window (window);
	pdm_show_page (window, PDM_COOKIES);
}

void
window_view_cookie_sites_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	pdm_show_page (window, PDM_COOKIESITES);
}

void
window_block_cookies_from_activate_cb (GtkMenuItem *menuitem,
				       GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_permission (window->active_embed, FALSE, COOKIEPERMISSION);
}

void
window_allow_cookies_from_activate_cb (GtkMenuItem *menuitem,
				       GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_permission (window->active_embed, TRUE, COOKIEPERMISSION);	
}

void
window_menu_zoom_in_activate_cb (GtkCheckMenuItem *menuitem,
				 GaleonWindow *window)
{
	GaleonEmbed *embed;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	if (window->zoom_spin)
	{
		gtk_spin_button_spin (GTK_SPIN_BUTTON (window->zoom_spin),
				      GTK_SPIN_STEP_FORWARD, 10); 
	}
	else
	{
		float zoom;
		mozilla_get_zoom (embed, &zoom);
		zoom = zoom*100+10;
		embed_set_zoom (embed, zoom, TRUE);
		history_set_zoom (embed->location, zoom);
	}
}

void
window_menu_zoom_100_activate_cb (GtkCheckMenuItem *menuitem,
				 GaleonWindow *window)
{
	GaleonEmbed *embed;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	if (window->zoom_spin &&
	    gtk_spin_button_get_value_as_int (
		    GTK_SPIN_BUTTON (window->zoom_spin)) != 100)
	{
		gtk_spin_button_set_value 
			(GTK_SPIN_BUTTON (window->zoom_spin), 100);
	}
	else
	{
		embed_set_zoom (embed, 100, TRUE);
		history_set_zoom (embed->location, 100);
	}
}

void
window_menu_zoom_out_activate_cb (GtkCheckMenuItem *menuitem,
				  GaleonWindow *window)
{
	GaleonEmbed *embed;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	if (window->zoom_spin)
	{
		gtk_spin_button_spin (GTK_SPIN_BUTTON (window->zoom_spin),
				      GTK_SPIN_STEP_BACKWARD, 10); 
	}
	else
	{
		float zoom;
		mozilla_get_zoom (embed, &zoom);
		zoom = zoom*100-10;
		embed_set_zoom (embed, zoom, TRUE);
		history_set_zoom (embed->location, zoom);
	}
}

void 
window_menu_encoding_activate_cb (GtkWidget *menuitem,
				  GaleonWindow *window)
{
	char *menulabel;

	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	gtk_label_get (GTK_LABEL(GTK_BIN(menuitem)->child), &menulabel);
	mozilla_force_character_set (window->active_embed,
				     g_hash_table_lookup (charsets,
							  menulabel));
	embed_reload (window->active_embed, GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
}

/*
 * window_menu_encoding_state_changed_cb: 
 * handle the first use of the encoding_menu to create the submenus
 */
void 
window_menu_encoding_state_changed_cb (GtkMenuItem *encoding_menu,
				       GtkStateType state, 
				       GaleonWindow *window)
{
	/* return immediately if this is not an entry to the widget */
	if (state != GTK_STATE_NORMAL)
	{
		return;
	}

	/* get the charset titles and names from mozilla, if still not done */
	if (charsets == NULL)
	{
		mozilla_get_charsets (&charsets, &sorted_charset_titles);
	}

	/* create the submenu if we already havent havent*/
	if (!window->encodings_menu_set)
	{
		/* kill the dummy */
		gtk_widget_destroy (encoding_menu->submenu);
		encoding_menu->submenu = NULL;
		/* set the menu */
		window_create_charset_submenus (window, sorted_charset_titles);
		window->encodings_menu_set = TRUE;
	}
}

void
window_view_passwords_activate_cb (GtkMenuItem *menuItem,
			           GaleonWindow *window)
{
	return_if_not_window (window);
	pdm_show_page (window, PDM_PASSWORDS);
}
void
window_view_password_sites_activate_cb (GtkMenuItem *menuItem,
				        GaleonWindow *window)
{
	return_if_not_window (window);
	pdm_show_page (window, PDM_PASSWORDSITES);
}

void
window_menu_offline_mode_activate_cb (GtkCheckMenuItem *menuitem,
				      GaleonWindow *window)
{
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	mozilla_set_offline(window->active_embed, menuitem->active);
}

void
window_menu_save_session_as_activate_cb (GtkWidget *menuitem,
					 GaleonWindow *window)
{
	gchar *file = NULL;
	gchar *sessions_dir;
	
	sessions_dir = g_concat_dir_and_file (g_get_home_dir (), 
					      "/.galeon/sessions");
	if (show_file_picker (window->wmain, _("Save session as..."),
			      sessions_dir, NULL, modeSave,
			      &file, NULL, NULL, NULL))
	{
		/* we don't need to check overwrite here, since the file
		 * picker already does it */
		session_save_to (file);
		session_history_add (file);
		g_free (file);
	}
	g_free (sessions_dir);
}

void
window_menu_open_session_activate_cb (GtkWidget *menuitem,
				      GaleonWindow *window)
{
	gchar *file = NULL;
	gchar *sessions_dir = g_concat_dir_and_file (g_get_home_dir (), 
						     "/.galeon/sessions");

	if (show_file_picker (window->wmain, _("Open session..."),
			      sessions_dir, NULL, modeOpen,
			      &file, NULL, NULL, NULL)) 
	{
		if (session_open_confirmation ())
			if (session_load_from (file, FALSE))
				session_history_add (file);
	}
	
	g_free (file);
	g_free (sessions_dir);
}

void
window_menu_show_java_console_cb (GtkWidget *meniutem,
				  GaleonWindow *window)
{
	mozilla_show_java_console ();
}

void
window_menu_show_javascript_console_cb (GtkWidget *meniutem,
					GaleonWindow *window)
{
	GaleonEmbed *embed;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	js_console_show (embed);
}

void
window_menu_show_autoreload_cb (GtkWidget *meniutem,
				GaleonWindow *window)
{
	autoreload_show_dialog (window->active_embed);
}

void
window_menu_show_page_info_cb (GtkWidget *menuitem,
			       GaleonWindow *window)
{
	page_info_show_dialog (window->active_embed, PAGE_INFO_GENERAL);
}

void
window_menu_no_proxies_activate_cb (GtkCheckMenuItem *menuitem,
				    GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active)
	{
		eel_gconf_set_integer (CONF_NETWORK_PROXY_MODE, 0);
	}
}

void
window_menu_manual_proxies_activate_cb (GtkCheckMenuItem *menuitem,
					GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active)
	{
		eel_gconf_set_integer (CONF_NETWORK_PROXY_MODE, 1);
	}
}

void
window_menu_automatic_proxies_activate_cb (GtkCheckMenuItem *menuitem,
					   GaleonWindow *window)
{
	return_if_not_window (window);
	if (menuitem->active)
	{
		eel_gconf_set_integer (CONF_NETWORK_PROXY_MODE, 2);
	}
}

void
window_menu_proxy_settings_activate_cb (GtkCheckMenuItem *menuitem,
					GaleonWindow *window)
{
	return_if_not_window (window);
	prefs_show_page (window, PREFS_PAGE_ID_ADVANCED_NETWORK);
}

void
window_menu_related_links_activate_cb (GtkMenuItem *menuitem, 
				       GaleonWindow *window)
{
	GtkMenu *menu;
	GList *child_list;
	GList *li;
	if (menuitem->submenu == NULL) 
	{
		menu = GTK_MENU (gtk_menu_new ());
		gtk_menu_item_set_submenu (menuitem, GTK_WIDGET (menu));
	}
	else
	{
		menu = GTK_MENU (menuitem->submenu);
	}

	/* remove old menu items */
	child_list = gtk_container_children (GTK_CONTAINER (menu));
	for (li = child_list; li != NULL; li = li->next)
	{
		gtk_widget_destroy (GTK_WIDGET (li->data));
	}
	g_list_free (child_list);

	/* create menu items */
	link_interfaces_build_menu (window, menu, TRUE);
	
	gtk_menu_reposition (menu);
}
