/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gnome-pilot-conduitcap.c
 *
 * Copyright (C) 1998 Red Hat Software       
 * Copyright (C) 1999-2000 Free Software Foundation
 * Copyright (C) 2001  Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Eskil Heyn Olsen
 *          Vadim Strizhevsky
 *          Michael Fulbright <msf@redhat.com>
 *          JP Rosevear <jpr@ximian.com>
 *
 */

#include <sys/stat.h>
#include <glade/glade.h>
#include "pilot.h"
#include "util.h"
#include "gnome-pilot-cdialog.h"
#include "gnome-pilot-conduitcap.h"

#define SMALL_ICON_SIZE 20

static GtkVBoxClass *parent_class = NULL;

struct _GnomePilotConduitCapPrivate 
{
	GladeXML *xml;

	GnomePilotClient *gpc;
	gint handle;
	
	PilotState *state;
	GPilotPilot *pilot;

	GHashTable *conduits;
	
	CappletWidget *parent;

	GtkWidget *vbox;

	GtkWidget *pilots_menu;
	GtkWidget *pilots_username;
	
	GtkWidget *conduit_clist;
	GtkWidget *conduit_enable;
	GtkWidget *conduit_disable;
	GtkWidget *conduit_settings;

	GtkWidget *conduit_popup;
	GtkWidget *conduit_popup_enable;
	GtkWidget *conduit_popup_disable;
	GtkWidget *conduit_popup_settings;
	
	GtkWidget *conduit_description;
};

static void class_init (GnomePilotConduitCapClass *klass);
static void init (GnomePilotConduitCap *gpcc);

static gboolean get_widgets (GnomePilotConduitCap *gpcc);
static void map_widgets (GnomePilotConduitCap *gpcc);
static void init_widgets (GnomePilotConduitCap *gpcc);
static void fill_widgets (GnomePilotConduitCap *gpcc);

static void check_conduits_buttons (GnomePilotConduitCap *gpcc);

static void gpcc_conduits_choose_pilot (GtkWidget *widget, gpointer user_data);

static void gpcc_conduits_enable (GtkWidget *widget, gpointer user_data);
static void gpcc_conduits_disable (GtkWidget *widget, gpointer user_data);
static void gpcc_conduits_settings (GtkWidget *widget, gpointer user_data);
static void gpcc_conduits_popup (GtkCList *list, GdkEventButton *event, gpointer user_data);

static void gpcc_conduits_selected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data);
static void gpcc_conduits_unselected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data);

static void gpcc_cap_try (GtkWidget *widget, gpointer user_data);
static void gpcc_cap_revert (GtkWidget *widget, gpointer user_data);
static void gpcc_cap_ok (GtkWidget *widget, gpointer user_data);
static void gpcc_cap_cancel (GtkWidget *widget, gpointer user_data);

static void gpcc_destroy (GtkObject *object);

GtkType
gnome_pilot_conduitcap_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "GnomePilotConduitCap",
        sizeof (GnomePilotConduitCap),
        sizeof (GnomePilotConduitCapClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (gtk_vbox_get_type (), &info);
    }

  return type;
}

static void
class_init (GnomePilotConduitCapClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	parent_class = gtk_type_class (gtk_vbox_get_type ());

	object_class->destroy = gpcc_destroy;
}

static void
init (GnomePilotConduitCap *gpcc)
{
	GnomePilotConduitCapPrivate *priv;
	GtkWidget *window;
	
	priv = g_new0 (GnomePilotConduitCapPrivate, 1);

	gpcc->priv = priv;

	/* State information */
	loadPilotState (&priv->state);
	
	/* Gui stuff */
	priv->xml = glade_xml_new (GLADEDATADIR "/gpilotd-capplet.glade", "Conduits");
	if (!priv->xml) {
		g_message ("gnome-pilot-conduitcap init(): Could not load the Glade XML file!");
		goto error;
	}

	if (!get_widgets (gpcc)) {
		g_message ("gnome-pilot-conduitcap init(): Could not find all widgets in the XML file!");
		goto error;
	}

	/* Ugly bits to suck the widget out */
	window = glade_xml_get_widget (priv->xml, "Conduits");
	gtk_object_ref (GTK_OBJECT (priv->vbox));
	gtk_container_remove (GTK_CONTAINER (window), priv->vbox);
	gtk_container_add (GTK_CONTAINER (gpcc), priv->vbox);
	
	map_widgets (gpcc);
	fill_widgets (gpcc);
	init_widgets (gpcc);

 error:
}



GtkWidget *
gnome_pilot_conduitcap_new (GnomePilotClient *gpc)
{
	GnomePilotConduitCap *gpcc;
	GtkWidget *widget;
	
	widget = gtk_type_new (GNOME_PILOT_TYPE_CONDUITCAP);
	
	gpcc = GNOME_PILOT_CONDUITCAP (widget);
	gpcc->priv->gpc = gpc;
	
	return widget;
}

static gboolean
get_widgets (GnomePilotConduitCap *gpcc)
{
	GnomePilotConduitCapPrivate *priv;

	priv = gpcc->priv;

#define GW(name) glade_xml_get_widget (priv->xml, name)

	priv->vbox = GW ("ConduitsMain");

	priv->pilots_menu = GW ("pilots_menu");
	priv->pilots_username = GW ("username_label");
	
	priv->conduit_clist = GW ("conduit_list");
	priv->conduit_enable = GW ("conduit_enable_button");
	priv->conduit_disable = GW ("conduit_disable_button");
	priv->conduit_settings = GW ("conduit_settings_button");

	priv->conduit_description = GW ("description_label");

#undef GW
	return (priv->vbox
		&& priv->pilots_menu
		&& priv->conduit_clist
		&& priv->conduit_enable
		&& priv->conduit_disable
		&& priv->conduit_settings
		&& priv->conduit_description);
}

static void
map_widgets (GnomePilotConduitCap *gpcc)
{
#if 0
	GnomePilotConduitCapPrivate *priv;
	
	priv = gpcc->priv;
#endif
}

static void 
init_widgets (GnomePilotConduitCap *gpcc)
{
	GnomePilotConduitCapPrivate *priv;

	priv = gpcc->priv;

	/* Button signals */
	gtk_signal_connect (GTK_OBJECT (priv->conduit_enable), "clicked",
			    GTK_SIGNAL_FUNC (gpcc_conduits_enable), gpcc);

	gtk_signal_connect (GTK_OBJECT (priv->conduit_disable), "clicked",
			    GTK_SIGNAL_FUNC (gpcc_conduits_disable), gpcc);

	gtk_signal_connect (GTK_OBJECT (priv->conduit_settings), "clicked",
			    GTK_SIGNAL_FUNC (gpcc_conduits_settings), gpcc);

	/* Row selection signals */
	gtk_signal_connect (GTK_OBJECT (priv->conduit_clist), "select-row",
			    GTK_SIGNAL_FUNC (gpcc_conduits_selected), gpcc);
	gtk_signal_connect (GTK_OBJECT (priv->conduit_clist), "unselect-row",
			    GTK_SIGNAL_FUNC (gpcc_conduits_unselected), gpcc);

	/* Popup menu signals */
	gtk_signal_connect (GTK_OBJECT (priv->conduit_clist), "button_press_event",
			    GTK_SIGNAL_FUNC (gpcc_conduits_popup), gpcc);
}

static gint
append_conduit_clist (GnomePilotConduitCap *gpcc, ConduitState *conduit_state)
{
	GnomePilotConduitCapPrivate *priv;
	GnomePixmap *icon_gpixmap = NULL;
	GdkPixmap *pmap, *mask;
	const gchar *row_text[3];
	gint row = 0;
	
	priv = gpcc->priv;

	row_text[0] = conduit_state->name;
	row_text[1] = display_sync_type_name (conduit_state->enabled,conduit_state->sync_type);
	row = gtk_clist_append (GTK_CLIST (priv->conduit_clist), (gchar **)row_text);

	if (conduit_state->icon) {
		icon_gpixmap = (GnomePixmap *)gnome_stock_pixmap_widget_at_size (NULL,
										 conduit_state->icon,
										 SMALL_ICON_SIZE,
										 SMALL_ICON_SIZE);
	}
	if (icon_gpixmap) {
		pmap = icon_gpixmap->pixmap;
		mask   = icon_gpixmap->mask;
		gtk_clist_set_pixtext (GTK_CLIST (priv->conduit_clist), row, 0, conduit_state->name, 3, pmap, mask);
	}

	gtk_clist_set_row_data (GTK_CLIST (priv->conduit_clist), row, conduit_state);

	return row;
}

static void
fill_pilots_menu (GnomePilotConduitCap *gpcc) 
{
	GnomePilotConduitCapPrivate *priv;
	GtkWidget *menu, *menu_item;
	GList *tmp;
	
	priv = gpcc->priv;
	
	menu = gtk_menu_new ();
	
	tmp = priv->state->pilots;
	while (tmp != NULL) {
		menu_item = gtk_menu_item_new_with_label (((GPilotPilot*)tmp->data)->name);
		gtk_widget_show (menu_item);
		gtk_object_set_data (GTK_OBJECT (menu_item), "pilot", tmp->data);
		gtk_signal_connect (GTK_OBJECT(menu_item),"activate",
				    GTK_SIGNAL_FUNC (gpcc_conduits_choose_pilot),
				    gpcc);
		gtk_menu_append (GTK_MENU (menu), menu_item);

		tmp = tmp->next;
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (priv->pilots_menu), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (priv->pilots_menu), 0);

	/* Get the activate signal to fire */
	menu_item = gtk_menu_get_active (GTK_MENU (menu));
	gtk_menu_item_activate (GTK_MENU_ITEM (menu_item));

}

static void
fill_conduit_clist (GnomePilotConduitCap *gpcc)
{
	GnomePilotConduitCapPrivate *priv;
	ConduitState *conduit_state;
	GList *conduit_states;

	priv = gpcc->priv;

	if (priv->pilot == NULL)
		return;
	
	if (priv->conduits == NULL )
		priv->conduits = g_hash_table_new (g_int_hash, g_int_equal);
	
	conduit_states = g_hash_table_lookup (priv->conduits, &priv->pilot->pilot_id);
	if (conduit_states == NULL) {		
		conduit_states = load_conduit_list (priv->pilot);
		g_hash_table_insert (priv->conduits, &priv->pilot->pilot_id, conduit_states);
	}
	
	gtk_clist_clear (GTK_CLIST (priv->conduit_clist));
	while (conduit_states != NULL) {
		conduit_state = (ConduitState*)conduit_states->data;

		append_conduit_clist (gpcc, conduit_state);
		conduit_states = conduit_states->next;
	}
}

static void
fill_widgets (GnomePilotConduitCap *gpcc)
{
	GnomePilotConduitCapPrivate *priv;

	priv = gpcc->priv;

	fill_pilots_menu (gpcc);
}

CappletWidget *
gnome_pilot_conduitcap_get_parent (GnomePilotConduitCap *gpcc) 
{
	GnomePilotConduitCapPrivate *priv;

	priv = gpcc->priv;

	return priv->parent;
}

void
gnome_pilot_conduitcap_set_parent (GnomePilotConduitCap *gpcc, CappletWidget *cap)
{
	GnomePilotConduitCapPrivate *priv;

	priv = gpcc->priv;
	
	priv->parent = cap;

	gtk_signal_connect (GTK_OBJECT (priv->parent), "try",
			   GTK_SIGNAL_FUNC (gpcc_cap_try), gpcc);
	gtk_signal_connect (GTK_OBJECT (priv->parent), "revert",
			   GTK_SIGNAL_FUNC (gpcc_cap_revert), gpcc);
	gtk_signal_connect (GTK_OBJECT (priv->parent), "ok",
			   GTK_SIGNAL_FUNC (gpcc_cap_ok), gpcc);
	gtk_signal_connect (GTK_OBJECT (priv->parent), "cancel",
			   GTK_SIGNAL_FUNC (gpcc_cap_cancel), gpcc);

#if 0
	gtk_signal_connect (GTK_OBJECT (capplet), "help",
			   GTK_SIGNAL_FUNC (gnome_help_display), 
			   &help_entry);
#endif
}

void
gnome_pilot_conduitcap_update (GnomePilotConduitCap *gpcc) 
{
	GnomePilotConduitCapPrivate *priv;
	
	priv = gpcc->priv;
	
	freePilotState (priv->state);
	loadPilotState (&priv->state);

	fill_widgets (gpcc);
}

static void
gpcc_set_changed_state (GnomePilotConduitCap *gpcc)
{
	GnomePilotConduitCapPrivate *priv;

	priv = gpcc->priv;
	
	if (priv->parent == NULL)
		return;
	
	capplet_widget_state_changed (CAPPLET_WIDGET (priv->parent), TRUE);
	
}


static ConduitState *
get_state (GnomePilotConduitCap *gpcc, int row)
{
	GnomePilotConduitCapPrivate *priv;
	
	priv = gpcc->priv;

	return (ConduitState*)gtk_clist_get_row_data (GTK_CLIST (priv->conduit_clist), row);
}


static ConduitState *
get_current_state (GnomePilotConduitCap *gpcc, int *row)
{
	GnomePilotConduitCapPrivate *priv;
	gint curr_row;
	
	priv = gpcc->priv;

	curr_row = GPOINTER_TO_INT (GTK_CLIST (priv->conduit_clist)->selection->data);
	if (row)
		*row = curr_row;

	return get_state (gpcc, curr_row);
}

static void
check_conduits_buttons (GnomePilotConduitCap *gpcc) 
{
	GnomePilotConduitCapPrivate *priv;
	ConduitState *state;
	gboolean test, enabled = FALSE;
	
	priv = gpcc->priv;

	test = check_selection (GTK_CLIST (priv->conduit_clist));
	if (test) {
		state = get_current_state (gpcc, NULL);
		enabled = state->enabled;
	}
	
	gtk_widget_set_sensitive (GTK_WIDGET (priv->conduit_enable), test && !enabled);
	gtk_widget_set_sensitive (GTK_WIDGET (priv->conduit_disable), test && enabled);
	gtk_widget_set_sensitive (GTK_WIDGET (priv->conduit_settings), test && enabled);
}

static void
gpcc_conduits_choose_pilot (GtkWidget *widget, gpointer user_data)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (user_data);
	GnomePilotConduitCapPrivate *priv;
	
	priv = gpcc->priv;

	priv->pilot = gtk_object_get_data (GTK_OBJECT (widget), "pilot");

	gtk_label_set_text (GTK_LABEL (priv->pilots_username), priv->pilot->pilot_username);

	fill_conduit_clist (gpcc);
	check_conduits_buttons (gpcc);
}

static void 
gpcc_conduits_enable (GtkWidget *widget, gpointer user_data)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (user_data);
	GnomePilotConduitCapPrivate *priv;
	ConduitState *state;
	gint row;
	
	priv = gpcc->priv;

	state = get_current_state (gpcc, &row);
	
	if (state->default_sync_type == GnomePilotConduitSyncTypeNotSet) {
		/* nothing? */
	} else if (state->default_sync_type == GnomePilotConduitSyncTypeCustom) {
		state->changed=TRUE;
		state->enabled=TRUE;
		state->sync_type = GnomePilotConduitSyncTypeCustom;
	} else {
		state->changed = TRUE;
		state->enabled = TRUE;
		state->sync_type = state->default_sync_type;
	}

	gtk_clist_set_text (GTK_CLIST (priv->conduit_clist), row, 1,
			    display_sync_type_name (state->enabled, state->sync_type));

	check_conduits_buttons (gpcc);
	gpcc_set_changed_state (gpcc);
	gpcc_conduits_settings (widget, user_data);
}

static void 
gpcc_conduits_disable (GtkWidget *widget, gpointer user_data)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (user_data);
	GnomePilotConduitCapPrivate *priv;
	ConduitState *state;
	gint row;
	
	priv = gpcc->priv;
	
	state = get_current_state (gpcc, &row);

	state->enabled=FALSE;
	state->changed=TRUE;

	gtk_clist_set_text (GTK_CLIST (priv->conduit_clist), row, 1,
			    display_sync_type_name (FALSE, GnomePilotConduitSyncTypeNotSet));

	check_conduits_buttons (gpcc);
	gpcc_set_changed_state (gpcc);
}

static void 
gpcc_conduits_settings (GtkWidget *widget, gpointer user_data)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (user_data);
	GnomePilotConduitCapPrivate *priv;
	ConduitState *state;
	gint row;
	
	priv = gpcc->priv;

	state = get_current_state (gpcc, &row);
	
	if (state->conduit == NULL) {
		/* must allocate conduit */
		if (gnome_pilot_conduit_management_instantiate_conduit (state->management,
									state->pilot, 
									&state->conduit)
		    != GNOME_PILOT_CONDUIT_MGMT_OK) {
			
			gchar *msg = _("Unable to instantiate %s conduit.");
			error_dialog (msg, state->name);
			return;
		}

		state->settings_widget2 = gnome_pilot_cdialog_new (state);
	}
	
	if (state->settings_widget2 != NULL) {
		if (gnome_pilot_cdialog_run_and_close (GNOME_PILOT_CDIALOG (state->settings_widget2))) {
			/* pressed ok */
			state->sync_type = gnome_pilot_cdialog_sync_type (GNOME_PILOT_CDIALOG (state->settings_widget2));
			state->enabled = (state->sync_type != GnomePilotConduitSyncTypeNotSet);
			state->changed=TRUE;
			
			gtk_clist_set_text (GTK_CLIST (priv->conduit_clist), row, 1,
					    display_sync_type_name (state->enabled, state->sync_type));
			
			gpcc_set_changed_state (gpcc);
		} else {
			/* pressed cancel */
			gnome_pilot_conduit_display_settings (state->conduit);
		}
	}
	
	check_conduits_buttons (gpcc);
}

static void
gpcc_conduits_popup (GtkCList *list, GdkEventButton *event, gpointer user_data)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (user_data);
	GnomePilotConduitCapPrivate *priv;
	ConduitState *state;
	gint row;
	gboolean test = FALSE, enabled = FALSE;
	
	GnomeUIInfo popup [] = {
		GNOMEUIINFO_ITEM_DATA (N_("_Enable this conduit"),
				       N_("Disable the currently selected conduit"),
				       gpcc_conduits_enable, gpcc, NULL),
		GNOMEUIINFO_ITEM_DATA (N_("_Disable this conduit"),
				       N_("Enable the currently selected conduit"),
				       gpcc_conduits_disable, gpcc, NULL),

		GNOMEUIINFO_SEPARATOR,

		GNOMEUIINFO_ITEM_DATA (N_("_Settings..."),
				       N_("Modify the currently selected conduit's settings"),
				       gpcc_conduits_settings, gpcc, NULL),
		
		GNOMEUIINFO_END
	};
 
	priv = gpcc->priv;

	if (priv->conduit_popup == NULL) {
		priv->conduit_popup = gtk_menu_new ();
		gnome_app_fill_menu (GTK_MENU_SHELL (priv->conduit_popup), popup, NULL, TRUE, 0);

		priv->conduit_popup_enable = popup[0].widget;
		priv->conduit_popup_disable = popup[1].widget;
		priv->conduit_popup_settings = popup[3].widget;
		
	}
	
	if (gtk_clist_get_selection_info (list, event->x, event->y, &row, NULL)) {
		state = get_state (gpcc, row);
		test = TRUE;
		enabled = state->enabled;
	}

	gtk_widget_set_sensitive (priv->conduit_popup_enable, test && !enabled);
	gtk_widget_set_sensitive (priv->conduit_popup_disable, test && enabled);
	gtk_widget_set_sensitive (priv->conduit_popup_settings, test && enabled);

	show_popup_menu (list, event, GTK_MENU (priv->conduit_popup));
}

static void
gpcc_conduits_selected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (user_data);
	GnomePilotConduitCapPrivate *priv;
	ConduitState *state;
	
	priv = gpcc->priv;
	
	state = get_current_state (gpcc, NULL);
	
	gtk_label_set_text (GTK_LABEL (priv->conduit_description), state->description);
	check_conduits_buttons (gpcc);
}

static void
gpcc_conduits_unselected (GtkCList *clist, gint row, gint col, GdkEventButton *event, gpointer user_data)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (user_data);
	
	check_conduits_buttons (gpcc);
}

static void 
conduit_try_foreach (gpointer key, gpointer value, gpointer func)
{
	GList *states = (GList*)value;
	
	while (states != NULL) {
		ConduitState *state = (ConduitState*)states->data;
		
		if (state->changed && state->enabled) {
			if (state->sync_type == GnomePilotConduitSyncTypeCustom)
				gnome_pilot_conduit_config_enable (state->config, GnomePilotConduitSyncTypeCustom);
			else
				gnome_pilot_conduit_config_enable_with_first_sync (state->config,
										   state->sync_type,
										   state->sync_type,
										   TRUE);
		}
		
		if (state->changed && !state->enabled)
			gnome_pilot_conduit_config_disable (state->config);
		
		if (state->conduit && state->changed)
			gnome_pilot_conduit_save_settings (state->conduit);
		
		states = states->next;
	}
}

static void 
gpcc_cap_try (GtkWidget *widget, gpointer user_data)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (user_data);
	GnomePilotConduitCapPrivate *priv;
	
	priv = gpcc->priv;
	
	g_hash_table_foreach (priv->conduits, conduit_try_foreach, NULL);
}

static void
conduit_revert_foreach (gpointer key, gpointer value, gpointer func)
{
	GList *states = (GList*)value;
	
	while (states != NULL) {
		ConduitState *state = (ConduitState*)states->data;
		
		if (state->changed && state->orig_enabled) {
			if (state->orig_sync_type == GnomePilotConduitSyncTypeCustom)
				gnome_pilot_conduit_config_enable (state->config, GnomePilotConduitSyncTypeCustom);
			else
				 gnome_pilot_conduit_config_enable (state->config, state->orig_sync_type);
			
			state->enabled = state->orig_enabled;
			state->sync_type = state->orig_sync_type;
		}
		
		if (state->changed && !state->orig_enabled) {
			gnome_pilot_conduit_config_disable (state->config);
			
			state->enabled = state->orig_enabled;
		}
		
		if (state->conduit && state->changed)
			gnome_pilot_conduit_revert_settings (state->conduit);
		
		states = states->next;
	}
}

static void
gpcc_cap_revert (GtkWidget *widget, gpointer user_data)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (user_data);
	GnomePilotConduitCapPrivate *priv;
	
	priv = gpcc->priv;
	
	g_hash_table_foreach (priv->conduits, conduit_revert_foreach, gpcc);

	fill_conduit_clist (gpcc);
}

static void 
gpcc_cap_ok (GtkWidget *widget, gpointer user_data)
{
	gpcc_cap_try (widget, user_data);
}

static void 
gpcc_cap_cancel (GtkWidget *widget, gpointer user_data)
{
	gpcc_cap_revert (widget, user_data);
}

static void
gpcc_destroy (GtkObject *object)
{
	GnomePilotConduitCap *gpcc = GNOME_PILOT_CONDUITCAP (object);
	GnomePilotConduitCapPrivate *priv;
	
	priv = gpcc->priv;

	if (priv->conduit_popup)
		gtk_widget_destroy (priv->conduit_popup);
	
	gtk_object_unref (GTK_OBJECT (priv->xml));

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}
